﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Edit-BusinessUnitName
// This is a Custom Cmdlet that will allow you to change the name of the business unit specified.
// If you supply the guid of the business unit id only that name will be changed
// If you supply only the name of the business unit and multiple business units exist with the same name, all will be changed.
//
// Required Parameters
// -ChangeTo
//
// Optional Parameters (Only one of these needs to be supplied for it to find a record. The Cmdlet will not produce an error
//                      if you do not supply either one of them however it will also not be able to make any changes)
// -BusinessUnitId
// -ChangeFrom
//
// **********************************************************************************************************
using System;
using System.IO;
using System.Collections.Generic;
using System.Management.Automation;
using System.Web.Services.Protocols;
using System.ServiceModel;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.Xrm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsSecurity.Grant, "CRMRoleToTeam", SupportsShouldProcess = true)]

    public class GrantRoleToTeam : CrmServiceCommand
    {

        private Entity Record { get; set; }

        [Parameter(Mandatory = true)]
        public String TeamName { get; set; }

        [Parameter(Mandatory = false)]
        public Guid TeamId { get; set; }

        [Parameter(Mandatory = true)]
        public String RoleName { get; set; }

        [Parameter(Mandatory = false)]
        public Guid RoleId { get; set; }

        [Parameter(Mandatory = false)]
        public String ParentBUId { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                IOrganizationService os = _service;
                Entity TeamEntity;
                Entity RoleEntity;
                Boolean Result = false;

                Console.WriteLine("Checking to see if the Role exists. If it exists it will return the Id.");

                // Get the Role Record from the Id if it was supplied. It must be from the same business unit of the team you are about to associate it to.
                if (RoleId != null && RoleId.ToString() != Guid.Empty.ToString())
                {
                    RoleEntity = GetRoleRecordById(RoleId);

                    RoleName = RoleEntity.Attributes["name"].ToString();

                    Console.WriteLine("Valid Role Id: " + RoleId.ToString().ToUpper() + " - " + RoleName);
                }
                else
                {
                    if (String.IsNullOrEmpty(RoleName) != true)
                    {
                        RoleEntity = GetRoleRecordByName(RoleName, ParentBUId);
                        if(RoleEntity != null && RoleEntity.Id != null && RoleEntity.Id.ToString() != Guid.Empty.ToString())
                        {
                            RoleId = RoleEntity.Id;

                            Console.WriteLine("Valid Role Id: " + RoleId.ToString().ToUpper() + " - " + RoleName);
                        }
                    }
                    else
                    {
                        Console.WriteLine("You Must supply at a minimum, a valid role name.");
                        return;
                    }
                }

                // Get the Role Record from the Id if it was supplied
                if (TeamId != null && TeamId.ToString() != Guid.Empty.ToString())
                {
                    TeamEntity = GetTeamRecordById(TeamId);

                    TeamName = TeamEntity.Attributes["name"].ToString();

                    Console.WriteLine("Valid Team Id: " + TeamId.ToString().ToUpper() + " - " + TeamName);
                }
                else
                {
                    if (String.IsNullOrEmpty(TeamName) != true)
                    {
                        TeamEntity = GetTeamRecordByName(TeamName);
                        if (TeamEntity != null && TeamEntity.Id != null && TeamEntity.Id.ToString() != Guid.Empty.ToString())
                        {
                            TeamId = TeamEntity.Id;

                            Console.WriteLine("Valid Team Id: " + TeamId.ToString().ToUpper() + " - " + TeamName);
                        }
                    }
                    else
                    {
                        Console.WriteLine("You Must supply at a minimum, a valid Team name.");
                        return;
                    }
                }

                Result = AssignRoleToTeam(_service, RoleEntity, TeamEntity);

                if (!Result)
                {
                    Console.WriteLine("*** Error: Role " + RoleName.ToString() + " was not associated to Team " + TeamName + " due to an unreported error.");
                
                }
                else
                {
                    Console.WriteLine("Role " + RoleName.ToString() + " was associated to Team " + TeamName);
                }
                Console.WriteLine("Assign Role to Team - Processing Completed...");
            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }

        public bool AssignRoleToTeam(IOrganizationService _service, Entity Role, Entity Team)
        {
            try
            {
                // Add the role to the team.
                _service.Associate(
                       "team",
                       Team.Id,
                       new Relationship("teamroles_association"),
                       new EntityReferenceCollection() { new EntityReference("role", Role.Id) });

                return true;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }
            return false;
        }

        public Entity GetTeamRecordByName(String TeamName)
        {
            try
            {
                //Console.WriteLine("Searching for Team Record " + TeamName.ToString());

                List<Entity> EntityRecordIDList = new List<Entity>();

                //Create a query expression specifying the link entity alias and the columns of the link entity that you want to return
                QueryExpression qe = new QueryExpression()
                {
                    EntityName = "team",
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                RetrieveMultipleRequest request = new RetrieveMultipleRequest()
                {
                    Query = qe
                };

                RetrieveMultipleResponse ec = (RetrieveMultipleResponse)Service.Execute(request);

                foreach (Entity ent in ec.EntityCollection.Entities)
                {
                    if (ent.Contains("name") && ent["name"].ToString().Equals(TeamName))
                    {
                        EntityRecordIDList.Add(ent);
                    }
                }

                if (EntityRecordIDList.Count != 1)
                {
                    return new Entity();
                }
                else
                {
                    return EntityRecordIDList[0];
                }

            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }


        public Entity GetTeamRecordById(Guid TeamId)
        {
            try
            {
                Entity ec = Service.Retrieve("team", TeamId, new ColumnSet(true));

                if (ec != null && ec.Id.ToString() != Guid.Empty.ToString())
                {
                    return ec;
                }
                else
                {
                    return new Entity();
                }
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }


        public Entity GetRoleRecordByName(String RoleName, String BusinessUnitId)
        {
            try
            {
                //Console.WriteLine("Searching for Role Record " + RoleName.ToString());

                List<Entity> EntityRecordIDList = new List<Entity>();

                //Create a query expression specifying the link entity alias and the columns of the link entity that you want to return
                QueryExpression qe = new QueryExpression()
                {
                    EntityName = "role",
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                RetrieveMultipleRequest request = new RetrieveMultipleRequest()
                {
                    Query = qe
                };

                RetrieveMultipleResponse ec = (RetrieveMultipleResponse)Service.Execute(request);

                foreach(Entity ent in ec.EntityCollection.Entities)
                {
                    if (ent.Contains("name") && ent.Contains("businessunitid") &&
                        ent.Attributes["name"].ToString().Equals(RoleName) && ((EntityReference)ent.Attributes["businessunitid"]).Id.ToString().ToUpper().Equals(BusinessUnitId))
                    {
                        EntityRecordIDList.Add(ent);
                    }
                }

                if(EntityRecordIDList.Count != 1)
                {
                    return new Entity();
                }
                else
                {
                    return EntityRecordIDList[0];
                }

            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }


        public Entity GetRoleRecordById(Guid RoleId)
        {
            try
            {
                Entity ec = Service.Retrieve("role", RoleId, new ColumnSet(true));

                if (ec != null && ec.Id.ToString() != Guid.Empty.ToString())
                {
                    return ec;
                }
                else
                {
                    return new Entity();
                }
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }
    }
}
