﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Export-Customizations
// This is a Custom Cmdlet that will allow you to Export the CRM Customizations for the organization identified by the OrganizationUri supplied
// Required Parameters
// There are not any required parameters over and above the Base Class required parameters.
// If you do not supply a filename the filename "default.zip" will be used
// If you do not supply a Solution Folder the current folder will be used to save the file to
// If you do not supply a Solution name the default solution will be used and all objects will be included.
//
// The export wizard has a list of special settings to be either included or excluded. They will be excluded unless they are specifically included as a parameter.
// Optional Parameters - Only use the parameter and nothing else to indicate that the specific item should be exported as well.
// -IncludeAutoNumberingSettings
// -IncludeCalendarSettings
// -IncludeCustomizationSettings
// -IncludeEmailTrackingSettings
// -IncludeGeneralSettings
// -IncludeIsvConfig
// -IncludeMarketingSettings
// -IncludeOutlookSynchronizationSettings
// -IncludeRelationshipRoles
// **********************************************************************************************************
using System;
using System.IO;
using System.Management.Automation;
using Microsoft.Crm.Sdk.Messages;
using System.Web.Services.Protocols;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet("Export", "Customizations", SupportsShouldProcess = true)]

    public class ExportCRMCustomizations : CrmServiceCommand
    {
        [Parameter(Mandatory = false)]
        public string SolutionFolder { get; set; }

        [Parameter(Mandatory = false)]
        public string FileName { get; set; }

        [Parameter(Mandatory = false)]
        public string Solution { get; set; }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeAutoNumberingSettings {
            get
            {
                return AutoNumberingSettings;
            }
            set
            {
                AutoNumberingSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeCalendarSettings
        {
            get
            {
                return CalendarSettings;
            }
            set
            {
                CalendarSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeCustomizationSettings
        {
            get
            {
                return CustomizationSettings;
            }
            set
            {
                CustomizationSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeEmailTrackingSettings
        {
            get
            {
                return EmailTrackingSettings;
            }
            set
            {
                EmailTrackingSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeGeneralSettings
        {
            get
            {
                return GeneralSettings;
            }
            set
            {
                GeneralSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeIsvConfig
        {
            get
            {
                return IsvConfig;
            }
            set
            {
                IsvConfig = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeMarketingSettings
        {
            get
            {
                return MarketingSettings;
            }
            set
            {
                MarketingSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeOutlookSynchSettings
        {
            get
            {
                return OutlookSynchSettings;
            }
            set
            {
                OutlookSynchSettings = value;
            }
        }

        [Parameter(Mandatory = false)]
        public SwitchParameter IncludeRelationshipRoles
        {
            get
            {
                return RelationshipRoles;
            }
            set
            {
                RelationshipRoles = value;
            }
        }

        private bool AutoNumberingSettings { get; set; }
        private bool CalendarSettings { get; set; }
        private bool CustomizationSettings { get; set; }
        private bool EmailTrackingSettings { get; set; }
        private bool GeneralSettings { get; set; }
        private bool IsvConfig { get; set; }
        private bool MarketingSettings { get; set; }
        private bool OutlookSynchSettings { get; set; }
        private bool RelationshipRoles { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                string SolutionToExport = @"default";

                if (Solution != null && Solution.Length != 0)
                    SolutionToExport = Solution;

                WriteVerbose(string.Concat(@"Exporting CRM customizations using the ", SolutionToExport.ToString(), @" solution..."));
                //var response = (ExportAllXmlResponse)Service.Execute(new ExportAllXmlRequest());
                //WriteObject(response.ExportXml);               
                var exportSolutionRequest = new ExportSolutionRequest
                {
                    Managed = false,
                    SolutionName = SolutionToExport,
                    ExportAutoNumberingSettings = AutoNumberingSettings,
                    ExportCalendarSettings = CalendarSettings,
                    ExportCustomizationSettings = CustomizationSettings,
                    ExportEmailTrackingSettings = EmailTrackingSettings,
                    ExportGeneralSettings = GeneralSettings,
                    ExportIsvConfig = IsvConfig,
                    ExportMarketingSettings = MarketingSettings,
                    ExportOutlookSynchronizationSettings = OutlookSynchSettings,
                    ExportRelationshipRoles = RelationshipRoles
                };

                var exportSolutionResponse = (ExportSolutionResponse)Service.Execute(exportSolutionRequest);
                byte[] exportXml = exportSolutionResponse.ExportSolutionFile;

                if (FileName == null || FileName.Length == 0)
                {
                    FileName = @"default.zip";
                }

                // If the solution directory is null or  blank use the current directory
                if (SolutionFolder == null || SolutionFolder.Length == 0)
                {
                    SolutionFolder = Directory.GetCurrentDirectory().ToString();
                }

                // Make sure that the directory entry ends with a backslash character so that the resulting url is valid
                if (!(SolutionFolder.EndsWith(@"\\")))
                {
                    SolutionFolder = String.Concat(SolutionFolder.ToString(), (@"\\"));
                }

                File.WriteAllBytes(SolutionFolder + FileName, exportXml);

            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }

    }
}
