﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// This is main class that drives this dll. All other classes are extensions of this class
//
// Implements the IOrganizationService and handles the connectivity to CRM via the Web Services.
//
// Derive additional classes from this class:
//      public class MyClass : CrmServiceCommand
// Then override the process method to create additional custom Cmdlets to this DLL
//
// Required Parameters - If these are not supplied and it requires them the user will be prompted for the parameter.
// -OrganizationUri the full http or https uri to the Organization Service of the CRM deployment and org you will be executing against.
//
// Optional Parameters
// -Username
// -Password
// -Domain
// 
// **********************************************************************************************************
using System;
using System.Management.Automation;
using System.Net;
using System.Web.Services.Protocols;
using System.ServiceModel.Description;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;


namespace Microsoft.MCS.Commandlets
{
    public class CrmServiceCommand : PSCmdlet
    {
        public IOrganizationService _service;

        [Parameter(Mandatory = true, HelpMessage = "Provide the full URI to the Organization Service endpoint for the Organization you intend this CmdLet to execute against.")]
        public string OrganizationUri { get; set; }

        [Parameter(Mandatory = false, HelpMessage = "If you are not logged in as a user with permissions to CRM you will need to provide a Username/Password")]
        public string Username { get; set; }

        [Parameter(Mandatory = false, HelpMessage = "If you supply a username you will need to supply a password")]
        public string Password { get; set; }

        [Parameter(Mandatory = false, HelpMessage = "The domain name that the user login account belongs to.")]
        public string Domain { get; set; }

        protected static string GetErrorMessageFromSoapException(SoapException exception)
        {
            if (exception.Detail == null)
                return String.Empty;

            return exception.Detail.InnerText;
        }

        protected IOrganizationService Service
        {
            get
            {
                if (_service == null)
                {
                    var credentials = new ClientCredentials();
                    if (!(Domain == null || Domain == ""))
                    {
                        credentials.Windows.ClientCredential = new System.Net.NetworkCredential(Username, Password, Domain);
                    }
                    else
                    {
                        credentials.Windows.ClientCredential = CredentialCache.DefaultNetworkCredentials;
                    }

                    var orgUri = new Uri(OrganizationUri);
                    var serviceProxy = new OrganizationServiceProxy(orgUri, null, credentials, null);
                    serviceProxy.ServiceConfiguration.CurrentServiceEndpoint.Behaviors.Add(new ProxyTypesBehavior());
                    var interval = new TimeSpan(7000000000);
                    serviceProxy.Timeout = interval;
                    _service = serviceProxy;
                }

                return _service;
            }
        }
    }
}


