﻿using System;
using System.ServiceModel;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Btsss.Plugins.VIMT.Appointments.Messages;

namespace Btsss.Plugins.VIMT.Appointments
{

    /// <summary>
    /// PluginEntryPoint plug-in.
    /// This is a generic entry point for a plug-in class. Use the Plug-in Registration tool found in the CRM SDK to register this class, import the assembly into CRM, and then create step associations.
    /// A given plug-in can have any number of steps associated with it. 
    /// </summary>    
    public class RetrievePostAppointmentRunner : AppointmentBase
    {
        private IOrganizationService _service { get; set; }

        private Uri vimtEndPoint { get; set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="RetrievePostAppointmentRunner"/> class.
        /// </summary>
        /// <param name="unsecure">Contains public (unsecured) configuration information.</param>
        /// <param name="secure">Contains non-public (secured) configuration information. 
        /// When using Microsoft Dynamics CRM for Outlook with Offline Access, 
        /// the secure string is not passed to a plug-in that executes while the client is offline.</param>
        public RetrievePostAppointmentRunner(string unsecure, string secure)
            : base(typeof(RetrievePostAppointmentRunner))
        {

            // TODO: Implement your custom configuration handling.
        }


        /// <summary>
        /// Main entry point for he business logic that the plug-in is to execute.
        /// </summary>
        /// <param name="localContext">The <see cref="LocalPluginContext"/> which contains the
        /// <see cref="IPluginExecutionContext"/>,
        /// <see cref="IOrganizationService"/>
        /// and <see cref="ITracingService"/>
        /// </param>
        /// <remarks>
        /// For improved performance, Microsoft Dynamics CRM caches plug-in instances.
        /// The plug-in's Execute method should be written to be stateless as the constructor
        /// is not called for every invocation of the plug-in. Also, multiple system threads
        /// could execute the plug-in at the same time. All per invocation state information
        /// is stored in the context. This means that you should not use global variables in plug-ins.
        /// </remarks>
        protected override void ExecuteCrmPlugin(LocalPluginContext localContext)
        {
            // Implement your custom plug-in business logic.
            try
            {
                Entity apptRecord = null;
                if (localContext == null || localContext.PluginExecutionContext.Depth == 1 && localContext.PluginExecutionContext.OutputParameters.ContainsKey("BusinessEntity"))
                {
                    _service = localContext.OrganizationService;

                    localContext.Trace("Starting Retrieve Plugin Runner");
                    var businessEntity = localContext.PluginExecutionContext.OutputParameters["BusinessEntity"] as Entity;

                    localContext.Trace("Check Business Entity to see if it is null");
                    // get the guid and identify what fields are in the property bag for the record that was sent in 
                    // Respective of the appointment identifier field.
                    if (businessEntity != null && businessEntity.Id != null && !string.IsNullOrEmpty(businessEntity.LogicalName) && businessEntity.LogicalName == "btsss_appointment")
                    {
                        apptRecord = (Entity)_service.Retrieve("btsss_appointment", businessEntity.Id, new ColumnSet("btsss_appointmentid", "btsss_externalappointmentkey", "btsss_externalsystemkey", "btsss_contactid"));

                    }
                    else
                    {
                        return;
                    }

                    if (apptRecord != null && apptRecord.Attributes.Contains("btsss_externalappointmentkey") && apptRecord.Attributes.Contains("btsss_externalsystemkey") &&
                        apptRecord.Attributes["btsss_externalappointmentkey"] != null && apptRecord.Attributes["btsss_externalsystemkey"] != null &&
                        apptRecord.Attributes.Contains("btsss_contactid") && apptRecord.Attributes["btsss_contactid"] != null)
                    {
                        Messages.LoadVISTAAppointmentDetailResponse response = new LoadVISTAAppointmentDetailResponse();
                        // use that information to get back to the originating system and get the information specific to that record.
                        switch (apptRecord.Attributes["btsss_externalsystemkey"].ToString())
                        {
                            case "VISTA":
                                // go to the VISTA VIMT adapters and get the record
                                response = GetVistaData(localContext, apptRecord);
                                break;
                            case "CaregiverSystem": // Does not exist or we do not have information on right now
                                break;
                            default:
                                // Since this is the only one for now go to the VISTA VIMT adapters to get the data
                                response = GetVistaData(localContext, apptRecord);
                                break;
                        }
                        if (response != null && !string.IsNullOrEmpty(response.AppointmentIdentifier) && !string.IsNullOrEmpty(response.OriginatingSystemIdentifier))
                        {
                            LoadAppointmentBU(ref businessEntity, response);
                            localContext.Trace("Processing completed in plugin. Context object loaded with record.");
                        }

                        localContext.Trace(businessEntity.LogicalName.ToString());

                    }
                    else
                    {
                        return;
                    }

                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                localContext.Trace(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
            catch (InvalidOperationException ex)
            {
                localContext.Trace(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
            catch (Exception ex)
            {
                localContext.Trace(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
        }

        private void LoadAppointmentBU(ref Entity businessEntity, LoadVISTAAppointmentDetailResponse response)
        {
            if (response.AppointmentDate != null)
                businessEntity.Attributes["btsss_appointmentscheduledate"] = response.AppointmentDate;
            if (response.AppointmentIdentifier != null)
                businessEntity.Attributes["btsss_externalappointmentkey"] = response.AppointmentIdentifier.ToString();
            if (response.AppointmentStatus != null)
                businessEntity.Attributes["btsss_completed"] = response.AppointmentStatus.ToString().Equals("Completed") ? new OptionSetValue(0) : new OptionSetValue(1);
            //if (response.FacilityIdentifier != null)
            //    businessEntity.Attributes["btsss_facilityidentifier"] = response.FacilityIdentifier.ToString();
            if (response.OriginatingSystemIdentifier != null)
                businessEntity.Attributes["btsss_externalsystemkey"] = "VISTA";
            //if (response.PatientIdentifier != null)
            //    businessEntity.Attributes["btsss_patientidentifier"] = response.PatientIdentifier.ToString();
            if (response.SOPCode != null)
                businessEntity.Attributes["btsss_name"] = response.SOPCode.ToString();

        }
        private Messages.LoadVISTAAppointmentDetailResponse GetVistaData(LocalPluginContext localContext, Entity apptRecord)
        {
            Messages.LoadVISTAAppointmentDetailResponse response = new Messages.LoadVISTAAppointmentDetailResponse();

            localContext.Trace("Starting Retrieve Plugin Runner");
            MCSUtilities2011.MCSSettings McsSettings = new MCSUtilities2011.MCSSettings();

            McsSettings.setService = localContext.OrganizationService;
            McsSettings.systemSetting = "Active Settings";

            vimtEndPoint = new Uri(McsSettings.GetSingleSetting("btsss_vimtendpoint", "string"));

            Messages.LoadVISTAAppointmentDetailRequest request = new Messages.LoadVISTAAppointmentDetailRequest()
            {
                OrganizationName = localContext.PluginExecutionContext.OrganizationName,
                UserId = localContext.PluginExecutionContext.UserId
            };

            if (apptRecord.Attributes["btsss_externalappointmentkey"] != null && apptRecord.Attributes["btsss_externalappointmentkey"].ToString() != string.Empty)
            {
                request.AppointmentIdentifier = apptRecord.Attributes["btsss_externalappointmentkey"].ToString();
            }
            else
            {
                return new LoadVISTAAppointmentDetailResponse();
            }

            Entity User = localContext.OrganizationService.Retrieve("systemuser", localContext.PluginExecutionContext.UserId, new ColumnSet(new string[] { "firstname", "lastname" }));

            if (User != null)
            {
                if (User.Attributes.Contains("firstname") && User.Attributes["firstname"] != null)
                {
                    request.UserFirstName = User.Attributes["firstname"].ToString();
                }
                if (User.Attributes.Contains("lastname") && User.Attributes["lastname"] != null)
                {
                    request.UserLastName = User.Attributes["lastname"].ToString();
                }
            }

            if (apptRecord.Attributes.Contains("btsss_contactid") && apptRecord.Attributes["btsss_contactid"] != null)
            {
                // We know there is a contact id or we would not be here... go get the ICN for the Vet from that contact record. If there is not one the bail
                Entity contEnt = _service.Retrieve("contact", ((EntityReference)apptRecord.Attributes["btsss_contactid"]).Id, new ColumnSet(true));
                if (contEnt != null && contEnt.Attributes.Contains("btsss_icn") && contEnt.Attributes["btsss_icn"] != null)
                {
                    request.PatientIdentifier = contEnt.Attributes["btsss_icn"].ToString();
                }
                else
                {
                    response.AppointmentIdentifier = request.AppointmentIdentifier;
                    response.AppointmentStatus = "No Veteran Identifier";
                    response.ExceptionOccured = false;
                    response.FacilityIdentifier = string.Empty;
                    response.IsInNetwork = false;
                    response.OriginatingSystemIdentifier = "VISTA";
                    response.PatientIdentifier = string.Empty;
                    response.SOPCode = string.Empty;
                    response.Message = "Missing Veteran Identifier: Contact record did not contain identifying veteran information";

                    return response;
                }

                request.MessageId = Guid.NewGuid();
                response = VRMRest.Utility.SendReceive<LoadVISTAAppointmentDetailResponse>(vimtEndPoint, MessageRegistry.LoadVISTAAppointmentDetailRequest, request, null);
            }
            else
            {
                // Do nothing the appointment is not associated with a contact. It must be associated with a contact in order to get the ICN necessary for retrieval
            }
            return response;
        }
    }
}
