﻿#region Directives

using MCS_AutoNumber.Common;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

#endregion Directives

namespace MCS_AutoNumber
{
    public class AutoNumber : IPlugin
    {
        #region Public Static Properties

        private static readonly Object SyncLock = new Object();

        #endregion Public Static Properties

        #region Properties
        
        public Guid? AutoNumberId 
        { 
            get; 
            set; 
        }
        
        public String Entity 
        { 
            get; 
            set; 
        }
        
        public String Field 
        { 
            get; 
            set; 
        }

        public String Format 
        { 
            get; 
            set; 
        }

        public Int32 NextValue 
        { 
            get; 
            set; 
        }

        public Int32 StepValue 
        { 
            get; 
            set; 
        }

        #endregion Properties

        #region Constructors

        public AutoNumber()
        {
        }

        public AutoNumber(Microsoft.Xrm.Sdk.Entity autoNumber)
        {
            if (autoNumber == null)
            {
                return;
            }

            this.AutoNumberId =
                (autoNumber.Attributes.Contains(Fields.mcs_autonumberid)) ?
                    new Guid(autoNumber.Attributes[Fields.mcs_autonumberid].ToString()) : null as Guid?;

            if (!this.AutoNumberId.HasValue)
            {
                return;
            }

            this.Entity =
                (autoNumber.Attributes.Contains(Fields.mcs_entity)) ?
                    autoNumber.Attributes[Fields.mcs_entity].ToString() : String.Empty;

            this.Field =
                (autoNumber.Attributes.Contains(Fields.mcs_field)) ?
                    autoNumber.Attributes[Fields.mcs_field].ToString() : String.Empty;

            this.NextValue =
                (autoNumber.Attributes.Contains(Fields.mcs_nextvalue)) ?
                    System.Convert.ToInt32(
                        autoNumber.Attributes[Fields.mcs_nextvalue],
                        System.Globalization.CultureInfo.CurrentCulture
                    ) : 0;

            this.StepValue =
                (autoNumber.Attributes.Contains(Fields.mcs_stepvalue)) ?
                    System.Convert.ToInt32(
                        autoNumber.Attributes[Fields.mcs_stepvalue],
                        System.Globalization.CultureInfo.CurrentCulture
                    ) : 0;

            this.Format =
                (autoNumber.Attributes.Contains(Fields.mcs_format)) ?
                    autoNumber.Attributes[Fields.mcs_format].ToString() : String.Empty;
        }        

        #endregion Constructors

        #region Public Methods

        public String getFormattedNumberValue()
        {
            String result = String.Empty; String nxtValue = string.Empty;
            try
            {
                //Claim number format
                if(this.NextValue > 0)
                {
                    nxtValue = this.NextValue.ToString().PadLeft(9, '0');
                }
                result = String.Format(System.Globalization.CultureInfo.CurrentCulture, this.Format, nxtValue);

                String currentMonth = DateTime.Now.Month.ToString().PadLeft(2,'0');
                String currentYear = DateTime.Now.Year.ToString().PadLeft(2, '0').Substring(2, 2);
                result = result.Replace("MM", currentMonth).Replace("YY", currentYear);
            }
            catch
            {
            }
            return result;
        }

        public void Execute(IServiceProvider sp)
        {
            Entity oDyna = null;
            ITracingService tracingService = null;
            try
            {
                tracingService = sp.GetService(typeof(ITracingService)) as ITracingService;
                tracingService.Trace("In AutoNumber Plugin...");

                IPluginExecutionContext context = sp.GetService(typeof(IPluginExecutionContext)) as IPluginExecutionContext;
                IOrganizationServiceFactory factory = sp.GetService(typeof(IOrganizationServiceFactory)) as IOrganizationServiceFactory;
                IOrganizationService service = factory.CreateOrganizationService(context.UserId);
                
                if (context.IsExecutingOffline)
                {
                    return;
                }

                if  (   
                        context.Stage != Stages.PreValidation 
                        && 
                        context.Stage != Stages.PreOperation
                    )
                {
                    return;
                }
                
                if (context.IsOfflinePlayback)
                {
                    if (context.InputParameters.Contains(Parameters.Target) &&
                        context.InputParameters[Parameters.Target] is Entity)
                    {
                        oDyna = context.InputParameters[Parameters.Target] as Entity;
                    }
                    else
                    {
                        return;
                    }

                    AutoNumber.processEntityForAutoNumber(ref oDyna, service, tracingService);

                    if (oDyna != null)
                    {
                        context.InputParameters[Parameters.Target] = oDyna;
                    }
                }
                else
                {
                    if  (
                            context.InputParameters.Contains(Parameters.Target) 
                            && 
                            context.InputParameters[Parameters.Target] is Entity
                        )
                    {
                        oDyna = context.InputParameters[Parameters.Target] as Entity;
                    }
                    else
                    {
                        return;
                    }

                    AutoNumber.processEntityForAutoNumber(ref oDyna, service, tracingService);
                    if (oDyna != null)
                    {
                        context.InputParameters[Parameters.Target] = oDyna;
                    }
                }

            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException("AutoNumber Plugin Error >> " + ex.Message, ex);
            }
        }

        /// <summary>
        /// Updates the Auto Number Entity
        /// </summary>
        /// <param name="autoNumberId">GUID</param>
        /// <param name="service">Service</param>
        public void updateAutoNumber(Guid? entityId, IOrganizationService service)
        {
            if (!this.AutoNumberId.HasValue)
            {
                return;
            }

            String functionName = "UpdateAutoNumber: ";
            EntityCollection oColl = new EntityCollection();
            try
            {
                Entity updEntity = new Entity(Entities.AutoNumber);

                updEntity.Attributes.Add(Fields.mcs_autonumberid, this.AutoNumberId.Value);
                updEntity.Attributes.Add(Fields.mcs_entityid, entityId.Value.ToString());

                service.Update(updEntity);
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                throw new Exception(functionName + "(F): " + ex.GetDetailedOutput());
            }
            catch (Exception ex)
            {
                throw new Exception(functionName + ex.GetDetailedOutput());
            }
        }

        #endregion Public Methods

        #region Public Static Methods

        public static List<AutoNumber> getAutoNumbers(String entity, IOrganizationService service)
        {
            return getAutoNumbers(AutoNumber.getAutoNumberRecords(entity, service));
        }

        public static List<AutoNumber> getAutoNumbers(EntityCollection entities)
        {
            List<AutoNumber> result = new List<AutoNumber>();
            foreach (Entity entity in entities.Entities)
            {
                var autoNumber = new AutoNumber(entity);
                result.Add(autoNumber);
            }
            return result;
        }

        /// <summary>
        /// Method to process AutoNumber logic
        /// </summary>
        /// <param name="oEntity">oEntity</param>
        /// <param name="service">service</param>
        public static void processEntityForAutoNumber(ref Entity oEntity, IOrganizationService service, ITracingService tracingService)
        {
            if (oEntity == null)
            {
                return;
            }

            String functionName = "ProcessEntityForAutoNumber: ";
            String entityName = String.Empty;
            String formattedNumberValue = String.Empty;
            try
            {
                entityName = oEntity.LogicalName;
                tracingService.Trace(" - Start Process for " + entityName);

                OrganizationServiceContext context = new OrganizationServiceContext(service);

                lock (SyncLock)
                {
                    var autoNumbers = AutoNumber.getAutoNumbers(entityName, service);
                    if (autoNumbers.Count == 0)
                    {
                        return;
                    }

                    tracingService.Trace(" - AutoNumber Records Found : " + autoNumbers.Count);
                    foreach (var autoNumber in autoNumbers)
                    {
                        tracingService.Trace(" - Read AutoNumber Properties");

                        if (String.IsNullOrEmpty(autoNumber.Field))
                        {
                            continue;
                        }

                        formattedNumberValue = autoNumber.getFormattedNumberValue();
                        if (String.IsNullOrEmpty(formattedNumberValue))
                        {
                            continue;
                        }

                        tracingService.Trace(" - AutoNumber Generated : " + formattedNumberValue);

                        if (!oEntity.Attributes.Contains(autoNumber.Field))
                        {
                            oEntity.Attributes.Add(autoNumber.Field, String.Empty);
                        }

                        if (string.IsNullOrEmpty(oEntity.GetAttributeValue<string>(autoNumber.Field)))
                        {
                            oEntity.Attributes[autoNumber.Field] = formattedNumberValue;

                            tracingService.Trace(" - New Value : " + autoNumber.NextValue);

                            //Send update call so the workflow can run and increment the next number.
                            autoNumber.updateAutoNumber(oEntity.Id, service);
                            tracingService.Trace(" - Updated in AutoNumber Entity");
                        }
                        else
                        {
                            tracingService.Trace(" - Auto Number not updated as autonumber field already has value");
                        }
                    }
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                throw new Exception(functionName + "(F): " + ex.GetDetailedOutput());
            }
            catch (Exception ex)
            {
                throw new Exception(functionName + ex.GetDetailedOutput());
            }
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Gets All the Auto Number Records for a specific entity
        /// </summary>
        /// <param name="entityName">Entity Mame</param>
        /// <param name="service">Service</param>
        /// <returns>Records for this entity</returns>
        public static EntityCollection getAutoNumberRecords(String entityName, IOrganizationService service)
        {
            String functionName = "GetAutoNumberRecords: ";
            EntityCollection oColl = new EntityCollection();
            try
            {
                QueryExpression query = new QueryExpression(Entities.AutoNumber);

                query.ColumnSet =
                    new ColumnSet(
                        new String[] { 
                            Fields.mcs_autonumberid, 
                            Fields.mcs_entity, 
                            Fields.mcs_field, 
                            Fields.mcs_format, 
                            Fields.mcs_name, 
                            Fields.mcs_nextvalue, 
                            Fields.mcs_stepvalue 
                        }
                    );

                //query.ColumnSet.AllColumns = true;

                query.Criteria = new FilterExpression(LogicalOperator.And);

                if (!String.IsNullOrEmpty(entityName))
                {
                    query.Criteria.AddCondition(Fields.mcs_entity, ConditionOperator.Equal, new Object[] { entityName });
                }
                query.Criteria.AddCondition(Fields.statecode, ConditionOperator.Equal, new Object[] { StateCodes.Active });

                oColl = service.RetrieveMultiple(query);
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                throw new Exception(functionName + "(F): " + ex.GetDetailedOutput());
            }
            catch (Exception ex)
            {
                throw new Exception(functionName + ex.GetDetailedOutput());
            }
            return oColl;
        }

        #endregion Public Static Methods
    }
}