﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Mdws;
using BMS.Facade;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Via.Commands.WF;
using BMS.VistaIntegration.Via.Commands;
using System.Linq;
using System.IO;

namespace BMS.VistaIntegration.Via.Tests
{
    [TestClass]
    public class ViaVistASessionFactoryTest
    {
        private bool _compareToMDWS = true;
        private SiteParameters _siteParams = new SiteParameters();
        private VistASite _vistaSite = null;
        private string _site = "CHEYL19";
        //private string _site = "HOUTEST";

        [TestInitialize()]
        public void InitTests()
        {
            switch (_site)
            {
                case "NTX":
                    _vistaSite = new VistASite("test", "V16NTX", "549", TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                    BuildParamsForHouston();
                    break;
                case "BILOXI":
                    _vistaSite = new VistASite("test", "V16BLX", "520", TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                    BuildParamsForHouston();
                    break;
                case "STL":
                    _vistaSite = new VistASite("test", "V16STL", "657", TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                    BuildParamsForHouston();
                    break;
                case "HOUSTON":
                    _vistaSite = new VistASite("test", "V16HOU", "580", TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                    BuildParamsForHouston();
                    break;
                case "CHEYL19":
                    _vistaSite = new VistASite("test", "CHEYL19", "516", TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                    BuildParamsForCheyl19();
                    break;
            }
        }

        public void BuildParamsForCheyl19()
        {
            _siteParams.PatientIEN = "3977";
            _siteParams.PatientIENWithAdmission = "7185012"; //7185012^DRALEAU,LORENE^F^2670331^101625668^3041105^73^202^204908~jhkdksdf^0^40
            _siteParams.NewPersonIEN = "202";
            _siteParams.PatientMovementIEN = "7886";
            _siteParams.CanceledOrdersIENs = new[] { "5587610", "5587620", "5587630" };
            _siteParams.listPatientStartDate = new DateTime(2010, 01, 6);
            _siteParams.listPatientEndDate = new DateTime(2011, 11, 8);
            _siteParams.listPatientClinics = new[] { "1690" , "1909", "1910"};
            //_siteParams.listOrderStartDate = new DateTime(2005, 04, 18);  //  missing record
            //_siteParams.listOrderEndDate = new DateTime(2005, 04, 20);
            _siteParams.listOrderStartDate = new DateTime(2002, 01, 01);
            _siteParams.listOrderEndDate = new DateTime(2004, 08, 12);
            _siteParams.orderableItems = new[] { "73", "75", "76", "2057" };
        }

        public void BuildParamsForHouston()
        {
            _siteParams = new SiteParameters();
            _siteParams.PatientIEN = "482243";
            _siteParams.PatientIENWithAdmission = "482238"; // 482238^SHUYL,ILYYN U^M^2420426^101523523^3160923^52^137729^2271498~MAJOR DEPRESSION/PSYCHOSIS^0^
            _siteParams.NewPersonIEN = "202";
            _siteParams.ClinicIEN = "109";
            _siteParams.PatientSSN = "101523523";
            _siteParams.PatientMovementIEN = "2160047";
            _siteParams.AdmissionIEN = "2255839";
            _siteParams.BulkPatientIENs = "504486,443077,79854,276189,306977,454825,250350,163669,446769,142893,397690,249660,465101,45193,13281,85039,387393,381871,365491,182748,13193,145871,106778,448422,7736,397841,101930,410648,286760,359564,84966,5549,325125,231082,309510,107599,429366,13778,243353,460530,264513,283083,361881,439994,140288,216720,454716,7898,72480,339985,214871,71909,292242,464450,461207,409766,158993,12867,312229,349321,127638,258465,447405,192168,268993,203451,99735,462522,143499,420623,62099,279151,447899,322359,152341,214962,152967,173852,256844,413739,249087,91209,13204,351365,35825,463915,80240,266182,418802,196832,434317,256842,133215,455282,449067,160373,144365,360442,264620,79215,282064,277103,185699,56846,465077,296586,61868,338945,429063,279825,281364,459060,141299,226870,341949,221930,447421,214732,174485,309876,82108,275593,446021,328089,37510,427742,463635,184015,291763,364734,365911,216422,242847,240349,418263,136319,15527,320674,463342,158328,181905,418977,42893,4198,26030,383419,423273,22140,140891,429765,261821,44207,106029,406466,442997,168141,54084,376559,441445,49,86227,167314,287840,68020,267474,314246,28343,186014,157552,108518,409634,222075,167638,68496,229673,431729,466155,457505,119459,318438,436105,458888,383155,122548,50423,72747,228170,316314,441344,171812,172978,309391,446050,265249,5374,464061,450146,137204,60393,433164,228019,428185,18470,360895,257394,325947,458644,242056,387625,18068,82123,194853,261524,444376,427085,80321,116473,369058,446986,214538,379061,270089,373536,248932,159400,266810,366640,445716,171769,462407,216918,362961,317853,47759,94729,135450,157188,182730,298275,14522,352041,367648,154439,434953,440195,190689,433684,334362,307429,67301,465888,305510,446053,350516,64977,433363,373879,37402,397190,210237,377882,145544,431316,351865,442697,132947,241447,217219,331609,18370,28915,77481,111144,466297,123209,420596,136748,40380,254627,428823,463470,72580,181475,78289,345041,217722,250195,410622,245948,207852,454337,220330,390807,411589,299472,433144,308373,222787,447607,242523,53097,446967,199481,356494,267269,455540,432358,182182,453964,221940,360568,303269,75429,29075,143163,307477,116586,384039,371995,364153,28360,90666,349058,105915,205681,345922,158229,333418,302257,116841,291090,228636,265976,268565,275729,2778,221056,317584,439086,300686,192473,230841,348943,236847,442494,445656,85466,117232,315611,78773,250299,222839,451618,316049,461742,143401,112047,269161,427432,156096,380951,14729,447315,189865,28640,287909,359999,92329,292017,449203,136310,439741,194533,238745,283849,173806,275356,423962,316029,139176,374151,357351,425635,93055,298880,285156,18959,83535,294790,125701,194486,448855,448614,372429,332122,47635,75700,279278,428441,330974,378170,441806,335174,418378,87300,360063,133418,321728,350058,422470,227754,339862,208970,21728,64217,143133,320626,76472,343110,229203,378029,326443,186824,448639,149094,285624,111974,66866,255439,205634,282065,4442,225843,448485,455554,402092,276716,459061,463414,373633,97878,314989,377218,110660,215027,380266,65463,350474,282413,251327,262021,246050,421604,223793,173085,447056,167273,212076,449126,436061,119704,378585,204856,298587,427502,383652,72515,275035,40624,165245,406005,190183,339578,55412,439181,333314,165117,320403,418262,228633,253541,272212,100473,415666,54715,28170,19424,24071,96840,257559,331982,422855,21220,203185,146585,108875,196920,4146,335625,312054,281142";
            _siteParams.CanceledOrdersIENs = new[] { "108465819", "108465820" , "108465821" };
            _siteParams.listPatientStartDate = new DateTime(2015, 01, 6);
            _siteParams.listPatientEndDate = new DateTime(2018, 3, 8);
            _siteParams.listPatientClinics = new[] { "5435", "5320" };
            _siteParams.listOrderStartDate = new DateTime(2017, 12, 22);
            _siteParams.listOrderEndDate = new DateTime(2017, 12, 28, 23, 59, 59);
            _siteParams.orderableItems = new[] { "73", "75", "76", "18005", "18003", "14795", "13974", "13975", "14215", "14214" };
        }

        public IVistAQuery BuildVistAQuery(DataRetrievalMethod retrievalMethod)
        {
            IVistAQuery vistaQuery = null;
            switch (retrievalMethod)
            {
                case DataRetrievalMethod.MDWS:
                    MdwsVistASession session = new MdwsVistASession(_vistaSite);
                    session.Open(new VistAConnectionInfo(_vistaSite.MdwsEndpointUrl, null, null));
                    vistaQuery = new MdwsVistAQuery(session);
                    break;
                case DataRetrievalMethod.VIA:
                    vistaQuery = new ViaVistAQuery(new ViaVistASession(_vistaSite));
                    break;
            }

            return vistaQuery;
        }

        [TestMethod]
        public void TestListBedSwitch()
        {
            try
            {
                List<string> iens = new List<string>() { "1,2,3" };

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<Data.WF.BedSwitch> bedSwitchVIA = viaQuery.GetBedsSwitch(iens);

                Assert.IsNotNull(bedSwitchVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Data.WF.BedSwitch> bedSwitchMDWS = mdwsQuery.GetBedsSwitch(iens);

                    Assert.IsNotNull(bedSwitchMDWS);

                    Assert.AreEqual(bedSwitchVIA.Count, bedSwitchMDWS.Count);

                    Assert.AreEqual(bedSwitchVIA.First().IEN, bedSwitchMDWS.First().IEN);
                    Assert.AreEqual(bedSwitchVIA.First().WardIen, bedSwitchMDWS.First().WardIen);

                    Assert.AreEqual(bedSwitchVIA.Last().IEN, bedSwitchMDWS.Last().IEN);
                    Assert.AreEqual(bedSwitchVIA.Last().WardIen, bedSwitchMDWS.Last().WardIen);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientByDate() // MDWS returns more results because it list patients by book-ended IENs instead of dates. Some entries do not have dates in HOUTEST. Test fails as a result
        {
            try
            {
                DateTime startDate = new DateTime(2018, 01, 01);
                DateTime endDate = new DateTime(2018, 01, 07);

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(startDate, endDate);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                SortedList<string, Patient> viaMap = new SortedList<string, Patient>(patientsVIA.ToDictionary(item => item.IEN));

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(startDate, endDate);

                    Assert.IsNotNull(patientsMDWS);

                    SortedList<string, Patient> mdwsMap = new SortedList<string, Patient>(patientsMDWS.ToDictionary(item => item.IEN));

                    SortedList<string, Patient> onlyInVia = new SortedList<string, Patient>();
                    SortedList<string, Patient> onlyInMdws = new SortedList<string, Patient>();

                    foreach (string patientIen in viaMap.Keys)
                    {
                        if (!mdwsMap.ContainsKey(patientIen))
                            onlyInVia.Add(patientIen, viaMap[patientIen]);
                    }

                    foreach (string patientIen in mdwsMap.Keys)
                    {
                        if (!viaMap.ContainsKey(patientIen))
                            onlyInMdws.Add(patientIen, mdwsMap[patientIen]);
                    }

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientByIEN() // VIA appears to work. MDWS seems to ignore PatientIEN filter.
        {
            // spiders out on Attending Physician to "NewPerson" and FacilityTreatingSpecialty - "TreatingSpecialty"
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(null, null, _siteParams.BulkPatientIENs);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                //if (_compareToMDWS)
                //{
                //    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                //    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(null, null, _siteParams.PatientIEN);

                //    Assert.IsNotNull(patientsMDWS);

                //    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                //}

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatient()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                Patient patientVIA = viaQuery.GetPatientByIen(_siteParams.PatientIEN);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    Patient patientMDWS = mdwsQuery.GetPatientByIen(_siteParams.PatientIEN);

                    Assert.IsNotNull(patientMDWS);

                    // check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientVIA.IEN, patientMDWS.IEN);
                    Assert.AreEqual(patientVIA.SocialSecurityNumber, patientMDWS.SocialSecurityNumber);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListAdmittedPatientsForUpdate()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetAdmittedPatientsForUpdate();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetAdmittedPatientsForUpdate();

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);

                    Assert.AreEqual(patientsVIA.First().IEN, patientsMDWS.First().IEN);
                    Assert.AreEqual(patientsVIA.First().SocialSecurityNumber, patientsMDWS.First().SocialSecurityNumber);
                    Assert.AreEqual(patientsVIA.First().AdmissionIen, patientsMDWS.First().AdmissionIen);
                    Assert.AreEqual(patientsVIA.First().AdmittingDiagnosis, patientsMDWS.First().AdmittingDiagnosis);
                    Assert.AreEqual(patientsVIA.First().TreatingSpecialtyId, patientsMDWS.First().TreatingSpecialtyId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestGetPatientBySSN()
        {
            try
            {
                IList<string> patientSSNs = new List<string>() { _siteParams.PatientSSN }; // TODO - add ssns

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(patientSSNs);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(patientSSNs);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);

                    // check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientsVIA.First().IEN, patientsMDWS.First().IEN);
                    Assert.AreEqual(patientsVIA.First().SocialSecurityNumber, patientsMDWS.First().SocialSecurityNumber);

                    Assert.AreEqual(patientsVIA.Last().IEN, patientsMDWS.Last().IEN);
                    Assert.AreEqual(patientsVIA.Last().SocialSecurityNumber, patientsMDWS.Last().SocialSecurityNumber);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListAdmittedPatients()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<AdmittedPatient> patientsVIA = viaQuery.GetAdmittedPatients(_siteParams.PatientMovementIEN);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<AdmittedPatient> patientsMDWS = mdwsQuery.GetAdmittedPatients(_siteParams.PatientMovementIEN);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);

                    Assert.AreEqual(patientsVIA.First().IEN, patientsMDWS.First().IEN);
                    Assert.AreEqual(patientsVIA.First().Name, patientsMDWS.First().Name);
                    Assert.AreEqual(patientsVIA.First().AdmissionIen, patientsMDWS.First().AdmissionIen);
                    Assert.AreEqual(patientsVIA.First().MovementIen, patientsMDWS.First().MovementIen);
                    Assert.AreEqual(patientsVIA.First().WardName, patientsMDWS.First().WardName);
                    Assert.AreEqual(patientsVIA.First().BedName, patientsMDWS.First().BedName);

                    Assert.AreEqual(patientsVIA.Last().IEN, patientsMDWS.Last().IEN);
                    Assert.AreEqual(patientsVIA.Last().Name, patientsMDWS.Last().Name);
                    Assert.AreEqual(patientsVIA.Last().AdmissionIen, patientsMDWS.Last().AdmissionIen);
                    Assert.AreEqual(patientsVIA.Last().MovementIen, patientsMDWS.Last().MovementIen);
                    Assert.AreEqual(patientsVIA.Last().WardName, patientsMDWS.Last().WardName);
                    Assert.AreEqual(patientsVIA.Last().BedName, patientsMDWS.Last().BedName);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetWardLocations()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<WardLocation> wardLocationsVIA = viaQuery.GetWardLocations();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(wardLocationsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<WardLocation> wardLocationsMDWS = mdwsQuery.GetWardLocations();

                    Assert.IsNotNull(wardLocationsMDWS);

                    Assert.AreEqual(wardLocationsVIA.Count, wardLocationsMDWS.Count);

                    // check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(wardLocationsMDWS.First().IEN, wardLocationsVIA.First().IEN);
                    Assert.AreEqual(wardLocationsMDWS.First().Specialty.Name, wardLocationsVIA.First().Specialty.Name);

                    Assert.AreEqual(wardLocationsMDWS.Last().IEN, wardLocationsVIA.Last().IEN);
                    Assert.AreEqual(wardLocationsMDWS.Last().Specialty.Name, wardLocationsVIA.Last().Specialty.Name);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetHospitalLocations()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<HospitalLocation> hospitalLocationsVIA = viaQuery.GetHospitalLocations();

                foreach (HospitalLocation hospitalLocation in hospitalLocationsVIA)
                {
                    if (hospitalLocation.WardLocation != null)
                        break;
                }

                Dictionary<string, HospitalLocation> viaMap = new Dictionary<string, HospitalLocation>();
                viaMap = hospitalLocationsVIA.ToDictionary(item => item.IEN);

                Assert.IsNotNull(hospitalLocationsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<HospitalLocation> hospitalLocationsMDWS = mdwsQuery.GetHospitalLocations();

                    Assert.IsNotNull(hospitalLocationsMDWS);

                    Dictionary<string, HospitalLocation> mdwsMap = new Dictionary<string, HospitalLocation>();
                    mdwsMap = hospitalLocationsVIA.ToDictionary(item => item.IEN);

                    Assert.AreEqual(hospitalLocationsVIA.Count, hospitalLocationsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod] // this method does not appear to have a use-case and does not work with MDWS. VIA seems to work.
        public void TestListNewPersonByDateRange()
        {
            try
            {
                DateTime startDate = new DateTime(2006, 02, 25); // TODO - set dates
                DateTime endDate = new DateTime(2006, 03, 01, 23, 59, 59);

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<NewPerson> newPersonsVIA = viaQuery.GetNewPersons(startDate, endDate);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(newPersonsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<NewPerson> newPersonsMDWS = mdwsQuery.GetNewPersons(startDate, endDate);

                    Assert.IsNotNull(newPersonsMDWS);

                    Assert.AreEqual(newPersonsVIA.Count, newPersonsMDWS.Count);

                    // Check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(newPersonsVIA.First().IEN, newPersonsMDWS.First().IEN);
                    Assert.AreEqual(newPersonsVIA.First().Name, newPersonsMDWS.First().Name);

                    Assert.AreEqual(newPersonsVIA.Last().IEN, newPersonsMDWS.Last().IEN);
                    Assert.AreEqual(newPersonsVIA.Last().Name, newPersonsMDWS.Last().Name);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListNewPersonByIEN()
        {
            // ListNewPerson is normally called dynamically in a "spider out" call from ListPatient
            try
            {
                ViaVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA) as ViaVistAQuery;

                // make dynamic call to VIA command for new person
                string target = BMS.VistaIntegration.Via.Commands.EIS.ListNewPersonCommand.Target;
                int argumentCount = BMS.VistaIntegration.Via.Commands.EIS.ListNewPersonCommand.ArgumentsCount;

                List<NewPerson> newPersonsVIA = viaQuery.GetResults(new EntitySetCache.BulkEntitiesListCommand<NewPerson>(viaQuery, target, new string[] { _siteParams.NewPersonIEN }, argumentCount, null));
                Assert.IsNotNull(newPersonsVIA);

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListSpecialty()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Specialty> specialtiesVIA = viaQuery.GetSpecialties();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(specialtiesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Specialty> specialtiesMDWS = mdwsQuery.GetSpecialties();

                    Assert.IsNotNull(specialtiesMDWS);

                    Assert.AreEqual(specialtiesVIA.Count, specialtiesMDWS.Count);

                    Assert.AreEqual(specialtiesVIA.First().Name, specialtiesMDWS.First().Name);
                    Assert.AreEqual(specialtiesVIA.First().IEN, specialtiesMDWS.First().IEN);

                    Assert.AreEqual(specialtiesVIA.Last().Name, specialtiesMDWS.Last().Name);
                    Assert.AreEqual(specialtiesVIA.Last().IEN, specialtiesMDWS.Last().IEN);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod] // Not implemented
        public void TestListWardLocation()
        {
            // can't find MDWS or VIA implementation. There is a GetWardLocations, but that's already tested above.
            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListFacilityMovementTypes()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<FacilityMovementType> facilityMovementTypesVIA = viaQuery.GetFacilityMovementTypes();

                Assert.IsNotNull(facilityMovementTypesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<FacilityMovementType> facilityMovementTypesMDWS = mdwsQuery.GetFacilityMovementTypes();

                    Assert.IsNotNull(facilityMovementTypesMDWS);

                    Assert.AreEqual(facilityMovementTypesVIA.Count, facilityMovementTypesMDWS.Count);

                    Assert.AreEqual(facilityMovementTypesVIA.First().Name, facilityMovementTypesMDWS.First().Name);
                    Assert.AreEqual(facilityMovementTypesVIA.First().IEN, facilityMovementTypesMDWS.First().IEN);
                    Assert.AreEqual(facilityMovementTypesVIA.First().Active, facilityMovementTypesMDWS.First().Active);

                    Assert.AreEqual(facilityMovementTypesVIA.Last().Name, facilityMovementTypesMDWS.Last().Name);
                    Assert.AreEqual(facilityMovementTypesVIA.Last().IEN, facilityMovementTypesMDWS.Last().IEN);
                    Assert.AreEqual(facilityMovementTypesVIA.Last().Active, facilityMovementTypesMDWS.Last().Active);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListFacilityTreatingSpecialties()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<FacilityTreatingSpecialty> facilityTreatingSpecialtiesVIA = viaQuery.GetFacilityTreatingSpecialties();

                Assert.IsNotNull(facilityTreatingSpecialtiesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<FacilityTreatingSpecialty> facilityTreatingSpecialtiesMDWS = mdwsQuery.GetFacilityTreatingSpecialties();

                    Assert.IsNotNull(facilityTreatingSpecialtiesMDWS);

                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.Count, facilityTreatingSpecialtiesMDWS.Count);

                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.First().Name, facilityTreatingSpecialtiesMDWS.First().Name);
                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.First().IEN, facilityTreatingSpecialtiesMDWS.First().IEN);
                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.First().SpecialtyId, facilityTreatingSpecialtiesMDWS.First().SpecialtyId);

                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.Last().Name, facilityTreatingSpecialtiesMDWS.Last().Name);
                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.Last().IEN, facilityTreatingSpecialtiesMDWS.Last().IEN);
                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.Last().SpecialtyId, facilityTreatingSpecialtiesMDWS.Last().SpecialtyId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestListMasMovementTransactionType()
        {
            // can't find MDWS or VIA implementation
            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListMedicalCenterDivision()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<MedicalCenterDivision> medicalCenterDivisionsVIA = viaQuery.GetMedicalCenterDivisions();

                Assert.IsNotNull(medicalCenterDivisionsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<MedicalCenterDivision> medicalCenterDivisionMDWS = mdwsQuery.GetMedicalCenterDivisions();

                    Assert.IsNotNull(medicalCenterDivisionMDWS);

                    Assert.AreEqual(medicalCenterDivisionsVIA.Count, medicalCenterDivisionMDWS.Count);

                    Assert.AreEqual(medicalCenterDivisionsVIA.First().Name, medicalCenterDivisionMDWS.First().Name);
                    Assert.AreEqual(medicalCenterDivisionsVIA.First().IEN, medicalCenterDivisionMDWS.First().IEN);
                    Assert.AreEqual(medicalCenterDivisionsVIA.First().FacilityNumber, medicalCenterDivisionMDWS.First().FacilityNumber);

                    Assert.AreEqual(medicalCenterDivisionsVIA.Last().Name, medicalCenterDivisionMDWS.Last().Name);
                    Assert.AreEqual(medicalCenterDivisionsVIA.Last().IEN, medicalCenterDivisionMDWS.Last().IEN);
                    Assert.AreEqual(medicalCenterDivisionsVIA.Last().FacilityNumber, medicalCenterDivisionMDWS.Last().FacilityNumber);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListOrderableItems()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderableItem> orderableItemsVIA = viaQuery.GetOrderableItems();

                Assert.IsNotNull(orderableItemsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderableItem> orderableItemsMDWS = mdwsQuery.GetOrderableItems();

                    Assert.IsNotNull(orderableItemsMDWS);

                    Assert.AreEqual(orderableItemsVIA.Count, orderableItemsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListOrderStatuses()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderStatus> orderStatusesVIA = viaQuery.GetOrderStatuses();

                Assert.IsNotNull(orderStatusesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderStatus> orderStatusesMDWS = mdwsQuery.GetOrderStatuses();

                    Assert.IsNotNull(orderStatusesMDWS);

                    Assert.AreEqual(orderStatusesVIA.Count, orderStatusesMDWS.Count);

                    Assert.AreEqual(orderStatusesVIA.First().Name, orderStatusesMDWS.First().Name);
                    Assert.AreEqual(orderStatusesVIA.First().IEN, orderStatusesMDWS.First().IEN);

                    Assert.AreEqual(orderStatusesVIA.Last().Name, orderStatusesMDWS.Last().Name);
                    Assert.AreEqual(orderStatusesVIA.Last().IEN, orderStatusesMDWS.Last().IEN);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        // not implemented exception
        private void TestGetPatientMovement(string patientIEN, DateTime movementDate, MovementTransactionType movementType)
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                PatientMovement patientMovementVIA = viaQuery.GetPatientMovement(patientIEN, movementDate, movementType);

                Assert.IsNotNull(patientMovementVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    PatientMovement patientMovementMDWS = mdwsQuery.GetPatientMovement(patientIEN, movementDate, movementType);

                    Assert.IsNotNull(patientMovementMDWS);

                    // Check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientMovementVIA.IEN, patientMovementMDWS.IEN);
                    Assert.AreEqual(patientMovementVIA.DateTime, patientMovementMDWS.DateTime);
                    Assert.AreEqual(patientMovementVIA.EnteredById, patientMovementMDWS.EnteredById);
                    Assert.AreEqual(patientMovementVIA.PatientId, patientMovementMDWS.PatientId);
                    Assert.AreEqual(patientMovementVIA.CurrentAdmissionIen, patientMovementMDWS.CurrentAdmissionIen);
                    Assert.AreEqual(patientMovementVIA.EnteredById, patientMovementMDWS.EnteredById);
                    Assert.AreEqual(patientMovementVIA.RoomBedId, patientMovementMDWS.RoomBedId);
                    Assert.AreEqual(patientMovementVIA.TransactionTypeId, patientMovementMDWS.TransactionTypeId);
                    Assert.AreEqual(patientMovementVIA.TypeOfMovementId, patientMovementMDWS.TypeOfMovementId);
                    Assert.AreEqual(patientMovementVIA.WardLocationId, patientMovementMDWS.WardLocationId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        // [TestMethod]  // not implemented exception
        public void TestGetPatientMovementForAdmission() 
        {
            try
            {
                DateTime movementDate = new DateTime(2018, 1, 26);

                TestGetPatientMovement("288567", movementDate, MovementTransactionType.ADMISSION);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestGetPatientMovementForDischarge()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.DISCHARGE);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestGetPatientMovementForSpecialtyTransfer()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.SPECIALTY_TRANSFER);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestGetPatientMovementForTransfer()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.TRANSFER);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementByIEN()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                PatientMovement patientMovementVIA = viaQuery.GetPatientMovementByIen(_siteParams.PatientMovementIEN);

                Assert.IsNotNull(patientMovementVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    PatientMovement patientMovementMDWS = mdwsQuery.GetPatientMovementByIen(_siteParams.PatientMovementIEN);

                    Assert.IsNotNull(patientMovementMDWS);

                    // Check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientMovementVIA.IEN, patientMovementMDWS.IEN);
                    Assert.AreEqual(patientMovementVIA.DateTime, patientMovementMDWS.DateTime);
                    Assert.AreEqual(patientMovementVIA.EnteredById, patientMovementMDWS.EnteredById);
                    Assert.AreEqual(patientMovementVIA.PatientId, patientMovementMDWS.PatientId);
                    Assert.AreEqual(patientMovementVIA.CurrentAdmissionIen, patientMovementMDWS.CurrentAdmissionIen);
                    Assert.AreEqual(patientMovementVIA.EnteredById, patientMovementMDWS.EnteredById);
                    Assert.AreEqual(patientMovementVIA.RoomBedId, patientMovementMDWS.RoomBedId);
                    Assert.AreEqual(patientMovementVIA.TransactionTypeId, patientMovementMDWS.TransactionTypeId);
                    Assert.AreEqual(patientMovementVIA.TypeOfMovementId, patientMovementMDWS.TypeOfMovementId);
                    Assert.AreEqual(patientMovementVIA.WardLocationId, patientMovementMDWS.WardLocationId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementByDateRange()
        {
            DateTime startDate = new DateTime(2018, 1, 26);
            DateTime endDate = new DateTime(2018, 1, 26, 23, 59, 59);

            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<PatientMovement> patientMovementsVIA = viaQuery.GetPatientMovements(startDate, endDate);

                Assert.IsNotNull(patientMovementsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<PatientMovement> patientMovementsMDWS = mdwsQuery.GetPatientMovements(startDate, endDate);

                    Assert.IsNotNull(patientMovementsMDWS);

                    Assert.AreEqual(patientMovementsVIA.Count, patientMovementsMDWS.Count);

                    // Check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientMovementsVIA[0].IEN, patientMovementsMDWS[0].IEN);
                    Assert.AreEqual(patientMovementsVIA[0].DateTime, patientMovementsMDWS[0].DateTime);
                    Assert.AreEqual(patientMovementsVIA[0].EnteredById, patientMovementsMDWS[0].EnteredById);
                    Assert.AreEqual(patientMovementsVIA[0].PatientId, patientMovementsMDWS[0].PatientId);
                    Assert.AreEqual(patientMovementsVIA[0].CurrentAdmissionIen, patientMovementsMDWS[0].CurrentAdmissionIen);
                    Assert.AreEqual(patientMovementsVIA[0].EnteredById, patientMovementsMDWS[0].EnteredById);
                    Assert.AreEqual(patientMovementsVIA[0].RoomBedId, patientMovementsMDWS[0].RoomBedId);
                    Assert.AreEqual(patientMovementsVIA[0].TransactionTypeId, patientMovementsMDWS[0].TransactionTypeId);
                    Assert.AreEqual(patientMovementsVIA[0].TypeOfMovementId, patientMovementsMDWS[0].TypeOfMovementId);
                    Assert.AreEqual(patientMovementsVIA[0].WardLocationId, patientMovementsMDWS[0].WardLocationId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementIENs()
        {
            DateTime startDate = new DateTime(2018, 1, 26);
            DateTime endDate = new DateTime(2018, 1, 30, 23, 59, 59);

            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<PatientMovementIen> patientMovementIensVIA = viaQuery.GetPatientMovementIens(startDate, endDate);

                Assert.IsNotNull(patientMovementIensVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<PatientMovementIen> patientMovementIensMDWS = mdwsQuery.GetPatientMovementIens(startDate, endDate);

                    Assert.IsNotNull(patientMovementIensMDWS);

                    // MDWS returns extra results. VIA returns correct number
                    //Assert.AreEqual(patientMovementIensVIA.Count, patientMovementIensMDWS.Count);

                    // check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientMovementIensVIA[0].IEN, patientMovementIensMDWS[0].IEN);
                    Assert.AreEqual(patientMovementIensVIA[0].PatientIen, patientMovementIensMDWS[0].PatientIen);
                    Assert.AreEqual(patientMovementIensVIA[0].CurrentAdmissionIen, patientMovementIensMDWS[0].CurrentAdmissionIen);
                    Assert.AreEqual(patientMovementIensVIA[0].EnteredDate, patientMovementIensMDWS[0].EnteredDate);
                    Assert.AreEqual(patientMovementIensVIA[0].MovementDate, patientMovementIensMDWS[0].MovementDate);
                    Assert.AreEqual(patientMovementIensVIA[0].RoomBedId, patientMovementIensMDWS[0].RoomBedId);
                    Assert.AreEqual(patientMovementIensVIA[0].TransactionTypeId, patientMovementIensMDWS[0].TransactionTypeId);
                    Assert.AreEqual(patientMovementIensVIA[0].TypeOfMovementIen, patientMovementIensMDWS[0].TypeOfMovementIen);
                    Assert.AreEqual(patientMovementIensVIA[0].WardLocationId, patientMovementIensMDWS[0].WardLocationId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]  // not implemented exception
        public void TestGetPatientMovementByAdmission()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<PatientMovement> patientMovementsVIA = viaQuery.GetPatientMovementsForAdmission(_siteParams.AdmissionIEN);

                Assert.IsNotNull(patientMovementsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<PatientMovement> patientMovementsMDWS = mdwsQuery.GetPatientMovementsForAdmission(_siteParams.AdmissionIEN);

                    Assert.IsNotNull(patientMovementsMDWS);

                    Assert.AreEqual(patientMovementsVIA.Count, patientMovementsMDWS.Count);

                    // Check fields have expected values and match between VIA/MDWS
                    Assert.AreEqual(patientMovementsVIA[0].IEN, patientMovementsMDWS[0].IEN);
                    Assert.AreEqual(patientMovementsVIA[0].DateTime, patientMovementsMDWS[0].DateTime);
                    Assert.AreEqual(patientMovementsVIA[0].EnteredById, patientMovementsMDWS[0].EnteredById);
                    Assert.AreEqual(patientMovementsVIA[0].PatientId, patientMovementsMDWS[0].PatientId);
                    Assert.AreEqual(patientMovementsVIA[0].CurrentAdmissionIen, patientMovementsMDWS[0].CurrentAdmissionIen);
                    Assert.AreEqual(patientMovementsVIA[0].EnteredById, patientMovementsMDWS[0].EnteredById);
                    Assert.AreEqual(patientMovementsVIA[0].RoomBedId, patientMovementsMDWS[0].RoomBedId);
                    Assert.AreEqual(patientMovementsVIA[0].TransactionTypeId, patientMovementsMDWS[0].TransactionTypeId);
                    Assert.AreEqual(patientMovementsVIA[0].TypeOfMovementId, patientMovementsMDWS[0].TypeOfMovementId);
                    Assert.AreEqual(patientMovementsVIA[0].WardLocationId, patientMovementsMDWS[0].WardLocationId);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        private void TestListScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIEN = null)
        {
            // "PATIEN","SDATE","EDATE","MAX","FROM"
            // <criteria>7200876^3100510^3130520^^</criteria>
            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            IList<ScheduledAdmission> callVIA = viaQuery.GetScheduledAdmissions(startDate, endDate, patientIEN);
            Assert.IsNotNull(callVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                IList<ScheduledAdmission> callMDWS = mdwsQuery.GetScheduledAdmissions(startDate, endDate, patientIEN);

                Assert.IsNotNull(callMDWS);

                Assert.AreEqual(callVIA.Count, callMDWS.Count);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByStartDate()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 5, 10); // TODO - get date

                TestListScheduledAdmissions(startDate);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByDateRange()
        {
            try
            {
                DateTime startDate = new DateTime(2016, 01, 01);
                DateTime endDate = new DateTime(2016, 01, 31, 23, 59, 59);

                TestListScheduledAdmissions(startDate, endDate);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByPatientIEN()
        {
            try
            {
                TestListScheduledAdmissions(null, null, _siteParams.PatientIEN);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        private void TestListPatientAppointments(DateTime? startDate, DateTime? endDate, string patientIEN, IEnumerable<string> clinics)
        {
            try
            {
                // "IENSB","MAX","SDATE","EDATE","CLNIEN","FROM"
                // <criteria>7208163^^3101206^3101206^1690^</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                DateTime beforeVIA = DateTime.Now;
                IList<PatientAppointment> callVIA = viaQuery.GetPatientAppointments(startDate, endDate, patientIEN, clinics);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<PatientAppointment> callMDWS = mdwsQuery.GetPatientAppointments(startDate, endDate, patientIEN, clinics);

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientAppointmentByStartDate()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 12, 6);
                IEnumerable<string> clinics = new[] { "1690" };

                TestListPatientAppointments(startDate, null, null, clinics);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientAppointmentByPatient()
        {
            try
            {
                TestListPatientAppointments(_siteParams.listPatientStartDate, _siteParams.listPatientEndDate, "7207908", null);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientAppointmentByDateRange()
        {
            try
            {
                TestListPatientAppointments(_siteParams.listPatientStartDate, _siteParams.listPatientEndDate, null, _siteParams.listPatientClinics);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestBulkCancelOrders()
        {
            try
            {
                StringBuilder sb = new StringBuilder();
                int limit = 30;
                int cnt = 0;
                foreach (string line in File.ReadLines("national-list.csv"))
                {
                    string[] site = line.Split(',');
                    if (!site[2].StartsWith("10N"))
                    {
                        _vistaSite = new VistASite("test", "VI" + site[1], site[2], TimeZoneInfo.Local, "http://DNS/QuerySvc.asmx");
                        BuildParamsForCheyl19();
                        IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                        IList<string> callVIA = viaQuery.GetCanceledOrders(_siteParams.CanceledOrdersIENs);
                        sb.Append(DateTime.Now + " ***** " + site[1] + " : " + site[2] + " : " + site[5] + " " + site[6] + " " + site[7]);
                        sb.AppendLine();
                        for (int i=0; i<callVIA.Count; i++) {
                            sb.Append(callVIA[i]);
                            sb.AppendLine();
                        }
                        cnt++;
                        if (cnt > limit)
                        {
                            break;
                        }
                        //System.Threading.Thread.Sleep(1000);
                    }
                }
                using (StreamWriter writer = new StreamWriter("cancelorder.txt"))
                {
                    writer.Write(sb.ToString());
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListCancelOrders()
        {
            //  doesn't return all the data
            try
            {
                // "IENSM"
                // <criteria>5587620</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<string> callVIA = viaQuery.GetCanceledOrders(_siteParams.CanceledOrdersIENs);

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<string> ordersMDWS = mdwsQuery.GetCanceledOrders(_siteParams.CanceledOrdersIENs);

                    Assert.IsNotNull(ordersMDWS);

                    Assert.AreEqual(callVIA.Count, ordersMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListRoomBed()
        {
            try
            {
                // "IENSM","FROM","MAX"
                // <criteria>^^1000</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<RoomBed> callVIA = viaQuery.GetRoomBeds();

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<RoomBed> callMDWS = mdwsQuery.GetRoomBeds();

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        //[TestMethod]
        public void TestListClinicAppointments()
        {
            // NOTE: This is not implemented specifically in any IVistAQuery. This is called by GetPatientAppointments, so this test is somewhat redundant.
            // There is no MDWS command for it.

            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListOrders()
        {
            DateTime startDate = _siteParams.listOrderStartDate;
            DateTime endDate = _siteParams.listOrderEndDate;

            ViaVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA) as ViaVistAQuery;

            DateTime beforeVIA = DateTime.Now;

            List<Order> orders = viaQuery.GetResults(new ListOrdersCommand(viaQuery)
            {
                StartDate = startDate,
                EndDate = endDate,
                OrderableItemIens = _siteParams.orderableItems,
                PatientIen = null,
                IsAnticipated = false,
            });

            DateTime afterVIA = DateTime.Now;
            TimeSpan durationVIA = afterVIA - beforeVIA;

            foreach (Order o in orders)
            {
                if (o.OrderOrderableItemIds.Count() > 1)
                {
                    int j = 0;
                }
            }

            Assert.IsNotNull(orders);
        }

        [TestMethod]
        public void TestListOrderActions()
        {
            try
            {
                // main use case has null patient IEN

                DateTime startDate = new DateTime(2006, 1, 23);
                DateTime endDate = new DateTime(2006, 1, 28, 23, 59, 59);

                IEnumerable<string> orderIENs = new[] { "73", "75", "76", "18005", "18003", "14795", "13974", "13975", "14215", "14214" };
                // "ORDIEN","IENSB","SDATE","EDATE","VALUE","MAX","FROM"
                // <criteria>73,75,76,360,740^1336650^3020220^3020220^1^^</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderAction> callVIA = viaQuery.GetOrderActions(startDate, endDate, null, orderIENs);

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderAction> callMDWS = mdwsQuery.GetOrderActions(startDate, endDate, null, orderIENs);

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }
    }
}
