﻿using BMS.DataContracts;
using InfoWorld.HL7.ITS;
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization;
using System.Xml;

namespace BMS.UnitTesting
{
    internal static class Builders
    {
        const string domain = "InfoWorld";
        #region admission
        /// <summary>
        /// Builds an admission intent object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Admission</returns>
        internal static Admission BuildAdmissionIntent(int seed)
        {
            Admission admission = new Admission();
            admission.CanceledDate = DateTime.Now;
            admission.CreationDate = DateTime.Now;// -new TimeSpan(24, 0, 0);
            admission.FlowId = Guid.NewGuid();
            admission.Id = new II(domain, seed + "");
            //admission.Parent = new Act(){Id = new II(domain, "parentAdmission" + seed)};
            admission.Patient = (Patient)BuildPerson("patient", seed);

            return admission;
        }

        /// <summary>
        /// Builds an admission request object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Admission</returns>
        internal static Admission BuildAdmissionRequest(int seed)
        {
            Admission admission = new Admission();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), admission);
            admission.Id = new II(domain, seed + "");
            admission.SignedBy = BuildPerson("signedBy", seed);
            admission.ReleasedDate = DateTime.Now + new TimeSpan(24, 0, 0);
            admission.Provider = BuildPerson("provider", seed);

            return admission;
        }

        /// <summary>
        /// Builds an admission event object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Admission</returns>
        internal static Admission BuildAdmissionEvent(int seed)
        {
            Admission admission = new Admission();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), admission);
            admission.Id = new II(domain, seed + "");
            admission.Bed = BuildPlace("assignedBed", seed);
            admission.AdmissionDate = DateTime.Now + new TimeSpan(2, 0, 0, 0);
            admission.AttendingNurse = BuildPerson("nurse", seed);
            admission.Bed = BuildPlace("bed", seed);
            admission.Ward = BuildPlace("ward", seed);

            return admission;
        }
        #endregion
        #region discharge
        /// <summary>
        /// Builds a discharge intent object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Discharge</returns>
        internal static Discharge BuildDischargeIntent(int seed)
        {
            Discharge discharge = new Discharge();
            discharge.Bed = BuildPlace("bed", seed);
            discharge.CanceledDate = DateTime.Now;
            discharge.CreationDate = DateTime.Now - new TimeSpan(24, 0, 0);
            discharge.FlowId = Guid.NewGuid();
            discharge.Id = new II(domain, seed + "");
            discharge.OrderText = "orderText" + seed;
            //discharge.Parent = new Act() { Id = new II(domain, "parentDischarge" + seed) };
            discharge.Patient = (Patient)BuildPerson("patient", seed);
            discharge.Provider = BuildPerson("provider", seed);            
            discharge.Ward = BuildPlace("ward", seed);

            return discharge;
        }

        /// <summary>
        /// Builds a discharge request object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Discharge</returns>
        internal static Discharge BuildDischargeRequest(int seed)
        {
            Discharge discharge = new Discharge();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), discharge);
            discharge.Id = new II(domain, seed + "");
            discharge.ReleasedDate = DateTime.Now + new TimeSpan(24, 0, 1);
            discharge.SignedBy = BuildPerson("signedBy", seed);
            discharge.SignedDate = DateTime.Now + new TimeSpan(24, 0, 1);

            return discharge;
        }

        /// <summary>
        /// Builds a discharge event object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Discharge</returns>
        internal static Discharge BuildDischargeEvent(int seed)
        {
            Discharge discharge = new Discharge();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), discharge);
            discharge.Id = new II(domain, seed + "");
            discharge.DischargedDate = DateTime.Now + new TimeSpan(24, 0, 0);
            discharge.MovementType = BuildConcept("typeOfMovement", seed);
            discharge.EnteredBy = BuildPerson("orderBy", seed);

            return discharge;
        }
        #endregion
        #region transfer
        /// <summary>
        /// Builds a transfer intent object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Transfer</returns>
        internal static Transfer BuildTransferIntent(int seed)
        {
            Transfer transfer = new Transfer();
            transfer.CanceledDate = DateTime.Now + new TimeSpan(24, 0, 0);
            transfer.CurrentLocation = "currentLocation" + seed;
            transfer.Contract = BuildConcept("contract", seed);
            transfer.CreationDate = DateTime.Now - new TimeSpan(24, 0, 0);
            transfer.Diagnosis = "diagnosis" + seed;
            transfer.DischargeComment = "dischargeComment" + seed;
            transfer.Facility = BuildPlace("facility", seed);
            transfer.FlowId = Guid.NewGuid();
            transfer.Id = new II(domain, seed + "");
            //transfer.Parent = new Act() { Id = new II(domain, "parentTransfer" + seed) };
            transfer.Patient = (Patient)BuildPerson("patient", seed);            
            transfer.Era = BuildConcept("era", seed);
            transfer.Speciality = BuildConcept("specialty", seed);


            return transfer;
        }

        /// <summary>
        /// Builds a transfer request object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Transfer</returns>
        internal static Transfer BuildTransferRequest(int seed)
        {
            Transfer transfer = new Transfer();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), transfer);
            transfer.Id = new II(domain, seed + "");
            transfer.Disposition = BuildConcept("disposition", seed);
            transfer.AdmissionFacility = BuildPlace("admissionFacility", seed);
            transfer.Comment = "comment" + seed;
            transfer.DispositionDate = DateTime.Now + new TimeSpan(24, 0, 1);
            transfer.Era = BuildConcept("era", seed);
            transfer.Facility = BuildPlace("facility", seed);
            transfer.Contract = BuildConcept("contract", seed);
            transfer.Speciality = BuildConcept("specialty", seed);
            return transfer;
        }

        /// <summary>
        /// Builds a transfer event object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Transfer</returns>
        internal static Transfer BuildTransferEvent(int seed)
        {
            Transfer transfer = new Transfer();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), transfer);
            transfer.Id = new II(domain, seed + "");

            return transfer;
        }
        #endregion

        /// <summary>
        /// Builds a diversion object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Diversion</returns>
        internal static Diversion BuildDiversion(int seed)
        {
            Diversion diversion = new Diversion();
            diversion.Active = seed % 2 == 0 ? true : false;
            diversion.Comments = "comments" + seed;
            diversion.EndDate = DateTime.Now + new TimeSpan(24, 0, 0);
            diversion.Facility = BuildPlace("facility", seed);
            diversion.Id = new II(domain, "" + seed);
            diversion.Location = BuildConcept("location", seed);
            diversion.StartDate = DateTime.Now - new TimeSpan(24, 0, 0);

            return diversion;
        }

        /// <summary>
        /// Builds a bed cleaning operation object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>BedCleaningOperation</returns>
        internal static BedCleaningOperation BuildBedCleaningOperation(int seed)
        {
            BedCleaningOperation bedCleaning = new BedCleaningOperation();
            bedCleaning.Bed = BuildPlace("bed", seed);
            bedCleaning.CanceledDate = DateTime.Now + new TimeSpan(24, 0, 0);
            bedCleaning.CreationDate = DateTime.Now - new TimeSpan(24, 0, 0);
            bedCleaning.FlowId = Guid.NewGuid();
            bedCleaning.Id = new II(domain, "" + seed);
            //bedCleaning.Parent = new Act() { Id = new II(domain, "parentBedCleaningOperation" + seed) };
            bedCleaning.Patient = (Patient)BuildPerson("patient", seed);
            bedCleaning.TypeOfClean = BuildConcept("typeOfClean", seed);

            return bedCleaning;
        }

        /// <summary>
        /// Builds a bed out of service object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>BedOutOfServiceEvent</returns>
        internal static BedUnavailable BuildBedOutOfServiceEvent(int seed)
        {
            BedUnavailable bedOutOfService = new BedUnavailable();
            bedOutOfService.Bed = BuildPlace("bed", seed);
            bedOutOfService.CanceledDate = DateTime.Now + new TimeSpan(24, 0, 0);
            bedOutOfService.CreationDate = DateTime.Now - new TimeSpan(24, 0, 0);
            bedOutOfService.FlowId = Guid.NewGuid();
            bedOutOfService.Id = new II(domain, "" + seed);
            //bedOutOfService.Parent = new Act() { Id = new II(domain, "parentBedOutOfServiceEvent" + seed) };
            bedOutOfService.Patient = (Patient)BuildPerson("patient", seed);
            bedOutOfService.Reason = BuildConcept("reason", seed);
            bedOutOfService.Type = BuildConcept("type", seed);

            return bedOutOfService;
        }

        /// <summary>
        /// Builds a bed occupancy info object for inserting into the database and then filter the bed occupancy data.
        /// </summary>
        /// <param name="admissionEvent">The admission event object.</param>
        /// <param name="dischargeIntent">The discharge intent object.</param>
        /// <param name="bedOutOfServiceEvent">The bed out of service event object.</param>
        /// <param name="bedCleaningOperation">The bed cleaning operation object.</param>
        /// <returns> BedOccupancyInfo </returns>
        internal static BedOccupancyInfo BuildBedOccupancy(Admission admissionEvent, Discharge dischargeIntent, BedUnavailable bedOutOfServiceEvent, BedCleaningOperation bedCleaningOperation)
        {
            BedOccupancyInfo bedOccupancyInfo = new BedOccupancyInfo();            
            bedOccupancyInfo.Id = admissionEvent.Id;            
            bedOccupancyInfo.FlowId = admissionEvent.FlowId;
            bedOccupancyInfo.BedId = admissionEvent.Bed.Id;
            bedOccupancyInfo.WardId = admissionEvent.Ward.Id;

            return bedOccupancyInfo;
        }

        /// <summary>
        /// Builds a transfer info object for inserting into the database and then filter the transfer data.
        /// </summary>
        /// <param name="transferIntent">The transfer intent object.</param>
        /// <param name="transferRequest">The transfer request object.</param>
        /// <returns>TransferInfo</returns>
        internal static TransferInfo BuildTransferInfo(Transfer transferIntent, Transfer transferRequest)
        {
            TransferInfo transferInfo = new TransferInfo();
            transferInfo.AdmissionFacilityId = transferRequest.AdmissionFacility.Id;
            transferInfo.CanceledDate = transferIntent.CanceledDate;
            transferInfo.Comment = transferIntent.DischargeComment;
            transferInfo.CurrentLocation = transferIntent.CurrentLocation;
            transferInfo.Contract = transferIntent.Contract;
            transferInfo.CreationDate = transferIntent.CreationDate;
            transferInfo.Diagnosis = transferIntent.Diagnosis;
            transferInfo.DispositionDate = transferRequest.DispositionDate;
            transferInfo.Era = transferRequest.Era;
            transferInfo.FacilityId = transferIntent.Facility.Id;
            transferInfo.FlowId = transferIntent.FlowId;
            transferInfo.Id = transferIntent.Id;
            transferInfo.Parent = transferIntent.Parent;
            transferInfo.Patient = transferIntent.Patient;

            return transferInfo;
        }

        /// <summary>
        /// Builds a movement intent object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Movement</returns>
        internal static Movement BuildMovementIntent(int seed)
        {
            Movement movement = new Movement();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), movement);
            movement.Bed = BuildPlace("bed", seed);
            movement.MovementType = BuildConcept("movementType", seed);            
            movement.Ward = BuildPlace("ward", seed);

            return movement;
        }

        /// <summary>
        /// Builds a movement request object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Movement</returns>
        internal static Movement BuildMovementRequest(int seed)
        {
            Movement movement = new Movement();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), movement);
            movement.MovementType = BuildConcept("movementType", seed); 

            return movement;
        }

        /// <summary>
        /// Builds a movement event object used for insert and update operations.
        /// </summary>
        /// <param name="seed">The seed(random number for creating dummy data).</param>
        /// <returns>Movement</returns>
        internal static Movement BuildMovementEvent(int seed)
        {
            Movement movement = new Movement();
            BuildAct(DateTime.Now, DateTime.Now - new TimeSpan(24, 0, 0), (Patient)BuildPerson("patient", seed), movement);
            movement.MovementType = BuildConcept("movementType", seed);    

            return movement;
        }

        internal static Configuration BuildConfiguration(int seed, ConfigType configType, BasicType basicType)
        {
            Configuration config = new Configuration();
            config.ConfigType = configType;
            config.Domain = domain;
            if (configType != ConfigType.USR)
                config.UserName = string.Empty;
            else
                config.UserName = "user " + seed;
            config.SectionList = new Dictionary<string, List<Element>>();
            List<Element> elementList = new List<Element>();
            Random random = new Random();
            int rndNo = 0;
            if (basicType == BasicType.ST)
            {
                for (int i = 1; i <= seed; i++)
                {
                    rndNo = random.Next();
                    elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = new ST("value" + rndNo) });
                }
            }
            if (basicType == BasicType.INT)
            {
                for (int i = 1; i <= seed; i++)
                {
                    rndNo = random.Next();
                    elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = new INT(rndNo) });
                }
            }
            if (basicType == BasicType.BL)
            {
                rndNo = random.Next();
                elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = new BL(true) });
                rndNo = random.Next();
                elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = new BL(false) });
            }
            if (basicType == BasicType.TS)
            {
                rndNo = random.Next();
                TS ts = new TS();
                ts.dateTimeValue = DateTime.Now;
                ts.value = ts.dateTimeValue.ToString();
                elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = ts });
            }
            if (basicType == BasicType.REAL)
            {
                rndNo = random.Next();
                REAL real = new REAL();
                real.DecimaValue = rndNo;
                real.value = rndNo.ToString();
                elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key " + rndNo, Value = real });
            }
            if (basicType == BasicType.EVS)
            {
                CD concept = null;
                for (int i = 1; i <= seed; i++)
                {
                    rndNo = random.Next();
                    concept = BuildConcept("concept", rndNo);
                    elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key" + rndNo, Value = concept });
                }
            }
            if (basicType == BasicType.EIS)
            {
                EntityIdentification ei = null;
                for (int i = 1; i <= seed; i++)
                {
                    rndNo = random.Next();
                    ei = new EntityIdentification() { EntityIdentifier = rndNo.ToString(), EntityName = "name " + rndNo, EntityTypeIdentifier = "type" + rndNo };
                    elementList.Add(new Element() { FriendlyName = "description" + rndNo, Key = "config key" + rndNo, Value = ei });
                }
            }

            config.SectionList.Add("section" + seed, elementList);
            return config;
        }

        /// <summary>
        /// Gets the data contract XML.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <param name="value">The val.</param>
        /// <returns></returns>
        internal static string GetDataContractXml(Type type, object value)
        {
            if (type == null) { throw new ArgumentNullException("type"); }
            if (value == null) { throw new ArgumentNullException("value"); }

            string xml = null;

            DataContractSerializer dcs = new DataContractSerializer(type);
            StringWriter stringWriter = null;
            try
            {
                stringWriter = new StringWriter();
                {
                    using (XmlWriter xmlWriter = XmlWriter.Create(stringWriter, new XmlWriterSettings() { Indent = true }))
                    {
                        dcs.WriteObject(xmlWriter, value);
                        xmlWriter.Flush();
                        xml = stringWriter.ToString();
                    }
                }
            }
            finally
            {
                if (stringWriter != null)
                    stringWriter.Dispose();
            }

            return xml;
        }

        #region private methods

        private static Person BuildPerson(string name, int seed)
        {
            Person person = new Patient()
            {
                FirstName = "testFirs" + name + seed,
                LastName = "testLast" + name + seed,
                Gender = BuildConcept("gender", seed),
                Id = new II(domain, Guid.NewGuid().ToString()),
                MiddleName = "I" + seed,
                SSN = new II("SSN", "testSSN" + seed)
            };
            return person;
        }

        private static CD BuildConcept(string concept, int seed)
        {
            CD conceptDescriptor = new CD()
            {
                code = concept + "code" + seed,
                codeSystem = concept + "codeSystem" + seed,
                codeSystemName = concept + "codeSystemName" + seed,
                displayName = concept + "displayName" + seed
            };
            return conceptDescriptor;
        }

        private static Place BuildPlace(string placeName, int seed)
        {
            Place place = new Place()
            {
                Id = new II(domain, Guid.NewGuid().ToString()),
                Name = placeName + seed
            };
            return place;
        }

        private static void BuildAct(DateTime canceledDate, DateTime creationDate, Patient patient, Act act)
        {
            act.CanceledDate = canceledDate;
            act.CreationDate = creationDate;
            act.FlowId = Guid.NewGuid();
            act.Id = new II(domain, Guid.NewGuid().ToString());
            act.Parent = null;
            act.Patient = patient;
        }

        #endregion
    }
}
