﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using System.Text.RegularExpressions;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class AdminWaitingAreaController : BaseController
    {
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingAreaViewModel model = new WaitingAreaViewModel();
                model.LoggedUser = loggedUser;
                model.WaitingAreaList = Facade.FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.WaitingArea).Where(a => a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)).ToList();                
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Waiting Area controller - used for the Add button.
        /// </summary>
        /// <param name="input">The input(unavailable reason view model).</param>
        /// <returns>The Index view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(WaitingAreaViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = this.loggedUser;
                if (string.IsNullOrWhiteSpace(model.WaitingAreaName))
                {
                    model.Message = Strings.YourDidNotEnterAWaitingArea;
                }
                else
                {
                    string pattern = "[\\~#%&*{}/:<>?|\"-'^()]";
                    string replacement = " ";
                    Regex regEx = new Regex(pattern);
                    string code = Regex.Replace(regEx.Replace(model.WaitingAreaName, replacement), @"\s+", " ");
                    code = Constants.NATIONAL_VOCABULARY_CODE_PREFIX + code.Trim().Replace(" ", "_");
                    CD concept = new CD() { code = code.ToUpper(), codeSystem = Constants.WaitingAreaCodeSystem, codeSystemName = Constants.WaitingArea, displayName = model.WaitingAreaName.ToUpper() };
                    List<CD> vocabulary = FacadeManager.VocabularyInterface.GetWaitingAreas();
                    if (vocabulary.Where(a => a.code.Equals(concept.code, StringComparison.InvariantCultureIgnoreCase) || a.displayName.Equals(concept.displayName, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                        model.Message = Strings.WaitingAreaAlreadyExists;
                    else
                    {                        
                        FacadeManager.VocabularyInterface.InsertConcept(concept);
                        model.Message = model.WaitingAreaName.ToUpper() + " " + Strings.WaitingAreaSaved;
                    }
                }
                return View("AddAction", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show Delete view.
        /// </summary>
        /// <param name="location">The concept Code and the waiting area name.</param>
        /// <returns>Returns the Delete view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingAreaViewModel model = new WaitingAreaViewModel();
                model.LoggedUser = loggedUser;
                model.Code = QueryStrings["code"];
                CD cd = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea).Where(a => a.code.Equals(model.Code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                model.WaitingAreaName = cd.displayName;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show DeleteAction view.
        /// </summary>
        /// <param name="location">The concept Code.</param>
        /// <returns>Returns the DeleteAction view result.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult DeleteAction(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CDList vocabulary = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);
                CD concept = vocabulary.SingleOrDefault(x => x.code == QueryStrings["code"]);

                if (concept != null)
                    FacadeManager.VocabularyInterface.DeleteConcept(concept);

                ModelState.Clear();
                BaseViewModel baseViewModel = new BaseViewModel();
                baseViewModel.LoggedUser = this.loggedUser;
                return View("DeleteAction", baseViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the Edit view.
        /// </summary>
        /// <param name="id">The id of the waiting area object selected.</param>
        /// <returns>The Edit view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Edit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingAreaViewModel model = new WaitingAreaViewModel();
                model.LoggedUser = loggedUser;
                model.Code = QueryStrings["code"];
                CD cd = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea).Where(a => a.code.Equals(model.Code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                model.WaitingAreaName = cd.displayName;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Edit view - used for the Submit button.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns>
        /// The Edit Confirmation view.
        /// </returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Edit(WaitingAreaViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ModelState.Clear();
                if (string.IsNullOrWhiteSpace(model.WaitingAreaName))
                {
                    model.Message = Strings.YourDidNotEnterAWaitingArea;
                    model.DisplayButtonWhenMissingWaitingAreaText = Strings.DisplayBlock;
                    model.DisplayButtonWhenSuccessfullyEditAreaText = Strings.DisplayNone;
                }
                else
                {
                    List<CD> vocabulary = FacadeManager.VocabularyInterface.GetWaitingAreas();
                    CD concept = vocabulary.SingleOrDefault(x => x.code == model.Code);
                    model.Message = Strings.EditConfirmation;
                    model.DisplayButtonWhenMissingWaitingAreaText = Strings.DisplayNone;
                    model.DisplayButtonWhenSuccessfullyEditAreaText = Strings.DisplayBlock;
                    if (concept != null)
                    {
                        concept.displayName = model.WaitingAreaName.ToUpper().Trim();                        
                        //check if another CD with the same display name already exists
                        if (vocabulary.Where(a => a.displayName.Equals(concept.displayName, StringComparison.InvariantCultureIgnoreCase) &&
                                !a.code.Equals(concept.code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                        {
                            model.Message = Strings.WaitingAreaAlreadyExists;
                            model.DisplayButtonWhenMissingWaitingAreaText = Strings.DisplayBlock;
                            model.DisplayButtonWhenSuccessfullyEditAreaText = Strings.DisplayNone;
                        }
                        else
                            //do the actual update
                            FacadeManager.VocabularyInterface.UpdateConcept(concept);
                    }
                }
                return View("EditAction", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
