using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using System.ComponentModel.DataAnnotations;
using BMS.Web.Models.Shared;
using BMS.Utils;
using BMS.Web.Controllers.Shared;
using System.Security;
using CompareAttribute = System.ComponentModel.DataAnnotations.CompareAttribute;

namespace BMS.Web.Models
{
    /// <summary>
    /// Facility Edit Model class.
    /// </summary>
    public class AdminFacilityEditViewModel : BaseViewModel
    {
        private SecureString _emsPassword, _emsConfirmPassword, _kioskPassword, _kioskConfirmPassword;

        /// <summary>
        /// Gets or sets the list of facilities.
        /// </summary>
        /// <value>
        /// The facilities.
        /// </value>
        public IList<Facility> Facilities { get; set; }

        /// <summary>
        /// Gets or sets the internal id of the selected facility.
        /// </summary>
        /// <value>
        /// The id of the selected facility.
        /// </value>
        public string FacilityIdentifier { get; set; }

        /// <summary>
        /// Gets or sets the vista division list.
        /// </summary>
        /// <value>
        /// The vista division list.
        /// </value>
        public List<CD> VistaDivisionList { get; set; }

        /// <summary>
        /// Gets or sets the medical center ID.
        /// </summary>
        /// <value>
        /// The medical center ID.
        /// </value>
        public string MedicalCenterID { get; set; }

        public string FacilityShortName { get; set; }

        public IList<CheckItem<UserOperations>> UsersOperations { get; set; }

        [ValidateXss]
        public string FacilityCode { get; set; }

        [ValidateXss]
        public string FacilityName { get; set; }        

        /// <summary>
        /// Gets or sets the facility POC email.
        /// </summary>
        /// <value>
        /// The facility POC email.
        /// </value>
        [ValidateXss]
        [RegularExpression(Constants.EMAIL_GROUP_REGULAR_EXPRESSION, ErrorMessage = "Invalid email.")]
        public string FacilityPOCEmail { get; set; }        

        /// <summary>
        /// Gets or sets the facility POC telephone.
        /// </summary>
        /// <value>
        /// The facility POC telephone.
        /// </value>
        [ValidateXss]
        public string FacilityPOCTelephone { get; set; }

        /// <summary>
        /// Gets or sets the facility POC contact.
        /// </summary>
        /// <value>
        /// The facility POC contact.
        /// </value>
        [ValidateXss]
        public string FacilityPOCContact { get; set; }

        [ValidateXss]
        public string FacilityAddress1 { get; set; }

        [ValidateXss]
        public string FacilityAddress2 { get; set; }

        [ValidateXss]
        public string FacilityCity { get; set; }        

        public string FacilityState { get; set; }

        [ValidateXss]
        public string FacilityZip { get; set; }

        [Range(1, 23, ErrorMessage = "VISN Code must be a positive integer between 1 and 23.")]
        public string VISNCode { get; set; }

        [Range(1, 4, ErrorMessage = "Region Code must be a positive integer between 1 and 4.")]
        public string RegionCode { get; set; }

        [ValidateXss]
        public string WardPrefix { get; set; }

        [ValidateXss]
        public string ADTPrefix { get; set; }

        [ValidateXss]
        public string WardSuffix { get; set; }

        [ValidateXss]
        public string ADTSuffix { get; set; }

        [ValidateXss]
        public string EMSMailSender { get; set; }

        [ValidateXss]
        public string EventMailSender { get; set; }

        [ValidateXss]
        public string SiteAlias1 { get; set; }

        [ValidateXss]
        public string SiteAlias2 { get; set; }

        [ValidateXss]
        public string SiteAlias3 { get; set; }

        public CD IsAliveSite { get; set; }

        public DateTime? BPLastRanDate { get; set; }

        public IList<CD> States { get; set; }

        public IList<CD> StrictDecisions { get; set; }
        
        public IList<CD> TimeZones { get; set; }
        
        [Range(-1, 1, ErrorMessage = "Local Time Adjust must be a positive or negative integer between -1 and 1.")]
        public int LocalTimeAdjust { get; set; }

        /// <summary>
        /// Gets or sets the integrated site list.
        /// </summary>
        /// <value>
        /// The integrated site list.
        /// </value>
        public IList<FacilitySisterSitesGroupViewModel> IntegratedSiteList { get; set; }

        /// <summary>
        /// Gets or sets the sister sites groups.
        /// </summary>
        /// <value>
        /// The sister sites groups.
        /// </value>
        public IList<SisterSiteGroupViewModel> SisterSitesGroups { get; set; }

        /// <summary>
        /// Gets or sets the integrated sister site id.
        /// </summary>
        /// <value>
        /// The integrated sister site id.
        /// </value>
        public int IntegratedSisterSiteId { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether [show allowed access sites].
        /// </summary>
        /// <value>
        /// 	<c>true</c> if [show allowed access sites]; otherwise, <c>false</c>.
        /// </value>
        public bool ShowAllowedAccessSites { get; set; }

        [Required(ErrorMessage = "The FacilitySiteTimeZone field is required.")]
        public string FacilitySiteTimeZoneCode { get; set; }

        public string AutoRemovalWaitingListCode { get; set; }

        public string AutoPlacementTransferListCode { get; set; }

        public string IntegratedFacilityCode { get; set; }

        public string BMSServerTimeZone { get; set; }

        /// <summary>
        /// Gets or sets the name of the EMS user.
        /// </summary>
        /// <value>
        /// The name of the EMS user.
        /// </value>
        [ValidateXss]
        public string EMSUserName { get; set; }

        /// <summary>
        /// Gets or sets the EMS password.
        /// </summary>
        /// <value>
        /// The EMS password.
        /// </value>
        //public string EMSPassword { get; set; }
        public string EMSPassword
        {
            get
            {
                return SecureStringUtils.ConvertToUnsecureString(_emsPassword);
            }

            set
            {
                _emsPassword = SecureStringUtils.ConvertToSecureString(value);
            }
        }

        /// <summary>
        /// Gets or sets the EMS confirm password.
        /// </summary>
        /// <value>
        /// The EMS confirm password.
        /// </value>        
        [Compare("EMSPassword", ErrorMessage = "EMS Password and EMS Password Confirm do not match!")]
        //public string EMSConfirmPassword { get; set; }
        public string EMSConfirmPassword
        {
            get
            {
                return SecureStringUtils.ConvertToUnsecureString(_emsConfirmPassword);
            }

            set
            {
                _emsConfirmPassword = SecureStringUtils.ConvertToSecureString(value);
            }
        }

        /// <summary>
        /// Gets or sets the name of the Kiosk user.
        /// </summary>
        /// <value>
        /// The name of the Kiosk user.
        /// </value>
        [ValidateXss]
        public string KioskUserName { get; set; }

        /// <summary>
        /// Gets or sets the Kiosk password.
        /// </summary>
        /// <value>
        /// The Kiosk password.
        /// </value>
        //public string KioskPassword { get; set; }
        public string KioskPassword
        {
            get
            {
                return SecureStringUtils.ConvertToUnsecureString(_kioskPassword);
            }

            set
            {
                _kioskPassword = SecureStringUtils.ConvertToSecureString(value);
            }
        }
        

        /// <summary>
        /// Gets or sets the Kiosk confirm password.
        /// </summary>
        /// <value>
        /// The Kiosk confirm password.
        /// </value>        
        [Compare("KioskPassword", ErrorMessage = "Whiteboard Kiosk Password and Whiteboard Kiosk Password Confirm do not match!")]
        //public string KioskConfirmPassword { get; set; }
        public string KioskConfirmPassword
        {
            get
            {
                return SecureStringUtils.ConvertToUnsecureString(_kioskConfirmPassword);
            }

            set
            {
                _kioskConfirmPassword = SecureStringUtils.ConvertToSecureString(value);
            }
        }

        public string BtnSelectFacilityEdit { get; set; }

        public string BtnRemoveUsers { get; set; }
    }
}