using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Web;
using System.Web.Helpers;
using System.Web.Mvc;
using System.Web.Script.Serialization;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.Facade.Translators;
using BMS.ServicesWrapper.BMService;
using BMS.ServicesWrapper.WF;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Controllers.Shared;
using BMS.Web.Models;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;
using System.Data.SqlClient;
using System.Data;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Ward Whiteboard Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class WardWhiteboardController : BaseController
    {
        /// <summary>
        /// Default action for the WardWhiteboard controller.
        /// </summary>
        /// <param name="id">The ward id.</param>
        /// <param name="splitScreen">The splitScreen.</param>
        /// <param name="displayPTCode">The displayPTCode.</param>
        /// <param name="genderColorCode">The genderColorCode.</param>
        /// <returns>Returns the WardWhiteboard view result.</returns>  
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WardWhiteboardViewModel wardWhiteboardVM = new WardWhiteboardViewModel();
                FillViewModel(wardWhiteboardVM);

                //if we have an WardId set it to the ViewModel.
                if (!string.IsNullOrEmpty(QueryStrings["divisionId"]))
                    wardWhiteboardVM.DivisionExtension = QueryStrings["divisionId"];

                wardWhiteboardVM.DisplayPTCode = string.IsNullOrEmpty(QueryStrings["displayPTCode"]) ? wardWhiteboardVM.DisplayPTList.Where(a => a.code == Constants.PERSON_LAST_NAME).First().code : QueryStrings["displayPTCode"];
                wardWhiteboardVM.GenderColorCode = string.IsNullOrEmpty(QueryStrings["genderColorCode"]) ? wardWhiteboardVM.GenderColorList.First().code : QueryStrings["genderColorCode"];
                wardWhiteboardVM.SplitScreen = string.IsNullOrEmpty(QueryStrings["splitScreen"]) ? wardWhiteboardVM.StrictDecisionList.First().code : QueryStrings["splitScreen"];
                //obvious, but checks if the strict decision codes are ok.
                wardWhiteboardVM.DisplayFooterCensus = string.IsNullOrEmpty(QueryStrings["displayFooterCensus"]) ? wardWhiteboardVM.StrictDecisionList.First(cd => cd.code == Constants.No).code : QueryStrings["displayFooterCensus"];

                wardWhiteboardVM.DisplayPT = wardWhiteboardVM.DisplayPTList.First(cd => wardWhiteboardVM.DisplayPTCode == cd.code).displayName;
                wardWhiteboardVM.GenderColor = wardWhiteboardVM.GenderColorList.First(cd => wardWhiteboardVM.GenderColorCode == cd.code).displayName;
                wardWhiteboardVM.DisplayStaffAttendingSelected = QueryStrings["displayStaffAttending"];
                int scrollRate;
                if (Int32.TryParse(QueryStrings["scrollRate"], out scrollRate) && scrollRate > 0)
                {
                    wardWhiteboardVM.ScrollRateSelected = scrollRate;
                    wardWhiteboardVM.DisplayScrollSelected = Constants.Yes;
                }
                else
                {
                    wardWhiteboardVM.ScrollRateSelected = 0;
                    wardWhiteboardVM.DisplayScrollSelected = Constants.No;
                }
                wardWhiteboardVM.SortBySelected = QueryStrings["sortBy"];
                SetWardSelected(wardWhiteboardVM);

                if (wardWhiteboardVM.DivisionList == null || wardWhiteboardVM.DivisionList.Count == 0)
                    //there are no wards
                    return new EmptyResult();
                if (wardWhiteboardVM.SelectedDivision == null)
                {
                    wardWhiteboardVM.SelectedDivision = wardWhiteboardVM.DivisionList[0];
                    wardWhiteboardVM.DivisionExtension = wardWhiteboardVM.SelectedDivision.Id.extension;
                    wardWhiteboardVM.DivisionName = wardWhiteboardVM.SelectedDivision.Name;
                }

                //if we don't have a WardId we select the 1st one in the list
                if (string.IsNullOrEmpty(wardWhiteboardVM.DivisionExtension) && wardWhiteboardVM.DivisionList != null && wardWhiteboardVM.DivisionList.Count > 0)
                    wardWhiteboardVM.DivisionExtension = wardWhiteboardVM.DivisionList[0].Id.extension;

                return View(wardWhiteboardVM);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        [ActionName("Index")]
        [MultiButtonAttribute]
        public ActionResult ShowReport([Bind(Exclude = "RememberMe")] WardWhiteboardViewModel input)
        {
            return Redirect(GetWhiteboardReportUrl(input.DisplayPTCode, input.GenderColorCode, input.SortBySelected, input.DisplayStaffAttendingSelected, input.DivisionExtension, ViewBag.FacilityCode));
        }

        /// <summary>
        /// Post action for the WardWhiteboard controller.
        /// </summary>
        /// <param name="input">The input(ward whiteboard model).</param>
        /// <param name="id">The ward id.</param>
        /// <returns>Returns the WardWhiteboard view result.</returns>        
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        [ActionName("Index")]
        [MultiButtonAttribute]
        public ActionResult Submit([Bind(Exclude = "RememberMe")] WardWhiteboardViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return this.RedirectToAction<WardWhiteboardController>(act => act.WardWhiteBoard(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "scrollValue", "scrollDirection" }, new string[] { input.DivisionExtension, input.SplitScreen, input.DisplayPTCode, input.GenderColorCode, input.SortBySelected, input.DisplayFooterCensus, input.DisplayStaffAttendingSelected, input.ScrollRateSelected.ToString(), null, null }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        public ActionResult WardWhiteBoard(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WardWhiteboardViewModel wardWhiteboardVM;
                if (ConfigurationManager.AppSettings["UseDirectWhiteboard"] != null && ConfigurationManager.AppSettings["UseDirectWhiteboard"].ToString() == "true")
                    wardWhiteboardVM = FillWhiteboardContentDirect(QueryStrings["divisionId"], QueryStrings["splitScreen"], QueryStrings["displayPTCode"], QueryStrings["genderColorCode"], QueryStrings["sortBy"], QueryStrings["displayFooterCensus"], QueryStrings["displayStaffAttending"], QueryStrings["scrollRate"], QueryStrings["scrollValue"], QueryStrings["scrollDirection"]);
                else
                    wardWhiteboardVM = FillWhiteboardContent(QueryStrings["divisionId"], QueryStrings["splitScreen"], QueryStrings["displayPTCode"], QueryStrings["genderColorCode"], QueryStrings["sortBy"], QueryStrings["displayFooterCensus"], QueryStrings["displayStaffAttending"], QueryStrings["scrollRate"], QueryStrings["scrollValue"], QueryStrings["scrollDirection"]);

                if (wardWhiteboardVM == null)
                    return new EmptyResult();
                if (QueryStrings.ContainsKey("facilityCode"))
                    wardWhiteboardVM.FacilityCode = QueryStrings["facilityCode"];
                if (QueryStrings.ContainsKey("isFromKiosk"))
                    wardWhiteboardVM.IsFromWhiteboardKiosk = QueryStrings["isFromKiosk"];
                if (QueryStrings.ContainsKey("ajaxRefreshCount"))
                    wardWhiteboardVM.AjaxRefreshCount = QueryStrings["ajaxRefreshCount"];
                else
                    wardWhiteboardVM.AjaxRefreshCount = "0";
                return View(wardWhiteboardVM);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult WardWhiteBoard(WardWhiteboardViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(input.ButtonRefreshHidden))
                    return this.RedirectToAction<WardWhiteboardController>(act => act.WardWhiteBoard(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "scrollValue", "scrollDirection", "isFromKiosk", "facilityCode" }, new string[] { input.DivisionExtension, input.SplitScreen, input.DisplayPTCode, input.GenderColorCode, input.SortBySelected, input.DisplayFooterCensus, input.DisplayStaffAttendingSelected, input.ScrollRateSelected.ToString(), input.ActualScrollValue.ToString(), input.ActualScrollDirection.ToString(), input.IsFromWhiteboardKiosk, input.FacilityCode }, loggedUser.Salt)));
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult WardWhiteBoardAjax(string model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                JavaScriptSerializer jss = new JavaScriptSerializer();
                WardWhiteboardViewModel input = jss.Deserialize<WardWhiteboardViewModel>(model);

                WardWhiteboardViewModel wwm;
                if (ConfigurationManager.AppSettings["UseDirectWhiteboard"] != null && ConfigurationManager.AppSettings["UseDirectWhiteboard"].ToString() == "true")
                    wwm = FillWhiteboardContentDirect(input.DivisionExtension, input.SplitScreen, input.DisplayPTCode, input.GenderColorCode, input.SortBySelected, input.DisplayFooterCensus, input.DisplayStaffAttendingSelected, input.ScrollRateSelected.ToString(), input.ActualScrollValue.ToString(), input.ActualScrollDirection.ToString());
                else
                    wwm = FillWhiteboardContent(input.DivisionExtension, input.SplitScreen, input.DisplayPTCode, input.GenderColorCode, input.SortBySelected, input.DisplayFooterCensus, input.DisplayStaffAttendingSelected, input.ScrollRateSelected.ToString(), input.ActualScrollValue.ToString(), input.ActualScrollDirection.ToString());

                if (!string.IsNullOrEmpty(input.FacilityCode))
                    wwm.FacilityCode = input.FacilityCode;
                if (!string.IsNullOrEmpty(input.IsFromWhiteboardKiosk))
                    wwm.IsFromWhiteboardKiosk = input.IsFromWhiteboardKiosk;
                int ajaxRefreshCount = 0;
                if (!string.IsNullOrEmpty(input.AjaxRefreshCount) && int.TryParse(input.AjaxRefreshCount, out ajaxRefreshCount))
                    wwm.AjaxRefreshCount = (ajaxRefreshCount + 1).ToString();
                else
                    wwm.AjaxRefreshCount = "0";
                return View("WardWhiteBoard", wwm);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult WardWhiteBoardStaff(string model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                JavaScriptSerializer jss = new JavaScriptSerializer();
                EditStaffViewModel viewModel = jss.Deserialize<EditStaffViewModel>(model);
                if (!string.IsNullOrEmpty(viewModel.BedId) && viewModel.BedId != Guid.Empty.ToString())
                {
                    Bed bed = FacadeManager.BedInterface.GetBed(new II(this.loggedUser.Domain, viewModel.BedId), loggedUser.VistaSite.Id);
                    bed.NurseAssignment = viewModel.Staff;
                    FacadeManager.BedInterface.UpdateBed(bed);
                    Division division = null;
                    if (bed.CommentList != null && bed.CommentList.Count > 0)
                        division = FacadeManager.EntityInterface.GetDivisionById(bed.CommentList[0].Division.Id);
                    Patient patient = null;
                    if (!string.IsNullOrEmpty(viewModel.PatientId) && viewModel.PatientId != Guid.Empty.ToString())
                        patient = FacadeManager.EntityInterface.GetPatientById(new II(this.loggedUser.Domain, viewModel.PatientId), null);
                    FacadeManager.BedInterface.ManageBedStaff(bed, division, bed.NurseAssignment, patient, this.loggedUser.Facility);
                }
                if (viewModel.Staff == null)
                    return Json(string.Empty);
                AntiXssEncoder xss = new AntiXssEncoder();
                return Json(xss.Decode(viewModel.Staff));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Edits the bed information.
        /// </summary>
        /// <param name="rootBedId">The root of BedId.</param>
        /// <param name="extensionBedId">The extension of BedId.</param>
        /// <param name="ward">Ward.</param>
        /// <param name="splitScreen">Split screen.</param>
        /// <param name="displayPTCode">Display Patient code.</param>
        /// <param name="genderColorCode">Gender color code.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Edit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WhiteboardRoomViewModel whiteboardRM = new WhiteboardRoomViewModel();

                //fill whiteboard room edit model
                int scrollRate;
                Int32.TryParse(QueryStrings["scrollRate"], out scrollRate);
                FillEditModel(loggedUser.Domain, QueryStrings["bedId"], QueryStrings["ward"], QueryStrings["splitScreen"], QueryStrings["displayPTCode"], QueryStrings["genderColorCode"], QueryStrings["sortBy"], QueryStrings["displayFooterCensus"], QueryStrings["displayStaffAttending"], scrollRate, whiteboardRM, QueryStrings["patientId"], loggedUser.Domain, QueryStrings["flowId"], bool.Parse(QueryStrings["hasEvacuationIcon"]));

                return View(whiteboardRM);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Edits the bed information.
        /// </summary>
        /// <param name="rootBedId">The root of BedId.</param>
        /// <param name="extensionBedId">The extension of BedId.</param>
        /// <param name="ward">Ward.</param>
        /// <param name="splitScreen">Split screen.</param>
        /// <param name="displayPTCode">Display Patient code.</param>
        /// <param name="genderColorCode">Gender color code.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult BeginEdit(WhiteboardParametersViewModel whiteboardPM)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WhiteboardRoomViewModel whiteboardRM = new WhiteboardRoomViewModel();

                //fill whiteboard room edit model
                int scrollRate;
                Int32.TryParse(whiteboardPM.ScrollRate, out scrollRate);
                FillEditModel(loggedUser.Domain, whiteboardPM.BedId, whiteboardPM.Ward, whiteboardPM.SplitScreen, whiteboardPM.DisplayPTCode, whiteboardPM.GenderColorCode, whiteboardPM.SortBy, whiteboardPM.DisplayFooterCensus, whiteboardPM.DisplayStaffAttending, scrollRate, whiteboardRM, whiteboardPM.PatientId, loggedUser.Domain, whiteboardPM.FlowId, bool.Parse(whiteboardPM.HasEvacuationIcon));

                return View(whiteboardRM);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for Edit view.
        /// </summary>
        /// <param name="buttonClearAll">The button clear all.</param>
        /// <param name="whiteboardRM">The edit whiteboard room model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Edit(WhiteboardRoomViewModel whiteboardRM)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                whiteboardRM.LoggedUser = loggedUser;
                if (whiteboardRM.SetVisibilityForExpectedOrActualCompletedDate != null)
                {
                    whiteboardRM.Bed = FacadeManager.BedInterface.GetBed(new II() { extension = whiteboardRM.ExtensionBedId, root = whiteboardRM.RootBedId }, loggedUser.VistaSite.Id);
                    if (whiteboardRM.Bed != null)
                        whiteboardRM.Bed.NurseAssignment = whiteboardRM.NurAssignment;
                    whiteboardRM.ReasonList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.UnavailableReason);
                    whiteboardRM.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.StrictDecision);
                    whiteboardRM.LoggedUser = this.loggedUser;

                    //get icon association
                    Guid bedUid = Guid.Parse(whiteboardRM.ExtensionBedId);
                    Guid? patientUid = null;
                    if (!string.IsNullOrEmpty(whiteboardRM.ExtensionPatientId) && whiteboardRM.ExtensionPatientId != (default(Guid)).ToString())
                        patientUid = Guid.Parse(whiteboardRM.ExtensionPatientId);
                    Guid facilityUid = Guid.Parse(this.loggedUser.Facility.Id.extension);
                    Dictionary<string, List<IconAssociationInfo>> oldIconAssociationList = FacadeManager.IconInterface.FilterBedPatientIconAssociation(bedUid, patientUid, facilityUid);
                    if (oldIconAssociationList == null)
                        oldIconAssociationList = new Dictionary<string, List<IconAssociationInfo>>();

                    Dictionary<string, List<IconAssociationInfo>> newIconAssociationList = new Dictionary<string, List<IconAssociationInfo>>();

                    //update code
                    if (oldIconAssociationList.Count > 0)
                    {
                        newIconAssociationList.Add(Constants.STANDARD_ICON, CreateIconAssociationList(oldIconAssociationList[Constants.STANDARD_ICON], whiteboardRM.StandardIconAssociationList));
                        newIconAssociationList.Add(Constants.EMERGENCY_ICON, CreateIconAssociationList(oldIconAssociationList[Constants.EMERGENCY_ICON], whiteboardRM.EmergencyIconAssociationList));
                        newIconAssociationList.Add(Constants.SITE_CONFIGURABLE_ICON, CreateIconAssociationList(oldIconAssociationList[Constants.SITE_CONFIGURABLE_ICON], whiteboardRM.SiteConfigurationIconAssociationList));
                    }
                    whiteboardRM.IconsAssociationList = newIconAssociationList;

                    //if bed has unavailable reason add option to delete this reason
                    if (!string.IsNullOrEmpty(whiteboardRM.UnavailableReasonCode))
                        whiteboardRM.ReasonList.Insert(0, new CD() { code = Strings.DeleteThisReason, displayName = Strings.DeleteThisReason });

                    IList<Ward> wardList = FacadeManager.EntityInterface.GetWardsByDivisionId(new II() { extension = whiteboardRM.RealDivisionSelectedExtension, root = whiteboardRM.RealDivisionSelectedRoot });
                    whiteboardRM.WardList = new List<Ward>();
                    Ward ward = null;
                    foreach (Ward w in whiteboardRM.Bed.WardList)
                    {
                        ward = wardList.Where(a => a.Id.extension.Equals(w.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ward != null)
                            whiteboardRM.WardList.Add(ward);
                    }

                    if (GetTypeCode(whiteboardRM.SelectedReason) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                    {
                        whiteboardRM.IsExpectedCompletedDateVisible = true;
                        if (whiteboardRM.ExpectedCompletedDate.HasValue)
                        {
                            whiteboardRM.ExpectedCompletedDateHourSelected = (whiteboardRM.ExpectedCompletedDate.Value.Hour < 10) ? "0" + whiteboardRM.ExpectedCompletedDate.Value.Hour.ToString() : whiteboardRM.ExpectedCompletedDate.Value.Hour.ToString();
                            whiteboardRM.ExpectedCompletedDateMinuteSelected = (whiteboardRM.ExpectedCompletedDate.Value.Minute < 10) ? "0" + whiteboardRM.ExpectedCompletedDate.Value.Minute.ToString() : whiteboardRM.ExpectedCompletedDate.Value.Minute.ToString();
                        }
                        else
                        {
                            whiteboardRM.ExpectedCompletedDateHourSelected = "00";
                            whiteboardRM.ExpectedCompletedDateMinuteSelected = "00";
                        }
                    }
                    else
                    {
                        whiteboardRM.IsExpectedCompletedDateVisible = false;
                        whiteboardRM.ExpectedCompletedDateHourSelected = "00";
                        whiteboardRM.ExpectedCompletedDateMinuteSelected = "00";
                        if (GetTypeCode(whiteboardRM.UnavailableReasonCode) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE && (string.IsNullOrWhiteSpace(whiteboardRM.SelectedReason) || whiteboardRM.SelectedReason == Strings.DeleteThisReason))
                        {
                            whiteboardRM.IsActualCompletedDateVisible = true;
                            whiteboardRM.ActualCompletedDate = DateTimeNowOnCurrentFacility;
                            whiteboardRM.ActualCompletedDateHourSelected = (whiteboardRM.ActualCompletedDate.Hour < 10) ? "0" + whiteboardRM.ActualCompletedDate.Hour.ToString() : whiteboardRM.ActualCompletedDate.Hour.ToString();
                            whiteboardRM.ActualCompletedDateMinuteSelected = (whiteboardRM.ActualCompletedDate.Minute < 10) ? "0" + whiteboardRM.ActualCompletedDate.Minute.ToString() : whiteboardRM.ActualCompletedDate.Minute.ToString();
                        }
                        else
                            whiteboardRM.IsActualCompletedDateVisible = false;
                    }
                    whiteboardRM.BedClean = FacadeManager.BedInterface.GetBedCleanByBedId(new II(whiteboardRM.RootBedId, whiteboardRM.ExtensionBedId), loggedUser.Facility.Id);
                    whiteboardRM.EMSNotifyIcon = FacadeManager.IconInterface.GetIconByCode(Constants.EMS_NOTIFIED);
                    if (whiteboardRM.BedClean != null)
                    {
                        if (whiteboardRM.BedClean.CompletedDate.HasValue)
                            whiteboardRM.IsBedCleanCompleted = true;
                        else
                        {
                            if (!string.IsNullOrEmpty(whiteboardRM.BedClean.AcceptedBy))
                                whiteboardRM.IsBedCleanAccepted = true;
                            else
                                whiteboardRM.IsBedCleanRequested = true;
                        }
                        if (!whiteboardRM.IsBedCleanCompleted)
                        {
                            whiteboardRM.RequestCleaning = whiteboardRM.BedClean.TypeOfClean.code;
                            whiteboardRM.Ward = whiteboardRM.BedClean.Ward.Id.extension;
                            whiteboardRM.CleaningDate = (whiteboardRM.BedClean.RequestedDate.HasValue) ? whiteboardRM.BedClean.RequestedDate.Value : DateTimeNowOnCurrentFacility;
                            whiteboardRM.SpecialInstructions = whiteboardRM.BedClean.SpecialInstructions;
                            if (whiteboardRM.CleaningDate.Hour.ToString().Length == 1)
                                whiteboardRM.HourSelected = whiteboardRM.CleaningDate.Hour.ToString().Replace(whiteboardRM.CleaningDate.Hour.ToString(), "0" + whiteboardRM.CleaningDate.Hour.ToString());
                            else
                                whiteboardRM.HourSelected = whiteboardRM.CleaningDate.Hour.ToString();
                            if (whiteboardRM.CleaningDate.Minute.ToString().Length == 1)
                                whiteboardRM.MinuteSelected = whiteboardRM.CleaningDate.Minute.ToString().Replace(whiteboardRM.CleaningDate.Minute.ToString(), "0" + whiteboardRM.CleaningDate.Minute.ToString());
                            else
                                whiteboardRM.MinuteSelected = whiteboardRM.CleaningDate.Minute.ToString();
                        }
                    }
                    else
                    {
                        whiteboardRM.IsBedCleanRequested = false;
                        whiteboardRM.IsBedCleanAccepted = false;
                    }
                    whiteboardRM.BedStaffHistoryList = FacadeManager.BedInterface.GetBedStaffHistory(new II(whiteboardRM.RootBedId, whiteboardRM.ExtensionBedId), this.loggedUser.Facility.Id);
                    FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);
                    if (settings != null && settings.FacilitySiteTimeZone != null)
                        whiteboardRM.FacilityTimeZone = settings.FacilitySiteTimeZone.code;

                    List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(this.loggedUser.Domain, this.loggedUser.Facility.Id).ToList();
                    if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    {
                        whiteboardRM.EMSModuleInactive = true;
                        whiteboardRM.DisplayEMSModuleInactive = @"style='display:none;'";
                    }
                    else
                    {
                        whiteboardRM.EMSModuleInactive = false;
                        whiteboardRM.DisplayEMSModuleInactive = string.Empty;
                    }

                    return View(whiteboardRM);
                }
                else
                {

                    //display ward whiteboard home page
                    if (!string.IsNullOrEmpty(whiteboardRM.NeverMindButton))
                        return this.RedirectToAction<WardWhiteboardController>(act => act.WardWhiteBoard(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "scrollValue", "scrollDirection" }, new string[] { whiteboardRM.WardParameter, whiteboardRM.SplitScreenParameter, whiteboardRM.DisplayPTCodeParameter, whiteboardRM.GenderColorCodeParameter, whiteboardRM.SortByParameter, whiteboardRM.DisplayFooterCensus, whiteboardRM.DisplayStaffAttendingParameter, whiteboardRM.ScrollRateParameter.ToString(), null, null }, loggedUser.Salt)));

                    Bed bed = FacadeManager.BedInterface.GetBed(whiteboardRM.Bed.Id, loggedUser.VistaSite.Id);
                    if (whiteboardRM.Bed != null)
                        whiteboardRM.Bed.NurseAssignment = whiteboardRM.NurAssignment;
                    DateTime dateNow = DateTimeNowOnCurrentFacility;
                    bool isFirstComment = true;
                    bool isChangedComment = false;
                    string lastComment = null;

                    BedComment bedComment = bed.CommentList.Where(a => a.Division.Id.root.Equals(whiteboardRM.RealDivisionSelectedRoot, StringComparison.InvariantCultureIgnoreCase) && a.Division.Id.extension.Equals(whiteboardRM.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (bedComment != null && !String.IsNullOrEmpty(bedComment.Comment))
                    {
                        lastComment = bed.CommentList.Where(a => a.Division.Id.root.Equals(whiteboardRM.RealDivisionSelectedRoot, StringComparison.InvariantCultureIgnoreCase) && a.Division.Id.extension.Equals(whiteboardRM.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Comment;
                        if (!string.IsNullOrWhiteSpace(lastComment))
                            isFirstComment = false;
                    }

                    //verify if comment changed
                    if (whiteboardRM.Comments != null && lastComment != whiteboardRM.Comments)
                        isChangedComment = true;

                    II patientId = null;
                    if (!string.IsNullOrEmpty(whiteboardRM.ExtensionPatientId) && !string.IsNullOrEmpty(whiteboardRM.RootPatientId))
                        patientId = new II { extension = whiteboardRM.ExtensionPatientId, root = whiteboardRM.RootPatientId };

                    IconAssociation iconAssociation = new IconAssociation();
                    if (patientId != null)
                        iconAssociation.Patient = new Patient() { Id = patientId };
                    else
                        iconAssociation.Patient = null;
                    iconAssociation.Bed = bed;
                    iconAssociation.Facility = this.loggedUser.Facility;

                    if (!string.IsNullOrEmpty(whiteboardRM.ClearAllButton))
                    {
                        whiteboardRM.Bed.Reason = null;
                        whiteboardRM.SelectedReason = string.Empty;
                        whiteboardRM.Comments = string.Empty;
                        whiteboardRM.Bed.NurseAssignment = null;
                        iconAssociation.ListIconFlagFacilityIdsOfBed = null;
                        iconAssociation.ListIconFlagFacilityIdsOfPatient = null;
                        iconAssociation.ListIconFlagIdsOfBed = null;
                        iconAssociation.ListIconFlagIdsOfPatient = null;
                        isChangedComment = true;
                        whiteboardRM.HasEvacuationIcon = false;
                    }
                    else
                    {
                        if (iconAssociation.ListIconFlagFacilityIdsOfBed == null)
                            iconAssociation.ListIconFlagFacilityIdsOfBed = new List<int>();
                        if (iconAssociation.ListIconFlagIdsOfBed == null)
                            iconAssociation.ListIconFlagIdsOfBed = new List<int>();
                        if (whiteboardRM.SiteConfigurationIconAssociationList != null)
                            iconAssociation.ListIconFlagFacilityIdsOfBed.AddRange(whiteboardRM.SiteConfigurationIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'R' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                        if (whiteboardRM.StandardIconAssociationList != null)
                            iconAssociation.ListIconFlagIdsOfBed.AddRange(whiteboardRM.StandardIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'R' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                        if (whiteboardRM.EmergencyIconAssociationList != null)
                            iconAssociation.ListIconFlagIdsOfBed.AddRange(whiteboardRM.EmergencyIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'R' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                        if (iconAssociation.Patient != null)
                        {
                            if (iconAssociation.ListIconFlagFacilityIdsOfPatient == null)
                                iconAssociation.ListIconFlagFacilityIdsOfPatient = new List<int>();
                            if (iconAssociation.ListIconFlagIdsOfPatient == null)
                                iconAssociation.ListIconFlagIdsOfPatient = new List<int>();
                            if (whiteboardRM.SiteConfigurationIconAssociationList != null)
                                iconAssociation.ListIconFlagFacilityIdsOfPatient.AddRange(whiteboardRM.SiteConfigurationIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                            if (whiteboardRM.StandardIconAssociationList != null)
                                iconAssociation.ListIconFlagIdsOfPatient.AddRange(whiteboardRM.StandardIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                            if (whiteboardRM.EmergencyIconAssociationList != null)
                                iconAssociation.ListIconFlagIdsOfPatient.AddRange(whiteboardRM.EmergencyIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes).Select(a => a.Icon.IconId));
                        }
                    }

                    // check if any of the associated icons are flagged isolation and add the isolation icon if they are
                    bool needsIsolation = false;
                    if (whiteboardRM.SiteConfigurationIconAssociationList != null && whiteboardRM.SiteConfigurationIconAssociationList.Where(a => a.Icon != null && a.Icon.IsIsolation).Count() > 0)
                        needsIsolation = true;
                    if (whiteboardRM.StandardIconAssociationList != null && whiteboardRM.StandardIconAssociationList.Where(a => a.Icon != null && a.Icon.IsIsolation).Count() > 0)
                        needsIsolation = true;
                    if (whiteboardRM.EmergencyIconAssociationList != null && whiteboardRM.EmergencyIconAssociationList.Where(a => a.Icon != null && a.Icon.IsIsolation).Count() > 0)
                        needsIsolation = true;

                    if (needsIsolation && !iconAssociation.ListIconFlagIdsOfBed.Contains(7))
                        iconAssociation.ListIconFlagIdsOfBed.Add(7);

                    //get new reason and type
                    CDWithProperties reason = null;
                    CD type = null;
                    bool isReasonAvailable = true;
                    GetReason(whiteboardRM.SelectedReason, out reason, out type, out isReasonAvailable);
                    whiteboardRM.IsSelectedReasonAvailable = isReasonAvailable;
                    //if reason is still available save items
                    if (whiteboardRM.IsSelectedReasonAvailable)
                    {
                        // save bed icon association
                        FacadeManager.IconInterface.InsertIconAssociation(iconAssociation);

                        BedUnavailable bedUnavailable = SaveBedUnavailable(whiteboardRM, bed, dateNow, isChangedComment, reason, type);
                        Patient patient = null;
                        if (bedUnavailable != null && bedUnavailable.Patient != null)
                            patient = bedUnavailable.Patient;
                        else
                        {
                            if (patientId != null && !string.IsNullOrEmpty(patientId.root) && !string.IsNullOrEmpty(patientId.extension) && !patientId.extension.Equals(Guid.Empty.ToString()))
                                patient = FacadeManager.EntityInterface.GetPatientById(patientId, null);
                            else
                                patient = null;
                        }
                        if (whiteboardRM.IsEvacuationFacility && patient != null && string.IsNullOrEmpty(whiteboardRM.ClearAllButton))
                        {
                            //verify if patient has evcautation icon
                            bool hasEvacuationIcon = false;
                            if (whiteboardRM.SiteConfigurationIconAssociationList != null && whiteboardRM.SiteConfigurationIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes && a.Icon.IconName.ToLower().Contains(Constants.EVACUATION.ToLower())).FirstOrDefault() != null)
                                hasEvacuationIcon = true;
                            if (whiteboardRM.StandardIconAssociationList != null && whiteboardRM.StandardIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes && a.Icon.IconName.ToLower().Contains(Constants.EVACUATION.ToLower())).FirstOrDefault() != null)
                                hasEvacuationIcon = true;
                            if (whiteboardRM.EmergencyIconAssociationList != null && whiteboardRM.EmergencyIconAssociationList.Where(a => a.Icon.PatientOrRoomBed == 'P' && a.Code == Constants.Yes && a.Icon.IconName.ToLower().Contains(Constants.EVACUATION.ToLower())).FirstOrDefault() != null)
                                hasEvacuationIcon = true;
                            //add patient into waiting list
                            if (hasEvacuationIcon && !whiteboardRM.HasEvacuationIcon)
                            {
                                whiteboardRM.HasEvacuationIcon = true;
                                CD waitingArea = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea).Where(a => a.code.Equals(Constants.WAITING_AREA_EVACUATION_CODE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                CD evacDispositionStatus = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.EvacDispositionStatus).Where(a => a.code.Equals(Constants.EVAC_DISPOSITION_STATUS_EVACUATE_CODE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                CD evacTransportationProvider = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TransportationProvider).Where(a => a.code.Equals(Constants.EVAC_TRANSPORTATION_PROVIDER_VA_CODE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                AddPatientIntoWaitingList(patient, whiteboardRM.FlowId.Value, waitingArea, evacDispositionStatus, evacTransportationProvider);
                            }
                            else
                            {
                                if (whiteboardRM.HasEvacuationIcon)
                                    whiteboardRM.HasEvacuationIcon = false;
                            }

                        }
                        SaveBed(whiteboardRM, bed);

                        if (isChangedComment || !whiteboardRM.BedUnavailableHistory.IsChangedOnlyComment)
                        {
                            //insert into history table
                            if (bedUnavailable != null)
                            {
                                if (!whiteboardRM.BedUnavailableHistory.EditedDate.HasValue && !whiteboardRM.BedUnavailableHistory.CanceledDate.HasValue)
                                {
                                    whiteboardRM.BedUnavailableHistory.EditedDate = dateNow;
                                    whiteboardRM.BedUnavailableHistory.EditedBy = this.loggedUser.UserName;
                                }
                            }
                            else
                            {
                                whiteboardRM.BedUnavailableHistory.Bed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null;
                                whiteboardRM.BedUnavailableHistory.CanceledBy = null;
                                whiteboardRM.BedUnavailableHistory.CanceledDate = null;
                                whiteboardRM.BedUnavailableHistory.CreatedBy = this.loggedUser.UserName;
                                whiteboardRM.BedUnavailableHistory.CreationDate = dateNow;
                                if (isFirstComment)
                                {
                                    whiteboardRM.BedUnavailableHistory.EditedBy = null;
                                    whiteboardRM.BedUnavailableHistory.EditedDate = null;
                                }
                                else
                                {
                                    whiteboardRM.BedUnavailableHistory.EditedBy = this.loggedUser.UserName;
                                    whiteboardRM.BedUnavailableHistory.EditedDate = dateNow;
                                }
                                whiteboardRM.BedUnavailableHistory.Id = new II(this.loggedUser.Domain, null);
                                whiteboardRM.BedUnavailableHistory.Parent = null;
                                whiteboardRM.BedUnavailableHistory.Patient = null;
                                whiteboardRM.BedUnavailableHistory.Reason = null;
                                whiteboardRM.BedUnavailableHistory.Type = null;
                                whiteboardRM.BedUnavailableHistory.VistaSite = this.loggedUser.VistaSite;
                            }

                            whiteboardRM.BedUnavailableHistory.Comment = whiteboardRM.Comments;
                            whiteboardRM.BedUnavailableHistory.Division = new Division() { Id = new II(whiteboardRM.RealDivisionSelectedRoot, whiteboardRM.RealDivisionSelectedExtension) };
                            if (!string.IsNullOrEmpty(whiteboardRM.ClearAllButton))
                                whiteboardRM.BedUnavailableHistory.IsClearAll = true;
                            foreach (BedComment bc in bed.CommentList)
                            {
                                if (whiteboardRM.BedUnavailableHistory.DivisionList == null)
                                    whiteboardRM.BedUnavailableHistory.DivisionList = new List<Division>();
                                if (whiteboardRM.BedUnavailableHistory.DivisionList.Where(a => a.Id.root.Equals(bc.Division.Id.root, StringComparison.InvariantCultureIgnoreCase) && a.Id.extension.Equals(bc.Division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                                    whiteboardRM.BedUnavailableHistory.DivisionList.Add(bc.Division);
                            }
                            FacadeManager.BedInterface.InsertBedUnavailableHistory(whiteboardRM.BedUnavailableHistory, this.loggedUser.Facility.Id);
                        }
                    }
                    else
                    {
                        //display message
                        return this.RedirectToAction<WardWhiteboardController>(act => act.NotifyChange(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "wardName", "bedName", "bedId", "displayFooterCensus", "displayStaffAttending", "scrollRate", "flowId", "hasEvacuationIcon", "isReasonAvailable" }, new string[] { whiteboardRM.WardParameter, whiteboardRM.SplitScreenParameter, whiteboardRM.DisplayPTCodeParameter, whiteboardRM.GenderColorCodeParameter, whiteboardRM.SortByParameter, whiteboardRM.WardName, whiteboardRM.Bed.Name, whiteboardRM.ExtensionBedId, whiteboardRM.DisplayFooterCensus, whiteboardRM.DisplayStaffAttendingParameter, whiteboardRM.ScrollRateParameter.ToString(), whiteboardRM.FlowId == null ? string.Empty : whiteboardRM.FlowId.Value.ToString(), whiteboardRM.HasEvacuationIcon.ToString(), whiteboardRM.IsSelectedReasonAvailable.ToString() }, loggedUser.Salt)));
                    }

                    //display clear all view
                    if (!string.IsNullOrEmpty(whiteboardRM.ClearAllButton))
                        return this.RedirectToAction<WardWhiteboardController>(act => act.ClearAll(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "wardName", "bedName", "bedId", "displayStaffAttending", "scrollRate", "patientId", "flowId", "hasEvacuationIcon" }, new string[] { whiteboardRM.WardParameter, whiteboardRM.SplitScreenParameter, whiteboardRM.DisplayPTCodeParameter, whiteboardRM.GenderColorCodeParameter, whiteboardRM.SortByParameter, whiteboardRM.DisplayFooterCensus, whiteboardRM.WardName, whiteboardRM.Bed.Name, whiteboardRM.ExtensionBedId, whiteboardRM.DisplayStaffAttendingParameter, whiteboardRM.ScrollRateParameter.ToString(), whiteboardRM.ExtensionPatientId, whiteboardRM.FlowId == null ? string.Empty : whiteboardRM.FlowId.Value.ToString(), whiteboardRM.HasEvacuationIcon.ToString() }, loggedUser.Salt)));

                    try
                    {
                        if (!whiteboardRM.RequestCleaning.Equals(Strings.No))
                        {
                            if (!whiteboardRM.IsBedCleanAccepted)
                                SaveManualCleaningRequest(whiteboardRM, bed, dateNow);
                        }
                        TempData["Msj"] = Strings.YouHaveSavedData;
                    }
                    catch (FaultException<GenericWFServiceFault>)
                    {
                        TempData["Msj"] = Strings.ErrorBedCleanRequest;
                    }
                    //display NotifyChange page
                    return this.RedirectToAction<WardWhiteboardController>(act => act.NotifyChange(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "wardName", "bedName", "displayFooterCensus", "displayStaffAttending", "scrollRate", "isReasonAvailable" }, new string[] { whiteboardRM.WardParameter, whiteboardRM.SplitScreenParameter, whiteboardRM.DisplayPTCodeParameter, whiteboardRM.GenderColorCodeParameter, whiteboardRM.SortByParameter, whiteboardRM.WardName, whiteboardRM.Bed.Name, whiteboardRM.DisplayFooterCensus, whiteboardRM.DisplayStaffAttendingParameter, whiteboardRM.ScrollRateParameter.ToString(), whiteboardRM.IsSelectedReasonAvailable.ToString() }, loggedUser.Salt)));
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Clears all.
        /// </summary>
        /// <param name="id">The id.</param>
        /// <param name="splitScreen">The split screen.</param>
        /// <param name="displayPTCode">The display PT code.</param>
        /// <param name="genderColorCode">The gender color code.</param>
        /// <param name="wardName">Name of the ward.</param>
        /// <param name="bedName">Name of the bed.</param>
        /// <param name="rootBedId">The root bed id.</param>
        /// <param name="extensionBedId">The extension bed id.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ClearAll(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WhiteboardClearAllViewModel model = new WhiteboardClearAllViewModel();
                model.WardParameter = QueryStrings["divisionId"];
                model.SplitScreenParameter = QueryStrings["splitScreen"];
                model.DisplayPTCodeParameter = QueryStrings["displayPTCode"];
                model.GenderColorCodeParameter = QueryStrings["genderColorCode"];
                model.WardName = QueryStrings["wardName"];
                model.BedName = QueryStrings["bedName"];
                model.RootBedId = loggedUser.Domain;
                model.ExtensionBedId = QueryStrings["bedId"];
                model.SortByParameter = QueryStrings["sortBy"];
                model.DisplayFooterCensus = QueryStrings["displayFooterCensus"];
                model.DisplayStaffAttendingParameter = QueryStrings["displayStaffAttending"];
                int scrollRate;
                Int32.TryParse(QueryStrings["scrollRate"], out scrollRate);
                model.ScrollRateParameter = scrollRate;
                model.RootPatientId = loggedUser.Domain;
                model.ExtensionPatientId = QueryStrings["patientId"];
                model.FlowId = QueryStrings["flowId"];
                model.HasEvacuationIcon = bool.Parse(QueryStrings["hasEvacuationIcon"]);
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for Clear all view.
        /// </summary>
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="btnReturnToWardWhiteboardPage">The button return to ward whiteboard page.</param>
        /// <returns></returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ClearAll(WhiteboardClearAllViewModel model, string btnReturnToWardWhiteboardPage)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(btnReturnToWardWhiteboardPage))
                    //return to ward whiteboard page
                    return this.RedirectToAction<WardWhiteboardController>(act => act.WardWhiteBoard(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "scrollValue", "scrollDirection" }, new string[] { model.WardParameter, model.SplitScreenParameter, model.DisplayPTCodeParameter, model.GenderColorCodeParameter, model.SortByParameter, model.DisplayFooterCensus, model.DisplayStaffAttendingParameter, model.ScrollRateParameter.ToString(), null, null }, loggedUser.Salt)));
                else
                    // return to edit bed unavailable page
                    //return this.RedirectToAction<WardWhiteboardController>(act => act.Edit(EncryptQueryString(new string[] { "bedId", "ward", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "patientId", "flowId", "hasEvacuationIcon" }, new string[] { model.ExtensionBedId, model.WardParameter, model.SplitScreenParameter, model.DisplayPTCodeParameter, model.GenderColorCodeParameter, model.SortByParameter, model.DisplayFooterCensus, model.DisplayStaffAttendingParameter, model.ScrollRateParameter.ToString(), model.ExtensionPatientId, model.FlowId, model.HasEvacuationIcon.ToString() }, loggedUser.Salt)));
                    return this.RedirectToAction<WardWhiteboardController>(act => act.BeginEdit(new WhiteboardParametersViewModel { BedId = model.ExtensionBedId, Ward = model.WardParameter, SplitScreen = model.SplitScreenParameter, DisplayPTCode = model.DisplayPTCodeParameter, GenderColorCode = model.GenderColorCodeParameter, SortBy = model.SortByParameter, DisplayFooterCensus = model.DisplayFooterCensus, DisplayStaffAttending = model.DisplayStaffAttendingParameter, ScrollRate = model.ScrollRateParameter.ToString(), PatientId = model.ExtensionPatientId, FlowId = model.FlowId, HasEvacuationIcon = model.HasEvacuationIcon.ToString() }));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        public ActionResult NotifyChange(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WhiteboardRoomViewModel model = new WhiteboardRoomViewModel();
                model.WardParameter = QueryStrings["divisionId"];
                model.SplitScreenParameter = QueryStrings["splitScreen"];
                model.DisplayPTCodeParameter = QueryStrings["displayPTCode"];
                model.GenderColorCodeParameter = QueryStrings["genderColorCode"];
                model.WardName = QueryStrings["wardName"];
                model.SortByParameter = QueryStrings["sortBy"];
                model.Bed = new Bed() { Name = QueryStrings["bedName"] };
                model.DisplayFooterCensus = QueryStrings["displayFooterCensus"];
                model.DisplayStaffAttendingParameter = QueryStrings["displayStaffAttending"];
                int scrollRate;
                Int32.TryParse(QueryStrings["scrollRate"], out scrollRate);
                model.ScrollRateParameter = scrollRate;
                model.LoggedUser = loggedUser;
                model.IsSelectedReasonAvailable = bool.Parse(QueryStrings["isReasonAvailable"]);
                if (QueryStrings.ContainsKey("flowId"))
                    if (!string.IsNullOrEmpty(QueryStrings["flowId"]))
                        model.FlowId = Guid.Parse(QueryStrings["flowId"]);
                if (QueryStrings.ContainsKey("hasEvacuationIcon"))
                    model.HasEvacuationIcon = bool.Parse(QueryStrings["hasEvacuationIcon"]);
                if (QueryStrings.ContainsKey("bedId"))
                    model.ExtensionBedId = QueryStrings["bedId"];
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult NotifyChange(WhiteboardRoomViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (input.IsSelectedReasonAvailable)
                    return this.RedirectToAction<WardWhiteboardController>(act => act.WardWhiteBoard(EncryptQueryString(new string[] { "divisionId", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "scrollValue", "scrollDirection" }, new string[] { input.WardParameter, input.SplitScreenParameter, input.DisplayPTCodeParameter, input.GenderColorCodeParameter, input.SortByParameter, input.DisplayFooterCensus, input.DisplayStaffAttendingParameter, input.ScrollRateParameter.ToString(), null, null }, loggedUser.Salt)));
                else
                    //return this.RedirectToAction<WardWhiteboardController>(act => act.Edit(EncryptQueryString(new string[] { "bedId", "ward", "splitScreen", "displayPTCode", "genderColorCode", "sortBy", "displayFooterCensus", "displayStaffAttending", "scrollRate", "patientId", "flowId", "hasEvacuationIcon" }, new string[] { input.ExtensionBedId, input.WardParameter, input.SplitScreenParameter, input.DisplayPTCodeParameter, input.GenderColorCodeParameter, input.SortByParameter, input.DisplayFooterCensus, input.DisplayStaffAttendingParameter, input.ScrollRateParameter.ToString(), input.ExtensionPatientId, input.FlowId.HasValue ? input.FlowId.Value.ToString() : string.Empty, input.HasEvacuationIcon.ToString() }, loggedUser.Salt)));
                    return this.RedirectToAction<WardWhiteboardController>(act => act.BeginEdit(new WhiteboardParametersViewModel { BedId = input.ExtensionBedId, Ward = input.WardParameter, SplitScreen = input.SplitScreenParameter, DisplayPTCode = input.DisplayPTCodeParameter, GenderColorCode = input.GenderColorCodeParameter, SortBy = input.SortByParameter, DisplayFooterCensus = input.DisplayFooterCensus, DisplayStaffAttending = input.DisplayStaffAttendingParameter, ScrollRate = input.ScrollRateParameter.ToString(), PatientId = input.ExtensionPatientId, FlowId = input.FlowId.HasValue ? input.FlowId.Value.ToString() : string.Empty, HasEvacuationIcon = input.HasEvacuationIcon.ToString() }));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirect to the EditPT view.
        /// </summary>
        /// <returns>The EditPT view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult EditPT(string p)
        //public ActionResult EditPT(string patientId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string patientId = QueryStrings["patientId"];
                Patient patient = FacadeManager.EntityInterface.GetPatientById(new II(loggedUser.Domain, patientId), null);
                //Patient patient = FacadeManager.EntityInterface.GetPatientById(new II(loggedUser.Domain, patientId), null);
                IReportInfo report = FacadeManager.ReportsInterface.GetPatientInquiryReport(FullUserName);
                report = report.Clone();
                if (report.Parameters == null)
                    report.Parameters = new Dictionary<string, string>();
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, string.Empty);
                //report.Parameters.Add(Constants.REP_EDIT_PT_IEN, patient.Ien.ToString());
                //report.Parameters.Add(Constants.REP_EDIT_PT_IDENTIFIER, QueryStrings["patientId"]);
                report.Parameters.Add(Constants.REP_EDIT_PT_IDENTIFIER, patientId);
                report.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION, FacadeManager.UserInterface.GetProfile().Facility.Id.extension);
                report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, string.Empty);
                report.Parameters.Add(Constants.REPORT_TITLE, Strings.BedControlPtInquiry);
                report.Parameters.Add(Constants.REP_TIME_ZONE_MINUTES_OFFSET, RepTimeZoneMinutesOffset);
                List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(this.loggedUser.Domain, this.loggedUser.Facility.Id).ToList();
                if (flags.Where(a => a.Code.Equals(Constants.DISCH_ORD, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_DISCH_ORDER_ACTIVE, Constants.No);
                else
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_DISCH_ORDER_ACTIVE, Constants.Yes);
                if (flags.Where(a => a.Code.Equals(Constants.ANTIC_DISCH, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_ANTICIPATED_DISCH_ORDER_ACTIVE, Constants.No);
                else
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_ANTICIPATED_DISCH_ORDER_ACTIVE, Constants.Yes);
                if (flags.Where(a => a.Code.Equals(Constants.DISCHARGE_APPOINTMENT, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_DISCH_APPOINTMENT_ACTIVE, Constants.No);
                else
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_DISCH_APPOINTMENT_ACTIVE, Constants.Yes);
                if (flags.Where(a => a.Code.Equals(Constants.TRSF_ORD, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_TRANSFER_ORDER_ACTIVE, Constants.No);
                else
                    report.Parameters.Add(Constants.REP_PATIENT_INQUIRY_IS_TRANSFER_ORDER_ACTIVE, Constants.Yes);
                ActionResult result = Redirect(report.Url);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        /// <summary>
        /// Fills the WardWhiteboardViewModel with data from Facade.
        /// </summary>
        /// <param name="wm">The ward whiteboard model.</param>
        private void FillViewModel(WardWhiteboardViewModel wm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (Session["DivisionList"] != null)
                    wm.DivisionList = (IList<Division>)Session["DivisionList"];
                else
                {
                    wm.DivisionList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).OrderBy(a => a.Name).ToList();
                    Session["DivisionList"] = wm.DivisionList;
                }

                // hard coded this to remove service call
                //wm.StrictDecisionList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                wm.StrictDecisionList = new CDList();
                wm.StrictDecisionList.Add(new CD { code = Constants.No, displayName = Constants.No });
                wm.StrictDecisionList.Add(new CD { code = Constants.Yes, displayName = Constants.Yes });

                // hard coded this to remove service call
                //wm.DisplayPTList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.PTDisplay);
                wm.DisplayPTList = new CDList();
                wm.DisplayPTList.Add(new CD { code = Constants.DISPLAY_PT_FIRST_AND_LAST4_CODE, displayName = Constants.DISPLAY_PT_FIRST_AND_LAST4_CODE });
                wm.DisplayPTList.Add(new CD { code = Constants.DISPLAY_PT_LAST_NAME_CODE, displayName = Constants.DISPLAY_PT_LAST_NAME_CODE });

                // hard coded this to remove service call
                //wm.GenderColorList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.GenderColor);
                wm.GenderColorList = new CDList();
                wm.GenderColorList.Add(new CD { code = Constants.GenderColorNone, displayName = Constants.GenderColorNone });
                wm.GenderColorList.Add(new CD { code = Constants.GenderColorBlueAndPink, displayName = Constants.GenderColorBlueAndPink });

                wm.WhiteboardList = new List<WhiteboardBedInfo>();
                wm.SortByList = new List<string>();
                wm.SortByList.Add(Constants.SORT_BY_STAFF_OR_ATTENDING_ASCENDING);
                wm.SortByList.Add(Constants.SORT_BY_STAFF_OR_ATTENDING_DESCENDING);
                wm.SortByList.Add(Constants.SORT_BY_WARD_ASCENDING);
                wm.SortByList.Add(Constants.SORT_BY_WARD_DESCENDING);
                wm.DisplayStaffAttendingList = new List<string>();
                wm.DisplayStaffAttendingList.Add(Constants.STAFF_ATTENDING);
                wm.DisplayStaffAttendingList.Add(Constants.STAFF);
                wm.DisplayStaffAttendingList.Add(Constants.ATTENDING);
                wm.ScrollRateList = new Dictionary<int, string>();
                wm.ScrollRateList.Add(2, "2 secs");
                wm.ScrollRateList.Add(5, "5 secs");
                wm.ScrollRateList.Add(10, "10 secs");
                wm.ScrollRateList.Add(20, "20 secs");
                wm.ScrollRateList.Add(30, "30 secs");

                //Insert defaul value for drop down list (Ward )
                wm.DivisionList.Insert(0, new Division() { Id = new II() { root = this.loggedUser.Domain, extension = (default(Guid)).ToString() }, Name = Strings.All });
                wm.FacilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                // todo cache facility time zone in session
                DateTime lastDate = TimeZoneInfo.ConvertTime(DateTime.UtcNow, FacadeUtil.GetFacilityTimeZoneInfo(this.loggedUser.Facility.Id));
                wm.LastUpdatedDate = string.Format("{0} {1} {2}", lastDate.ToString(Strings.USDate), Strings.At, lastDate.ToString("HH:mm"));
                if (wm.FacilitySettings != null && wm.FacilitySettings.FacilitySiteTimeZone != null)
                    wm.FacilitySiteCodeTimeZone = string.Format("({0})", wm.FacilitySettings.FacilitySiteTimeZone.code);
                else
                    wm.FacilitySiteCodeTimeZone = string.Empty;
                try
                {
                    wm.RefreshPageRate = (Int32.Parse(ConfigurationManager.AppSettings["WhiteboardAjaxRefreshRate"]) * 1000).ToString();
                }
                catch { wm.RefreshPageRate = "600000"; }
                try
                {
                    wm.RealRefreshRate = (Int32.Parse(ConfigurationManager.AppSettings["WhiteboardRealRefreshRate"]) * 60 * 1000).ToString();
                }
                catch { wm.RealRefreshRate = (120 * 60 * 1000).ToString(); } // 2 hours
                wm.AjaxRefreshCount = "0";
                wm.UserHasUpdateStaffPermission = FacadeManager.UserInterface.CheckPermission(Constants.Whiteboard_Edit_Staff_Permission);
                wm.LoggedUser = loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the sort by selected.
        /// </summary>
        /// <param name="wardWhiteboardVM">The ward whiteboard VM.</param>
        private string GetSortValue(WardWhiteboardViewModel wardWhiteboardVM)
        {
            string result = string.Empty;
            if (string.IsNullOrEmpty(wardWhiteboardVM.SortBySelected))
                return null;
            if (wardWhiteboardVM.SortBySelected.Equals(Constants.SORT_BY_STAFF_OR_ATTENDING_ASCENDING))
            {
                switch (wardWhiteboardVM.DisplayStaffAttendingSelected)
                {
                    case Constants.STAFF:
                    case Constants.STAFF_ATTENDING:
                        result = Constants.SORT_BY_STAFF_ASCENDING;
                        break;
                    case Constants.ATTENDING:
                        result = Constants.SORT_BY_ATTENDING_ASCENDING;
                        break;
                }
            }
            else if (wardWhiteboardVM.SortBySelected.Equals(Constants.SORT_BY_STAFF_OR_ATTENDING_DESCENDING))
            {
                switch (wardWhiteboardVM.DisplayStaffAttendingSelected)
                {
                    case Constants.STAFF:
                    case Constants.STAFF_ATTENDING:
                        result = Constants.SORT_BY_STAFF_DESCENDING;
                        break;
                    case Constants.ATTENDING:
                        result = Constants.SORT_BY_ATTENDING_DESCENDING;
                        break;
                }
            }
            else
                result = wardWhiteboardVM.SortBySelected;
            return result;
        }

        /// <summary>
        /// Sets the selected ward within the viewModel.
        /// </summary>
        /// <param name="viewModel">The ward whiteboard model.</param>
        private void SetWardSelected(WardWhiteboardViewModel viewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (viewModel.DivisionList == null)
                {
                    if (viewModel.LoggedUser == null)
                        viewModel.LoggedUser = FacadeManager.UserInterface.GetProfile();

                    if (Session["DivisionList"] != null)
                        viewModel.DivisionList = (IList<Division>)Session["DivisionList"];
                    else
                    {
                        viewModel.DivisionList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).OrderBy(a => a.Name).ToList();
                        Session["DivisionList"] = viewModel.DivisionList;
                    }

                }
                foreach (Division division in viewModel.DivisionList)
                {
                    if (division.Id.extension.Equals(viewModel.DivisionExtension, StringComparison.InvariantCultureIgnoreCase) || division.Name == viewModel.DivisionName)
                    {
                        viewModel.SelectedDivision = division;
                        viewModel.DivisionExtension = division.Id.extension;
                        break;
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the selected split screen option.
        /// </summary>
        /// <param name="viewModel">The ward whiteboard model.</param>
        private bool GetSelectedStrictDecision(string selectedValue)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (selectedValue == Constants.Yes)
                    return true;
                return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the selected ward group (division).
        /// </summary>
        /// <param name="whiteboardRM">The edit whiteboard room model.</param>
        /// <returns>
        /// Division.
        /// </returns>
        private Division GetWardGroupSelected(WhiteboardRoomViewModel whiteboardRM)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (whiteboardRM.WardGroupList == null)
                {
                    if (whiteboardRM.LoggedUser == null)
                        whiteboardRM.LoggedUser = FacadeManager.UserInterface.GetProfile();
                    whiteboardRM.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id);
                }
                foreach (Division division in whiteboardRM.WardGroupList)
                {
                    if (division.Name == whiteboardRM.WardName)
                        return division;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CDWithProperties GetStrictDecision(List<CDWithProperties> ptRiskList, string decisionCode, string ptRiskCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(decisionCode) && decisionCode.Equals(Constants.Yes))
                    return ptRiskList.Where(a => a.code == ptRiskCode).FirstOrDefault();
                else
                    return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the correspoding flag (CD) based on the risk code.
        /// </summary>
        /// <param name="risk">The risk.</param>        
        /// <param name="flagList">List of flag values.</param>        
        private string GetWhiteboardRoomModelFlag(CDWithProperties risk, CDList flagList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (risk != null && !string.IsNullOrEmpty(risk.code))
                    return flagList.Where(cd => cd.code == Constants.Yes).FirstOrDefault().code;
                else
                    return flagList.Where(cd => cd.code == Constants.No).FirstOrDefault().code;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the manual cleaning request.
        /// </summary>
        /// <param name="whiteboardRM">The whiteboard RM.</param>
        /// <param name="bed">The bed.</param>
        /// <param name="ward">The ward.</param>
        /// <param name="division">The division.</param>
        /// <param name="facility">The facility.</param>
        /// <param name="dateNow">The date now.</param>
        private void SaveManualCleaningRequest(WhiteboardRoomViewModel whiteboardRM, Bed bed, DateTime dateNow)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (whiteboardRM.RequestCleaning.Equals(Strings.Yes) || whiteboardRM.RequestCleaning.Equals(Strings.Stat))
                {
                    if (whiteboardRM.IsBedCleanRequested)
                    {
                        BedClean bc = FacadeManager.BedInterface.GetBedCleanByBedId(bed.Id, this.loggedUser.Facility.Id);
                        if (bc != null)
                        {
                            if (!bc.TypeOfClean.code.Equals(whiteboardRM.RequestCleaning, StringComparison.InvariantCultureIgnoreCase) ||
                                bc.SpecialInstructions != whiteboardRM.SpecialInstructions)
                            {
                                bc.TypeOfClean = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Decision).ToList().Where(a => a.displayName == whiteboardRM.RequestCleaning).FirstOrDefault();
                                bc.SpecialInstructions = whiteboardRM.SpecialInstructions;
                                DateTimeConverter.ConvertDateToUTC(bc, loggedUser.Facility.Id);
                                BMSFactory.BedManagerOperationsClient.UpdateBedCleaningOperation(bc.ToDataContract());
                            }
                        }
                    }
                    else
                    {
                        DateTime? requestDate = whiteboardRM.CleaningDate;
                        double value = 0;
                        if (double.TryParse(whiteboardRM.HourSelected, out value))
                            requestDate = requestDate.Value.AddHours(value);
                        if (double.TryParse(whiteboardRM.MinuteSelected, out value))
                            requestDate = requestDate.Value.AddMinutes(value);

                        BedClean bedClean = new BedClean()
                        {
                            Id = new II(this.loggedUser.Domain, null),
                            Bed = new Place() { Id = bed.Id, Name = bed.Name, Ien = bed.Ien },
                            LastEditDate = dateNow,
                            LastEditedBy = this.loggedUser.UserName,
                            ManualRequest = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision).ToList().Where(a => a.displayName == Strings.Yes).FirstOrDefault(),
                            RequestedDate = requestDate,
                            TypeOfClean = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Decision).ToList().Where(a => a.displayName == whiteboardRM.RequestCleaning).FirstOrDefault(),
                            Ward = FacadeManager.EntityInterface.GetWardById(new II(bed.Id.root, whiteboardRM.Ward)),
                            VistaSite = this.loggedUser.VistaSite,
                            SpecialInstructions = whiteboardRM.SpecialInstructions
                        };
                        DateTimeConverter.ConvertDateToUTC(bedClean, loggedUser.Facility.Id);
                        bedClean.Id = BMSFactory.BedManagerOperationsClient.CreateBedCleaningOperation(bedClean.ToDataContract());
                        FacadeManager.BedInterface.SendCleanDirtyBedNotifications(bedClean);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the bed unavailable.
        /// </summary>
        /// <param name="whiteboardRM">The whiteboard RM.</param>
        /// <param name="bed">The bed.</param>
        /// <param name="ward">The ward.</param>
        /// <param name="division">The division.</param>
        /// <param name="facility">The facility.</param>
        /// <param name="dateNow">The date now.</param>
        private BedUnavailable SaveBedUnavailable(WhiteboardRoomViewModel whiteboardRM, Bed bed, DateTime dateNow, bool isChangedComment, CDWithProperties reason, CD type)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedUnavailable bedUnavailable = FacadeManager.BedInterface.GetBedUnavailable(new II(whiteboardRM.RootBedId, whiteboardRM.ExtensionBedId), loggedUser.Facility.Id);
                whiteboardRM.BedUnavailableHistory = new BedUnavailableHistory();
                whiteboardRM.BedUnavailableHistory.IsChangedOnlyComment = true;
                if (bedUnavailable != null)
                    whiteboardRM.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, whiteboardRM.BedUnavailableHistory);
                if (string.IsNullOrEmpty(whiteboardRM.UnavailableReasonCode))
                {
                    if (!string.IsNullOrEmpty(whiteboardRM.SelectedReason))
                    {
                        //create
                        bedUnavailable = new BedUnavailable()
                        {
                            Bed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null,
                            CanceledBy = null,
                            CanceledDate = null,
                            CreatedBy = this.loggedUser.UserName,
                            CreationDate = dateNow,
                            EditedBy = null,
                            EditedDate = null,
                            Id = new II(this.loggedUser.Domain, null),
                            Parent = null,
                            Patient = null,
                            Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null),
                            Type = type,
                            OutOfServiceVistA = false,
                            WardList = bed.WardList,
                            VistaSite = this.loggedUser.VistaSite,
                            ExpectedCompletedDate = CreateExpectedCompletedDate(type, whiteboardRM.ExpectedCompletedDate, whiteboardRM.ExpectedCompletedDateHourSelected, whiteboardRM.ExpectedCompletedDateMinuteSelected)
                        };
                        whiteboardRM.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, whiteboardRM.BedUnavailableHistory);
                        whiteboardRM.BedUnavailableHistory.IsChangedOnlyComment = false;
                        FacadeManager.WorkflowInterface.CreateBedUnavailable(bedUnavailable, loggedUser.Facility);
                    }
                }
                else
                {
                    string newReason = string.Empty;
                    if (!string.IsNullOrEmpty(whiteboardRM.SelectedReason))
                        newReason = whiteboardRM.SelectedReason;
                    //update
                    DateTime? expectedCompletedDate = CreateExpectedCompletedDate(type, whiteboardRM.ExpectedCompletedDate, whiteboardRM.ExpectedCompletedDateHourSelected, whiteboardRM.ExpectedCompletedDateMinuteSelected);

                    if (bedUnavailable.Reason != null && (newReason != bedUnavailable.Reason.code || expectedCompletedDate != bedUnavailable.ExpectedCompletedDate || isChangedComment))
                    {
                        if (!string.IsNullOrEmpty(newReason) && newReason != Strings.DeleteThisReason)
                        {
                            bedUnavailable.EditedBy = this.loggedUser.UserName;
                            bedUnavailable.EditedDate = dateNow;
                            bedUnavailable.Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null);
                            bedUnavailable.Type = type;
                            bedUnavailable.OutOfServiceVistA = false;
                            bedUnavailable.ExpectedCompletedDate = expectedCompletedDate;
                            whiteboardRM.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, whiteboardRM.BedUnavailableHistory);
                            FacadeManager.WorkflowInterface.UpdateBedUnavailable(bedUnavailable, loggedUser.Facility);

                        }
                        else
                        {
                            //cancel
                            bedUnavailable.CanceledBy = this.loggedUser.UserName;
                            bedUnavailable.CanceledDate = CreateActualOOSCompletedDate(whiteboardRM.ActualCompletedDate, whiteboardRM.ActualCompletedDateHourSelected, whiteboardRM.ActualCompletedDateMinuteSelected, dateNow);
                            bedUnavailable.OutOfServiceVistA = false;
                            whiteboardRM.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, whiteboardRM.BedUnavailableHistory);
                            FacadeManager.WorkflowInterface.CancelBedUnavailable(bedUnavailable, loggedUser.Facility);
                        }
                        whiteboardRM.BedUnavailableHistory.IsChangedOnlyComment = false;
                    }
                }

                return bedUnavailable;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the expected completed date.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <param name="expectedDate">The expected date.</param>
        /// <param name="expectedHour">The expected hour.</param>
        /// <param name="expectedMinute">The expected minute.</param>
        /// <returns></returns>
        private DateTime? CreateExpectedCompletedDate(CD type, DateTime? expectedDate, string expectedHour, string expectedMinute)
        {
            DateTime? returnExpectedDate = null;
            if (type != null && type.code == Constants.BED_OUT_OF_SERVICE_TYPE_CODE && expectedDate.HasValue)
            {
                returnExpectedDate = expectedDate;
                double value = 0;
                if (double.TryParse(expectedHour, out value))
                    returnExpectedDate = returnExpectedDate.Value.AddHours(value);
                if (double.TryParse(expectedMinute, out value))
                    returnExpectedDate = returnExpectedDate.Value.AddMinutes(value);
            }
            return returnExpectedDate;
        }

        /// <summary>
        /// Creates the actual OOS completed date.
        /// </summary>
        /// <param name="actualDate">The actual date.</param>
        /// <param name="actualHour">The actual hour.</param>
        /// <param name="actualMinute">The actual minute.</param>
        /// <param name="dateNow">The date now.</param>
        /// <returns></returns>
        private DateTime CreateActualOOSCompletedDate(DateTime actualDate, string actualHour, string actualMinute, DateTime dateNow)
        {
            if (actualDate == null || actualDate == DateTime.MinValue)
                return dateNow;
            else
            {
                DateTime returnActualDate = actualDate;
                double value = 0;
                if (double.TryParse(actualHour, out value))
                    returnActualDate = returnActualDate.AddHours(value);
                if (double.TryParse(actualMinute, out value))
                    returnActualDate = returnActualDate.AddMinutes(value);
                return returnActualDate;
            }
        }

        /// <summary>
        /// Saves the bed.
        /// </summary>
        /// <param name="whiteboardRM">The whiteboard RM.</param>
        /// <param name="bed">The bed.</param>
        private void SaveBed(WhiteboardRoomViewModel whiteboardRM, Bed bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(whiteboardRM.ClearAllButton))
                {
                    List<Division> divisions = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                    foreach (BedComment bc in bed.CommentList)
                    {
                        if (bc.Division != null && bc.Division.Id != null && divisions.Where(a => a.Id.extension.Equals(bc.Division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                            bc.Comment = string.Empty;
                    }
                }
                else
                {
                    if (bed.CommentList.Where(a => a.Division.Id.extension.Equals(whiteboardRM.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                        bed.CommentList.Where(a => a.Division.Id.extension.Equals(whiteboardRM.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Comment = whiteboardRM.Comments;
                    else
                        bed.CommentList.Add(new BedComment() { Division = new Division() { Id = new II(whiteboardRM.RealDivisionSelectedRoot, whiteboardRM.RealDivisionSelectedExtension) }, Comment = whiteboardRM.Comments });
                }

                bed.NurseAssignment = whiteboardRM.Bed.NurseAssignment;
                FacadeManager.BedInterface.UpdateBed(bed);
                Division division = FacadeManager.EntityInterface.GetDivisionById(new II(whiteboardRM.RealDivisionSelectedRoot, whiteboardRM.RealDivisionSelectedExtension));
                Patient patient = null;
                if (!string.IsNullOrEmpty(whiteboardRM.ExtensionPatientId) && whiteboardRM.ExtensionPatientId != Guid.Empty.ToString())
                    patient = FacadeManager.EntityInterface.GetPatientById(new II(this.loggedUser.Domain, whiteboardRM.ExtensionPatientId), null);
                FacadeManager.BedInterface.ManageBedStaff(bed, division, bed.NurseAssignment, patient, this.loggedUser.Facility);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the reason.
        /// </summary>
        /// <param name="reasonCode">The reason code.</param>
        /// <param name="reason">The reason.</param>
        /// <param name="type">The type.</param>
        private void GetReason(string reasonCode, out CDWithProperties reason, out CD type, out bool isReasonAvailable)
        {
            List<CDWithProperties> reasonList = FacadeManager.VocabularyInterface.GetReasons().ToList();
            List<CD> typeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.AdminURType).ToList();
            reason = null;
            type = null;
            isReasonAvailable = true;
            if (!string.IsNullOrWhiteSpace(reasonCode) && reasonCode != Strings.DeleteThisReason)
            {
                reason = reasonList.Where(a => a.code == reasonCode).FirstOrDefault();
                if (reason == null)
                {
                    isReasonAvailable = false;
                }
                else
                {
                    string typeCode = reason.Properties.Where(a => a.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text;
                    type = typeList.Where(a => a.code == typeCode).FirstOrDefault();
                }
            }
        }

        /// <summary>
        /// Gets the type code.
        /// </summary>
        /// <param name="reasonCode">The reason code.</param>
        /// <returns></returns>
        private string GetTypeCode(string reasonCode)
        {
            List<CDWithProperties> reasonList = FacadeManager.VocabularyInterface.GetReasons().ToList();
            if (!string.IsNullOrWhiteSpace(reasonCode))
            {
                CDWithProperties reason = reasonList.Where(a => a.code == reasonCode).FirstOrDefault();
                if (reason != null)
                    return reason.Properties.Where(a => a.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text;
                else
                    return string.Empty;
            }
            return string.Empty;
        }

        /// <summary>
        /// Translates the bed unavailable history.
        /// </summary>
        /// <param name="bedUnavailable">The bed unavailable.</param>
        /// <param name="bedUnavailableHistory">The bed unavailable history.</param>
        /// <returns></returns>
        private BedUnavailableHistory TranslateBedUnavailableHistory(BedUnavailable bedUnavailable, BedUnavailableHistory bedUnavailableHistory)
        {
            bedUnavailableHistory.Bed = bedUnavailable.Bed;
            bedUnavailableHistory.CanceledBy = bedUnavailable.CanceledBy;
            bedUnavailableHistory.CanceledDate = bedUnavailable.CanceledDate;
            bedUnavailableHistory.CreatedBy = bedUnavailable.CreatedBy;
            bedUnavailableHistory.CreationDate = bedUnavailable.CreationDate;
            bedUnavailableHistory.EditedBy = bedUnavailable.EditedBy;
            bedUnavailableHistory.EditedDate = bedUnavailable.EditedDate;
            bedUnavailableHistory.Id = bedUnavailable.Id;
            bedUnavailableHistory.Parent = bedUnavailable.Parent;
            bedUnavailableHistory.Patient = bedUnavailable.Patient;
            bedUnavailableHistory.Reason = bedUnavailable.Reason;
            bedUnavailableHistory.Type = bedUnavailable.Type;
            bedUnavailableHistory.VistaSite = bedUnavailable.VistaSite;
            bedUnavailableHistory.ExpectedCompletedDate = bedUnavailable.ExpectedCompletedDate;
            return bedUnavailableHistory;
        }

        private List<IconAssociationInfo> CreateIconAssociationList(List<IconAssociationInfo> oldList, IList<IconAssociationInfo> newList)
        {
            List<IconAssociationInfo> result = new List<IconAssociationInfo>();
            if (oldList != null && oldList.Count > 0 && newList != null)
            {
                var pairs = from oldItem in oldList
                            join newItem in newList
                            on oldItem.Icon.IconId equals newItem.Icon.IconId
                            select new { oldItem, newItem };
                foreach (var pair in pairs)
                {
                    if (pair.newItem.Code != pair.oldItem.Code)
                    {
                        IconAssociationInfo icon = new IconAssociationInfo();
                        icon = pair.oldItem;
                        icon.Code = pair.newItem.Code;
                        result.Add(icon);
                    }
                    else
                        result.Add(pair.oldItem);
                }
            }
            return result;
        }

        private string GetWhiteboardReportUrl(string displayPTCode, string genderColorCode, string sortBy, string displayStaffAttending, string divisionExtension, string facilityCode)
        {
            StringBuilder url = new StringBuilder();
            url.Append("displayPTCode=" + displayPTCode);
            url.Append("&genderColorCode=" + genderColorCode);
            url.Append("&sortBy=" + sortBy);
            url.Append("&displayStaffAttending=" + displayStaffAttending);
            url.Append("&divisionId=" + divisionExtension);
            url.Append("&facilityCode=" + facilityCode);
            return "/Reporting/LocalReportViewer.aspx?" + CustomEncryption.Encrypt(url.ToString(), loggedUser.Salt) + "*ww";
        }

        /// <summary>
        /// Adds the ward group census.
        /// </summary>
        /// <param name="bedsOfDivision">The beds of division.</param>
        /// <param name="divisionName">Name of the division.</param>
        /// <returns></returns>
        private WardGroupCensus AddWardGroupCensus(IEnumerable<BedInfo> bedsOfDivision, string divisionName, IEnumerable<II> wardIds)
        {
            WardGroupCensus wardGroupCensusItem = new WardGroupCensus();
            wardGroupCensusItem.TotalBeds = bedsOfDivision.Count() - bedsOfDivision.Count(a => a.BedAvailabilityStatusCode.Equals(Constants.BED_OOS_FROM_BEDBOARD, StringComparison.InvariantCultureIgnoreCase) && String.IsNullOrEmpty(a.PatientId.extension));
            if (wardIds != null)
            {
                wardGroupCensusItem.FemaleBeds = (from bed in bedsOfDivision.Where(a => a.PatientGender == "Female")
                                                  join wardId in wardIds on new { bed.WardId.extension, bed.WardId.root } equals new { wardId.extension, wardId.root }
                                                  select bed).Count();
                wardGroupCensusItem.MaleBeds = (from bed in bedsOfDivision.Where(a => a.PatientGender == "Male")
                                                join wardId in wardIds on new { bed.WardId.extension, bed.WardId.root } equals new { wardId.extension, wardId.root }
                                                select bed).Count();
            }
            else
            {
                wardGroupCensusItem.FemaleBeds = bedsOfDivision.Count(a => a.PatientGender == "Female");
                wardGroupCensusItem.MaleBeds = bedsOfDivision.Count(a => a.PatientGender == "Male");
            }
            wardGroupCensusItem.OccupiedBeds = bedsOfDivision.Count(a => !String.IsNullOrEmpty(a.PatientId.extension));
            wardGroupCensusItem.AssignedBeds = bedsOfDivision.Count(a => (a.Reason.Contains("BED ASSIGNED") || a.Comment.Contains("BED ASSIGNED") || !string.IsNullOrEmpty(a.WaitingListInfo)));
            wardGroupCensusItem.AvailableBeds = wardGroupCensusItem.TotalBeds - wardGroupCensusItem.OccupiedBeds;
            wardGroupCensusItem.WardGroup = divisionName;
            return wardGroupCensusItem;
        }

        /// <summary>
        /// Fills the whiteboard room edit model.
        /// </summary>
        /// <param name="rootBedId">The root bed id.</param>
        /// <param name="extensionBedId">The extension bed id.</param>
        /// <param name="wardCode">Ward.</param>
        /// <param name="splitScreen">Split screen.</param>
        /// <param name="displayPTCode">Display PT code.</param>
        /// <param name="genderColorCode">Gender color code.</param>
        /// <param name="whiteboardRM">Whiteboard Room Model.</param>
        private void FillEditModel(string rootBedId, string extensionBedId, string wardCode, string splitScreen, string displayPTCode, string genderColorCode, string sortBy, string displayFooterCensus, string displayStaffAttending, int? scrollRate, WhiteboardRoomViewModel whiteboardRM, string extensionPatientId, string rootPatientId, string flowId, bool hasEvacuationIcon)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //set parameters from ward whiteboard home page
                whiteboardRM.WardParameter = wardCode;
                whiteboardRM.SplitScreenParameter = splitScreen;
                whiteboardRM.DisplayPTCodeParameter = displayPTCode;
                whiteboardRM.GenderColorCodeParameter = genderColorCode;
                whiteboardRM.SortByParameter = sortBy;
                whiteboardRM.DisplayFooterCensus = displayFooterCensus;
                whiteboardRM.DisplayStaffAttendingParameter = displayStaffAttending;
                whiteboardRM.ScrollRateParameter = (scrollRate.HasValue) ? scrollRate.Value : 0;

                if (string.IsNullOrEmpty(flowId))
                    whiteboardRM.FlowId = null;
                else
                    whiteboardRM.FlowId = Guid.Parse(flowId);
                whiteboardRM.HasEvacuationIcon = hasEvacuationIcon;
                whiteboardRM.IsEvacuationFacility = FacadeManager.ConfigurationInterface.GetIsEvacuation(this.loggedUser.Domain, this.loggedUser.Facility.Id.extension);
                II bedId = new II { root = rootBedId, extension = extensionBedId };
                whiteboardRM.RootBedId = rootBedId;
                whiteboardRM.ExtensionBedId = extensionBedId;
                whiteboardRM.Bed = FacadeManager.BedInterface.GetBed(bedId, loggedUser.VistaSite.Id);
                whiteboardRM.NurAssignment = whiteboardRM.Bed == null ? string.Empty : whiteboardRM.Bed.NurseAssignment;
                Division division = null;
                if (string.IsNullOrEmpty(wardCode) || wardCode.Equals(Guid.Empty.ToString()))
                {
                    IList<Division> divisionList = FacadeManager.EntityInterface.GetDivisionsByBedAndFacility(whiteboardRM.Bed.Id, this.loggedUser.Facility.Id);
                    if (divisionList != null && divisionList.Count > 0)
                        division = divisionList[0];
                }
                else
                    division = FacadeManager.EntityInterface.GetDivisionById(new II(rootBedId, wardCode));
                if (division != null)
                {
                    whiteboardRM.RealDivisionSelectedExtension = division.Id.extension;
                    whiteboardRM.RealDivisionSelectedRoot = division.Id.root;
                    BedComment bc = whiteboardRM.Bed.CommentList.Where(a => a.Division.Id.extension.Equals(whiteboardRM.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    whiteboardRM.Comments = (bc != null) ? bc.Comment : string.Empty;
                    whiteboardRM.ReasonList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.UnavailableReason);
                    AntiXssEncoder xss = new AntiXssEncoder();
                    whiteboardRM.WardName = xss.Decode(division.Name);
                }
                BedUnavailable bedUnavailable = FacadeManager.BedInterface.GetBedUnavailable(bedId, loggedUser.Facility.Id);
                if (bedUnavailable != null)
                {
                    whiteboardRM.Bed.Reason = bedUnavailable.Reason;
                    whiteboardRM.SelectedReason = bedUnavailable.Reason.code;
                    whiteboardRM.UnavailableReasonCode = bedUnavailable.Reason.code;

                    whiteboardRM.ExpectedCompletedDate = bedUnavailable.ExpectedCompletedDate;
                    if (bedUnavailable.ExpectedCompletedDate.HasValue)
                    {
                        whiteboardRM.ExpectedCompletedDateHourSelected = (whiteboardRM.ExpectedCompletedDate.Value.Hour < 10) ? "0" + whiteboardRM.ExpectedCompletedDate.Value.Hour.ToString() : whiteboardRM.ExpectedCompletedDate.Value.Hour.ToString();
                        whiteboardRM.ExpectedCompletedDateMinuteSelected = (whiteboardRM.ExpectedCompletedDate.Value.Minute < 10) ? "0" + whiteboardRM.ExpectedCompletedDate.Value.Minute.ToString() : whiteboardRM.ExpectedCompletedDate.Value.Minute.ToString();
                    }
                    else
                    {
                        whiteboardRM.ExpectedCompletedDateHourSelected = "00";
                        whiteboardRM.ExpectedCompletedDateMinuteSelected = "00";
                    }
                }
                else
                    whiteboardRM.UnavailableReasonCode = string.Empty;

                //set visibility for expected completed date

                if (GetTypeCode(whiteboardRM.UnavailableReasonCode) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                    whiteboardRM.IsExpectedCompletedDateVisible = true;
                else
                {
                    whiteboardRM.IsExpectedCompletedDateVisible = false;
                    whiteboardRM.ExpectedCompletedDateHourSelected = "00";
                    whiteboardRM.ExpectedCompletedDateMinuteSelected = "00";
                }

                //if bed has unavailable reason add option to delete this reason
                if (!string.IsNullOrEmpty(whiteboardRM.UnavailableReasonCode))
                {
                    whiteboardRM.ReasonList.Insert(0, new CD() { code = Strings.DeleteThisReason, displayName = Strings.DeleteThisReason });
                    if (whiteboardRM.ReasonList.Where(a => a.code.Equals(whiteboardRM.UnavailableReasonCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                        whiteboardRM.ReasonList.Add(whiteboardRM.Bed.Reason);
                }

                whiteboardRM.LoggedUser = this.loggedUser;

                IList<Ward> wardList = null;
                if (division != null)
                {
                    wardList = FacadeManager.EntityInterface.GetWardsByDivisionId(division.Id);
                }
                whiteboardRM.WardList = new List<Ward>();
                Ward ward = null;
                foreach (Ward w in whiteboardRM.Bed.WardList)
                {
                    ward = wardList.Where(a => a.Id.extension.Equals(w.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (ward != null)
                        whiteboardRM.WardList.Add(ward);
                }
                whiteboardRM.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.StrictDecision);

                whiteboardRM.RootPatientId = rootPatientId;
                whiteboardRM.ExtensionPatientId = extensionPatientId;
                Guid bedUid = Guid.Parse(extensionBedId);
                Guid? patientUid = null;
                if (!string.IsNullOrEmpty(extensionPatientId) && extensionPatientId != (default(Guid)).ToString())
                    patientUid = Guid.Parse(extensionPatientId);
                Guid facilityUid = Guid.Parse(this.loggedUser.Facility.Id.extension);
                whiteboardRM.IconsAssociationList = FacadeManager.IconInterface.FilterBedPatientIconAssociation(bedUid, patientUid, facilityUid);
                whiteboardRM.IsActualCompletedDateVisible = false;

                whiteboardRM.BedClean = FacadeManager.BedInterface.GetBedCleanByBedId(bedId, loggedUser.Facility.Id);
                whiteboardRM.EMSNotifyIcon = FacadeManager.IconInterface.GetIconByCode(Constants.EMS_NOTIFIED);
                if (whiteboardRM.BedClean != null)
                {
                    if (whiteboardRM.BedClean.CompletedDate.HasValue)
                        whiteboardRM.IsBedCleanCompleted = true;
                    else
                    {
                        if (!string.IsNullOrEmpty(whiteboardRM.BedClean.AcceptedBy))
                            whiteboardRM.IsBedCleanAccepted = true;
                        else
                            whiteboardRM.IsBedCleanRequested = true;
                    }
                    if (!whiteboardRM.IsBedCleanCompleted)
                    {
                        whiteboardRM.RequestCleaning = whiteboardRM.BedClean.TypeOfClean.code;
                        whiteboardRM.Ward = whiteboardRM.BedClean.Ward.Id.extension;
                        whiteboardRM.CleaningDate = (whiteboardRM.BedClean.RequestedDate.HasValue) ? whiteboardRM.BedClean.RequestedDate.Value : DateTimeNowOnCurrentFacility;
                        whiteboardRM.SpecialInstructions = whiteboardRM.BedClean.SpecialInstructions;
                    }
                    else
                    {
                        whiteboardRM.RequestCleaning = Strings.No;
                        whiteboardRM.CleaningDate = DateTimeNowOnCurrentFacility;
                        whiteboardRM.IsBedCleanRequested = false;
                        whiteboardRM.IsBedCleanAccepted = false;
                    }
                }
                else
                {
                    whiteboardRM.RequestCleaning = Strings.No;
                    whiteboardRM.CleaningDate = DateTimeNowOnCurrentFacility;
                    whiteboardRM.IsBedCleanRequested = false;
                    whiteboardRM.IsBedCleanAccepted = false;
                }
                if (whiteboardRM.CleaningDate.Hour.ToString().Length == 1)
                    whiteboardRM.HourSelected = whiteboardRM.CleaningDate.Hour.ToString().Replace(whiteboardRM.CleaningDate.Hour.ToString(), "0" + whiteboardRM.CleaningDate.Hour.ToString());
                else
                    whiteboardRM.HourSelected = whiteboardRM.CleaningDate.Hour.ToString();
                if (whiteboardRM.CleaningDate.Minute.ToString().Length == 1)
                    whiteboardRM.MinuteSelected = whiteboardRM.CleaningDate.Minute.ToString().Replace(whiteboardRM.CleaningDate.Minute.ToString(), "0" + whiteboardRM.CleaningDate.Minute.ToString());
                else
                    whiteboardRM.MinuteSelected = whiteboardRM.CleaningDate.Minute.ToString();
                whiteboardRM.BedStaffHistoryList = FacadeManager.BedInterface.GetBedStaffHistory(bedId, this.loggedUser.Facility.Id);
                FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);
                if (settings != null && settings.FacilitySiteTimeZone != null)
                    whiteboardRM.FacilityTimeZone = settings.FacilitySiteTimeZone.code;

                List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(this.loggedUser.Domain, this.loggedUser.Facility.Id).ToList();
                if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                {
                    whiteboardRM.EMSModuleInactive = true;
                    whiteboardRM.DisplayEMSModuleInactive = @"style='display:none;'";
                }
                else
                {
                    whiteboardRM.EMSModuleInactive = false;
                    whiteboardRM.DisplayEMSModuleInactive = string.Empty;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Direct Whiteboard Population from database
        private static string QuickParse(SqlDataReader reader, string columnName)
        {
            var obj = reader[columnName];

            if (obj == null || obj == DBNull.Value)
                return null;

            return obj.ToString();
        }

        private static T QuickParse<T>(SqlDataReader reader, string columnName, T defaultValue = default(T))
        {
            var obj = reader[columnName];

            if (obj == null || obj == DBNull.Value)
                return defaultValue;

            return (T)obj;
        }

        private static T QuickParse<T>(SqlDataReader reader, int columnIndex, T defaultValue = default(T))
        {
            var obj = reader[columnIndex];

            if (obj == null || obj == DBNull.Value)
                return defaultValue;

            return (T)obj;
        }

        private WardWhiteboardViewModel FillWhiteboardContentDirect(string divisionId, string splitScreen, string displayPTCode, string genderColorCode, string sortBy, string displayFooterCensus, string displayStaffAttending, string pScrollRate, string pScrollValue, string pScrollDirection)
        {
            WardWhiteboardViewModel wardWhiteboardVM = new WardWhiteboardViewModel();
            FillViewModel(wardWhiteboardVM); // todo - see if we can skip some of the lookups in here
            wardWhiteboardVM.IconFlagsList = FacadeManager.IconInterface.GetAllIconsByFacility(Guid.Parse(this.loggedUser.Facility.Id.extension)).Where(a => a.Active == true).ToList();

            Dictionary<string, int> nameAndSSNIndex = new Dictionary<string, int>();

            string directSqlConStr = ConfigurationManager.ConnectionStrings["DirectSQL"].ToString();
            using (SqlConnection con = new SqlConnection(directSqlConStr))
            {
                using (SqlCommand cmd = new SqlCommand("usp_Rpt_Whiteboard", con))
                {
                    cmd.CommandType = CommandType.StoredProcedure;

                    cmd.Parameters.Add("@FACILITY_UID", SqlDbType.UniqueIdentifier).Value = Guid.Parse(this.loggedUser.Facility.Id.extension);

                    Guid divisionUid = Guid.Parse(divisionId);

                    if (divisionUid == Guid.Empty)
                        cmd.Parameters.Add("@DIVISION_UID", SqlDbType.UniqueIdentifier).Value = DBNull.Value;
                    else
                        cmd.Parameters.Add("@DIVISION_UID", SqlDbType.UniqueIdentifier).Value = divisionUid;

                    cmd.Parameters.Add("@GENDER_COLOR_CODE", SqlDbType.VarChar).Value = genderColorCode;

                    if (displayFooterCensus == Constants.Yes)
                        cmd.Parameters.Add("@DISPLAY_FOOTER_CENSUS", SqlDbType.Bit).Value = 1;

                    con.Open();
                    SqlDataReader reader = cmd.ExecuteReader();

                    wardWhiteboardVM.WhiteboardList = new List<WhiteboardBedInfo>();

                    while (reader.Read())
                    {
                        wardWhiteboardVM.OnEvac = QuickParse<bool>(reader, "FACILITY_EVAC");

                        WhiteboardBedInfo bedInfo = new WhiteboardBedInfo();

                        bedInfo.BedIdExtension = QuickParse(reader, "BED_UID");
                        bedInfo.PatientIdExtension = QuickParse(reader, "PATIENT_UID");
                        bedInfo.PatientLastName = QuickParse<string>(reader, "PATIENT_NAME_FAMILY", String.Empty);

                        // last initial and 4 SSN
                        bedInfo.Last4SSN = QuickParse<string>(reader, "PATIENT_LAST4", string.Empty);
                        if (!string.IsNullOrEmpty(bedInfo.Last4SSN))
                            bedInfo.Last4SSN = bedInfo.Last4SSN.Substring(1);

                        bedInfo.Reason = QuickParse<string>(reader, "COMMENT");
                        string rowColor = QuickParse<string>(reader, "ROWCOLOR");

                        bedInfo.Attending = QuickParse<string>(reader, "ATTPHYS_NAME");
                        bedInfo.StaffName = QuickParse<string>(reader, "Nurse");
                        bedInfo.BedName = QuickParse<string>(reader, "BED_NAME");
                        bedInfo.WardName = QuickParse<string>(reader, "WARD_AND_TS");
                        bedInfo.StayLength = QuickParse<string>(reader, "LOS");
                        bedInfo.ObservationCountdown = QuickParse<string>(reader, "OBSCOUNTDOWN");
                        bedInfo.EvacuationDispostionStatus = QuickParse<string>(reader, "EVAC_STATUS");
                        bedInfo.HasMultipleDivisionsAssigned = QuickParse<bool>(reader, "HAS_MULTIPLE_DIVISIONS_ASSIGNED");

                        int bedAvailabilityStatusIconId = QuickParse<int>(reader, "OUT_OF_SERVICE_ICON_ID", 0);
                        int dischargeStatusId = QuickParse<int>(reader, "DISCHARGE_STATUS_ICON_ID", 0);
                        string iconListStr = QuickParse<string>(reader, "ICON_IMAGE_IDs");
                        int numiIconId = QuickParse<int>(reader, "NUMI_ICON_ID", 0);
                        int bedStatusIconId = QuickParse<int>(reader, "BEDSTATUSICON", 0);

                        string classColor = string.Empty;

                        if (rowColor == "BLUE")
                            classColor = "class=\"colorBlue SmallText\"";
                        else if (rowColor == "PINK")
                            classColor = "class=\"colorPink SmallText\"";
                        else if (rowColor == "RED")
                            classColor = "class=\"colorRed SmallText\"";
                        else if (rowColor == "ORANGE")
                            classColor = "class=\"colorOrange SmallText\"";
                        else if (rowColor == "GREEN")
                            classColor = "class=\"colorYellowOrange SmallText\"";
                        else
                            classColor = "class=\"colorLightGray SmallText\"";


                        bedInfo.GenderColor = classColor;

                        //// icons

                        //// set bed availability status icon
                        if (bedAvailabilityStatusIconId != 0)
                            bedInfo.BedAvailabilityStatusIcon = wardWhiteboardVM.IconFlagsList.Where(i => i.IconId == bedAvailabilityStatusIconId).FirstOrDefault();

                        if (bedInfo.BedAvailabilityStatusIcon == null)
                            bedInfo.BedAvailabilityStatusIcon = new IconInfo();
                        else
                            bedInfo.BedAvailabilityStatusIconBytes = bedInfo.BedAvailabilityStatusIcon.ImageBytes;


                        // set discharge status icon - interward transfer icon is a discharge icon
                        if (dischargeStatusId != 0)
                            bedInfo.DischargeIcon = wardWhiteboardVM.IconFlagsList.Where(i => i.IconId == dischargeStatusId).FirstOrDefault();

                        if (bedInfo.DischargeIcon == null)
                            bedInfo.DischargeIcon = new IconInfo();
                        else
                            bedInfo.DischargeIconBytes = bedInfo.DischargeIcon.ImageBytes;

                        // set bed cleaning status icon
                        if (bedStatusIconId != 0)
                            bedInfo.BedCleaningStatusIcon = wardWhiteboardVM.IconFlagsList.Where(i => i.IconId == bedStatusIconId).FirstOrDefault();

                        if (bedInfo.BedCleaningStatusIcon == null)
                            bedInfo.BedCleaningStatusIcon = new IconInfo();
                        else
                            bedInfo.BedCleaningStatusIconBytes = bedInfo.BedCleaningStatusIcon.ImageBytes;

                        //// set bed cleaning status icon
                        //bedInfo.BedCleaningStatusIcon = wardWhiteboardVM.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.BedCleaningStatusCode).FirstOrDefault();
                        //if (bedInfo.BedCleaningStatusIcon == null)
                        //    bedInfo.BedCleaningStatusIcon = new IconInfo();
                        //else
                        //{
                        //    // If IsBedCleanManual, then assign the bed clean image and proceed
                        //    if (model.IsBedCleanManual == true)
                        //    {
                        //        bedInfo.BedCleaningStatusIconBytes = bedInfo.BedCleaningStatusIcon.ImageBytes;
                        //    }
                        //    // If IsBedCleanManual is false (i.e, automated bed clean request), then 
                        //    // verify if there is a EMS Notification Config for the specific ward group.
                        //    else if (model.IsBedCleanManual == false && emsList != null)
                        //    {
                        //        foreach (EMSNotification ems in emsList)
                        //        {
                        //            // Get ems notification config location and match with model.DivisionName 
                        //            // Alternatively :
                        //            //   Get VistA ward name based on BMS ward group and match with model.WardName
                        //            if (ems.Location.Equals(model.DivisionName))
                        //            {
                        //                bedInfo.BedCleaningStatusIconBytes = bedInfo.BedCleaningStatusIcon.ImageBytes;
                        //                break;
                        //            }
                        //        }
                        //    }
                        //    else
                        //    {
                        //        bedInfo.BedCleaningStatusIcon = new IconInfo();
                        //    }
                        //}

                        bedInfo.IconFlagsList = new List<IconInfo>();
                        if (!string.IsNullOrEmpty(iconListStr))
                        {
                            string[] iconIds = iconListStr.Split(',');

                            // set icons for flags
                            bedInfo.IconFlagsList = (from icon in wardWhiteboardVM.IconFlagsList
                                                     where (from iconAssociationId in iconIds
                                                            select iconAssociationId).Contains(icon.IconId.ToString())
                                                     select icon).OrderBy(o => o.Subgroup == Constants.STANDARD_ICON ? 1 : (o.Subgroup == Constants.EMERGENCY_ICON ? 2 : 3)).ThenBy(o => o.OrderIndex).ToList();

                        }

                        if (bedInfo.IconFlagsList != null && bedInfo.IconFlagsList.Where(a => a.IconName.ToLower().Contains(Constants.EVACUATION)).FirstOrDefault() != null)
                            bedInfo.HasEvacuationIcon = true;
                        else
                            bedInfo.HasEvacuationIcon = false;

                        //combine all iconFlags for report
                        if (wardWhiteboardVM.IsFromWhiteboardKiosk == "1" && bedInfo.IconFlagsList != null && bedInfo.IconFlagsList.Count > 0)
                        {
                            List<string> imagesParam = new List<string>();
                            foreach (IconInfo icon in bedInfo.IconFlagsList)
                                imagesParam.Add(icon.ImageBytes);
                            bedInfo.CombineIconBinaryList = ImageConverter.MergeMultipleImages(imagesParam);
                        }

                        // numi icon
                        if (numiIconId != 0)
                            bedInfo.LevelOfCareIcon = wardWhiteboardVM.IconFlagsList.Where(i => i.IconId == numiIconId).FirstOrDefault();

                        if (bedInfo.LevelOfCareIcon == null)
                            bedInfo.LevelOfCareIcon = new IconInfo();
                        else
                            bedInfo.LevelOfCareIconBytes = bedInfo.DischargeIcon.ImageBytes;

                        // Construct the PT based on the Display PT selected value.
                        StringBuilder pt = new StringBuilder();

                        if (!string.IsNullOrEmpty(bedInfo.PatientLastName) && !string.IsNullOrEmpty(bedInfo.Last4SSN))
                        {
                            if (bedInfo.IconFlagsList != null && bedInfo.IconFlagsList.Where(a => (a.PatientOrRoomBed == 'P' && (a.IconName.ToLower().Contains("optout") || a.IconName.ToLower().Contains("opt_out") || a.IconName.ToLower().Contains("opt-out") || a.IconName.ToLower().Contains("opt out")))).FirstOrDefault() != null)
                                pt.Append("X X X X");
                            else
                            {
                                if (displayPTCode == Constants.DISPLAY_PT_FIRST_AND_LAST4_CODE)
                                    pt.Append(bedInfo.Last4SSN);
                                if (displayPTCode == Constants.DISPLAY_PT_LAST_NAME_CODE)
                                    pt.Append(bedInfo.PatientLastName);
                            }
                        }

                        bedInfo.PT = pt.ToString();

                        // gather same/similar name and SSN info
                        bedInfo.First4LastName = bedInfo.PatientLastName.Length > 3 ? bedInfo.PatientLastName.ToLower().Substring(0, 4) : bedInfo.PatientLastName;
                        if (!String.IsNullOrEmpty(bedInfo.First4LastName))
                        {
                            if (!nameAndSSNIndex.ContainsKey(bedInfo.First4LastName))
                                nameAndSSNIndex.Add(bedInfo.First4LastName, 0);

                            ++nameAndSSNIndex[bedInfo.First4LastName];
                        }

                        if (!String.IsNullOrEmpty(bedInfo.Last4SSN))
                        {
                            if (!nameAndSSNIndex.ContainsKey(bedInfo.Last4SSN))
                                nameAndSSNIndex.Add(bedInfo.Last4SSN, 0);

                            ++nameAndSSNIndex[bedInfo.Last4SSN];
                        }

                        // display staff/attending
                        switch (displayStaffAttending)
                        {
                            case Constants.STAFF:
                                bedInfo.IsAttendingVisible = false;
                                bedInfo.IsStaffVisible = true;
                                break;
                            case Constants.ATTENDING:
                                bedInfo.IsAttendingVisible = true;
                                bedInfo.IsStaffVisible = false;
                                break;
                            case Constants.STAFF_ATTENDING:
                                bedInfo.IsAttendingVisible = true;
                                bedInfo.IsStaffVisible = true;
                                break;
                        }

                        wardWhiteboardVM.WhiteboardList.Add(bedInfo);
                    }

                    reader.NextResult();
                    if (reader.HasRows)
                    {
                        // get facility census percent
                        while (reader.Read())
                        {
                            wardWhiteboardVM.FacilityOccupancyPercent = QuickParse<int>(reader, 0, 0);
                        }
                    }

                    reader.NextResult();
                    if (reader.HasRows)
                    {
                        // get ward facility census percent
                        while (reader.Read())
                        {
                            wardWhiteboardVM.DivisionOccupancyPercent = QuickParse<int>(reader, 0, 0);
                        }
                    }

                    reader.NextResult();
                    if (reader.HasRows)
                    {
                        wardWhiteboardVM.WardGroupCensusList = new List<WardGroupCensus>();
                        // footer census
                        while (reader.Read())
                        {
                            WardGroupCensus groupCensus = new WardGroupCensus();
                            groupCensus.WardGroup = QuickParse<string>(reader, "WARD_AND_TS", string.Empty);
                            groupCensus.OccupiedBeds = QuickParse<int>(reader, "OCCUPIED", 0);
                            groupCensus.FemaleBeds = QuickParse<int>(reader, "Female", 0);
                            groupCensus.MaleBeds = QuickParse<int>(reader, "Male", 0);
                            groupCensus.AssignedBeds = QuickParse<int>(reader, "ASSIGNED", 0);
                            groupCensus.AvailableBeds = QuickParse<int>(reader, "AVAILABLE", 0);
                            groupCensus.TotalBeds = QuickParse<int>(reader, "TOTAL", 0);

                            wardWhiteboardVM.WardGroupCensusList.Add(groupCensus);
                        }
                    }
                }
            }

            // set similar name icon
            foreach (WhiteboardBedInfo bedInfo in wardWhiteboardVM.WhiteboardList)
            {
                bedInfo.IsSimilarName = (nameAndSSNIndex.ContainsKey(bedInfo.Last4SSN) && nameAndSSNIndex[bedInfo.Last4SSN] > 1) || (nameAndSSNIndex.ContainsKey(bedInfo.First4LastName) && nameAndSSNIndex[bedInfo.First4LastName] > 1);

                if (bedInfo.IsSimilarName && wardWhiteboardVM.IconFlagsList != null)
                    bedInfo.SimilarNameIcon = wardWhiteboardVM.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == Constants.SIMILAR_SAME_NAME).First();

                if (bedInfo.SimilarNameIcon == null)
                    bedInfo.SimilarNameIcon = new IconInfo();
                else
                    bedInfo.SimilarNameIconBytes = bedInfo.SimilarNameIcon.ImageBytes;
            }

            //if we have an WardId set it to the ViewModel.
            if (!string.IsNullOrEmpty(divisionId))
            {
                Guid divisionExtension = new Guid(divisionId);

                if (divisionExtension != Guid.Empty && wardWhiteboardVM.WhiteboardList.Count > 0)
                    wardWhiteboardVM.DivisionName = wardWhiteboardVM.WhiteboardList[0].WardName;
                else
                    wardWhiteboardVM.DivisionName = "All";

                wardWhiteboardVM.DivisionExtension = divisionId;
            }

            wardWhiteboardVM.DisplayPTCode = displayPTCode ?? wardWhiteboardVM.DisplayPTList.First().code;
            wardWhiteboardVM.GenderColorCode = genderColorCode ?? wardWhiteboardVM.GenderColorList.First().code;
            wardWhiteboardVM.SplitScreen = splitScreen ?? wardWhiteboardVM.StrictDecisionList.First().code;
            wardWhiteboardVM.DisplayFooterCensus = displayFooterCensus ?? wardWhiteboardVM.StrictDecisionList.First(cd => cd.code == Constants.No).code;

            wardWhiteboardVM.DisplayPT = wardWhiteboardVM.DisplayPTList.First(cd => wardWhiteboardVM.DisplayPTCode == cd.code).displayName;
            wardWhiteboardVM.GenderColor = wardWhiteboardVM.GenderColorList.First(cd => wardWhiteboardVM.GenderColorCode == cd.code).displayName;
            wardWhiteboardVM.DisplayStaffAttendingSelected = displayStaffAttending;

            int scrollDirection, scrollRate;
            decimal scrollValue;
            decimal.TryParse(pScrollValue, out scrollValue);
            wardWhiteboardVM.ActualScrollValue = scrollValue;

            if (Int32.TryParse(pScrollDirection, out scrollDirection))
                wardWhiteboardVM.ActualScrollDirection = scrollDirection;
            else
                wardWhiteboardVM.ActualScrollDirection = 1;

            if (Int32.TryParse(pScrollRate, out scrollRate))
            {
                wardWhiteboardVM.ScrollRateSelected = scrollRate;
                wardWhiteboardVM.DisplayScrollSelected = Constants.Yes;
            }
            else
            {
                wardWhiteboardVM.ScrollRateSelected = 0;
                wardWhiteboardVM.DisplayScrollSelected = Constants.No;
            }
            wardWhiteboardVM.SortBySelected = sortBy;

            ViewBag.StartIndex = 0;
            ViewBag.EndIndex = wardWhiteboardVM.WhiteboardList.Count - 1;

            DateTime dtUtc = DateTime.UtcNow;
            DateTime dtTz = TimeZoneInfo.ConvertTimeFromUtc(dtUtc, FacadeUtil.GetFacilityTimeZoneInfo(this.loggedUser.Facility.Id));
            string dateOffset = dtTz.Subtract(dtUtc).TotalMinutes.ToString();

            // todo - consider making a direct call for the waiting count, this makes a service call
            wardWhiteboardVM.PatientsWaitingCount = FacadeManager.ADTInterface.GetPatientWaitingCount(this.loggedUser.Facility.Id, this.loggedUser.VistaSite.Id, wardWhiteboardVM.FacilitySettings.MedicalCenterID, int.Parse(dateOffset));

            wardWhiteboardVM.WardWhiteboardReportUrl = GetWhiteboardReportUrl(displayPTCode, genderColorCode, wardWhiteboardVM.SortBySelected, wardWhiteboardVM.DisplayStaffAttendingSelected, divisionId, ViewBag.FacilityCode);

            wardWhiteboardVM.SplitScreenSelected = GetSelectedStrictDecision(wardWhiteboardVM.SplitScreen);
            wardWhiteboardVM.DisplayFooterCensusSelected = GetSelectedStrictDecision(wardWhiteboardVM.DisplayFooterCensus);

            return wardWhiteboardVM;
        }

        #endregion

        private WardWhiteboardViewModel FillWhiteboardContent(string divisionId, string splitScreen, string displayPTCode, string genderColorCode, string sortBy, string displayFooterCensus, string displayStaffAttending, string pScrollRate, string pScrollValue, string pScrollDirection)
        {
            WardWhiteboardViewModel wardWhiteboardVM = new WardWhiteboardViewModel();
            FillViewModel(wardWhiteboardVM);

            //if we have an WardId set it to the ViewModel.
            if (!string.IsNullOrEmpty(divisionId))
                wardWhiteboardVM.DivisionExtension = divisionId;

            wardWhiteboardVM.DisplayPTCode = displayPTCode ?? wardWhiteboardVM.DisplayPTList.First().code;
            wardWhiteboardVM.GenderColorCode = genderColorCode ?? wardWhiteboardVM.GenderColorList.First().code;
            wardWhiteboardVM.SplitScreen = splitScreen ?? wardWhiteboardVM.StrictDecisionList.First().code;
            wardWhiteboardVM.DisplayFooterCensus = displayFooterCensus ?? wardWhiteboardVM.StrictDecisionList.First(cd => cd.code == Constants.No).code;

            wardWhiteboardVM.DisplayPT = wardWhiteboardVM.DisplayPTList.First(cd => wardWhiteboardVM.DisplayPTCode == cd.code).displayName;
            wardWhiteboardVM.GenderColor = wardWhiteboardVM.GenderColorList.First(cd => wardWhiteboardVM.GenderColorCode == cd.code).displayName;
            wardWhiteboardVM.DisplayStaffAttendingSelected = displayStaffAttending;

            int scrollDirection, scrollRate;
            decimal scrollValue;
            decimal.TryParse(pScrollValue, out scrollValue);
            wardWhiteboardVM.ActualScrollValue = scrollValue;

            if (Int32.TryParse(pScrollDirection, out scrollDirection))
                wardWhiteboardVM.ActualScrollDirection = scrollDirection;
            else
                wardWhiteboardVM.ActualScrollDirection = 1;

            if (Int32.TryParse(pScrollRate, out scrollRate))
            {
                wardWhiteboardVM.ScrollRateSelected = scrollRate;
                wardWhiteboardVM.DisplayScrollSelected = Constants.Yes;
            }
            else
            {
                wardWhiteboardVM.ScrollRateSelected = 0;
                wardWhiteboardVM.DisplayScrollSelected = Constants.No;
            }
            wardWhiteboardVM.SortBySelected = sortBy;

            SetWardSelected(wardWhiteboardVM);

            if (wardWhiteboardVM.DivisionList == null || wardWhiteboardVM.DivisionList.Count == 0)
                //there are no wards
                return null;
            if (wardWhiteboardVM.SelectedDivision == null)
            {
                wardWhiteboardVM.SelectedDivision = wardWhiteboardVM.DivisionList[0];
                wardWhiteboardVM.DivisionExtension = wardWhiteboardVM.SelectedDivision.Id.extension;
                wardWhiteboardVM.DivisionName = wardWhiteboardVM.SelectedDivision.Name;
            }

            //if we don't have a WardId we select the 1st one in the list
            if (string.IsNullOrEmpty(wardWhiteboardVM.DivisionExtension) && wardWhiteboardVM.DivisionList != null && wardWhiteboardVM.DivisionList.Count > 0)
                wardWhiteboardVM.DivisionExtension = wardWhiteboardVM.DivisionList[0].Id.extension;

            wardWhiteboardVM.IconFlagsList = FacadeManager.IconInterface.GetAllIconsByFacility(Guid.Parse(this.loggedUser.Facility.Id.extension)).Where(a => a.Active == true).ToList();

            bool isVirtual = true;
            bool allDivisionsSelected = true;
            List<Ward> wards = new List<Ward>();
            II selectedDivision = new II("", (default(Guid)).ToString());
            if (wardWhiteboardVM.SelectedDivision != null && wardWhiteboardVM.SelectedDivision.Id != null && !wardWhiteboardVM.SelectedDivision.Id.extension.Equals((default(Guid)).ToString()))
            {
                selectedDivision = wardWhiteboardVM.SelectedDivision.Id;
                allDivisionsSelected = false;
                wards = FacadeManager.EntityInterface.GetWardsByDivisionId(selectedDivision).ToList();

                // check to see if selected wards are virtual
                foreach (Ward ward in wards)
                {
                    if (!ward.IsVirtual)
                        isVirtual = false;
                }

                if (isVirtual)
                {
                    IList<Ward> allWards = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id);
                    List<Ward> childWards = new List<Ward>();
                    foreach (Ward ward in allWards)
                    {
                        if (ward.AssignedVirtualWard != null && wards.Count(item => String.Compare(item.Id.extension, ward.AssignedVirtualWard.Id.extension, StringComparison.InvariantCultureIgnoreCase) == 0) > 0)
                            childWards.Add(ward);
                    }

                    wards = childWards;
                }
            }
            else
            {
                isVirtual = false;
                wards = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).ToList();
            }

            string sort = GetSortValue(wardWhiteboardVM);


            Dictionary<string, List<BedInfo>> bedsByDivision;
            List<BedOccupancyCount> bedOccupancyList = FacadeManager.ADTInterface.GetBedOccupancyForFacility(this.loggedUser.Facility, out bedsByDivision);

            Dictionary<string, SpecialtiesHidden> specialtiesHidden = FacadeManager.VocabularyInterface.GetHiddenVistaSpecialties(wards);
            List<EMSNotification> emsList = FacadeManager.ConfigurationInterface.GetEMSNotifications(this.loggedUser.Domain, this.loggedUser.Facility.Id);

            wardWhiteboardVM.WhiteboardList = WhiteboardBedInfo.GetWhiteboardData(wardWhiteboardVM.IconFlagsList, wardWhiteboardVM.GenderColorCode, wardWhiteboardVM.DisplayPTCode, wardWhiteboardVM.DisplayStaffAttendingSelected, bedsByDivision[selectedDivision.extension], false, wards, specialtiesHidden, wardWhiteboardVM.IsFromWhiteboardKiosk == "1", emsList);

            wardWhiteboardVM.WardWhiteboardReportUrl = GetWhiteboardReportUrl(displayPTCode, genderColorCode, wardWhiteboardVM.SortBySelected, wardWhiteboardVM.DisplayStaffAttendingSelected, divisionId, ViewBag.FacilityCode);

            wardWhiteboardVM.WardGroupCensusList = new List<WardGroupCensus>();
            if (allDivisionsSelected)
            {
                foreach (Division division in wardWhiteboardVM.DivisionList)
                {
                    if (division.Id.extension != (default(Guid)).ToString())
                    {
                        var wardIds = from ward in wards
                                      where (ward.DivisionList.Where(a => a.Id.extension.Equals(division.Id.extension, StringComparison.InvariantCultureIgnoreCase) && a.Id.root.Equals(division.Id.root, StringComparison.InvariantCultureIgnoreCase)).Count() > 0)
                                      select ward.Id;
                        wardWhiteboardVM.WardGroupCensusList.Add(AddWardGroupCensus(bedsByDivision[division.Id.extension], division.Name, wardIds));
                    }
                }
            }
            else if (isVirtual)
            {
                float occupiedBeds = 0;
                float totalBeds = 0;
                foreach (Ward ward in wards)
                {
                    WardGroupCensus wardGroupCensus = AddWardGroupCensus(bedsByDivision[ward.Division.Id.extension], ward.Division.Name, null);
                    wardWhiteboardVM.WardGroupCensusList.Add(wardGroupCensus);

                    occupiedBeds += wardGroupCensus.OccupiedBeds;
                    totalBeds += wardGroupCensus.TotalBeds;

                    wardWhiteboardVM.DivisionOccupancyPercent = (int)System.Math.Round((occupiedBeds / totalBeds) * 100f, 0);
                }
            }
            else
            {
                wards = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).ToList();
                wardWhiteboardVM.WardGroupCensusList.Add(AddWardGroupCensus(bedsByDivision[wardWhiteboardVM.SelectedDivision.Id.extension], wardWhiteboardVM.SelectedDivision.Name, null));
            }

            if (wards != null)
            {
                BedOccupancyCount facilityBeds = bedOccupancyList.Where(item => item.DivisionName == "All").FirstOrDefault();
                if (facilityBeds != null)
                    wardWhiteboardVM.FacilityOccupancyPercent = facilityBeds.OccupancyPercent;
            }

            if (!isVirtual && wardWhiteboardVM.SelectedDivision.Id.extension != default(Guid).ToString())
            {
                BedOccupancyCount bedOccupancyCount = bedOccupancyList.Where(item => item.DivisionName == wardWhiteboardVM.SelectedDivision.Name).FirstOrDefault();
                if (bedOccupancyCount != null)
                    wardWhiteboardVM.DivisionOccupancyPercent = bedOccupancyCount.OccupancyPercent;
            }

            wardWhiteboardVM.SplitScreenSelected = GetSelectedStrictDecision(wardWhiteboardVM.SplitScreen);
            wardWhiteboardVM.DisplayFooterCensusSelected = GetSelectedStrictDecision(wardWhiteboardVM.DisplayFooterCensus);

            ViewBag.StartIndex = 0;
            ViewBag.EndIndex = wardWhiteboardVM.WhiteboardList.Count - 1;
            DateTime dtUtc = DateTime.UtcNow;
            DateTime dtTz = TimeZoneInfo.ConvertTimeFromUtc(dtUtc, FacadeUtil.GetFacilityTimeZoneInfo(this.loggedUser.Facility.Id));
            string dateOffset = dtTz.Subtract(dtUtc).TotalMinutes.ToString();
            wardWhiteboardVM.PatientsWaitingCount = FacadeManager.ADTInterface.GetPatientWaitingCount(this.loggedUser.Facility.Id, this.loggedUser.VistaSite.Id, wardWhiteboardVM.FacilitySettings.MedicalCenterID, int.Parse(dateOffset));
            return wardWhiteboardVM;
        }

        #endregion
    }
}
