﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using BMS.Facade.Data;
using BMS.Web.Models;
using InfoWorld.HL7.ITS;
using BMS.Web.App_GlobalResource;
using Microsoft.Web.Mvc;
using BMS.Facade;
using BMS.Web.Controllers.Shared;
using BMS.Utils;
using BMS.ServicesWrapper.EIS;
namespace BMS.Web.Controllers
{
    /// <summary>
    /// Ward Configuration Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class WardConfigurationController : BaseController
    {
        #region Public Methods

        /// <summary>
        ///  Default action for the WardConfiguration controller.
        /// </summary>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WardConfigurationViewModel model = new WardConfigurationViewModel();

                FillModel(model);

                if (string.IsNullOrEmpty(QueryStrings["wardId"]))
                    SetDefaultValueForDropDownList(model);
                else
                {
                    model.RootId = loggedUser.Domain;
                    model.ExtensionId = QueryStrings["wardId"];

                    WardViewModel ward = model.AllWards.Where(a => a.Model.Id.root.Equals(loggedUser.Domain, StringComparison.InvariantCultureIgnoreCase) && a.Model.Id.extension.Equals(QueryStrings["wardId"], StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.SelectedWardExtension = ward == null ? null : ward.Model.Id.extension;
                    model.TypeGroup = ward == null ? null : ward.TypeGroup;
                    model.WardGroupText = ward == null ? null : ward.DivisionName;

                    model.DisplaySpecialtyDesktop = ward == null ? true : ward.DisplaySpecialtyDesktop;
                    model.DisplaySpecialtyKiosk = ward == null ? true : ward.DisplaySpecialtyKiosk;
                    model.IsVirtual = ward == null ? false : ward.IsVirtual;
                    model.SelectedWardType = model.IsVirtual ? Strings.Virtual : Strings.VISTAWard;
                    model.WardName = ward == null ? null : ward.Model.Name;

                    if (model.IsVirtual)
                        model.SelectedWardType = Strings.Virtual;
                    else
                        model.SelectedWardType = Strings.VISTAWard;

                    model.AssignedVirtualWardExtension = ward == null || ward.Model.AssignedVirtualWard == null ? null : ward.Model.AssignedVirtualWard.Id.extension;

                    model.SelectedCensusCategory = ward == null || ward.Model.CensusCategory == null ? string.Empty : ward.Model.CensusCategory.code;
                }

                if (TempData.ContainsKey("ModelState"))
                {
                    ModelStateDictionary externalModelState = (ModelStateDictionary)TempData["ModelState"];
                    foreach (KeyValuePair<string, ModelState> valuePair in externalModelState)
                        ModelState.Add(valuePair.Key, valuePair.Value);
                }
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the WardConfiguration controller.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="btnSave">The BTN save.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(WardConfigurationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool hasWardName = !String.IsNullOrWhiteSpace(model.WardName);

                if (model.IsVistAWard || hasWardName)
                {
                    Ward ward = null;
                    if (!string.IsNullOrEmpty(model.SelectedWardExtension) && !model.SelectedWardExtension.Equals(Guid.Empty.ToString()))
                    {
                        II wardId = new II() { root = Constants.ADMISSIONROOT, extension = model.SelectedWardExtension };
                        ward = Facade.FacadeManager.EntityInterface.GetWardById(wardId);
                    }
                    else
                    {
                        if (QueryStrings.ContainsKey("wardId") && !String.IsNullOrEmpty(QueryStrings["wardId"]))
                        {
                            II wardId = new II() { root = loggedUser.Domain, extension = QueryStrings["wardId"] };
                            ward = Facade.FacadeManager.EntityInterface.GetWardById(wardId);
                        }
                        else
                            ward = new Ward() { Name = model.WardName };
                    }

                    FillModel(model);

                    if (ward != null)
                    {
                        bool isAdded = false;
                        ward.TypeGroup = model.TypeGroup.ToUpper();
                        //HHG INC000001259968: added AntiXssEncoder xss in order to decode model.WardGroupText to fix comparison bug.
                        AntiXssEncoder xss = new AntiXssEncoder(); 
                        if (ward.Division == null || ward.Division.Name != xss.Decode(model.WardGroupText.ToUpper()))
                        {
                            if (ward.Division == null)
                                isAdded = true;
                            ward.Division = new Division { Name = model.WardGroupText.ToUpper() };
                        }

                        model.IsVirtual = model.SelectedWardType == Strings.Virtual;
                        ward.IsVirtual = model.IsVirtual;
                        ward.HideSpecialtyDesktop = !model.DisplaySpecialtyDesktop;
                        ward.HideSpecialtyKiosk = !model.DisplaySpecialtyKiosk;

                        if (!String.IsNullOrEmpty(model.SelectedCensusCategory))
                            ward.CensusCategory = new CD { code = model.SelectedCensusCategory };
                        else
                            ward.CensusCategory = null;

                        ward.Name = model.WardName;

                        if (!String.IsNullOrEmpty(model.AssignedVirtualWardExtension) && model.AssignedVirtualWardExtension != Guid.Empty.ToString())
                        {
                            II vartualWardId = new II() { root = Constants.ADMISSIONROOT, extension = model.AssignedVirtualWardExtension };
                            ward.AssignedVirtualWard = Facade.FacadeManager.EntityInterface.GetWardById(vartualWardId);
                        }
                        else
                            ward.AssignedVirtualWard = null;

                        if (ward.Id == null)
                            EISFactory.Instance.CreateWard(ward);

                        Facade.FacadeManager.EntityInterface.UpdateWard(ward);
                        return this.RedirectToAction<WardConfigurationController>(act => act.AddEditWardConfirmation(EncryptQueryString(new string[] { "wardId", "sortByWard", "sortBySpecialty", "sortByTypeGroup", "sortByWardGroup", "isAddedMessageConfirmation" }, new string[] { ward.Id.extension, model.SortByVistaWardNameButton, model.SortByVistaSpecialtyButton, model.SortByVistaTypeGroupButton, model.SortByVistaWardGroupTextButton, isAdded.ToString() }, loggedUser.Salt)));
                    }
                    ModelState.Clear();
                    FillModel(model);
                    SetDefaultValueForDropDownList(model);
                }
                else
                {
                    ModelState.AddModelError("SelectedWardExtension", Strings.YouMustSelectAWard);
                    //Store modelstate from tempdata
                    TempData.Add("ModelState", ModelState);
                    FillModel(model);
                }
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Go to delete page.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Ward ward = FacadeManager.EntityInterface.GetWardById(new II() { root = loggedUser.Domain, extension = QueryStrings["wardId"] });
                AntiXssEncoder xss = new AntiXssEncoder();
                ward.TypeGroup = xss.Decode(ward.TypeGroup);
                ward.Division.Name = xss.Decode(ward.Division.Name);
                WardViewModel model = new WardViewModel() { Model = ward };
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Delete the ward.
        /// </summary>
        /// <param name="identifier">The identifier.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult DeleteWard(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facade.FacadeManager.EntityInterface.DeleteWard(new II() { root = loggedUser.Domain, extension = QueryStrings["wardId"] });
                return this.RedirectToAction<WardConfigurationController>(act => act.DeleteConfirmation(p));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Display delete confirmation page.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <param name="wardName">Name of the ward.</param>
        /// <param name="typeGroup">The type group.</param>
        /// <param name="wardGroup">The ward group.</param>
        /// <returns>DeleteConfirmation view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult DeleteConfirmation(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WardViewModel model = new WardViewModel();
                model.Model = new Ward();
                model.Model.Ien = QueryStrings["ien"];
                model.Model.Name = QueryStrings["wardName"];
                model.Model.TypeGroup = QueryStrings["typeGroup"];
                model.Model.Division = new Division() { Name = QueryStrings["wardGroup"] };
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Display update (add\edit) ward confirmation page.
        /// </summary>
        /// <param name="rootId">RootId of ward.</param>
        /// <param name="extensionId">ExtensionId of ward.</param>
        /// <param name="sortByVistaWardNameButton">Sort by VistA ward name.</param>
        /// <param name="sortByVistaSpecialtyButton">Sort by VistA specialty.</param>
        /// <param name="sortByVistaTypeGroupButton">Sort by type group.</param>
        /// <param name="sortByVistaWardGroupTextButton">Sort by division.</param>
        /// <param name="isAddedMessageConfirmation">True if is added new ward; otherwise,false.</param>
        /// <returns>DeleteConfirmation view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddEditWardConfirmation(string p)
        {
            AddEditWardViewModel model = new AddEditWardViewModel();
            model.SortByVistaWardNameButton = QueryStrings["sortByWard"];
            model.SortByVistaSpecialtyButton = QueryStrings["sortBySpecialty"];
            model.SortByVistaTypeGroupButton = QueryStrings["sortByTypeGroup"];
            model.SortByVistaWardGroupTextButton = QueryStrings["sortByWardGroup"];
            model.IsAddedMessageConfirmation = bool.Parse(QueryStrings["isAddedMessageConfirmation"]);
            model.RootId = loggedUser.Domain;
            model.ExtensionId = QueryStrings["wardId"];
            model.LoggedUser = loggedUser;
            Ward ward = FacadeManager.EntityInterface.GetWardById(new II() { root = model.RootId, extension = model.ExtensionId });
            AntiXssEncoder xss = new AntiXssEncoder();
            ward.TypeGroup = xss.Decode(ward.TypeGroup);
            ward.Division.Name = xss.Decode(ward.Division.Name);
            model.Ward = new WardViewModel() { Model = ward };
            return View(model);
        }

        /// <summary>
        /// Sorts the wards.
        /// </summary>
        /// <param name="buttonId">The button id.</param>
        /// <returns>Wards partial view.</returns>
        [HttpPost]
        public ActionResult SortWards(string buttonId)
        {
            AntiXssEncoder xss = new AntiXssEncoder();
            WardConfigurationViewModel model = new WardConfigurationViewModel();
            model.LoggedUser = loggedUser;

            model.ActiveWards = Facade.FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).Where(a => a.IsActive == true && a.Division != null).Select(x => new WardViewModel() { Model = x }).ToList();
            model.ActiveWards.ToList().ForEach(a => { a.Model.TypeGroup = xss.Decode(a.Model.TypeGroup); a.Model.Division.Name = xss.Decode(a.Model.Division.Name); });

            switch (buttonId)
            {
                case "VistaWardName":
                    model.ActiveWards = model.ActiveWards.OrderBy(a => a.Model.Name).ToList();
                    break;
                case "VistaSpeciality":
                    model.ActiveWards = model.ActiveWards.OrderBy(a => a.Model.Specialty.displayName).ToList();
                    break;
                case "VistaTypeGroup":
                    model.ActiveWards = model.ActiveWards.OrderBy(a => a.Model.TypeGroup).ToList();
                    break;
                case "VistaWardGroupText":
                    model.ActiveWards = model.ActiveWards.OrderBy(a => a.Model.Division.Name).ToList();
                    break;
            }

            return PartialView("Wards", model);
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Fill the model.
        /// </summary>
        /// <param name="model">The model.</param>
        private void FillModel(WardConfigurationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AntiXssEncoder xss = new AntiXssEncoder();
                model.Wards = Facade.FacadeManager.EntityInterface.GetWardsByVista(this.loggedUser.VistaSite.Id).Where(a => a.IsCurrentlyOOS == false && String.Compare(a.Name, "0") > 0 && String.Compare(a.Name, "ZY") < 0).Select(x => new WardViewModel() { Model = new Ward() { Address1 = x.Address1, Division = x.Division, Id = x.Id, Ien = x.Ien, IsActive = x.IsActive, Name = x.Name + " {" + Strings.IEN + ":" + x.Ien.ToString() + "}", Specialty = x.Specialty, TypeGroup = x.TypeGroup } }).ToList();

                Dictionary<string, Ward> activeWards = Facade.FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).ToDictionary(w => w.Id.extension);
                model.AllWards = new List<WardViewModel>();
                model.ActiveWards = new List<WardViewModel>();
                model.VirtualWards = new List<WardViewModel>();
                model.VirtualWardList = new List<Ward>();

                foreach (Ward ward in activeWards.Values)
                {
                    if (!ward.IsActive || ward.Division == null)
                        continue;

                    Ward virtualWard = null;
                    if (ward.AssignedVirtualWard != null)
                    {
                        activeWards.TryGetValue(ward.AssignedVirtualWard.Id.extension, out virtualWard);
                        ward.AssignedVirtualWard = virtualWard;
                    }

                    WardViewModel wvm = new WardViewModel() { Model = ward };
                    wvm.Model.TypeGroup = xss.Decode(ward.TypeGroup);
                    wvm.Model.Division.Name = xss.Decode(ward.Division.Name);

                    model.AllWards.Add(wvm);

                    if (!ward.IsVirtual)
                        model.ActiveWards.Add(wvm);
                    else
                    {
                        model.VirtualWardList.Add(ward);
                        model.VirtualWards.Add(wvm);
                    }
                }

                model.VirtualWardList.Insert(0, new Ward() { Name = Strings.SelectAWardUpperCase, Id = new II() { root = Strings.SelectAWardUpperCase, extension = Guid.Empty.ToString() } });

                //Add Select A WARD option to the drop down list.
                model.Wards.Insert(0, new WardViewModel() { Model = new Ward() { Name = Strings.SelectAWardUpperCase, Id = new II() { root = Strings.SelectAWardUpperCase, extension = Guid.Empty.ToString() } } });

                //Add the current facility name to the model.
                model.FacilityName = this.loggedUser.Facility.Name;
                model.LoggedUser = loggedUser;

                model.CensusCategories = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.CensusCategory);
                model.CensusCategories.Insert(0, new CD { code = string.Empty, displayName = Strings.None });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the default value for drop down list.
        /// </summary>
        /// <param name="model">The model.</param>
        private void SetDefaultValueForDropDownList(WardConfigurationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.SelectedWardExtension = Guid.Empty.ToString();
                model.TypeGroup = string.Empty;
                model.WardGroupText = string.Empty;
                model.SelectedCensusCategory = string.Empty;
                model.DisplaySpecialtyKiosk = true;
                model.DisplaySpecialtyDesktop = true;
                model.SelectedWardType = Strings.VISTAWard;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
