﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Models;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class UserConfigurationController : BaseController
    {
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            UserConfigurationViewModel model = new UserConfigurationViewModel();
            model.SiteName = this.loggedUser.Facility.Code + " - " + this.loggedUser.Facility.Name;
            model.VISNNo = this.loggedUser.Visn.Number;
            model.RegionNo = this.loggedUser.Visn.Region.Number.ToString();
            model.StrictDecisions = Facade.FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.StrictDecision);
            model.LoggedUser = this.loggedUser;

            if (QueryStrings.ContainsKey("bulkMode"))
            {
                List<SearchUser> usersToConfigure = TempData["AddedUsers"] as List<SearchUser>;

                if (QueryStrings["bulkMode"] == "Add")
                {
                    model.BulkUserIds = String.Empty;

                    foreach (SearchUser userToAdd in usersToConfigure)
                    {
                        model.BulkUserIds += userToAdd.User.Id.extension + "|";
                    }

                    model.Data = usersToConfigure.First().User;
                }
                else
                {
                    model.BulkRevokeUsers = new List<string>();
                    foreach (SearchUser userToAdd in usersToConfigure)
                    {
                        model.BulkRevokeUsers.Add(userToAdd.User.FullName);
                    }

                    model.Data = usersToConfigure.First().User;
                    TempData["AddedUsers"] = usersToConfigure;
                    return View("ConfirmBulkRevoke", model);
                }
            }
            else
                model.Data = GetUser(string.IsNullOrEmpty(p) ? null : QueryStrings["id"]);

            if (!model.IsDefaultUser)
            {
                FacadeManager.UserInterface.FillUserData(model.Data);
                PermissionFacility permissionFacility = FacadeManager.UserInterface.GetFacilitiesForReadAndWrite(model.Data.UserName, new List<Facility>() { this.loggedUser.Facility }).FirstOrDefault();
                if (permissionFacility != null)
                {
                    model.Data.ReadAccess = permissionFacility.ReadAccess == true ? model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() : model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.Data.WriteAccess = permissionFacility.WriteAccess == true ? model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() : model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.Data.WriteAccessWhiteboard = permissionFacility.WriteAccessWhiteboard == true ? model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() : model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
                else
                {
                    model.Data.ReadAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.Data.WriteAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.Data.WriteAccessWhiteboard = model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                }
            }
            model.LoggedUser = loggedUser;

            IReportInfo userAccessReport = FacadeManager.ReportsInterface.GetOtherReports(FullUserName).Where(reportInfo => reportInfo.Name == "UserAccess").FirstOrDefault();
            if (userAccessReport != null)
            {
                userAccessReport.Parameters = new Dictionary<string, string>();
                userAccessReport.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, Url.Action("Index", "UserConfiguration"));
                userAccessReport.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToUserConfiguration);
                userAccessReport.Parameters.Add(Constants.REPORT_TITLE, "User Access Report");
                userAccessReport.Parameters.Add(Constants.REP_CURRENT_USER, this.loggedUser.UserName);
                userAccessReport.Parameters.Add(Constants.REP_VISN_EXTENSIONS, this.loggedUser.Visn.Id.extension.ToUpper());
                userAccessReport.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION_TO_UPPER, this.loggedUser.Facility.Id.extension.ToUpper());

                model.UserAccessReportLink = userAccessReport.Url;
            }

            return View(model);
        }

        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index([Bind(Exclude = "RememberMe")] UserConfigurationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = loggedUser;
                if (model.Data.UserName == Constants.DEFAULT_USER_PROFILE)
                {
                    ModelState.AddModelError(string.Empty, "Select an existing NT User Name.");
                    return View(model);
                }

                if (model.Data.ReadAccess == null || model.Data.ReadAccess.code == Constants.No)
                {
                    ModelState.AddModelError(string.Empty, "Read Access is required for this facility.");
                    model.SiteName = this.loggedUser.Facility.Code + " - " + this.loggedUser.Facility.Name;
                    model.VISNNo = this.loggedUser.Visn.Number;
                    model.RegionNo = this.loggedUser.Visn.Region.Number.ToString();
                    model.StrictDecisions = Facade.FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.StrictDecision);
                    return View(model);
                }

                if (!String.IsNullOrEmpty(model.BulkUserIds))
                {
                    model.Data.Facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, this.loggedUser.Facility.Id.extension));
                    string[] userIds = model.BulkUserIds.Split(new char[] { '|' }, StringSplitOptions.RemoveEmptyEntries);

                    foreach (string userId in userIds)
                    {
                        User userToConfigure = GetUser(userId);
                        model.Data.CopyPermissionsTo(userToConfigure);

                        FacadeManager.ConfigurationInterface.SaveUserProfile(userToConfigure);
                        if (!userToConfigure.IsDefaultUser)
                        {
                            PermissionFacility permissionFacility = new PermissionFacility();
                            permissionFacility.Facility = this.loggedUser.Facility;
                            permissionFacility.ReadAccess = true;

                            if (model.Data.WriteAccess != null && model.Data.WriteAccess.code == Constants.Yes)
                                permissionFacility.WriteAccess = true;
                            else
                                permissionFacility.WriteAccess = false;

                            if (model.Data.WriteAccessWhiteboard != null && model.Data.WriteAccessWhiteboard.code == Constants.Yes)
                                permissionFacility.WriteAccessWhiteboard = true;
                            else
                                permissionFacility.WriteAccessWhiteboard = false;

                            FacadeManager.UserInterface.SaveUserData(userToConfigure, permissionFacility);
                        }
                    }
                }
                else
                {
                    model.Data.Facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, this.loggedUser.Facility.Id.extension));
                    FacadeManager.ConfigurationInterface.SaveUserProfile(model.Data);
                    if (!model.IsDefaultUser)
                    {
                        PermissionFacility permissionFacility = new PermissionFacility();
                        permissionFacility.Facility = this.loggedUser.Facility;
                        permissionFacility.ReadAccess = true;

                        if (model.Data.WriteAccess != null && model.Data.WriteAccess.code == Constants.Yes)
                            permissionFacility.WriteAccess = true;
                        else
                            permissionFacility.WriteAccess = false;

                        if (model.Data.WriteAccessWhiteboard != null && model.Data.WriteAccessWhiteboard.code == Constants.Yes)
                            permissionFacility.WriteAccessWhiteboard = true;
                        else
                            permissionFacility.WriteAccessWhiteboard = false;

                        FacadeManager.UserInterface.SaveUserData(model.Data, permissionFacility);
                    }
                }

                return View("UserEditHasSaved", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        public ActionResult SelectUser()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SelectUserModel model = new SelectUserModel();
                
                model.SearchedUserIdsSelected = new List<string>();
                model.AddedUserIdsSelected = new List<string>();
                
                model.Users = FacadeManager.UserInterface.SearchUsers(model.SelectedDomain, model.SearchText);                
                model.LoggedUser = loggedUser;

                ConfigureSelectUserModel(model);

                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult SearchUser([Bind(Exclude = "RememberMe")] SelectUserModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ConfigureSelectUserModel(model);
                model.SearchedUserIdsSelected = new List<string>();
                model.AddedUserIdsSelected = new List<string>();

                model.Users = FacadeManager.UserInterface.SearchUsers(model.SelectedDomain, !string.IsNullOrEmpty(model.SearchText) ? model.SearchText.Trim() : model.SearchText);

                TempData["AddedUsers"] = model.AddedUsers;
                TempData["SearchedUsers"] = model.SearchedUsers;

                model.LoggedUser = loggedUser;

                return View("SelectUser", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult SelectUser([Bind(Exclude = "RememberMe")] SelectUserModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ConfigureSelectUserModel(model);

                if (!String.IsNullOrEmpty(model.Add))
                {
                    // Add users
                    foreach (string searchUserId in model.SearchedUserIdsSelected)
                    {
                        SearchUser searchUser = model.SearchedUsers.Where(u => u.User.Id.extension == searchUserId).FirstOrDefault();
                        bool alreadyAdded = false;

                        foreach (SearchUser addedUser in model.AddedUsers)
                        {
                            if (addedUser.User.Id == searchUser.User.Id)
                            {
                                alreadyAdded = true;
                                break;
                            }
                        }

                        if (alreadyAdded)
                            continue;

                        model.AddedUsers.Add(searchUser);
                    }

                    TempData["AddedUsers"] = model.AddedUsers;
                    TempData["SearchedUsers"] = model.SearchedUsers;
                    return View("SelectUser", model);
                }
                else if (!String.IsNullOrEmpty(model.Remove))
                {
                    // Remove added users
                    foreach (string addedUserIdToRemove in model.AddedUserIdsSelected)
                    {
                        SearchUser addedUserToRemove = model.AddedUsers.Where(u => u.User.Id.extension == addedUserIdToRemove).FirstOrDefault();
                        model.AddedUsers.Remove(addedUserToRemove);
                    }

                    TempData["AddedUsers"] = model.AddedUsers;
                    TempData["SearchedUsers"] = model.SearchedUsers;
                    return View("SelectUser", model);
                }
                else if (!String.IsNullOrEmpty(model.BulkMode))
                {
                    TempData["AddedUsers"] = model.AddedUsers;
                    string bulkMode = model.BulkMode == "Bulk Activate Users" ? "Add" : "Revoke";

                    if (bulkMode == "Add")
                    {
                        // make sure no user has roles
                        List<string> invalidUsers = new List<string>();
                        foreach (SearchUser addedUser in model.AddedUsers)
                        {
                            if (addedUser.HasRole)
                                invalidUsers.Add(addedUser.User.FullName);
                        }

                        if (invalidUsers.Count > 0)
                        {
                            TempData["SearchedUsers"] = model.SearchedUsers;
                            model.AddRevokeMessage = "The following users have roles and cannot be part of a bulk activate: " + string.Join("; ", invalidUsers);
                            return View("SelectUser", model);
                        }
                    }

                    return this.RedirectToAction<UserConfigurationController>(act => act.Index(EncryptQueryString(new string[] { "id", "bulkMode" }, new string[] { model.SelectedUserId, bulkMode }, loggedUser.Salt)));
                }

                return this.RedirectToAction<UserConfigurationController>(act => act.Index(EncryptQueryString(new string[] { "id" }, new string[] { model.SelectedUserId }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter(OperationOverride = "UserConfiguration, Index")]
        public ActionResult RevokeUsers([Bind(Exclude = "RememberMe")] UserConfigurationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.LoggedUser = loggedUser;

                List<SearchUser> usersToConfigure = TempData["AddedUsers"] as List<SearchUser>;
                foreach (SearchUser userToAdd in usersToConfigure)
                {
                    userToAdd.User.Facility = this.loggedUser.Facility;

                    FacadeManager.UserInterface.RevokeUserPermissions(userToAdd.User);
                }

                return View("UserEditHasSaved", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void ConfigureSelectUserModel(SelectUserModel model)
        {
            model.AddRevokeMessage = String.Empty;
            model.Domains = FacadeManager.UserInterface.GetUserDomains();

            if (String.IsNullOrEmpty(model.SelectedDomain))
            {
                string adDomain = this.loggedUser.GetADDomain();
                if (model.Domains.Contains(adDomain))
                    model.SelectedDomain = adDomain;
                else
                    model.SelectedDomain = model.Domains.First();
            }

            if (TempData.ContainsKey("SearchedUsers") && TempData["SearchedUsers"] != null)
                model.SearchedUsers = TempData["SearchedUsers"] as List<SearchUser>;
            else
                model.SearchedUsers = new List<SearchUser>();

            if (TempData.ContainsKey("AddedUsers") && TempData["AddedUsers"] != null)
                model.AddedUsers = TempData["AddedUsers"] as List<SearchUser>;
            else
                model.AddedUsers = new List<SearchUser>();
        }
    }
}
