﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.Utils.Properties;
using Microsoft.Web.Mvc;
using BMS.Web.App_GlobalResource;
using BMS.Facade.Fault;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Ems Notification Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class EmsNotificationController : BaseController
    {
        /// <summary>
        /// Default action for the EmsNotificationController.
        /// </summary>
        /// <returns>Returns the Ems Notification view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsNotificationViewModel emsNotificationViewModel = new EmsNotificationViewModel();
                bool isLocationRequired = false;
                if (!string.IsNullOrEmpty(p) && bool.TryParse(QueryStrings["isLocationRequired"], out isLocationRequired) && isLocationRequired)
                    ModelState.AddModelError("Location", Strings.YourDidNotSelectAWardGroup);
                FillModel(emsNotificationViewModel);
                return View(emsNotificationViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult Index(EmsNotificationViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (input.Location == Strings.SelectWardGroup)
                    return this.RedirectToAction<EmsNotificationController>(act => act.Index(EncryptQueryString(new string[] { "isLocationRequired" }, new string[] { "true" }, loggedUser.Salt)));
                return this.RedirectToAction<EmsNotificationController>(act => act.AddEdit(EncryptQueryString(new string[] { "location", "isAddOperation" }, new string[] {input.Location, "true" }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show AddEdit view when click Add button or Edit link.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns>Returns the AddEdit view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddEdit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsNotificationViewModel emsNotificationViewModel = new EmsNotificationViewModel();
                emsNotificationViewModel.LoggedUser = loggedUser;
                emsNotificationViewModel.IsAddOperation = bool.Parse(QueryStrings["isAddOperation"]);
                string location = GetLocationName(QueryStrings["location"]);
                EMSNotification emsNotification = FacadeManager.ConfigurationInterface.GetEMSNotification(location, this.loggedUser.Domain, this.loggedUser.Facility.Id);                
                // Add button
                if (emsNotificationViewModel.IsAddOperation)
                {
                    emsNotificationViewModel.Location = location;
                    emsNotificationViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                    emsNotificationViewModel.EmsDirty = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.EmsClean = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.BedControllerDirty = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.BedControllerClean = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.VistaDirty = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.VistaClean = GetFlag(false, emsNotificationViewModel.FlagsList);
                    emsNotificationViewModel.IsAddOperation = true;
                    emsNotificationViewModel.Header = Constants.NotificationsAdd;
                    emsNotificationViewModel.DisplayTextForSubmitButton = Strings.Save;
                    if (emsNotification == null)
                    {
                        emsNotificationViewModel.EMSAddEditDisplay = string.Empty;
                        emsNotificationViewModel.EMSAlreadyExistsDisplay = Strings.StyleDisplayNone;
                        emsNotificationViewModel.DisplayMessageWhenEmsNotificationAlreadyExist = string.Empty;
                    }
                    else
                    {
                        emsNotificationViewModel.EMSAlreadyExistsDisplay = string.Empty;
                        emsNotificationViewModel.EMSAddEditDisplay = Strings.StyleDisplayNone;
                        emsNotificationViewModel.DisplayMessageWhenEmsNotificationAlreadyExist = string.Format(Strings.NotificationLocationAlreadyExist, emsNotificationViewModel.Location);
                    }
                    return View(emsNotificationViewModel);
                }
                else
                {
                    // Edit link                    
                    if (emsNotification != null)
                    {
                        emsNotificationViewModel.Location = emsNotification.Location;
                        emsNotificationViewModel.EmsEmail = emsNotification.EMSEmail;
                        emsNotificationViewModel.BedControllerEmail = emsNotification.BedControllerEmail;
                        emsNotificationViewModel.VistaEmail = emsNotification.VistaGroupMail;
                        emsNotificationViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);

                        emsNotificationViewModel.EmsDirty = emsNotification.EMSDirty;
                        emsNotificationViewModel.EmsClean = emsNotification.EMSCleaned;
                        emsNotificationViewModel.BedControllerDirty = emsNotification.BedControllerDirty;
                        emsNotificationViewModel.BedControllerClean = emsNotification.BedControllerCleaned;
                        emsNotificationViewModel.VistaDirty = emsNotification.VistaGroupDirty;
                        emsNotificationViewModel.VistaClean = emsNotification.VistaGroupCleaned;
                        emsNotificationViewModel.Header = Constants.EMSBedNotificationEdit;
                        emsNotificationViewModel.DisplayTextForSubmitButton = Strings.Submit;                        
                        emsNotificationViewModel.EMSAddEditDisplay = string.Empty;
                        emsNotificationViewModel.EMSAlreadyExistsDisplay = Strings.StyleDisplayNone;
                        return View(emsNotificationViewModel);
                    }
                    else
                        throw new ConfigurationFault(Resources.CONFIGURATION_GET_EMS_NOTIFICATION_NOT_EXISTS);                 
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for AddEdit view.
        /// </summary>
        /// <param name="emsNotificationViewModel">The ems notification view model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult AddEdit(EmsNotificationViewModel emsNotificationViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!ModelState.IsValid)
                {
                    emsNotificationViewModel.Header = Constants.EMSBedNotificationEdit;
                    emsNotificationViewModel.DisplayTextForSubmitButton = Strings.Submit;
                    emsNotificationViewModel.LoggedUser = loggedUser;
                    emsNotificationViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                    return View("AddEdit", emsNotificationViewModel);
                }
                if (emsNotificationViewModel.IsAddOperation)
                    emsNotificationViewModel.DisplayMessageWhenSuccessfullyAddEditNotification = Strings.YouHaveAddedThisRecordSuccessfullyInTheLocalDatabase;                
                else
                    emsNotificationViewModel.DisplayMessageWhenSuccessfullyAddEditNotification = Strings.YouHaveEditedThisRecordSuccessfullyInTheLocalDatabase;
                emsNotificationViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                emsNotificationViewModel.EmsClean = GetFlag(emsNotificationViewModel.EmsClean.code, emsNotificationViewModel.FlagsList);
                emsNotificationViewModel.EmsDirty = GetFlag(emsNotificationViewModel.EmsDirty.code, emsNotificationViewModel.FlagsList);
                emsNotificationViewModel.BedControllerClean = GetFlag(emsNotificationViewModel.BedControllerClean.code, emsNotificationViewModel.FlagsList);
                emsNotificationViewModel.BedControllerDirty = GetFlag(emsNotificationViewModel.BedControllerDirty.code, emsNotificationViewModel.FlagsList);
                emsNotificationViewModel.VistaClean = GetFlag(emsNotificationViewModel.VistaClean.code, emsNotificationViewModel.FlagsList);
                emsNotificationViewModel.VistaDirty = GetFlag(emsNotificationViewModel.VistaDirty.code, emsNotificationViewModel.FlagsList);
                FacadeManager.ConfigurationInterface.SaveEMSNotification(EmsNotificationViewModel.TranslateFromViewModelToDomainObject(emsNotificationViewModel), this.loggedUser.Domain, emsNotificationViewModel.IsAddOperation, this.loggedUser.Facility.Id);
                return this.RedirectToAction<EmsNotificationController>(act => act.AddEditAction(EncryptQueryString(new string[] { "displayMessageSuccess" }, new string[] { emsNotificationViewModel.DisplayMessageWhenSuccessfullyAddEditNotification }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show AddEditAction view.
        /// </summary>
        /// <returns>Returns the AddEditAction view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddEditAction(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsNotificationViewModel model = new EmsNotificationViewModel();
                model.DisplayMessageWhenSuccessfullyAddEditNotification = QueryStrings["displayMessageSuccess"];
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }   

        /// <summary>
        /// Show Delete view when click Delete button.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns>Returns the Delete view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsNotificationViewModel emsNotificationViewModel = new EmsNotificationViewModel();
                emsNotificationViewModel.Location = GetLocationName(QueryStrings["location"]);
                emsNotificationViewModel.LoggedUser = loggedUser;
                emsNotificationViewModel.EmsClean = new CD();
                emsNotificationViewModel.EmsDirty = new CD();
                emsNotificationViewModel.BedControllerClean = new CD();
                emsNotificationViewModel.BedControllerDirty = new CD();
                emsNotificationViewModel.VistaClean = new CD();
                emsNotificationViewModel.VistaDirty = new CD();
                return View(emsNotificationViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show DeleteAction view.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns>Returns the DeleteAction view result.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Delete([Bind(Exclude = "RememberMe")] EmsNotificationViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AntiXssEncoder xss = new AntiXssEncoder();
                EMSNotification emsNotification = FacadeManager.ConfigurationInterface.GetEMSNotification(xss.Decode(input.Location), this.loggedUser.Domain, this.loggedUser.Facility.Id);
                FacadeManager.ConfigurationInterface.DeleteEMSNotification(emsNotification, this.loggedUser.Domain, this.loggedUser.Facility.Id);
                input.LoggedUser = loggedUser;
                return View("DeleteAction", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        /// <summary>
        /// Gets the flag.
        /// </summary>
        /// <param name="isFlagged">if set to <c>true</c> [is flagged].</param>
        /// <param name="flagList">The flag list.</param>
        /// <returns></returns>
        private CD GetFlag(bool isFlagged, CDList flagList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (isFlagged)
                    return flagList.Find(cd => cd.code == Constants.Yes);
                else
                    return flagList.Find(cd => cd.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the flag.
        /// </summary>
        /// <param name="isFlagged">The is flagged.</param>
        /// <param name="flagList">The flag list.</param>
        /// <returns></returns>
        private CD GetFlag(string isFlagged, CDList flagList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (isFlagged == Constants.Yes)
                    return flagList.Find(cd => cd.code == Constants.Yes);
                else
                    return flagList.Find(cd => cd.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the model.
        /// </summary>
        /// <param name="emsNotificationViewModel">The ems notification view model.</param>
        private void FillModel(EmsNotificationViewModel emsNotificationViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                emsNotificationViewModel.LoggedUser = this.loggedUser;
                EmsNotificationViewModel.FacilityName = emsNotificationViewModel.LoggedUser.Facility.Name;
                emsNotificationViewModel.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                List<EMSNotification> emsList = FacadeManager.ConfigurationInterface.GetEMSNotifications(emsNotificationViewModel.LoggedUser.Domain, this.loggedUser.Facility.Id);
                List<EMSNotification> tempEms = new List<EMSNotification>();
                foreach (EMSNotification ems in emsList)
                {
                    if (emsNotificationViewModel.WardGroupList.Where(a => a.Name.Equals(ems.Location, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                        tempEms.Add(ems);
                }
                emsNotificationViewModel.EMSNotificationList = tempEms;
                tempEms = null;
                foreach (EMSNotificationItemViewModel item in emsNotificationViewModel.EMSNotificationListExtended)
                    item.LocationExtension = emsNotificationViewModel.WardGroupList.Where(a => a.Name.Equals(item.Model.Location, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Id.extension;
                emsNotificationViewModel.IsAddOperation = false;
                emsNotificationViewModel.EmsClean = new CD();
                emsNotificationViewModel.EmsDirty = new CD();
                emsNotificationViewModel.BedControllerClean= new CD();
                emsNotificationViewModel.BedControllerDirty = new CD();
                emsNotificationViewModel.VistaClean = new CD();
                emsNotificationViewModel.VistaDirty = new CD();
                emsNotificationViewModel.Header = Constants.NotificationsAdd;
                emsNotificationViewModel.DisplayTextForSubmitButton = Strings.Save;
                //Add Select a Ward Group option to the drop down list.
                emsNotificationViewModel.WardGroupList.Insert(0, new Division() { Id = new II() { root = this.loggedUser.Domain, extension = Strings.SelectWardGroup }, Name = Strings.SelectWardGroup });                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the name of the location.
        /// </summary>
        /// <param name="emsNotificationViewModel">The ems notification view model.</param>
        /// <param name="location">The location.</param>
        /// <returns>Name of location.</returns>
        private string GetLocationName(string location)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Division> wardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id);                
                foreach (Division obj in wardGroupList)
                {
                    if (obj.Id.extension.Equals(location, StringComparison.InvariantCultureIgnoreCase))
                        return obj.Name;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        
        #endregion
    }
}
