﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.ServicesWrapper.EVS;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Models;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// ADT Orderable Item Controller class.
    /// </summary>
    public class AdtOrderableItemsController : BaseController
    {
        /// <summary>
        ///  Default action for the ADT Orderable Item controller.
        /// </summary>
        /// <returns>The index view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdtOrderableItemsViewModel viewModel = new AdtOrderableItemsViewModel();
                FillViewModel(viewModel, false);
                return View(viewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the ADT Orderable Item controller - used for the Add button.
        /// </summary>
        /// <param name="input">The input(ADT Orderable Item view model).</param>
        /// <returns>The Confirmation view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(AdtOrderableItemsViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(input.ButtonLoadOI))
                {
                    FillViewModel(input, true);
                    ModelState.Clear();
                    return View(input);
                }
                else
                {
                    input.LoggedUser = this.loggedUser;
                    input.OrderableItemTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.OrderableItemType);
                    input.ADTOrderableItemList = FacadeManager.VocabularyInterface.GetAdtOrderableItems();

                    string selectedOrderableItemTypeText = input.OrderableItemTypeList.SingleOrDefault(a => a.code == input.OrderableItemTypeCode).displayName;
                    CD _adtOrderableItemSelected = EVSFactory.InstanceFromWCF.SearchConceptsDB(new CodeFilterParameters() { MatchAlgorithm = MatchAlgorithm.IdenticalIgnoreCase, MatchText = input.ADTOrderableItemCode, VocabularyDomain = Util.Vocabulary.OrderableItem.ToString(), MatchProperty = MatchProperty.Code }).FirstOrDefault();
                    string[] adtOrderableItemArray = _adtOrderableItemSelected.code.Split('_');
                    string adtCode = adtOrderableItemArray[0] + "_" + input.LoggedUser.Facility.Code + "_" + adtOrderableItemArray[1];
                    bool _isExistsAnOrderableItem = false;

                    //check to see if there is an orderable item in the DB with a different type
                    if (input.ADTOrderableItemListExtended != null)
                        _isExistsAnOrderableItem = (from orderableItemListExtended in input.ADTOrderableItemListExtended
                                                    where (orderableItemListExtended.OrderableItemIEN == adtOrderableItemArray[1] && orderableItemListExtended.OrderableItemTypeText != selectedOrderableItemTypeText)
                                                    select orderableItemListExtended).SingleOrDefault() == null ? false : true;


                    //check to see if already exists an orderableitem in DB            
                    OrderableItemViewModel itemViewModel = (from orderableItemListExtended in input.FacilityADTOrderableItemListExtended
                                                            where orderableItemListExtended.Model.code == adtCode
                                                            select orderableItemListExtended).SingleOrDefault();
                    if (!_isExistsAnOrderableItem)
                    {
                        if (itemViewModel == null)
                        {
                            itemViewModel = new OrderableItemViewModel();
                            itemViewModel.LoggedUser = this.loggedUser;

                            itemViewModel.Model = new CDWithProperties(adtCode, Constants.ADTOrderableItemCodeSystem, Constants.ADTOrderableItem, null, _adtOrderableItemSelected.displayName, null, null, null, new List<ConceptProperty>());
                            ConceptProperty property = new ConceptProperty(new ST("OrderableItemType"), new ST(input.OrderableItemTypeCode), new ST(Constants.EVS_Language_code), new ST(Constants.MIME_TEXT), null);
                            itemViewModel.Model.Properties.Add(property);
                            itemViewModel.OrderableItemTypeText = selectedOrderableItemTypeText;
                            try
                            {
                                itemViewModel.OrderableItemIEN = input.ADTOrderableItemCode.Substring(input.ADTOrderableItemCode.IndexOf("_") + 1);
                            }
                            catch (Exception) { itemViewModel.OrderableItemIEN = string.Empty; }

                            itemViewModel.AddAlreadyExistsConfirmationVisible = Strings.None.ToLower();
                            itemViewModel.AddConfirmationVisible = Strings.Block.ToLower();
                            itemViewModel.DeleteConfirmationVisible = Strings.None.ToLower();
                            itemViewModel.AddADTOrderableItemWithADifferentTypeConfirmationVisible = Strings.None.ToLower();
                            FacadeManager.VocabularyInterface.InsertConcept(itemViewModel.Model);
                        }
                        else
                        {
                            itemViewModel.LoggedUser = this.loggedUser;
                            itemViewModel.AddAlreadyExistsConfirmationVisible = Strings.Block.ToLower();
                            itemViewModel.AddConfirmationVisible = Strings.None.ToLower();
                            itemViewModel.DeleteConfirmationVisible = Strings.None.ToLower();
                            itemViewModel.AddADTOrderableItemWithADifferentTypeConfirmationVisible = Strings.None.ToLower();
                        }
                    }
                    else
                    {
                        if (itemViewModel == null)
                        {
                            itemViewModel = new OrderableItemViewModel();
                            try
                            {
                                itemViewModel.OrderableItemIEN = input.ADTOrderableItemCode.Substring(input.ADTOrderableItemCode.IndexOf("_") + 1);
                            }
                            catch (Exception) { itemViewModel.OrderableItemIEN = string.Empty; }
                            itemViewModel.OrderableItemTypeText = selectedOrderableItemTypeText;
                            itemViewModel.Model = new CDWithProperties();
                            itemViewModel.Model.displayName = _adtOrderableItemSelected.displayName;
                        }
                        itemViewModel.LoggedUser = this.loggedUser;
                        itemViewModel.AddADTOrderableItemWithADifferentTypeConfirmationVisible = Strings.Block.ToLower();
                        itemViewModel.AddAlreadyExistsConfirmationVisible = Strings.None.ToLower();
                        itemViewModel.AddConfirmationVisible = Strings.None.ToLower();
                        itemViewModel.DeleteConfirmationVisible = Strings.None.ToLower();
                    }

                    return View("Confirmation", itemViewModel);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for Delete operation.
        /// </summary>
        /// <param name="id">The id of the Orderable Item selected.</param>
        /// <returns>The Confirmation view.</returns>
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdtOrderableItemsViewModel adtViewmodel = new AdtOrderableItemsViewModel();
                adtViewmodel.LoggedUser = this.loggedUser;
                adtViewmodel.OrderableItemTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.OrderableItemType);
                adtViewmodel.ADTOrderableItemList = FacadeManager.VocabularyInterface.GetAdtOrderableItems();
                OrderableItemViewModel itemViewModel = adtViewmodel.FacilityADTOrderableItemListExtended.SingleOrDefault(a => a.Model.code == QueryStrings["code"]);
                itemViewModel.LoggedUser = this.loggedUser;
                itemViewModel.AddAlreadyExistsConfirmationVisible = Strings.None.ToLower();
                itemViewModel.AddConfirmationVisible = Strings.None.ToLower();
                itemViewModel.DeleteConfirmationVisible = Strings.Block.ToLower();
                itemViewModel.AddADTOrderableItemWithADifferentTypeConfirmationVisible = Strings.None.ToLower();

                if (itemViewModel != null)
                    FacadeManager.VocabularyInterface.DeleteConcept(itemViewModel.Model);

                return View("Confirmation", itemViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillViewModel(AdtOrderableItemsViewModel viewModel, bool loadOI)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                viewModel.LoggedUser = this.loggedUser;
                viewModel.OrderableItemList = new List<CD>();
                if (loadOI)
                {
                    viewModel.OrderableItemList = FacadeManager.VocabularyInterface.GetOrderableItemsDB(loggedUser.VistaSite.Code)
                                                               .OrderBy(cd => cd.displayName)
                                                               .Select(cd => new CD { code = cd.code, codeSystem = cd.codeSystem, codeSystemName = cd.codeSystemName, displayName = cd.displayName + "{" + Strings.IEN + ":" + cd.code.Substring(cd.code.IndexOf("_") + 1) + "}" })
                                                               .ToList();
                }
                viewModel.OrderableItemTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.OrderableItemType);
                viewModel.ADTOrderableItemList = FacadeManager.VocabularyInterface.GetAdtOrderableItems();
                //Add Select An Orderable Item option to the drop down list.
                viewModel.OrderableItemList.Insert(0, new CD() { displayName = Strings.SelectAnOrderableItem, code = string.Empty });
                //Add Select Type option to the drop down list.
                viewModel.OrderableItemTypeList.Insert(0, new CD() { displayName = Strings.SelectType, code = string.Empty });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
