﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Configuration;
using System.Linq;
using System.Threading.Tasks;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.ServicesWrapper.BMService;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.Security;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Operations.Concrete.EntryProcessors;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.VistaWorker2.Writer.Implementation.Concrete.EIS;
using InfoWorld.HL7.ITS;

namespace BMS.VistaIntegration.Operations.Concrete
{
    public class Operation : BaseOperation
    {
        private static readonly bool IsUpdatePatientsEnabled;
        private const string UpdateBmsPatientsKey = "VistaIntegration.UpdateBmsPatients";

        private ProcessTypeCallback processTypeCallback;
        private BmsLogger logger;

        static Operation()
        {
            IsUpdatePatientsEnabled = bool.Parse(ConfigurationManager.AppSettings[UpdateBmsPatientsKey]);
        }

        public Operation(VistASite site, IList<Parameter> parameters)
            : base(site, parameters)
        {
            this.logger = new BmsLogger(string.Format("Operation:{0} - ", site.Name));
        }

        private void ProcessCommonType<T>(VistaDataType type, Func<Parameter, IList<T>> fetchFunc) where T : IEntity
        {
            Func<Parameter, IEntryProcessor> processorMaker = (param) =>
            {
                var result = new SingleEntryProcessor<T>(base.writer, param);
                result.Init(p => fetchFunc(p));
                return result;
            };
            processTypeCallback(type, processorMaker);
        }

        private II GetVistAId()
        {
            string domain = null;
            try
            {
                domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
            }
            catch (Exception)
            {
                domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
            }
            II id = new InfoWorld.HL7.ITS.II(domain, base.writer.Site.Id);
            return id;
        }

        private void UpdateAuditLog(VistaDataType dataType, string errorMessage)
        {
            logger.LogFormat(BmsLogger.Level.Verbose, "Updating failed audit log for {0} with error message '{1}'", dataType, errorMessage);

            try
            {
                lock (eventArgsList)
                {
                    eventArgsList[dataType].CompletedOperations += 1;
                    eventArgsList[dataType].State = ParameterEventArgs.OperationState.Failed;

                    if (string.IsNullOrEmpty(eventArgsList[dataType].ErrorMessage))
                        eventArgsList[dataType].ErrorMessage = errorMessage;
                    else
                        eventArgsList[dataType].ErrorMessage += " \r\n " + errorMessage;

                    if ((dataType == VistaDataType.ADT && eventArgsList[dataType].CompletedOperations == 5)
                            || (dataType == VistaDataType.Patient && eventArgsList[dataType].CompletedOperations == 2))
                    {
                        logger.LogFormat(BmsLogger.Level.Verbose, "Finalizing failed audit log for {0} with completed operations count {1}", dataType, eventArgsList[dataType].CompletedOperations);

                        FinaliseAuditLog(eventArgsList[dataType]);
                        eventArgsList.Remove(dataType);
                    }
                }
            }
            catch (Exception ex)
            {
                logger.LogFormat(BmsLogger.Level.Error, "Exception while updating AuditLog for {0}\r\n{1}", dataType, ex);

                throw;
            }
        }

        private static List<OccupiedBedInfo> GetBmsBedSwitch(II id)
        {
            List<OccupiedBedInfo> beds = FacadeManager.BedInterface.GetOccupiedBeds(id).ToList();
            return beds;
        }

        private void ChangePatientBedSwitch(OccupiedBedInfo bmsOccupiedBed, Data.WF.BedSwitch vistaBedInfo, ParameterEventArgs.Info info)
        {
            try
            {
                string domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
                II vistaSiteId = new II(domain, VistASite.Id);
                Bed bed = EISFactory.InstanceWindows.GetBed(vistaBedInfo.RoomBedIen, vistaSiteId);
                Ward ward = EISFactory.InstanceWindows.GetWard(vistaBedInfo.WardIen, vistaSiteId);

                if (ward != null && bed != null)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].RetrievedEntries++;
                    }

                    Bed oldBed = null;
                    if (!string.IsNullOrEmpty(bmsOccupiedBed.BedIEN))
                        oldBed = EISFactory.InstanceWindows.GetBed(bmsOccupiedBed.BedIEN, vistaSiteId);

                    if (bmsOccupiedBed.WardIEN != vistaBedInfo.WardIen)
                    {
                        BMS.DataContracts.AdmissionEvent adm = null;
                        if (bmsOccupiedBed.IsAdmission)
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(bmsOccupiedBed.Id);
                        else
                        {
                            BMS.DataContracts.MovementEvent mov = BMSFactory.BedManagerOperationsClientWindows.GetMovementEvent(bmsOccupiedBed.Id);
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(mov.AdmissionId);
                            mov.WardId = ward.Id;
                            mov.BedId = bed.Id;
                            BMSFactory.BedManagerOperationsClientWindows.UpdateMovementEvent(mov);
                        }
                        adm.WardId = ward.Id;
                        adm.BedId = bed.Id;
                        BMSFactory.BedManagerOperationsClientWindows.UpdateAdmissionEvent(adm);
                        logger.LogFormat(BmsLogger.Level.Info, "Bed switch with ward changed - update movement for patient ssn {0}", bmsOccupiedBed.PatientUID);
                    }
                    else
                    {
                        DataContracts.BedSwitch bedSwitch = new DataContracts.BedSwitch()
                        {
                            Id = new II(domain, null),
                            BedId = bed.Id,
                            OldBedId = oldBed != null ? oldBed.Id : null,
                            EnteredDate = DateTime.UtcNow,
                            PatientId = new II(domain, bmsOccupiedBed.PatientUID),
                            WardId = ward.Id,
                            VistaSiteId = vistaSiteId,
                            Ien = vistaBedInfo.IEN,
                        };
                        BMSFactory.BedManagerOperationsClientWindows.CreateBedSwitch(bedSwitch);
                        if (!bmsOccupiedBed.IsAdmission)
                        {
                            BMS.DataContracts.MovementEvent mov = BMSFactory.BedManagerOperationsClientWindows.GetMovementEvent(bmsOccupiedBed.Id);
                            mov.WardId = bedSwitch.WardId;
                            mov.BedId = bedSwitch.BedId;
                            BMSFactory.BedManagerOperationsClientWindows.UpdateMovementEvent(mov);
                        }
                        if (bedSwitch.OldBedId != null)
                            base.writer.UtilsInstance.GenerateBedCleanRequest(vistaBedInfo.IEN, bedSwitch.OldBedId, bedSwitch.WardId, bedSwitch.VistaSiteId, VistASite.TimeZone, DateTime.UtcNow, DateTime.UtcNow, "BSW");
                        base.writer.UtilsInstance.SendMailNotification(bedSwitch, bedSwitch.VistaSiteId, VistASite.TimeZone, Constants.BED_SWITCH, null, null);
                        logger.LogFormat(BmsLogger.Level.Info, "Bed switch for patient ssn {0}", bmsOccupiedBed.PatientUID);
                    }
                    BMSFactory.BedManagerOperationsClientFromWCF.DeleteIconAssociation(Guid.Parse(bmsOccupiedBed.PatientUID));
                    info.ProcessedWithSuccesEntries++;
                }
                else
                {
                    logger.LogFormat(BmsLogger.Level.Warning, "Unable to get bed or ward details of a bed switch with IEN {0}", vistaBedInfo.IEN);
                }
            }
            catch (WriterException we)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                    info.ProcessedWithFaildEntries++;
                    eventArgsList[VistaDataType.ADT].WriterExceptions.Add(we);
                    logger.LogFormat(BmsLogger.Level.Error, "Bed switch for patient ssn {0} failed: {1}", bmsOccupiedBed.PatientUID, we);
                }
            }
            catch (Exception e)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                    info.ProcessedWithFaildEntries++;
                    logger.LogFormat(BmsLogger.Level.Error, "Bed switch for patient ssn {0} failed: {1}", bmsOccupiedBed.PatientUID, e);
                }
            }
        }

        private void ProcessBedSwitch(II vistAId)
        {
            logger.LogInformation("Start Operation ProcessBedSwitch");

            try
            {
                List<OccupiedBedInfo> list = GetBmsBedSwitch(vistAId);
                logger.LogFormat(BmsLogger.Level.Verbose, "Got {0} occupied beds from BMS database", (list == null ? "no" : list.Count.ToString()));

                Dictionary<string, OccupiedBedInfo> bmsDictionaryBedSwitch = list.ToDictionary(a => a.ActIEN);
                long maxIen = 0;
                long currentIen = 0;
                foreach (string ien in bmsDictionaryBedSwitch.Keys)
                {
                    currentIen = long.Parse(ien);
                    if (currentIen > maxIen)
                        maxIen = currentIen;
                }
                if (!bmsDictionaryBedSwitch.Keys.Any())
                    return;

                List<Data.AdmittedPatient> vistaAdmittedPatients = base.query.GetAdmittedPatients((maxIen + 1).ToString()).ToList();
                logger.LogFormat(BmsLogger.Level.Verbose, "Got {0} admitted patients from VistA", (vistaAdmittedPatients == null ? "no" : vistaAdmittedPatients.Count.ToString()));

                vistaAdmittedPatients.RemoveAll(a => string.IsNullOrEmpty(a.WardName) || string.IsNullOrEmpty(a.BedName));
                logger.LogFormat(BmsLogger.Level.Verbose, "{0} admitted patients remaining after removing null ward names and bed names", (vistaAdmittedPatients == null ? "no" : vistaAdmittedPatients.Count.ToString()));

                vistaAdmittedPatients.RemoveAll(a => !bmsDictionaryBedSwitch.ContainsKey(a.MovementIen));
                logger.LogFormat(BmsLogger.Level.Verbose, "{0} admitted patients remaining after removing records not in occupied beds list", (vistaAdmittedPatients == null ? "no" : vistaAdmittedPatients.Count.ToString()));

                var bedSwitchChanged = from b in vistaAdmittedPatients
                                       let bmsBedSwitch = bmsDictionaryBedSwitch[b.MovementIen]
                                       where (b.BedName != bmsBedSwitch.BedName || b.WardName != bmsBedSwitch.WardName)
                                       select new { bmsBedSwitch, b };

                logger.LogFormat(BmsLogger.Level.Verbose, "{0} admitted patients remaining after removing records with bed or ward names not matching", (vistaAdmittedPatients == null ? "no" : vistaAdmittedPatients.Count.ToString()));

                if (bedSwitchChanged != null && bedSwitchChanged.Count() > 0)
                {
                    logger.LogFormat(BmsLogger.Level.Verbose, "Got {0} bed switch changes", bedSwitchChanged.Count());
                    List<string> bedSwitchIens = new List<string>();

                    bedSwitchChanged.ForEach(a => bedSwitchIens.Add(a.b.MovementIen));
                    List<Data.WF.BedSwitch> vistaBedSwitches = base.query.GetBedsSwitch(bedSwitchIens).ToList();
                    logger.LogFormat(BmsLogger.Level.Verbose, "Got {0} bed switches from VistA", vistaBedSwitches.Count());

                    ParameterEventArgs.Info info = null;
                    Data.WF.BedSwitch vistaBedSwitch = null;
                    var counter = 0;
                    bedSwitchChanged.ForEach(s =>
                    {
                        vistaBedSwitch = vistaBedSwitches.Where(a => a.IEN == s.b.MovementIen).FirstOrDefault();
                        if (s.bmsBedSwitch.BedIEN != vistaBedSwitch.RoomBedIen || s.bmsBedSwitch.WardIEN != vistaBedSwitch.WardIen)
                        {
                            if (info == null)
                                info = GetInfo("BedSwitch", eventArgsList[VistaDataType.ADT]);
                            ChangePatientBedSwitch(s.bmsBedSwitch, vistaBedSwitch, info);
                        }
                        else
                        {
                            counter++;
                        }
                    });

                    if (counter > 0)
                    {
                        logger.LogFormat(BmsLogger.Level.Verbose, "Not processed {0} bed switches due to no changes", counter);
                    }

                    vistaBedSwitches = null;
                }

                list = null;
                bmsDictionaryBedSwitch = null;
                vistaAdmittedPatients = null;
                bedSwitchChanged = null;
            }
            catch (Exception ex)
            {
                logger.LogFormat(BmsLogger.Level.Error, "Exception ProcessBedSwitch.\r\n{0}", ex.ToString());
                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessBedSwitch");
            }
        }

        private void ProcessCancelOrders(II vistAId)
        {
            logger.LogInformation("Start Operation ProcessCancelOrders");

            try
            {
                IList<DataContracts.Order> acts = ServicesWrapper.BMService.BMSFactory.BedManagerQueryClientWindows.GetCancelableOrders(vistAId);
                if (!acts.Any())
                    return;

                IEnumerable<string> actIens = acts.Select(s => s.Ien);
                IList<string> canceledIens = base.query.GetCanceledOrders(actIens);
                if (canceledIens == null || canceledIens.Count == 0)
                    return;
                ParameterEventArgs.Info info = null;
                acts.Where(s => s != null && !string.IsNullOrEmpty(s.Ien) && canceledIens.Contains(s.Ien)).ForEach(c =>
                {
                    if (info == null)
                        info = GetInfo("CancelOrder", eventArgsList[VistaDataType.ADT]);
                    CancelOrder(c, info);
                });
                acts = null;
                actIens = null;
                canceledIens = null;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessCancelOrders");
            }
        }

        private void CancelOrder(DataContracts.Order order, ParameterEventArgs.Info info)
        {
            DataContracts.AdmissionOrder admission = order as DataContracts.AdmissionOrder;
            if (admission != null)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].RetrievedEntries++;
                }

                try
                {
                    admission = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionOrder(admission.Id);
                    admission.IsActive = false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateAdmissionOrder(admission);
                    info.ProcessedWithSuccesEntries++;
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel admission order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel admission order with ien {0} failed: {1}", order.Ien, e);
                    }
                }

                return;
            }

            DataContracts.MovementOrder movement = order as DataContracts.MovementOrder;
            if (movement != null)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].RetrievedEntries++;
                }

                try
                {
                    movement = BMSFactory.BedManagerOperationsClientWindows.GetMovementOrder(movement.Id);
                    movement.IsActive = false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateMovementOrder(movement);
                    info.ProcessedWithSuccesEntries++;
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel movement order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel movement order with ien {0} failed: {1}", order.Ien, e);
                    }
                }

                return;
            }

            DataContracts.DischargeOrder discharge = order as DataContracts.DischargeOrder;
            if (discharge != null)
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.ADT].RetrievedEntries++;
                }

                try
                {
                    discharge = BMSFactory.BedManagerOperationsClientWindows.GetDischargeOrder(discharge.Id);
                    discharge.IsActive = false;
                    BMSFactory.BedManagerOperationsClientWindows.UpdateDischargeOrder(discharge);
                    info.ProcessedWithSuccesEntries++;
                    logger.LogFormat(BmsLogger.Level.Info, "Cancel discharge order with ien {0} with succes", order.Ien);
                }
                catch (Exception e)
                {
                    lock (eventArgsList)
                    {
                        eventArgsList[VistaDataType.ADT].State = ParameterEventArgs.OperationState.PartiallySucceeded;
                        info.ProcessedWithFaildEntries++;
                        logger.LogFormat(BmsLogger.Level.Error, "Cancel discharge order with ien {0} failed: {1}", order.Ien, e);
                    }
                }

                return;
            }

        }

        private void ProcessWorkFlowTypes()
        {
            logger.LogInformation("Start Operation ProcessWorkFlowTypes.");

            Task[] tasks = new Task[] {
                Task.Factory.StartNew(() => ProcessOrders()),
                Task.Factory.StartNew(() => TaskPatientMovements()),
                Task.Factory.StartNew(() => ProcessPatientAppointments()),
                Task.Factory.StartNew(() => ProcessScheduledAdmissions()),
                Task.Factory.StartNew(() => ProcessEDISPatientAdmissions())
            };

            logger.LogInformation("Operation ADT - Before wait tasks");

            try
            {
                Task.WaitAll(tasks, TimeSpan.FromHours(120));
            }
            catch (AggregateException ex)
            {
                logger.LogError("Exception ProcessWorkflowTypes wait tasks");
                ex.Handle((x) => { logger.LogError(x.ToString()); return true; });
            }
            finally
            {
                for (int i = 0; i < tasks.Length; i++)
                {
                    tasks[i].Dispose();
                    tasks[i] = null;
                }

                tasks = null;

                logger.LogInformation("Operation ADT - After wait tasks");
            }
        }

        private void TaskPatientMovements()
        {
            ProcessPatientMovements();
            ProcessDeletedPatientMovements();
        }

        private void ProcessOrders()
        {
            logger.LogInformation("Start Operation ProcessOrders");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    II vistAId = GetVistAId();
                    ProcessCancelOrders(vistAId);
                    return new OrderProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.Order, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessOrders - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessOrders.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessOrders");
            }
        }

        private void ProcessPatientMovements()
        {
            logger.LogInformation("Start Operation ProcessPatientMovements");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    II vistAId = GetVistAId();
                    ProcessBedSwitch(vistAId);
                    return new PatientMovementProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientMovement, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessPatientMovements - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessPatientMovements.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessPatientMovements");
            }
        }

        private void ProcessDeletedPatientMovements()
        {
            logger.LogInformation("Start Operation ProcessDeletedPatientMovements");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new DeletedPatientMovementsProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientMovementIen, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessDeletedPatientMovements\r\n" + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessDeletedPatientMovements.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessDeletedPatientMovements");
            }
        }

        private void ProcessPatientAppointments()
        {
            logger.LogInformation("Start Operation ProcessPatientAppointments");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new PatientAppointmentsProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.PatientAppointment, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessPatientAppointments - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessPatientAppointments.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessPatientAppointments");
            }
        }

        private void ProcessScheduledAdmissions()
        {
            logger.LogInformation("Start Operation ProcessScheduledAdmissions");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new ScheduledAdmissionProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.ScheduledAdmission, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessScheduledAdmissions - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessScheduledAdmissions.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessScheduledAdmissions");
            }
        }

        private void ProcessEDISPatientAdmissions()
        {
            if (ConfigurationManager.AppSettings["EDISBackgroundProcess.Enabled"] != null && ConfigurationManager.AppSettings["EDISBackgroundProcess.Enabled"].ToString() == "false")
                return;

            logger.LogInformation("Start Operation ProcessEDISPatientAdmissions");

            try
            {
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    return new EDISPatientAdmissionProcessor(base.writer, param, base.query);
                };
                processTypeCallback(VistaDataType.EDISPatientAdmission, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception ProcessEDISPatientAdmissions - " + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.ADT))
                    UpdateAuditLog(VistaDataType.ADT, ex.ToString());
                else
                {
                    logger.LogWarning("Audit log for ADT is missing in ProcessEDISPatientAdmissions.");
                }

                throw;
            }
            finally
            {
                logger.LogInformation("End Operation ProcessEDISPatientAdmissions");
            }
        }

        private void ProcessWaitingList()
        {
            Func<Parameter, IEntryProcessor> processorMaker = (param) =>
            {
                var result = new WaitingListProcessor(base.writer, param, base.query);
                result.Init();
                return result;
            };

            processTypeCallback(VistaDataType.WaitingList, processorMaker);
        }

        private void ProcessPatients()
        {
            Task[] tasks = new Task[] {
                Task.Factory.StartNew(() => 
                                {
                                    if (IsUpdatePatientsEnabled)
                                        ProcessUpdatingPatients();
                                }),
                Task.Factory.StartNew(() => TaskNewPatients())
            };

            logger.LogInformation("Operation Patients - Before wait tasks");
            try
            {
                Task.WaitAll(tasks);
            }
            catch (AggregateException ex)
            {
                logger.LogError("Exception ProcessPatient wait tasks");
                ex.Handle((x) => { logger.LogError(x.ToString()); return true; });
            }
            finally
            {
                foreach (Task t in tasks)
                    t.Dispose();
                tasks = null;
            }

            logger.LogInformation("Operation Patients - After wait tasks");
        }

        private void TaskNewPatients()
        {
            logger.LogInformation("Start Operation TaskNewPatients");
            try
            {
                string patientIen = BMSFactory.BedManagerOperationsClientWindows.GetVistaPatientIen(VistASite.Name);
                Func<Parameter, IEntryProcessor> processorMaker = (param) =>
                {
                    var result = new SingleEntryProcessor<BMS.VistaIntegration.Data.Patient>(base.writer, param);
                    result.Init(p => base.query.GetPatients(p.StartDateParam, p.EndDateParam, patientIen));
                    return result;
                };

                processTypeCallback(VistaDataType.Patient, processorMaker);
            }
            catch (Exception ex)
            {
                logger.LogError("Exception TaskNewPatients\r\n" + ex.ToString());
                if (eventArgsList.ContainsKey(VistaDataType.Patient))
                    UpdateAuditLog(VistaDataType.Patient, ex.ToString());
                throw;
            }

            logger.LogInformation("Start Operation TaskNewPatients");
        }

        private void ProcessUpdatingPatients()
        {
            logger.LogInformation("Start Operation ProcessUpdatingPatients");

            try
            {
                Parameter param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Patient);
                if (param == null)
                    return;
                lock (eventArgsList)
                {
                    if (!eventArgsList.ContainsKey(VistaDataType.Patient))
                    {
                        eventArgsList.Add(param.Type, new ParameterEventArgs(param));
                        eventArgsList[param.Type].State = ParameterEventArgs.OperationState.Running;
                        eventArgsList[param.Type].CompletedOperations = 0;
                        OnTypeUpdating(eventArgsList[param.Type]);
                    }
                }

                II vistaSite = GetVistAId();
                // get admitted and waiting list patients from BMS
                IList<II> patientIds = ServicesWrapper.BMService.BMSFactory.BedManagerQueryClientWindows.FilterUpdatingPatients(vistaSite);
                if (!patientIds.Any())
                    return;
                IList<Facade.Data.Patient> bmsPatients = EISFactory.InstanceWindows.GetPatientsByIds(patientIds, null);
                List<string> iens = new List<string>();
                II ien = null;
                foreach (Facade.Data.Patient patient in bmsPatients)
                {
                    if (patient.IENList != null && patient.IENList.Count > 0)
                    {
                        ien = patient.IENList.Where(a => a.root.Equals(VistASite.Name, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ien != null)
                            iens.Add(ien.extension);
                    }
                }
                //get admitted patients from Vista
                IList<VistaIntegration.Data.Patient> vistaPatients = base.query.GetAdmittedPatientsForUpdate();
                // remove the iens for admitted patients
                foreach (VistaIntegration.Data.Patient p in vistaPatients)
                {
                    if (iens.Contains(p.IEN))
                        iens.Remove(p.IEN);
                }
                // get the rest of the patients by ien from Vista  
                if (base.query.GetType().Name.Contains("ViaVistAQuery"))
                {
                    string bulkIENs = String.Join(",", iens);
                    IList<Data.Patient> bulkPatients = base.query.GetPatients(null, null, bulkIENs);
                    foreach (Data.Patient pat in bulkPatients)
                        vistaPatients.Add(pat);
                }
                else
                {
                    foreach (string i in iens)
                        vistaPatients.Add(base.query.GetPatientByIen(i));
                }

                IList<Person> attendingPhysicians = EISFactory.InstanceWindows.GetMedicalPersonsByIds(bmsPatients.Select<Facade.Data.Patient, II>(a => a.AttendingPhysicianId).Where(b => b != null && !string.IsNullOrEmpty(b.extension) && !b.extension.Equals(Guid.Empty.ToString())).Distinct().ToList());
                Facade.Data.Patient bmsPatient = null;
                string bmsAttendingIen = null, vistaAttendingIen = null;
                using (var patientWriter = base.writer.MakeEntityWriter<Data.Patient>() as PatientWriter)
                {
                    foreach (VistaIntegration.Data.Patient p in vistaPatients)
                    {
                        if (p == null)
                            continue;
                        bmsPatient = bmsPatients.Where(a => a.SSN.extension == p.SocialSecurityNumber).FirstOrDefault();
                        if (bmsPatient != null)
                        {
                            if (bmsPatient.AttendingPhysicianId != null)
                            {
                                try
                                {
                                    bmsAttendingIen = attendingPhysicians.Where(a => a.Id.extension.Equals(bmsPatient.AttendingPhysicianId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Ien;
                                }
                                catch { bmsAttendingIen = null; }
                            }
                            else
                                bmsAttendingIen = null;
                            if (p.AttendingPhysician != null)
                                vistaAttendingIen = p.AttendingPhysician.IEN;
                            else
                                vistaAttendingIen = null;

                            if (p.AttendingPhysician != null && bmsAttendingIen == vistaAttendingIen)
                                p.AttendingPhysician = new NewPerson() { IEN = "-100", Name = p.AttendingPhysician.Name }; //do not upate attending physician
                            CheckForUpdatePatient(patientWriter, p, bmsPatient);
                        }
                    }
                }
                patientIds = null;
                bmsPatients = null;
                vistaPatients = null;
                attendingPhysicians = null;
            }
            catch (Exception ex)
            {
                lock (eventArgsList)
                {
                    logger.LogError("Exception ProcessUpdatingPatients\r\n" + ex.ToString());
                    eventArgsList[VistaDataType.Patient].State = ParameterEventArgs.OperationState.Failed;
                    if (string.IsNullOrEmpty(eventArgsList[VistaDataType.Patient].ErrorMessage))
                        eventArgsList[VistaDataType.Patient].ErrorMessage = ex.ToString();
                    else
                        eventArgsList[VistaDataType.Patient].ErrorMessage += " \r\n " + ex.ToString();
                }

                throw;
            }
            finally
            {
                lock (eventArgsList)
                {
                    eventArgsList[VistaDataType.Patient].CompletedOperations += 1;
                    if (eventArgsList[VistaDataType.Patient].CompletedOperations == 2)
                    {
                        FinaliseAuditLog(eventArgsList[VistaDataType.Patient]);
                        eventArgsList.Remove(VistaDataType.Patient);
                    }
                }

                logger.LogInformation("End Operation ProcessUpdatingPatients");
            }
        }

        private void CheckForUpdatePatient(PatientWriter patientWriter, Data.Patient vistaPatient, Facade.Data.Patient bmsPatient)
        {
            try
            {
                bool isUpdated = patientWriter.UpdateOldEntity(vistaPatient, bmsPatient);
                if (isUpdated)
                {
                    ParameterEventArgs.Info info = GetInfo("UpdatePacients", eventArgsList[VistaDataType.Patient]);
                    info.ProcessedWithSuccesEntries++;
                }
            }
            catch (Exception e)
            {
                ParameterEventArgs.Info info = GetInfo("UpdatePacients", eventArgsList[VistaDataType.Patient]);
                info.ProcessedWithFaildEntries++;
                logger.LogFormat(BmsLogger.Level.Error, "Update patient data with ssn {0} failed: {1}", bmsPatient.SSN.extension, e);

            }
        }

        protected override void ProcessTypes(ProcessTypeCallback callback)
        {
            this.processTypeCallback = callback;
            Parameter param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Specialty);
            if (param != null)
                ProcessCommonType(VistaDataType.Specialty, p => base.query.GetSpecialties());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.TreatingSpecialty);
            if (param != null)
                ProcessCommonType(VistaDataType.TreatingSpecialty, p => base.query.GetFacilityTreatingSpecialties());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.FacilityMovementType);
            if (param != null)
                ProcessCommonType(VistaDataType.FacilityMovementType, p => base.query.GetFacilityMovementTypes());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.OrderableItem);
            if (param != null)
                ProcessCommonType(VistaDataType.OrderableItem, p => base.query.GetOrderableItems());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.HospitalLocation);
            if (param != null)
                ProcessCommonType(VistaDataType.HospitalLocation, p => base.query.GetHospitalLocations());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.WardLocation);
            if (param != null)
                ProcessCommonType(VistaDataType.WardLocation, p => base.query.GetWardLocations());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.RoomBed);
            if (param != null)
                ProcessCommonType(VistaDataType.RoomBed, p => base.query.GetRoomBeds());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.MedicalCenterDivision);
            if (param != null)
                ProcessCommonType(VistaDataType.MedicalCenterDivision, p => base.query.GetMedicalCenterDivisions());

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.Patient);
            if (param != null)
                ProcessPatients();

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.WaitingList);
            if (param != null)
                ProcessWaitingList();

            param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.ADT);
            if (param != null)
                ProcessWorkFlowTypes();
        }
    }
}
