﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker2.Writer;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.LogCheckers;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.Security;
using BMS.ServicesWrapper.EVS;
using FC = BMS.Facade.Data;
using BMS.ServicesWrapper.BMService;
using BMS.Utils;
using BMS.VistaWorker2.Writer.Implementation.Concrete;
using BMS.DataContracts;
using InfoWorld.HL7.ITS;

namespace BMS.VistaIntegration.Operations.Concrete.EntryProcessors
{
    public class PatientMovementProcessor : MultipleEntryProcessor
    {
        private static readonly BmsLogger logger = new BmsLogger("PatientMovementProcessor: ");

        public Parameter Parameter { get; private set; }
        public IVistAQuery Query { get; private set; }

        public PatientMovementProcessor(IWriterManager writerManager, Parameter parameter, IVistAQuery query)
            : base(writerManager)
        {
            if (parameter.Type != VistaDataType.PatientMovement)
                throw new InvalidOperationException();
            Parameter = parameter;
            Query = query;
            InitEntriesProcessor(GetEntitiesProcessor());
        }

        ListEntryProcessor<T> GetEntitiesProcessor<T>(Func<IList<T>> fetchFunc, Comparison<T> comparison, Func<T, DateTime?> filterDateTimeFunc)
        {
            IList<T> list = fetchFunc();
            if (list != null && list.Count > 0)
                list = (from a in (list as List<PatientMovement>) orderby a.EnteredOnDateTime select a).ToList() as List<T>;
            return new ListEntryProcessor<T>(WriterManager, list, filterDateTimeFunc);
        }

        public IVistaIntegrationLogChecker LogChecker { get; private set; }

        private IList<IEntryProcessor> GetEntitiesProcessor()
        {
            List<IEntryProcessor> result = new List<IEntryProcessor>();
            List<VistaIntegrationLog> logs = new List<VistaIntegrationLog>();

            string domain = SecurityFactory.InstanceWindows.GetCurrentDomain();
            II vistASiteId = new II(domain, WriterManager.Site.Id);

            var s1 = GetEntitiesProcessor<PatientMovement>(() => Query.GetPatientMovements(Parameter.StartDateParam, Parameter.EndDateParam), null, null);
            //remove movements without ward and bed
            List<PatientMovement> movementsToRemove = new List<PatientMovement>();
            decimal transaction = 0;
            List<string> admissionIens = new List<string>();
            foreach (PatientMovement pm in s1.Entries)
            {
                decimal.TryParse(pm.TransactionTypeId, out transaction);
                if ((transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MAX)
                    || (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MAX))
                {
                    if (string.IsNullOrEmpty(pm.WardLocationId) && string.IsNullOrEmpty(pm.RoomBedId))
                        movementsToRemove.Add(pm);
                }
                if ((transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MAX)
                    || (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MAX)
                    || (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MAX))
                {
                    if (!movementsToRemove.Contains(pm))
                        admissionIens.Add(pm.CurrentAdmissionIen);
                }
            }

            foreach (PatientMovement pm in movementsToRemove)
                s1.Entries.Remove(pm);

            s1.RetrievedEntries = s1.Entries.Count;
            StringBuilder sb = new StringBuilder();
            int count = 0;
            List<string> missingAdmissions = new List<string>();
            foreach (string ien in admissionIens)
            {
                if (s1.Entries.Where(a => a.IEN == ien).FirstOrDefault() == null)
                {
                    sb.Append(ien);
                    sb.Append(",");
                    count = count + 1;
                }
                if (count == 50)
                {
                    missingAdmissions.AddRange(BMSFactory.BedManagerQueryClientWindows.GetMissingAdmissions(sb.ToString(), vistASiteId));
                    count = 0;
                    sb.Clear();
                }
            }
            if (count > 0)
                missingAdmissions.AddRange(BMSFactory.BedManagerQueryClientWindows.GetMissingAdmissions(sb.ToString(), vistASiteId));

            if (missingAdmissions != null && missingAdmissions.Count > 0)
            {
                logger.LogFormat(BmsLogger.Level.Verbose, "There are total {0} missing admissions in BMS database", missingAdmissions.Count);

                PatientMovement obj = null;
                transaction = 0;
                foreach (string ien in missingAdmissions)
                {
                    if (!string.IsNullOrEmpty(ien))
                    {
                        obj = Query.GetPatientMovementByIen(ien);
                        if (obj == null)
                        {
                            logger.LogFormat(BmsLogger.Level.Warning, "Missing admission lookup for {0} was not returned by VistA.", ien);
                            continue;
                        }

                        decimal.TryParse(obj.TransactionTypeId, out transaction);
                        if (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MAX)
                        {
                            if (string.IsNullOrEmpty(obj.WardLocationId) && string.IsNullOrEmpty(obj.RoomBedId))
                                continue;
                        }
                        s1.Entries.Add(obj);
                    }
                }
            }

            s1.RetrievedEntries = s1.Entries.Count;
            if (s1.Entries != null && s1.Entries.Count > 0)
                s1.Entries = (from a in (s1.Entries as List<PatientMovement>) orderby a.EnteredOnDateTime select a).ToList();
            s1.Entries.ForEach(i => logs.Add(new VistaIntegrationLog() { File = VistaFiles.PatientMovement, Ien = i.IEN }));
            result.Add(s1);
            LogChecker = new HashSetLogChecker(logs, WriterManager.Site.Name);
            result.ForEach(s => s.InitLogCheck(LogChecker));
            return result;
        }
    }
}
