﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Operations;
using System.Threading.Tasks;
using BMS.VistaIntegration.Data;
using BMS.VistaWorker2;
using BMS.VistaIntegration.Exceptions;
using BMS.Utils;
using BMS.VistaWorker2.Writer;
using BMS.VistaIntegration.Operations.Concrete;
using System.Threading;
using BMS.VistaIntegration.Dal;
using BMS.VistaIntegration.Audit;
using BMS.VistaIntegration.Managers;
using BMS.Schedulers;
using BMS.ServicesWrapper.BMService;
using System.Configuration;

namespace BMS.VistaIntegration.Commands
{
    public class RunSchedulerCommand : ICommand
    {
        private readonly Crawler crawler;
        private static readonly BmsLogger Logger = new BmsLogger("VistaIntegration RunSchedulerCommand Message: ");
        private SchedulerData[] data;
        private Dictionary<string, Task> runningTasks;
        private const string Mdws_Query_Exception_Text_1 = "A connection attempt failed";
        private const string Mdws_Query_Exception_Text_2 = "[MAP]";
        private const string MaxCounterFailed_CustomMessage = "The MDWS query has failed already for the maximum consecutive number of trials. The process is incrementing the start date.";
        private static object _syncLock;

        public RunSchedulerCommand(Crawler crawler, IEnumerable<SchedulerData> data)
        {
            this.crawler = crawler;
            this.data = data.ToArray();
            runningTasks = new Dictionary<string, Task>(StringComparer.InvariantCultureIgnoreCase);
            _syncLock = new object();
        }

        public void Run(Scheduler scheduler)
        {            
            for (int i = 0; i < data.Length; i++)
            {
                SchedulerData con = data[i];
                List<VistaDataType> dataTypes = Utilities.GetVistaTypes(con.Types).ToList();               
                
                foreach (VistaDataType dt in dataTypes)
                {
                    string key = dt.ToString() + "_" + con.Site.Id;
                    lock (_syncLock)
                    {
                        if (runningTasks.ContainsKey(key) && runningTasks[key] != null && !runningTasks[key].IsCompleted)
                            continue;
                    }
                    SchedulerData sData = new SchedulerData(con.Site, con.ConnectionInfo, con.SessionFactory, dt);
                    Task task = new Task(() => StartRunSchedulerTask(sData, scheduler));
                    lock (_syncLock)
                    {                        
                        runningTasks[key] = task;
                    }
                    task.ContinueWith((t) => 
                    {
                        string type = crawler.OperationManager.EndRunTask(sData.Site.Id, t);
                        lock (_syncLock)
                        {
                            if (type != VistaDataType.None.ToString())
                                runningTasks.Remove(type);
                            else if (runningTasks.ContainsValue(t))
                            {
                                type = runningTasks.Where(a => a.Value == t).FirstOrDefault().Key;
                                runningTasks.Remove(type);
                            }
                        }
                        sData = null;
                        t.Dispose(); 
                    });
                    crawler.OperationManager.RunOperation(con.Site.Id, task, sData.Types);
                }
            }
        }

        private string GetSchedulerInfo(SchedulerData data, Scheduler scheduler)
        {
            return string.Format("site={0} , scheduler={1} , types={2}",
                     data.Site.Name,
                     scheduler.Name,
                     Utilities.ToString(data.Types));
        }

        private void RunOperation(BaseOperation operation, SchedulerData data, Scheduler scheduler)
        {
            using (var session = data.SessionFactory.MakeVistASession(data.Site))
            {
                session.Open(data.ConnectionInfo);
                operation.Run(session, crawler.WriterManagerFactory, crawler.CancellationTokenSource.Token);
            }
        }

        private IList<Parameter> GetParameters(VistASite site, SchedulerData data)
        {
            DateTime now = DateTime.UtcNow;
            now = site.ConvertToSiteDateTime(new DateTime(now.Year, now.Month, now.Day, now.Hour, now.Minute, 0));
            var operations = VistaOperationsCache.Instance.Get(data.Site.Id);
            SiteData siteData = new SiteData(site, operations);
            List<Parameter> parameters = Utilities.GetVistaTypes(data.Types)
                .Select(s => new Parameter(s, site.ConvertToSiteNullableDateTime(siteData[s].LastRunning), now))
                .ToList();

            int index = parameters.FindIndex(p => p.Type == VistaDataType.WaitingList);
            if (index >= 0)
            {
                parameters[index] = new Parameter(VistaDataType.WaitingList , now.Date,now.Date.AddDays(1));
            }

            index = parameters.FindIndex(p => p.Type == VistaDataType.Patient);
            if (index >= 0)
            {
                Parameter p = parameters[index];
                if (p.StartDateParam.HasValue)
                {
                    parameters[index] = new Parameter(VistaDataType.Patient, p.StartDateParam.Value.Date, p.EndDateParam);
                }
            }
            return parameters;
        }

        private BMS.DataContracts.JobLogInfo MakeJobLog(Parameter param, VistASite site, Scheduler schedule, DataRetrievalMethod dataRetrievalMethod)
        {
            BMS.DataContracts.JobLogInfo jobLog = AuditUtilities.MakeJobLog(param, site, dataRetrievalMethod);
            jobLog.LaunchType = JobLaunchType.Automatic;
            jobLog.Details = "Scheduler=" + schedule.Name;
            return jobLog;
        }

        private void OnJobLogFailed(VistASite site, BMS.DataContracts.JobLogInfo jobLog, Exception e)
        {
            try
            {    
                string errorMessage = e.Message;
                if (jobLog.RetrievedDataType == VistaDataType.ADT)
                {
                    if (e != null && ((!string.IsNullOrEmpty(e.Message) && (e.Message.Contains(Mdws_Query_Exception_Text_1) || e.Message.Contains(Mdws_Query_Exception_Text_2)))
                        || (e.InnerException != null && !string.IsNullOrEmpty(e.InnerException.Message) && (e.InnerException.Message.Contains(Mdws_Query_Exception_Text_1) || e.InnerException.Message.Contains(Mdws_Query_Exception_Text_2)))))
                    {
                        int jobFailedMaxCount = 0;
                        try
                        {
                            jobFailedMaxCount = int.Parse(ConfigurationManager.AppSettings["VistaIntegrationJobFailedMaxCount"]);
                        }
                        catch { jobFailedMaxCount = 10; }
                        IList<BMS.DataContracts.VistaOperation> operations = VistaOperationsCache.Instance.Get(site.Id);
                        BMS.DataContracts.VistaOperation operation = operations.First(s => s.DataType == jobLog.RetrievedDataType);
                        if (operation != null && operation.CounterFailed < jobFailedMaxCount)
                        {
                            operation.CounterFailed = operation.CounterFailed + 1;
                            VistaOperationsCache.Instance.Update(operation);
                        }
                        else if (operation != null && operation.CounterFailed >= jobFailedMaxCount)
                        {
                            StringBuilder sb = new StringBuilder();
                            sb.AppendLine(MaxCounterFailed_CustomMessage);
                            sb.AppendLine(e.Message);
                            errorMessage = sb.ToString();
                            SendEmailNotification(jobLog);
                        }
                    }
                }
                AuditUtilities.OperationFailed(site, jobLog, errorMessage, e.ToString());                
            }
            catch (Exception ex)
            {
                string msj = string.Format("Exception on update log faild for jobid {0} :\n{1}", jobLog.Id, ex);
                Logger.LogError(msj);
            }
        }

        private void UpdateVistAOperationData(VistASite site, ParameterEventArgs eventArgs)
        {
            if (eventArgs.State == ParameterEventArgs.OperationState.Succeeded ||
                eventArgs.State == ParameterEventArgs.OperationState.PartiallySucceeded)
            {
                DateTime? endDateTime = site.ConvertToUtcNullableDateTime(eventArgs.Parameter.EndDateParam);
                BMS.DataContracts.VistaOperation operation = new DataContracts.VistaOperation()
                {
                    DataType = eventArgs.Parameter.Type,
                    LastRunning = endDateTime,
                    VistaSiteId = site.Id,
                    CounterFailed = 0
                };
                VistaOperationsCache.Instance.Update(operation);
            }
            else if (eventArgs.State == ParameterEventArgs.OperationState.Failed)
            {
                if (eventArgs.Parameter.Type == VistaDataType.ADT)
                {
                    if (!string.IsNullOrEmpty(eventArgs.ErrorMessage) && (eventArgs.ErrorMessage.Contains(Mdws_Query_Exception_Text_1) || eventArgs.ErrorMessage.Contains(Mdws_Query_Exception_Text_2)))
                    {
                        int jobFailedMaxCount = 0;
                        try
                        {
                            jobFailedMaxCount = int.Parse(ConfigurationManager.AppSettings["VistaIntegrationJobFailedMaxCount"]);
                        }
                        catch { jobFailedMaxCount = 10; }
                        IList<BMS.DataContracts.VistaOperation> operations = VistaOperationsCache.Instance.Get(site.Id);
                        BMS.DataContracts.VistaOperation operation = operations.First(s => s.DataType == eventArgs.Parameter.Type);
                        if (operation != null && operation.CounterFailed < jobFailedMaxCount)
                        {
                            operation.CounterFailed = operation.CounterFailed + 1;
                            VistaOperationsCache.Instance.Update(operation);
                        }
                    }
                }
            }
        }

        private void UpdateAuditData(VistASite site, Scheduler scheduler, DataRetrievalMethod retrievalMethod, ParameterEventArgs eventArgs)
        {
            BMS.DataContracts.JobLogInfo jobLog = null;
            if (eventArgs.State == ParameterEventArgs.OperationState.Failed && eventArgs.Parameter.Type == VistaDataType.ADT)
            {                
                int jobFailedMaxCount = 0;
                try
                {
                    jobFailedMaxCount = int.Parse(ConfigurationManager.AppSettings["VistaIntegrationJobFailedMaxCount"]);
                }
                catch { jobFailedMaxCount = 10; }
                IList<BMS.DataContracts.VistaOperation> operations = VistaOperationsCache.Instance.Get(site.Id);
                BMS.DataContracts.VistaOperation operation = operations.First(s => s.DataType == eventArgs.Parameter.Type);
                if (operation != null && operation.CounterFailed >= jobFailedMaxCount)
                {
                    StringBuilder sb = new StringBuilder();
                    sb.AppendLine(MaxCounterFailed_CustomMessage);
                    sb.AppendLine(eventArgs.ErrorMessage);
                    eventArgs.ErrorMessage = sb.ToString();
                    jobLog = MakeJobLog(eventArgs.Parameter, site, scheduler, retrievalMethod);
                    SendEmailNotification(jobLog);
                }
            }
            if (jobLog == null)
                jobLog = MakeJobLog(eventArgs.Parameter, site, scheduler, retrievalMethod);            
            AuditUtilities.OperationUpdated(site, jobLog, eventArgs);
        }

        private static bool SameParameters(IList<Parameter> var1, IList<Parameter> var2)
        {
            if (var1.Count != var2.Count)
                return false;
            foreach (var par in var1)
                if (var2.FirstOrDefault(s => s.Type == par.Type) == null)
                    return false;
            return true;
        }

        private void StartRunSchedulerTask(SchedulerData data, Scheduler scheduler)
        {            
            BaseOperation operation = null;
            VistASite site = data.Site;
            try
            {
                var parameters = GetParameters(site, data);
                foreach (Parameter param in parameters)
                    param.JobLaunchType = JobLaunchType.Automatic;
                operation = new Operation(site, parameters);
                operation.TypeUpdating += (s, e) => UpdateAuditData(site, scheduler, data.SessionFactory.RetrievalMethod, e);
                operation.TypeUpdated += (s, e) => UpdateAuditData(site, scheduler, data.SessionFactory.RetrievalMethod, e);
                operation.TypeUpdated += (s, e) => UpdateVistAOperationData(site, e);
                RunOperation(operation, data, scheduler);
            }
            catch (Exception e)
            {
                IDictionary<Parameter, BMS.DataContracts.JobLogInfo> logDictionary = operation.Parameters.ToDictionary(p => p, p => MakeJobLog(p, site, scheduler, data.SessionFactory.RetrievalMethod));
                Logger.LogError(string.Format("Run on scheduler task exception for {0}:\n{1}", GetSchedulerInfo(data, scheduler), e));
                if (logDictionary != null)
                {
                    logDictionary.Values.Where(s => s.Status == JobStatus.Running).ForEach(j => OnJobLogFailed(site, j, e));
                }
            }
        }

        private void SendEmailNotification(BMS.DataContracts.JobLogInfo jobLog)
        {
            string from = string.Empty;
            string to = string.Empty;
            string subject = string.Empty;
            string body = string.Empty;
            try
            {
                to = ConfigurationManager.AppSettings["VistaIntegrationJobFailedNotificationEmailTo"];
                if (String.IsNullOrEmpty(to))
                {
                    Logger.LogError("The destination email address used to notify in case of repeated job failing is empty");
                    return;
                }
            }
            catch (Exception e)
            {
                Logger.LogError(string.Format("The destination email address used to notify in case of repeated job failing is not specified. {0}", e));
                return;
            }

            try
            {
                from = ConfigurationManager.AppSettings["VistaIntegrationJobFailedNotificationEmailFrom"];
                if (String.IsNullOrEmpty(to))
                {
                    Logger.LogError("The source email address used to notify in case of repeated job failing is empty");
                    return;
                }
            }
            catch (Exception e)
            {
                Logger.LogError(string.Format("The source email address used to notify in case of repeated job failing is not specified. {0}", e));
                return;
            }
 
            subject = String.Format("The Background Processor for operation {0} and site {1} has repeatedly failed to succeed", Utilities.ToString(jobLog.RetrievedDataType), jobLog.VistaName);
            body = String.Format("The Background Processor for operation {0} and site {1} has repeatedly failed to succeed. The DateTime interval used was between {2} and {3}",
                Utilities.ToString(jobLog.RetrievedDataType), jobLog.VistaName, jobLog.QueryStartDate, jobLog.QueryEndDate);
            try
            {
                SendMail.Execute(from, to, subject, body, ConfigurationManager.AppSettings[Constants.SMTP_HOST], 25, null);
            }
            catch (Exception e)
            {
                Logger.LogError(string.Format("The job failing email could not be sent. {0}", e));
            }
        }
    }
}
