﻿using System;
using System.Configuration;
using System.ServiceModel;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Exceptions;
using BMS.VistaIntegration.Via.BackgroundProcessService;
using BMS.VistaIntegration.VistA;

namespace BMS.VistaIntegration.Via
{
    public class ViaVistASession : IVistASession
    {
        private static string appToken = null, appPassword = null, viaURL = null;
        private static batchLoginResponse loginResponse;
        private readonly BmsLogger logger;
        private static object _loginLock = new object();

        public ViaVistASession(VistASite site)
        {
            this.logger = new BmsLogger(string.Format("ViaVistASession:{0} - ", site.Name));

            VistASite = site;
        }

        static ViaVistASession()
        {
            InitUserSettings();
        }

        public VistASite VistASite { get; private set; }

        private static batchLoginResponse LoginResponse
        {
            get
            {
                lock (_loginLock)
                {
                    if (loginResponse == null)
                    {
                        loginResponse = ViaVistASession.Login("580"); // 580 is the site code for Houston. This is just used to for login purposes. The site code actually gets replaced for each subsequent data call for the appropriate invoking site.
                    }

                    return loginResponse;
                }
            }
        }

        private static void InitUserSettings()
        {
            try
            {
                appToken = ConfigurationManager.AppSettings["ViaAppToken"];
                appPassword = ConfigurationManager.AppSettings["ViaAppPassword"];
                viaURL = ConfigurationManager.AppSettings["ViaEndpointUrl"];
            }
            catch
            {
                throw new VistAConnectionException("Default config parameters are incorrect!");
            }
        }

        public void Open(VistAConnectionInfo vistAConnection)
        {
            // VIA is stateless. Nothing to open.
        }

        public void RunClientAction(Action action)
        {
            try
            {
                action();
            }
            catch (CommunicationException e)
            {
                throw new VistAConnectionException(e);
            }
            catch (TimeoutException e)
            {
                throw new VistAConnectionException(e);
            }
        }

        public void Close()
        {
            // VIA is stateless. Nothing to close.
        }

        public IVistAQuery MakeQuery()
        {
            return new ViaVistAQuery(this);
        }

        public void Dispose()
        {
            // VIA is stateless. Nothing to dispose of.
        }

        public bool IsAlive()
        {
            return true; // VIA is stateless and is always "alive".
        }

        public void Test(VistAConnectionInfo connection)
        {
            try
            {
                var response = Login("580");
                if (string.IsNullOrEmpty(response.UserTO.DUZ))
                {
                    throw new VistAConnectionException(response.UserTO.fault == null ? "Server returned an empty DUZ with no fault" : response.UserTO.fault.message);
                }
            }
            catch (Exception ex)
            {
                if (ex is VistAConnectionException)
                {
                    throw;
                }

                throw new VistAConnectionException(ex);
            }
        }

        internal static batchLoginResponse Login(string vistaSiteNumber)
        {
            var viaClient = GetClient();

            var loginReq = new batchLogin()
            {
                siteCode = vistaSiteNumber,
                queryBean = CreateQueryBean(),
            };

            return viaClient.batchLogin(loginReq);
        }

        internal textArray GetBMSDataFromVia(string target, string criteria)
        {
            this.logger.LogFormat(BmsLogger.Level.Verbose, "Calling VIA with target '{0}' and criteria '{1}'", target, criteria);

            var retry = true;

            while (true)
            {
                try
                {
                    batchLoginResponse loginResp = LoginResponse;

                    var qBean = ViaVistASession.CreateQueryBean();
                    qBean.target = target;
                    qBean.criteria = criteria;
                    qBean.provider = new provider()
                    {
                        name = ConfigurationManager.AppSettings[Constants.Via_Requesting_App],
                        loginSiteCode = VistASite.Number,
                        userId = loginResp.UserTO.DUZ,
                    };

                    var getBMSData = new getBMSData(qBean);

                    var viaClient = ViaVistASession.GetClient();
                    var resp = viaClient.getBMSData(getBMSData);

                    var textArray = resp.TextArray;
                    if (retry && textArray.fault != null && textArray.fault.message.Contains("userId value has expired") && textArray.fault.message.Contains("login again"))
                    {
                        loginResponse = null;
                        retry = false; // Only retry once
                        this.logger.LogFormat(BmsLogger.Level.Verbose, "VIA call failed because user DUZ expired.");
                        continue;
                    }
                    else if (!retry && textArray.fault != null && textArray.fault.message.Contains("userId value has expired") && textArray.fault.message.Contains("login again"))
                    {
                        // we already retried and we're still having trouble
                        this.logger.LogFormat(BmsLogger.Level.Error, "Failed to re-login to VIA after user DUZ expired. ADT aborted for site {0}.", VistASite.Number);
                    }

                    return textArray;
                }
                catch (Exception ex)
                {
                    this.logger.LogFormat(BmsLogger.Level.Error, "Exception occurred when calling VIA with target '{0}' and criteria '{1}'.\r\n{2}", target, criteria, ex);
                    throw;
                }
            }
        }

        private static BackgroundProcessServiceInterfaceClient GetClient()
        {
            var viaUri = new Uri(ConfigurationManager.AppSettings[Constants.Via_Endpoint_Url]);
            var configName = "BackgroundProcessServiceImplPort";
            if (string.Equals(viaUri.Scheme, "https", StringComparison.OrdinalIgnoreCase))
            {
                configName += "SSL";
            }

            var viaClient = new BackgroundProcessServiceInterfaceClient(configName, new EndpointAddress(viaUri));
            return viaClient;
        }

        private static queryBean CreateQueryBean()
        {
            return new queryBean()
            {
                requestingApp = ConfigurationManager.AppSettings[Constants.Via_Requesting_App],
                consumingAppToken = ConfigurationManager.AppSettings[Constants.Via_App_Token],
                consumingAppPassword = ConfigurationManager.AppSettings[Constants.Via_App_Password],
            };
        }
    }
}
