﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Via.Commands.EIS;
using BMS.VistaIntegration.Via.Commands.EntityCommands;
using BMS.VistaIntegration.Via.Commands.EVS;

namespace BMS.VistaIntegration.Via.Commands.WF
{
    public class ListOrdersCommand : BaseListPeriodCommand<Order>
    {
        private string currentOrderableItemIen;

        public ListOrdersCommand(ViaVistAQuery query)
            : base(query)
        {
        }

        public string PatientIen
        {
            get;
            set;
        }

        public bool IsAnticipated
        {
            get;
            set;
        }

        public IEnumerable<string> OrderableItemIens
        {
            get;
            set;
        }

        public override List<Order> Execute(ViaVistASession session)
        {
            return this.IsAnticipated ? this.GetAnticipatedOrders(session) : this.GetOrders(session);
        }

        protected override string GetTarget()
        {
            return "ListOrders";
        }

        protected override IEnumerable<object> GetCriteria()
        {
            if (!this.IsAnticipated && string.IsNullOrEmpty(this.currentOrderableItemIen))
            {
                throw new InvalidOperationException("CurrentOrderableItemIen is not set.");
            }

            yield return this.currentOrderableItemIen;
            yield return this.PatientIen;
            yield return this.StartDate.GetValueOrDefault(MinDate);
            yield return this.EndDate.GetValueOrDefault(this.MaxDate);
            yield return this.IsAnticipated ? 2 : 1;
            yield return this.MaxCount;
            yield return this.From;
        }

        protected override IDependencySource GetDependencySource()
        {
            return OrderDependencySource.Instance;
        }

        protected override string TranslateValue(string fieldIen, string value)
        {
            this.Logger.LogFormat(Utils.BmsLogger.Level.Verbose, "Converting value '{1}' for field '{0}'", fieldIen, value);

            switch (fieldIen)
            {
                case ".02":
                case "6":
                    return value.Split(';').First();

                default:
                    return base.TranslateValue(fieldIen, value);
            }
        }

        private List<Order> GetOrders(ViaVistASession session)
        {
            if (this.OrderableItemIens == null || !this.OrderableItemIens.Any())
            {
                return new List<Order>(0);
            }

            var results = new List<Order>();

            this.currentOrderableItemIen = String.Join(",", this.OrderableItemIens.ToArray());
            results = base.Execute(session);

            return results;
        }

        private List<Order> GetAnticipatedOrders(ViaVistASession session)
        {
            this.currentOrderableItemIen = null;
            return base.Execute(session);
        }
    }

    internal sealed class OrderDependencySource : IDependencySource
    {
        private static IDependencySource instance;

        private OrderDependencySource()
        {
        }

        public static IDependencySource Instance
        {
            get
            {
                if (instance == null)
                {
                    instance = new OrderDependencySource();
                }

                return instance;
            }
        }

        public IEnumerable<DependentEntityInfo> GetDependentTypes()
        {
            yield return new DependentEntityInfo<OrderStatus>(ListOrderStatusesCommand.Target, ListOrderStatusesCommand.ArgumentsCount, "5");
            yield return new DependentEntityInfo<Patient>(ListPatientCommand.Target, ListPatientCommand.ArgumentsCount, ListPatientCommand.CheckIen, ".02");
            yield return new DependentEntityInfo<HospitalLocation>(ListHospitalLocationCommand.Target, ListHospitalLocationCommand.ArgumentsCount, "6");
        }
    }
}
