﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Via.Commands.EntityCommands;
using BMS.Utils;
using BMS.VistaIntegration.Via.Commands.EIS;

namespace BMS.VistaIntegration.Via.Commands.WF
{
    public class ListOrderActionsCommand : BaseListPeriodCommand<OrderAction>
    {
        public static TimeSpan SplitDuration = TimeSpan.FromDays(1);

        private bool isAnticipated;

        private IEnumerable<Order> orders;

        private string currentOrdersSet;

        public ListOrderActionsCommand(ViaVistAQuery query)
            : base(query)
        {
        }

        public string PatientIen
        {
            get;
            set;
        }

        public IEnumerable<string> OrderableItemIens
        {
            get;
            set;
        }

        private DateTime? origStartDate
        {
            get;
            set;
        }

        private DateTime? origEndDate
        {
            get;
            set;
        }

        public override List<OrderAction> Execute(ViaVistASession session)
        {
            this.origStartDate = this.StartDate;
            this.origEndDate = this.EndDate;
            bool startInRange = true;
            bool endInRange = true;

            //  duplicates are not being removed - JNL
            List<OrderAction> t1 = Enumerable.Union(
                this.GetOrderActions(session),
                this.GetAnticipatedOrderActions(session),
                EqualityComparer).ToList();
            List<OrderAction> t2 = new List<OrderAction>();
            foreach (OrderAction oa1 in t1)
            {
                bool found = false;
                foreach (OrderAction oa2 in t2)
                {
                    if (oa2.OrderId == oa1.OrderId)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    //  date range check
                    startInRange = true;
                    endInRange = true;
                    if (origStartDate != null && origStartDate > oa1.DateTimeOrdered) startInRange = false;
                    if (origEndDate != null && origEndDate < oa1.DateTimeOrdered) endInRange = false;
                    if (startInRange && endInRange)
                    {
                        t2.Add(oa1);
                    }
                }
            }
            return t2;

            //return Enumerable.Union(
            //    this.GetOrderActions(session),
            //    this.GetAnticipatedOrderActions(session),
            //    EqualityComparer).ToList();
        }

        protected override string GetTarget()
        {
            return "ListOrderActions";
        }

        protected override IEnumerable<object> GetCriteria()
        {
            yield return string.Join(",", this.OrderableItemIens);
            yield return this.currentOrdersSet;
            yield return this.StartDate.GetValueOrDefault(MinDate);
            yield return this.EndDate.GetValueOrDefault(this.MaxDate);
            yield return this.isAnticipated ? 2 : 1;
            yield return this.MaxCount;
            yield return this.From;
        }

        protected override IDependencySource GetDependencySource()
        {
            return OrderActionDependencySource.Instance;
        }

        protected override IEnumerable<OrderAction> ProcessSingleDictionary(ViaVistASession session, string key, Dictionary<string, List<string>> dictionary)
        {
            return base.ProcessSingleDictionary(session, key, dictionary).DoAction(x =>
            {
                //  key is always null, so this logic to assign the correct order does not work -- JNL
                    //x.OrderId = key;
                    //x.Order = this.orders.SingleOrDefault(y => y.IEN == key);

                x.Order = this.orders.FirstOrDefault(y => y.IEN == x.OrderId);
            });
        }

        private IEnumerable<OrderAction> GetAnticipatedOrderActions(ViaVistASession session)
        {
            var result = Enumerable.Empty<OrderAction>();
            if (!this.OrderableItemIens.IsNullOrEmpty())
            {
                this.isAnticipated = true;
                var startDate = this.StartDate.GetValueOrDefault(MinDate);
                var endDate = this.EndDate.GetValueOrDefault(this.MaxDate);
                try
                {
                    for (var date = startDate; date <= endDate; date += SplitDuration)
                    {
                        this.StartDate = date;
                        this.EndDate = date + SplitDuration;
                        this.orders = this.GetOrders();

                        result = this.MergeResults(session, result);
                    }
                }
                finally
                {
                    this.StartDate = startDate;
                    this.EndDate = endDate;
                }
            }

            return result;
        }

        private IEnumerable<OrderAction> GetOrderActions(ViaVistASession session)
        {
            var result = Enumerable.Empty<OrderAction>();
            if (!this.OrderableItemIens.IsNullOrEmpty())
            {
                var startDate = this.StartDate.GetValueOrDefault(MinDate);
                var endDate = this.EndDate.GetValueOrDefault(this.MaxDate);
                try
                {
                    this.isAnticipated = false;
                    for (var date = startDate; date <= endDate; date += SplitDuration)
                    {
                        this.StartDate = date;
                        this.EndDate = date + SplitDuration;
                        this.orders = this.GetOrders();

                        result = this.MergeResults(session, result);
                    }
                }
                finally
                {
                    this.StartDate = startDate;
                    this.EndDate = endDate;
                }
            }

            return result;
        }

        private IEnumerable<OrderAction> MergeResults(ViaVistASession session, IEnumerable<OrderAction> result)
        {
            foreach (var set in this.orders.Select(x => x.IEN).Distinct().JoinAndSplitByLength(',', MaxCriteriaLength))
            {
                this.currentOrdersSet = set;
                result = result.Union(base.Execute(session), EqualityComparer);
            }

            return result;
        }

        private IEnumerable<Order> GetOrders()
        {
            return this.VistAQuery.GetResults(new ListOrdersCommand(this.VistAQuery)
            {
                StartDate = this.StartDate,
                EndDate = this.EndDate,
                OrderableItemIens = this.OrderableItemIens,
                PatientIen = this.PatientIen,
                IsAnticipated = this.isAnticipated,
            }).Distinct(ListOrdersCommand.EqualityComparer);
        }
    }

    internal sealed class OrderActionDependencySource : IDependencySource
    {
        private static IDependencySource instance;

        private OrderActionDependencySource()
        {
        }

        public static IDependencySource Instance
        {
            get
            {
                if (instance == null)
                {
                    instance = new OrderActionDependencySource();
                }

                return instance;
            }
        }

        public IEnumerable<DependentEntityInfo> GetDependentTypes()
        {
            yield return new DependentEntityInfo<NewPerson>(ListNewPersonCommand.Target, ListNewPersonCommand.ArgumentsCount, "5", "3");
        }
    }
}
