﻿using System;
using System.Security.Cryptography;
using System.Text;
using System.IO;
using System.Collections.Generic;

namespace BMS.Utils
{
    class Encryptors
    {
        public ICryptoTransform Encryptor { get; set; }
        public ICryptoTransform Decryptor { get; set; }
    }

    public class CustomEncryption
    {
        const string str = "LgBTAHUAcABlAHIALgBTAGUAYwByAGUAdAAuAFAAQABhAHMAcwB3AG8AcgBkAC4A";
        static Dictionary<string, Encryptors> saltEncryptors = new Dictionary<string, Encryptors>();

        /// <summary>
        /// Encryption method.
        /// </summary>
        /// <param name="input">string to encrypt</param>
        /// <param name="salt">salt used for encryption</param>
        /// <returns>encrypted string</returns>
        public static string Encrypt(string input, string salt)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {               
                if (!saltEncryptors.ContainsKey(salt))
                {
                    // symmetric encryption algorithm
                    using (AesManaged aes = new AesManaged())
                    {                        
                        byte[] saltBytes = UTF8Encoding.UTF8.GetBytes(salt);
                        // We're using the PBKDF2 standard for password-based key generation
                        using (Rfc2898DeriveBytes rfc = new Rfc2898DeriveBytes(UTF8Encoding.Unicode.GetString(Convert.FromBase64String(str)), saltBytes))
                        {
                            aes.Key = rfc.GetBytes(aes.KeySize / 8);
                            aes.IV = rfc.GetBytes(aes.BlockSize / 8);
                        }
                        saltEncryptors.Add(salt, new Encryptors() { Encryptor = aes.CreateEncryptor(), Decryptor = aes.CreateDecryptor() });
                    }
                }
                
                using (MemoryStream encryptStream = new MemoryStream())
                {
                    byte[] utfdata = UTF8Encoding.UTF8.GetBytes(input);
                    CryptoStream encryptor = new CryptoStream(encryptStream, saltEncryptors[salt].Encryptor, CryptoStreamMode.Write);

                    encryptor.Write(utfdata, 0, utfdata.Length);
                    encryptor.Flush();
                    encryptor.FlushFinalBlock();

                    return Encryptor.ConvertToAlphabetString(encryptStream.ToArray());
                }                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Decryption method.
        /// </summary>
        /// <param name="input">String to decrypt.</param>
        /// <param name="salt">salt used for decryption</param>
        /// <returns>Decrypted string.</returns>
        public static string Decrypt(string input, string salt)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!saltEncryptors.ContainsKey(salt))
                {
                    // symmetric encryption algorithm
                    using (AesManaged aes = new AesManaged())
                    {                        
                        byte[] saltBytes = UTF8Encoding.UTF8.GetBytes(salt);
                        // We're using the PBKDF2 standard for password-based key generation
                        using (Rfc2898DeriveBytes rfc = new Rfc2898DeriveBytes(UTF8Encoding.Unicode.GetString(Convert.FromBase64String(str)), saltBytes))
                        {
                            aes.Key = rfc.GetBytes(aes.KeySize / 8);
                            aes.IV = rfc.GetBytes(aes.BlockSize / 8);
                        }
                        saltEncryptors.Add(salt, new Encryptors() { Encryptor = aes.CreateEncryptor(), Decryptor = aes.CreateDecryptor() });
                    }
                }

                // Output stream, can be also a FileStream
                using (MemoryStream decryptStream = new MemoryStream())
                {
                    byte[] encryptBytes = Encryptor.ConvertFromAlphabetString(input);
                    CryptoStream decryptor = new CryptoStream(decryptStream, saltEncryptors[salt].Decryptor, CryptoStreamMode.Write);

                    decryptor.Write(encryptBytes, 0, encryptBytes.Length);
                    decryptor.Flush();
                    decryptor.FlushFinalBlock();

                    // Showing our decrypted content
                    byte[] decryptBytes = decryptStream.ToArray();
                    return UTF8Encoding.UTF8.GetString(decryptBytes, 0, decryptBytes.Length);                        
                }                
            }
            catch (Exception ex)
            {
                throw new Exception("String to decrypt is incorrect!", ex);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetRandomSalt(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return Convert.ToBase64String(ASCIIEncoding.ASCII.GetBytes(userName));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
