﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS.Extension.EIS;
using InfoWorld.HL7.ITS;
using BMS.Utils;

namespace BMS.ServicesWrapper.EIS
{
    internal static class TraitManager
    {

        #region Eis Traits Manager

        /// <summary>
        /// Creates the trait pair.
        /// </summary>
        /// <param name="traitIdentifier">The trait identifier.</param>
        /// <param name="traitValue">The trait value.</param>
        /// <param name="algorithm">The algorithm.</param>
        /// <returns></returns>
        private static TraitPair CreateTraitPair(string traitIdentifier, ANY traitValue, MatchingAlgorithm algorithm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair t = new TraitPair();
                t.Identifier = traitIdentifier;
                t.Value = traitValue;
                t.MatchingAlgorithm = algorithm;
                return t;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair.
        /// </summary>
        /// <param name="traitIdentifier">The trait identifier.</param>
        /// <param name="traitValue">The trait value.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPair(string traitIdentifier, ANY traitValue)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                MatchingAlgorithm m = traitValue is R_Base ? new MatchingAlgorithmRole() :
                    traitValue is TS || traitValue is INT || traitValue is REAL ? new MatchingAlgorithmIntRealDateTime() :
                    traitValue is BL ? new MatchingAlgorithmBool() : (MatchingAlgorithm)new MatchingAlgorithmString();
                return CreateTraitPair(traitIdentifier, traitValue, m);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Creates the traits list.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        public static TraitList CreateTraitsList(string name, string ien)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitList traitsList = new TraitList();
                traitsList.Add(CreateTraitPairForName(name));
                traitsList.Add(CreateTraitPairForIen(ien));
                return traitsList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the the trait pair for name.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForName(string name)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.NAME, new ST(name));
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for ien.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForIen(string ien)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.UID, new II { root = EISConstants.IEN, extension = ien });
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for code.
        /// </summary>
        /// <param name="ien">The code.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null; 
                trait = CreateTraitPair(EISConstants.UID, new II { root = EISConstants.CODE, extension = code });
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for number.
        /// </summary>
        /// <param name="number">The number.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForNumber(string number)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.UID, new II { root = EISConstants.NUMBER, extension = number });
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the type of the trait pair for place.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForPlaceType(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string dispayName = string.Empty;
                switch (code)
                {
                    case EISConstants.IDENTIFIER_FOR_REGION_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_REGION_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_VISN_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_VISN_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_VISTA_SITE_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_VISTA_SITE_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_HOSPITAL_LOCATION_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_HOSPITAL_LOCATION_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_FACILITY_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_FACILITY_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_WARD_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_WARD_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_WARDGROUP_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_WARDGROUP_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_BED_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_BED_NAME;
                        break;
                }

                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.CODE, new CS(code, EISConstants.ORGANIZATION_CHART_CODE_SYSTEM, EISConstants.ORGANIZATION_CHART_CODE, null, dispayName, null));
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for vista Specialty.
        /// </summary>
        /// <param name="vistaSpecialty">The vista Specialty.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForVistaSpecialty(string vistaSpecialty)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.VISTASPECIALTY, new ST(vistaSpecialty));
                if (trait != null)
                {
                    MatchingAlgorithmString role = trait.MatchingAlgorithm as MatchingAlgorithmString;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsString.IdenticalIgnoreCase;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for member.
        /// </summary>
        /// <param name="r_Member">The r_ member.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForMember(R_Member r_Member)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.MBR, r_Member);
                if (trait != null)
                {
                    MatchingAlgorithmRole role = trait.MatchingAlgorithm as MatchingAlgorithmRole;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsRole.IsEqualTo;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the trait pair for service delivery location.
        /// </summary>
        /// <param name="serviceDeliveryLocation">The service delivery location.</param>
        /// <returns></returns>
        public static TraitPair CreateTraitPairForServiceDeliveryLocation(R_ServiceDeliveryLocation serviceDeliveryLocation)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TraitPair trait = null;
                trait = CreateTraitPair(EISConstants.SDLOC, serviceDeliveryLocation);
                if (trait != null)
                {
                    MatchingAlgorithmRole role = trait.MatchingAlgorithm as MatchingAlgorithmRole;
                    if (role != null)
                    {
                        role.value = MatchingAlgorithmsRole.IsEqualTo;
                        trait.MatchingAlgorithm = role;
                    }
                }
                return trait;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region EIS Identifier Manager

        /// <summary>
        /// Creates the identifiers list.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        public static List<IdentifierValuePair> CreateIdentifiersList(string name, string ien)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IdentifierValuePair> identifiersList = new List<IdentifierValuePair>();
                identifiersList.Add(CreateIdentifierForName(name));
                identifiersList.Add(CreateIdentifierForIen(ien));
                return identifiersList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the name of the identidier for.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForName(string name)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(name);
                identifier.TraitIdentifier = EISConstants.NAME;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for number.
        /// </summary>
        /// <param name="number">The number.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForNumber(string number)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new II { root = EISConstants.NUMBER, extension = number };
                identifier.TraitIdentifier = EISConstants.UID;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for ien.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForIen(string ien)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new II { root = EISConstants.IEN, extension = ien };
                identifier.TraitIdentifier = EISConstants.UID;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new II { root = EISConstants.CODE, extension = code };
                identifier.TraitIdentifier = EISConstants.UID;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the type of the identifier for place.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForPlaceType(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                string dispayName = string.Empty;
                switch (code)
                {
                    case EISConstants.IDENTIFIER_FOR_REGION_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_REGION_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_VISN_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_VISN_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_VISTA_SITE_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_VISTA_SITE_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_HOSPITAL_LOCATION_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_HOSPITAL_LOCATION_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_FACILITY_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_FACILITY_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_WARD_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_WARD_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_WARDGROUP_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_WARDGROUP_NAME;
                        break;
                    case EISConstants.IDENTIFIER_FOR_BED_CODE:
                        dispayName = EISConstants.IDENTIFIER_FOR_BED_NAME;
                        break;
                }
                identifier.TraitValue = new CS(code, EISConstants.ORGANIZATION_CHART_CODE_SYSTEM, EISConstants.ORGANIZATION_CHART_CODE, null, dispayName, null);
                identifier.TraitIdentifier = EISConstants.CODE;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for member.
        /// </summary>
        /// <param name="member">The member.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForMember(R_Member member)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = member;
                identifier.TraitIdentifier = EISConstants.MBR;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for service delivery location.
        /// </summary>
        /// <param name="playerId">The player.</param>
        /// <param name="scoperId">The scoper.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForServiceDeliveryLocation(II playerId, II scoperId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                R_ServiceDeliveryLocation serviceDeliveryLocation = new R_ServiceDeliveryLocation()
                {
                    Code = CreateRoleCode(EISConstants.WARD_CODE, EISConstants.WARD_NAME),
                    Player = playerId == null ? null : new EntityInRole() { DomainIdentifier = playerId.root, EntityTypeIdentifier = EISConstants.PLACE, Identifier = playerId.extension },
                    Scoper = scoperId == null ? null : new EntityInRole() { DomainIdentifier = scoperId.root, EntityTypeIdentifier = EISConstants.ORGANIZATON, Identifier = scoperId.extension }
                };
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = serviceDeliveryLocation;
                identifier.TraitIdentifier = EISConstants.SDLOC;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for vista Specialty.
        /// </summary>
        /// <param name="vistaSpecialty">The vista Specialty.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForVistaSpecialty(string vistaSpecialty)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(vistaSpecialty);
                identifier.TraitIdentifier = EISConstants.VISTASPECIALTY;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the IS_ACTIVE trait 
        /// </summary>
        /// <param name="isActive">active or inactive</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsActive(bool isActive)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_ACTIVE;
                identifier.TraitValue = new ST() { text = isActive ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the HIDE_SPECIALTY_DESKTOP trait 
        /// </summary>
        /// <param name="hideSpecialtyDesktop">display or not</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForHideSpecialtyDesktop(bool hideSpecialtyDesktop)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.HIDE_SPECIALTY_DESKTOP;
                identifier.TraitValue = new ST() { text = hideSpecialtyDesktop ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the HIDE_SPECIALTY_WHITEBOARD trait 
        /// </summary>
        /// <param name="hideSpecialtyKiosk">display or not</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForHideSpecialtyKiosk(bool hideSpecialtyKiosk)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.HIDE_SPECIALTY_KIOSK;
                identifier.TraitValue = new ST() { text = hideSpecialtyKiosk ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the IS_VIRTUAL_WARD trait 
        /// </summary>
        /// <param name="isVirtual">virtual or not</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsVirtualWard(bool isVirtual)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_VIRTUAL_WARD;
                identifier.TraitValue = new ST() { text = isVirtual ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the CENSUS_CATEGORY trait 
        /// </summary>
        /// <param name="censusCategory">the census category</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForCensusCategory(string censusCategory)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(censusCategory);
                identifier.TraitIdentifier = EISConstants.CENSUS_CATEGORY;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the ASSIGNED_VIRTUAL_WARD trait 
        /// </summary>
        /// <param name="censusCategory">the census category</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForAssignedVirtualWard(string assignedVirtualWard)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(assignedVirtualWard);
                identifier.TraitIdentifier = EISConstants.ASSIGNED_VIRTUAL_WARD;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates IS_ALIVE trait.
        /// </summary>
        /// <param name="isAlive">true/false</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsAlive(bool isAlive)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_ALIVE;
                identifier.TraitValue = new ST() { text = isAlive ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates IS_VISTA_VALIDATED trait.
        /// </summary>
        /// <param name="isVistaValidated">true/false</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsVistaValidated(bool isVistaValidated)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_VISTA_VALIDATED;
                identifier.TraitValue = new ST() { text = isVistaValidated ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates IS_PATIENT_OPTOUT trait.
        /// </summary>
        /// <param name="isPatientOptOut">true/false</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsPatientOptOut(bool isPatientOptOut)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_PATIENT_OPTOUT;
                identifier.TraitValue = new INT() { IntValue = isPatientOptOut ? 1 : 0 };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates IS_CURRENTLY_OOS trait.
        /// </summary>
        /// <param name="isCurrentlyOOS">true/false</param>
        /// <returns>the trait for create/update</returns>
        public static IdentifierValuePair CreateIdentifierForIsCurrentlyOOS(bool isCurrentlyOOS)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitIdentifier = EISConstants.IS_CURRENTLY_OOS;
                identifier.TraitValue = new ST() { text = isCurrentlyOOS ? EISConstants.TRUE : EISConstants.FALSE };
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the identifier for type group.
        /// </summary>
        /// <param name="typeGroup">The type group.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForTypeGroup(string typeGroup)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(typeGroup);
                identifier.TraitIdentifier = EISConstants.TYPEGROUP;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the name of the identifier for person contact.
        /// </summary>
        /// <param name="personName">Name of the person.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForPersonContactName(string personName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IdentifierValuePair identifier = new IdentifierValuePair();
                identifier.TraitValue = new ST(personName);
                identifier.TraitIdentifier = EISConstants.PERSON_CONTACT_NAME;
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates an identifier value pair for a boolean value.
        /// </summary>
        /// <param name="traitIdentifier">the trait identifier</param>
        /// <param name="value">the boolean value</param>
        /// <returns>the identifier value pair</returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, bool value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = new INT() { IntValue = value ? 1 : 0 }
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates an identifier value pair for a string value.
        /// </summary>
        /// <param name="traitIdentifier">the trait identifier</param>
        /// <param name="value">the string value</param>
        /// <returns>the identifier value pair</returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, string value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = new ST() { text = value }
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates an identifier value pair for a CD value.
        /// </summary>
        /// <param name="traitIdentifier">the trait identifier</param>
        /// <param name="value">the CD value</param>
        /// <returns>the identifier value pair</returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, CD value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = new ST() { text = value == null ? "" : value.code }
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates an identifier value pair for an address.
        /// </summary>
        /// <param name="traitIdentifier">the trait identifier</param>
        /// <param name="value">the address value</param>
        /// <returns>the identifier value pair</returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, AD value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = value
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Creates the identifier.
        /// </summary>
        /// <param name="traitIdentifier">The trait identifier.</param>
        /// <param name="value">The value.</param>
        /// <returns></returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, ST value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = value
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates an identifier value pair for a contact.
        /// </summary>
        /// <param name="traitIdentifier">the trait identifier</param>
        /// <param name="tel">the contact value</param>
        /// <param name="contactType">the contact type</param>
        /// <returns>the identifier value pair</returns>
        internal static IdentifierValuePair CreateIdentifier(string traitIdentifier, TEL tel, ContactType contactType)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                tel.use = new TelecommunicationAddressUseList();
                switch (contactType)
                {
                    case ContactType.Fax:
                        tel.use.Add(TelecommunicationAddressUse.PUB);
                        tel.value = EISConstants.FAX_CONTACT_VALUE + tel.value;
                        break;
                    case ContactType.Email:
                        tel.use.Add(TelecommunicationAddressUse.PUB);
                        tel.value = EISConstants.EMAIL_CONTACT_VALUE + tel.value;
                        break;
                    case ContactType.Phone:
                        tel.use.Add(TelecommunicationAddressUse.PUB);
                        tel.value = EISConstants.TELEPHONE_CONTACT_VALUE + tel.value;
                        break;
                    case ContactType.Mobile:
                        tel.use.Add(TelecommunicationAddressUse.MC);
                        tel.value = EISConstants.TELEPHONE_CONTACT_VALUE + tel.value;
                        break;
                }
                return new IdentifierValuePair()
                {
                    TraitIdentifier = traitIdentifier,
                    TraitValue = tel
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        #endregion

        #region Helper Methods

        /// <summary>
        /// Creates the role code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <param name="dispayName">Name of the dispay.</param>
        /// <returns></returns>
        internal static CE CreateRoleCode(string code, string dispayName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new CE(code, EISConstants.ROLE_CODE_SYSTEM, EISConstants.ROLE_CODE, "", dispayName, null, null, null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Copies the entity into role.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        internal static EntityInRole CopyEntityId(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null || string.IsNullOrEmpty(entity.Identifier))
                    return null;
                return new EntityInRole()
                {
                    DomainIdentifier = entity.DomainIdentifier,
                    EntityTypeIdentifier = entity.EntityTypeIdentifier,
                    Identifier = entity.Identifier
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Copies the organization identificator into role.
        /// </summary>
        /// <param name="id">The organization id.</param>
        /// <returns></returns>
        internal static EntityInRole CreateOrganizationEntityInRole(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (id == null || string.IsNullOrEmpty(id.extension))
                    return null;
                return new EntityInRole()
                {
                    DomainIdentifier = id.root,
                    EntityTypeIdentifier = EISConstants.ORGANIZATON,
                    Identifier = id.extension
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Copies the entity identificator into role.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        internal static EntityInRole CopyEntityId(EntityIdentification entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null || string.IsNullOrEmpty(entity.EntityIdentifier))
                    return null;
                return new EntityInRole()
                {
                    DomainIdentifier = entity.DomainIdentifier,
                    EntityTypeIdentifier = entity.EntityTypeIdentifier,
                    Identifier = entity.EntityIdentifier
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static II FindScoper(Entity entity, string traitIdentifier)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                foreach (IdentifierValuePair identifierValuePair in entity.TraitList.Where(a => a.TraitIdentifier == traitIdentifier))
                {
                    R_Base role = identifierValuePair.TraitValue as R_Base;
                    if (role != null && role.Scoper != null && role.Scoper.Identifier != null)
                        return new II() { root = role.Scoper.DomainIdentifier, extension = role.Scoper.Identifier };
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static List<II> FindScopers(Entity entity, string traitIdentifier)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<II> scopers = new List<II>();
                foreach (IdentifierValuePair identifierValuePair in entity.TraitList.Where(a => a.TraitIdentifier == traitIdentifier))
                {
                    R_Base role = identifierValuePair.TraitValue as R_Base;
                    if (role != null && role.Scoper != null && role.Scoper.Identifier != null)
                        scopers.Add(new II() { root = role.Scoper.DomainIdentifier, extension = role.Scoper.Identifier });
                }
                if (scopers.Count > 0)
                    return scopers;
                else
                    return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Creates the identifier for phone.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        public static IdentifierValuePair CreateIdentifierForPhone(string phone)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TEL traitValue = CreatePhoneTraitValue(phone);
                IdentifierValuePair identifier = new IdentifierValuePair(EISConstants.PHONE, traitValue);
                return identifier;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static TEL CreatePhoneTraitValue(string phone)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return CreateTELTraitValue("PHONE", phone);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static TEL CreateTELTraitValue(string type, string v)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(v))
                    return null;

                TEL tel = new TEL();
                tel.value = GetTELPrefix(type) + v.ToString();
                tel.use = new TelecommunicationAddressUseList();
                tel.use.Add(TelecommunicationAddressUse.PUB);
                if (type == "MOBILE")
                    tel.use.Add(TelecommunicationAddressUse.MC);
                return tel;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static string GetTELPrefix(string type)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (type == "MOBILE" || type == "PHONE")
                    return "tel:";
                if (type == "FAX")
                    return "fax:";
                if (type == "EMAIL")
                    return "mailto:";
                return "";
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static bool Equals(II x, II y)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (x == null && y == null)
                    return true;
                if (x == null || y == null)
                    return false;
                return string.Equals(x.root, y.root, StringComparison.InvariantCultureIgnoreCase)
                    && string.Equals(x.extension, y.extension, StringComparison.InvariantCultureIgnoreCase);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
