﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.EVS;
using BMS.ServicesWrapper.EIS;
using BMS.Utils;
using EisPatient = BMS.Facade.Data.Patient;
using SC = BMS.DataContracts;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.Facade;
using FC = BMS.Facade.Data;
using BMS.VistaWorker2.Writer.Implementation.Concrete.EVS;

namespace BMS.VistaWorker2.Writer.Implementation.Concrete.EIS
{
    public class PatientWriter : BaseEisWriter<Patient, EisPatient>
    {
        public override void InitCache()
        {
            Data.SetCacheForTreatingSpecialty();
        }

        private FC.Person attendingPhysician;

        protected override void LoadData(Patient entity)
        {
            if (entity.AttendingPhysician != null && entity.AttendingPhysician.IEN == "-100") // avoid updating attending physician on update patients scheduler
                attendingPhysician = new FC.Person() { Ien = "-100" };
            else
                attendingPhysician = InsertIfNullOrUpdateIfDirty<NewPerson, FC.Person>(entity.AttendingPhysician);
        }

        protected override EisPatient GetBmsEntity(Patient entity)
        {
            return base.GetBmsEntity(entity);
        }


        protected override void TranslateFromVistaToBMS(Patient entity, EisPatient bmsEntity)
        {
            IList<CD> genderList = EVS.GetCodes(new CodeFilterParameters() { MaxSelectedCodes = int.MaxValue, VocabularyDomain = BMS.Facade.Data.Util.Vocabulary.Gender.ToString() });
            Dictionary<string, string> names = Utilities.SplitPersonFullName(entity.Name);
            bmsEntity.FirstName = names[Constants.PERSON_FIRST_NAME];
            CD gender = genderList.Where(a => a.code.ToLower() == entity.Sex.ToString().ToLower()).FirstOrDefault();
            if (gender == null)
                gender = genderList.Where(a => a.code == Constants.GENDER_UNKNOWN_CODE).First();
            bmsEntity.Gender = gender;
            
            bmsEntity.LastName = names[Constants.PERSON_LAST_NAME];
            bmsEntity.MiddleName = names[Constants.PERSON_MIDDLE_NAME];
            bmsEntity.SSN = new II(Constants.SSNROOT, entity.SocialSecurityNumber);
            bmsEntity.IsVistaValidated = true;
            bmsEntity.TreatingSpecialty = InsertIfNullOrUpdateIfDirty<FacilityTreatingSpecialty, CD>(entity.TreatingSpecialty);
            bmsEntity.AdmittingDiagnosis = entity.AdmittingDiagnosis;
            bmsEntity.IsOptOut = entity.IsOptOut;
            bmsEntity.ServiceConnectedPercentage = entity.ServiceConnectedPercentage.HasValue ?
                entity.ServiceConnectedPercentage.ToString() : "";

            bmsEntity.DateOfBirth = entity.DateOfBirth;
            bmsEntity.AttendingPhysicianId = attendingPhysician != null ? (attendingPhysician.Ien != "-100" ? attendingPhysician.Id : bmsEntity.AttendingPhysicianId) : null;

            if (bmsEntity.IENList != null && bmsEntity.IENList.Count > 0)
            {
                if (bmsEntity.IENList.Where(ii => ii.root == this.VistaSite.Code && ii.extension == entity.IEN).FirstOrDefault() == null)
                    bmsEntity.IENList.Add(new II { root = this.VistaSite.Code, extension = entity.IEN });
            }
            else
            {
                bmsEntity.IENList = new List<II>();
                bmsEntity.IENList.Add(new II { root = this.VistaSite.Code, extension = entity.IEN });
            }
        }

        private bool SameAttendingPhysician(EisPatient bmsEntity)
        {
            if (attendingPhysician != null && attendingPhysician.Ien == "-100")
                return true;

            if (attendingPhysician == null && bmsEntity.AttendingPhysicianId == null)
                return true;

            if (attendingPhysician == null || bmsEntity.AttendingPhysicianId == null)
                return false;

            return attendingPhysician.Id.extension.Equals(bmsEntity.AttendingPhysicianId.extension, StringComparison.InvariantCultureIgnoreCase) &&
                attendingPhysician.Id.root.Equals(bmsEntity.AttendingPhysicianId.root, StringComparison.InvariantCultureIgnoreCase);
        }

        protected override bool AreFieldsEqual(Patient entity, EisPatient bmsEntity)
        {
            bool result;
            Dictionary<string, string> names = Utilities.SplitPersonFullName(entity.Name);
            result = bmsEntity.FirstName == names[Constants.PERSON_FIRST_NAME] &&
                     string.Compare(GetGender(bmsEntity), GetGender(entity), true) == 0 &&
                     bmsEntity.LastName == names[Constants.PERSON_LAST_NAME] &&
                     bmsEntity.MiddleName == names[Constants.PERSON_MIDDLE_NAME] &&
                     bmsEntity.SSN.extension == entity.SocialSecurityNumber &&
                     bmsEntity.AdmittingDiagnosis == entity.AdmittingDiagnosis &&
                     bmsEntity.IsOptOut == entity.IsOptOut &&
                     CompareTreatingSpecialty(bmsEntity.TreatingSpecialty, entity.TreatingSpecialtyId) &&
                     bmsEntity.ServiceConnectedPercentage == ((entity.ServiceConnectedPercentage.HasValue) ? entity.ServiceConnectedPercentage.Value.ToString() : string.Empty) &&
                     SameAttendingPhysician(bmsEntity) &&
                     CheckPatientIen(bmsEntity, entity);

            return result;
        }

        /// <summary>
        /// Checks if IEN already exists in the patient list of IEN's.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        /// <param name="entity">The entity.</param>
        /// <returns>True - if IEN exists, otherwise False - if IEN does not exists.</returns>
        private bool CheckPatientIen(EisPatient bmsEntity, Patient entity)
        {
            if (bmsEntity.IENList == null)
                return false;
            if (bmsEntity.IENList != null && bmsEntity.IENList.Count == 0)
                return false;
            if (bmsEntity.IENList != null && bmsEntity.IENList.Count > 0)
            {
                if (bmsEntity.IENList.Where(ii => ii.root == this.VistaSite.Code && ii.extension == entity.IEN).FirstOrDefault() != null)
                    return true;
            }
            return false;
        }

        private static string GetGender(Patient patient)
        {
            return patient.Sex == null ? null : patient.Sex.Value.ToString();
        }

        private static string GetGender(EisPatient patient)
        {
            return patient.Gender == null || patient.Gender.code == Constants.GENDER_UNKNOWN_CODE ? null : patient.Gender.code;
        }

        private bool CompareTreatingSpecialty(CD specialty, string treatingSpecialtyId)
        {
            if (specialty == null && string.IsNullOrEmpty(treatingSpecialtyId)) return true;
            if (specialty == null || string.IsNullOrEmpty(treatingSpecialtyId)) return false;
            return specialty.code == VistaSite.Code + "_" + treatingSpecialtyId;
        }

        protected override IEntityDal<EisPatient> Dal
        {
            get { return Data.PatientDal; }
        }
    }
}
