﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.ServicesWrapper.EVS;

using FC = BMS.Facade.Data;

namespace BMS.VistaWorker.Writer.Concrete.EIS
{
    /// <summary>
    /// The ward location writer.
    /// </summary>
    class WardLocationWriter : BaseEisWriter<WardLocation, FC.Ward>
    {

        /// <summary>
        /// Gets the BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        protected override FC.Ward GetBmsEntity(WardLocation entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EIS.GetWard(entity.IEN, VistaSite.Id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Adds the modifications for BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void TranslateFromVistaToBMS(WardLocation entity, FC.Ward bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FC.HospitalLocation hospitalLocation = EIS.GetHospitalLocation(entity.HospitalLocationId, VistaSite.Id);                
                bmsEntity.Ien = entity.IEN;
                bmsEntity.Name = entity.Name;
                if (entity.SpecialtyId.HasValue)
                {
                    CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                    codeFilterParam.VocabularyDomain = FC.Util.Vocabulary.VistaSpecialty.ToString();
                    bmsEntity.Specialty = EVSFactory.InstanceWindows.GetCodes(codeFilterParam).Where(a => a.code == entity.SpecialtyId.Value.ToString()).FirstOrDefault();
                }
                else
                    bmsEntity.Specialty = null;
                bmsEntity.VistaSite = VistaSite;
                bmsEntity.IsCurrentlyOOS = entity.IsCurrentlyOutOfService;

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Inserts the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void InsertBmsEntity(FC.Ward bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.CreateWard(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void UpdateBmsEntity(FC.Ward bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.UpdateWard(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected override void DeleteBmsEntity(FC.Ward bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.DeleteWard(bmsEntity.Id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
