﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Cache.Commands;
using InterSystems.Data.CacheClient;
using BMS.VistaIntegration.Data.WF;


namespace BMS.VistaIntegration.Cache
{
    public class CacheVistAQuery : IVistAQuery
    {
        private readonly CacheVistASession session;
        private const int DEFAULT_BULK_COUNT = 1000;
        private DateTime minStartDate = DateTime.UtcNow.AddYears(-100);
        private DateTime maxEndDate = DateTime.UtcNow.AddDays(1);
        private CacheCommandInfo hospitalLocationCommand = new CacheCommandInfo(CacheCommandQueries.HOSPITAL_LOCATION_QUERY);
        private CacheCommandInfo wardLocationCommand = new CacheCommandInfo(CacheCommandQueries.WARD_LOCATION_QUERY);
        private CacheCommandInfo roomBedCommand = new CacheCommandInfo(CacheCommandQueries.ROOM_BED_QUERY);
        private CacheCommandInfo patientCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_QUERY);
        private CacheCommandInfo patientByIenCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_BY_IEN_QUERY);
        private CacheCommandInfo patientBySsnsCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_BY_SSNS_QUERY);
        private CacheCommandInfo newPersonCommand = new CacheCommandInfo(CacheCommandQueries.NEW_PERSON_QUERY);
        private CacheCommandInfo facilityTreatingSpecialtyCommand = new CacheCommandInfo(CacheCommandQueries.FACILITY_TREATING_SPECIALTY_QUERY);
        private CacheCommandInfo facilityMovementTypeCommand = new CacheCommandInfo(CacheCommandQueries.FACILITY_MOVEMENT_TYPE_QUERY);
        private CacheCommandInfo orderableItemCommand = new CacheCommandInfo(CacheCommandQueries.ORDERABLE_ITEM_QUERY);
        private CacheCommandInfo specialtyCommand = new CacheCommandInfo(CacheCommandQueries.SPECIALTY_QUERY);
        private CacheCommandInfo orderStatusCommand = new CacheCommandInfo(CacheCommandQueries.ORDER_STATUS_QUERY);
        private CacheCommandInfo medicalCenterDivisionCommand = new CacheCommandInfo(CacheCommandQueries.MEDICAL_CENTER_DIVISION_QUERY);
        private CacheCommandInfo orderCommand = new CacheCommandInfo(CacheCommandQueries.ORDER_QUERY);
        private CacheCommandInfo patientAppointmentCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_APPOINTMENT_QUERY);
        private CacheCommandInfo patientMovementCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_QUERY);
        private CacheCommandInfo patientMovementByPatientIenCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_BY_PATIENT_IEN_QUERY);
        private CacheCommandInfo patientMovementByAdmissionCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_BY_ADMISSION_QUERY);
        private CacheCommandInfo scheduledAdmissionCommand = new CacheCommandInfo(CacheCommandQueries.SCHEDULED_ADMISSION_QUERY);
        private CacheCommandInfo bedSwitchCommand = new CacheCommandInfo(CacheCommandQueries.BED_SWITCH_QUERY);
        private CacheCommandInfo cancelledOrdersCommand = new CacheCommandInfo(CacheCommandQueries.CANCELLED_ORDER_QUERY);
        private CacheCommandInfo admittedPatientsCommand = new CacheCommandInfo(CacheCommandQueries.ADMITTED_PATIENTS_QUERY);
        private CacheCommandInfo patientMovementIensCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_IENS_QUERY);
        private CacheCommandInfo patientMovementByIenCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_BY_IEN_QUERY);

        public int BulkCount { get; set; }

        public CacheVistAQuery(CacheVistASession _session)
        {
            this.BulkCount = DEFAULT_BULK_COUNT;
            this.session = _session;
            bool bhSupported = VistasBedHoldSupported.IsBedHoldSupported(_session.VistASite.Name);
            roomBedCommand.IsBedHoldSupported = bhSupported;
            patientMovementCommand.IsBedHoldSupported = bhSupported;
            patientMovementByPatientIenCommand.IsBedHoldSupported = bhSupported;
            patientMovementByAdmissionCommand.IsBedHoldSupported = bhSupported;
            patientMovementByIenCommand.IsBedHoldSupported = bhSupported;
        }

        private IList<T> ExecuteCommand<T>(CacheCommandInfo commandInfo, DateTime? startDate = null, DateTime? endDate = null) where T : class
        {
            if (!commandInfo.HasMore) return null;
            IList<T> result = new List<T>();
            commandInfo.RowCount = BulkCount;
            if (string.IsNullOrEmpty(commandInfo.LastIen))
                commandInfo.LastIen = "0";
            commandInfo.FromIen = "'" + commandInfo.LastIen + "'";
            commandInfo.StartDate = startDate;
            commandInfo.EndDate = endDate;
            string commandQuery = commandInfo.CommandQuery;
            commandQuery = commandQuery.Replace(Constants.SCHEMA, this.session.Schema);
            commandQuery = Utils.ReplaceQuery(commandInfo, commandQuery);
                        
            CacheCommand cmd = new CacheCommand(commandQuery, this.session.OdbcConnection);
            CacheDataReader reader = cmd.ExecuteReader();
            commandInfo.LastIen = Translators.Translate<T>(reader, commandInfo.IsBedHoldSupported, out result);
            reader.Close();
            cmd.Dispose();
            if (result != null && result.Count == BulkCount)
                commandInfo.HasMore = true;
            else
                commandInfo.HasMore = false;
            return result;
        }        

        #region EIS

        public IList<HospitalLocation> GetHospitalLocations()
        {
            return ExecuteCommand<HospitalLocation>(hospitalLocationCommand);
        }

        public IList<WardLocation> GetWardLocations()
        {
            return ExecuteCommand<WardLocation>(wardLocationCommand);
        }

        public IList<RoomBed> GetRoomBeds()
        {
            return ExecuteCommand<RoomBed>(roomBedCommand);
        }

        public IList<Patient> GetPatients(DateTime? startDate = null, DateTime? endDate = null, string vistaPatientIen = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            return ExecuteCommand<Patient>(patientCommand, startDate, endDate);
        }

        public Patient GetPatientByIen(string ien)
        {
            patientByIenCommand.Ien = ien;
            return ExecuteCommand<Patient>(patientByIenCommand, null, null).FirstOrDefault();
        }

        public IList<Patient> GetPatients(IEnumerable<string> soseclist)
        {
            StringBuilder sb = new StringBuilder();
            foreach (string x in soseclist)
            {
                sb.Append(x.Replace("-", string.Empty));
                sb.Append(",");
            }
            sb.Remove(sb.Length - 1, 1);
            patientBySsnsCommand.PatientSsns = sb.ToString();
            return ExecuteCommand<Patient>(patientBySsnsCommand, null, null);
        }

        public IList<NewPerson> GetNewPersons(DateTime? startDate = null, DateTime? endDate = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            return ExecuteCommand<NewPerson>(newPersonCommand, startDate, endDate);
        }

        public IList<Patient> GetAdmittedPatientsForUpdate()
        {
            throw new NotImplementedException();
        }

        #endregion

        #region EVS

        public IList<FacilityTreatingSpecialty> GetFacilityTreatingSpecialties()
        {
            return ExecuteCommand<FacilityTreatingSpecialty>(facilityTreatingSpecialtyCommand);
        }

        public IList<FacilityMovementType> GetFacilityMovementTypes()
        {
            return ExecuteCommand<FacilityMovementType>(facilityMovementTypeCommand);
        }

        public IList<OrderableItem> GetOrderableItems()
        {
            return ExecuteCommand<OrderableItem>(orderableItemCommand);
        }

        public IList<Specialty> GetSpecialties()
        {
            return ExecuteCommand<Specialty>(specialtyCommand);
        }

        public IList<OrderStatus> GetOrderStatuses()
        {
            return ExecuteCommand<OrderStatus>(orderStatusCommand);
        }

        public IList<MedicalCenterDivision> GetMedicalCenterDivisions()
        {
            return ExecuteCommand<MedicalCenterDivision>(medicalCenterDivisionCommand);
        }

        #endregion

        #region WF

        public IList<OrderAction> GetOrderActions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> orderableItemsIen = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            if (!string.IsNullOrEmpty(patientIen))
                orderCommand.Ien = patientIen;
            if (orderableItemsIen != null && orderableItemsIen.Count() > 0)
                orderCommand.OrderableItemsIen = orderableItemsIen;
            return ExecuteCommand<OrderAction>(orderCommand, startDate, endDate);
        }

        public IList<PatientAppointment> GetPatientAppointments(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> clinicIens = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            if (!string.IsNullOrEmpty(patientIen))
                patientAppointmentCommand.Ien = patientIen;
            if (clinicIens != null && clinicIens.Count() > 0)
                patientAppointmentCommand.ClinicsIen = clinicIens;
            if (string.IsNullOrEmpty(patientIen) && (clinicIens == null || clinicIens.Count() == 0))
                return new List<PatientAppointment>();
            return ExecuteCommand<PatientAppointment>(patientAppointmentCommand, startDate, endDate);
        }

        public IList<PatientMovement> GetPatientMovements(DateTime? startDate = null, DateTime? endDate = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            return ExecuteCommand<PatientMovement>(patientMovementCommand, startDate, endDate);
        }

        public PatientMovement GetPatientMovement(string patientIen, DateTime dateTime, BMS.Utils.MovementTransactionType movementType)
        {
            patientMovementByPatientIenCommand.Ien = patientIen;
            patientMovementByPatientIenCommand.StartDate = dateTime;
            patientMovementByPatientIenCommand.TransactionType = movementType;
            return ExecuteCommand<PatientMovement>(patientMovementByPatientIenCommand).FirstOrDefault();
        }

        public IList<PatientMovement> GetPatientMovementsForAdmission(string admissionIen)
        {
            patientMovementByAdmissionCommand = new CacheCommandInfo(CacheCommandQueries.PATIENT_MOVEMENT_BY_ADMISSION_QUERY);
            patientMovementByAdmissionCommand.Ien = admissionIen;
            return ExecuteCommand<PatientMovement>(patientMovementByAdmissionCommand, null, null);
        }

        public IList<AdmittedPatient> GetAdmittedPatients(string lastPatientMovementIen)
        {
            admittedPatientsCommand.Ien = lastPatientMovementIen;
            return ExecuteCommand<AdmittedPatient>(admittedPatientsCommand, null, null);
        }

        public IList<ScheduledAdmission> GetScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null)
        {
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;
            if (!string.IsNullOrEmpty(patientIen))
                scheduledAdmissionCommand.Ien = patientIen;
            return ExecuteCommand<ScheduledAdmission>(scheduledAdmissionCommand, startDate, endDate);
        }

        public IList<BedSwitch> GetBedsSwitch(IEnumerable<string> iens)
        {
            if (iens != null && iens.Count() > 0)
            {
                StringBuilder sb = new StringBuilder();
                foreach (string ien in iens)
                {
                    sb.Append(ien);
                    sb.Append(",");
                }
                sb.Remove(sb.Length - 1, 1);
                bedSwitchCommand.Ien = sb.ToString();
            }
            return ExecuteCommand<BedSwitch>(bedSwitchCommand, null, null);
        }

        public IList<string> GetCanceledOrders(IEnumerable<string> iens)
        {
            if (iens != null && iens.Count() > 0)
            {
                StringBuilder sb = new StringBuilder();
                foreach (string ien in iens)
                {
                    sb.Append(ien);
                    sb.Append(",");
                }
                sb.Remove(sb.Length - 1, 1);
                cancelledOrdersCommand.Ien = sb.ToString();
            }
            return ExecuteCommand<string>(cancelledOrdersCommand, null, null);
        }

        public IList<PatientMovementIen> GetPatientMovementIens(DateTime startDate, DateTime endDate)
        {
            patientMovementIensCommand.StartDate = startDate;
            patientMovementIensCommand.EndDate = endDate;
            return ExecuteCommand<PatientMovementIen>(patientMovementIensCommand);
        }

        public PatientMovement GetPatientMovementByIen(string patientMovementIen)
        {
            patientMovementByIenCommand.Ien = patientMovementIen;
            return ExecuteCommand<PatientMovement>(patientMovementByIenCommand).FirstOrDefault();
        }

        public IList<EDISPatientAdmission> GetEDISPatientAdmission(DateTime? startDate = null, DateTime? endDate = null)
        {
            // not cached
            if (!startDate.HasValue)
                startDate = minStartDate;
            if (!endDate.HasValue)
                endDate = maxEndDate;

            return new List<EDISPatientAdmission>();
        }

        #endregion        
    }
}
