﻿using InfoWorld.HL7.ITS;
using BMS.ServiceContracts;
using System.ServiceModel;
using BMS.MT;
using System;
using ePractice.MiddleTier;
using BMS.Translators;
using BMS.Utils;
using System.Transactions;
using System.Collections.Generic;
using BMS.DataContracts;
using BMS.FaultContracts;
using BMS.ServiceImplementation.Properties;
using System.Linq;
using BMS.ServiceImplementation.Utils;
using System.Threading;
using System.Collections.Concurrent;

namespace BMS.ServiceImplementation
{
    [ErrorHandlerAttribute]
    [ServiceBehavior(IgnoreExtensionDataObject = true, InstanceContextMode = InstanceContextMode.PerSession, ConcurrencyMode = ConcurrencyMode.Multiple, IncludeExceptionDetailInFaults = true, UseSynchronizationContext = false)]
    public class ConfigurationOperationsCore : IConfigurationOperations
    {
        private static readonly ReaderWriterLockSlim cacheLock = new ReaderWriterLockSlim();
        static ConcurrentDictionary<string, Dictionary<string, DataContracts.Configuration>> _applicationConfigurationCache;
        static ConfigurationOperationsCore()
        {
            _applicationConfigurationCache = new ConcurrentDictionary<string, Dictionary<string, DataContracts.Configuration>>(24, 1500, StringComparer.InvariantCultureIgnoreCase);
        }

        static void FlushCache()
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_applicationConfigurationCache != null)
                    _applicationConfigurationCache.Clear();
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        static void RemoveCache(string parentSection, string childSection)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_applicationConfigurationCache.ContainsKey(parentSection))
                {
                    if (!string.IsNullOrEmpty(childSection))
                    {
                        if (_applicationConfigurationCache[parentSection].ContainsKey(childSection))
                            _applicationConfigurationCache[parentSection].Remove(childSection);
                    }
                    else
                    {
                        Dictionary<string, DataContracts.Configuration> removed;
                        _applicationConfigurationCache.TryRemove(parentSection, out removed);
                    }
                }
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        static void Cache(string parentSection, string childSection, BMS.DataContracts.Configuration configuration)
        {
            cacheLock.EnterWriteLock();
            try
            {
                if (_applicationConfigurationCache.ContainsKey(parentSection))
                {
                    if (_applicationConfigurationCache[parentSection].ContainsKey(childSection))
                        _applicationConfigurationCache[parentSection][childSection] = configuration;
                    else
                        _applicationConfigurationCache[parentSection].Add(childSection, configuration);
                }
                else
                {
                    Dictionary<string, BMS.DataContracts.Configuration> childElement = new Dictionary<string, BMS.DataContracts.Configuration>(StringComparer.InvariantCultureIgnoreCase);
                    childElement.Add(childSection, configuration);
                    _applicationConfigurationCache.TryAdd(parentSection, childElement);
                }
            }
            finally { cacheLock.ExitWriteLock(); }
        }

        static DataContracts.Configuration GetCachedConfiguration(string parentSection, string childSection)
        {
            DataContracts.Configuration result = null;
            if (!string.IsNullOrEmpty(parentSection))
            {
                if (_applicationConfigurationCache.ContainsKey(parentSection))
                {
                    if (cacheLock.TryEnterReadLock(0))
                    {
                        try
                        {
                            if (!string.IsNullOrEmpty(childSection))
                            {
                                if (_applicationConfigurationCache[parentSection].ContainsKey(childSection))
                                    return _applicationConfigurationCache[parentSection][childSection];
                                else
                                    return result;
                            }
                            else
                            {
                                foreach (KeyValuePair<string, DataContracts.Configuration> config in _applicationConfigurationCache[parentSection])
                                {
                                    if (result == null)
                                    {
                                        result = new DataContracts.Configuration();
                                        result.ConfigType = config.Value.ConfigType;
                                        result.Domain = config.Value.Domain;
                                        result.UserName = config.Value.UserName;
                                        result.SectionList = new Dictionary<string, List<Element>>();
                                    }
                                    foreach (KeyValuePair<string, List<Element>> element in config.Value.SectionList)
                                        result.SectionList.Add(element.Key, element.Value);
                                }

                                return result;
                            }
                        }
                        finally { cacheLock.ExitReadLock(); }
                    }
                }
                return result;
            }
            return result;
        }

        #region Get Methods

        public DataContracts.Configuration GetUserProfile(string userName, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Read, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                DataContracts.Configuration userConfiguration = new DataContracts.Configuration();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.USR, userName, domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                {
                    masterConfigBO = GetConfigurationBOs(ConfigType.DEFUSR, String.Empty, domainId, String.Empty);
                    if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                        throw new FaultException<DefaultConfigException>(
                            new DefaultConfigException() { UserName = userName, Message = Resources.ERR_DEFAULT_USER_PROFILE_NOT_FOUND });
                    else
                        masterConfigBO.CopyTo(userConfiguration);
                }
                else
                    masterConfigBO.CopyTo(userConfiguration);

                return userConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public DataContracts.Configuration GetDefaultProfile(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Read, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                DataContracts.Configuration userConfiguration = new DataContracts.Configuration();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.DEFUSR, String.Empty, domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                    throw new FaultException<DefaultConfigException>(
                        new DefaultConfigException() { UserName = String.Empty, Message = Resources.ERR_DEFAULT_USER_PROFILE_NOT_FOUND });
                else
                    masterConfigBO.CopyTo(userConfiguration);

                return userConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<DataContracts.Configuration> GetUserProfiles(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Read, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                List<DataContracts.Configuration> result = new List<DataContracts.Configuration>();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.USR, Constants.ALL.ToUpper(), domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                {
                    masterConfigBO = GetConfigurationBOs(ConfigType.DEFUSR, String.Empty, domainId, String.Empty);
                    if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                        throw new FaultException<DefaultConfigException>(
                            new DefaultConfigException() { UserName = String.Empty, Message = Resources.ERR_DEFAULT_USER_PROFILE_NOT_FOUND });
                    else
                    {
                        DataContracts.Configuration userConfiguration = new DataContracts.Configuration();
                        masterConfigBO.CopyTo(userConfiguration);
                        result.Add(userConfiguration);
                    }
                }
                else
                    masterConfigBO.CopyTo(result);

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public DataContracts.Configuration GetApplicationConfiguration(string domainId, string parentSection, string childSection)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Read, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                DataContracts.Configuration appConfiguration = GetCachedConfiguration(parentSection, childSection);

                if (appConfiguration == null)
                    appConfiguration = new DataContracts.Configuration();
                else
                    return appConfiguration;

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.APP, String.Empty, domainId, string.IsNullOrEmpty(childSection) ? parentSection : childSection);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                {
                    masterConfigBO = GetConfigurationBOs(ConfigType.DEFAPP, String.Empty, domainId, string.IsNullOrEmpty(childSection) ? parentSection : childSection);
                    if (masterConfigBO != null && masterConfigBO.ConfigList.Count > 0)
                        masterConfigBO.CopyTo(appConfiguration);
                }
                else
                    masterConfigBO.CopyTo(appConfiguration);


                Cache(parentSection, appConfiguration);

                return appConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public DataContracts.Configuration GetDefaultApplicationConfiguration(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Read, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                DataContracts.Configuration appConfiguration = new DataContracts.Configuration();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.DEFAPP, String.Empty, domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                    throw new FaultException<DefaultConfigException>(
                        new DefaultConfigException() { UserName = String.Empty, Message = Resources.ERR_DEFAULT_APP_CONFIG_NOT_FOUND });
                else
                    masterConfigBO.CopyTo(appConfiguration);

                return appConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Dictionary<string, DataContracts.Configuration> GetApplicationConfigurationForHomePage(string domainId, List<string> configKeys)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Dictionary<string, DataContracts.Configuration> result = new Dictionary<string, DataContracts.Configuration>();
                foreach (string key in configKeys)
                {
                    if (key.Contains(Constants.IS_EVACUATION))
                        result.Add(key, GetApplicationConfiguration(domainId, Constants.IS_EVACUATION, key));
                    else
                        result.Add(key, GetApplicationConfiguration(domainId, key, string.Empty));
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Set Methods

        public void SetUserProfile(DataContracts.Configuration config)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (config == null)
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "config", MethodType = MethodType.Create, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                SaveConfiguration(config, string.Empty, string.Empty);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SetApplicationConfiguration(DataContracts.Configuration config, string parentSection, string childSection)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (config == null)
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "config", MethodType = MethodType.Create, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                SaveConfiguration(config, parentSection, childSection);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Reset Methods

        public DataContracts.Configuration ResetUserProfile(string userName, string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(userName))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "userName", MethodType = MethodType.Delete, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Delete, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                FlushCache();
                DataContracts.Configuration userConfiguration = new DataContracts.Configuration();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.DEFUSR, String.Empty, domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                    throw new FaultException<DefaultConfigException>(
                        new DefaultConfigException() { UserName = String.Empty, Message = Resources.ERR_DEFAULT_USER_PROFILE_NOT_FOUND });

                MasterConfigBO masterConfigBO_user = GetConfigurationBOs(ConfigType.USR, userName, domainId, String.Empty);
                if (masterConfigBO_user == null || masterConfigBO_user.ConfigList.Count == 0)
                    throw new FaultException<DefaultConfigException>(
                        new DefaultConfigException() { UserName = userName, Message = Resources.ERR_USER_PROFILE_NOT_FOUND });

                masterConfigBO_user.Delete();

                masterConfigBO.CopyTo(userConfiguration);
                userConfiguration.UserName = userName;
                userConfiguration.ConfigType = ConfigType.USR;
                SaveConfiguration(userConfiguration, string.Empty, string.Empty);

                return userConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public DataContracts.Configuration ResetApplicationConfiguration(string domainId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(domainId))
                    throw new FaultException<NullParameterException>(new NullParameterException() { ParameterName = "domainId", MethodType = MethodType.Delete, ErrorMessage = Resources.ERR_CONFIG_NULL_PARAMETER });

                FlushCache();
                DataContracts.Configuration appConfiguration = new DataContracts.Configuration();

                MasterConfigBO masterConfigBO = GetConfigurationBOs(ConfigType.DEFAPP, String.Empty, domainId, String.Empty);
                if (masterConfigBO == null || masterConfigBO.ConfigList.Count == 0)
                    throw new FaultException<DefaultConfigException>(
                        new DefaultConfigException() { UserName = String.Empty, Message = Resources.ERR_DEFAULT_APP_CONFIG_NOT_FOUND });

                masterConfigBO.CopyTo(appConfiguration);
                appConfiguration.ConfigType = ConfigType.APP;
                SaveConfiguration(appConfiguration, string.Empty, string.Empty);

                return appConfiguration;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Internal

        internal void Cache(string parentSection, DataContracts.Configuration configuration)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                DataContracts.Configuration appConfig = null;
                if (configuration.SectionList != null)
                    foreach (KeyValuePair<string, List<DataContracts.Element>> config in configuration.SectionList)
                    {
                        appConfig = new DataContracts.Configuration();
                        appConfig.ConfigType = configuration.ConfigType;
                        appConfig.Domain = configuration.Domain;
                        appConfig.UserName = configuration.UserName;
                        appConfig.SectionList = new Dictionary<string, List<Element>>();
                        appConfig.SectionList.Add(config.Key, config.Value);
                        Cache(parentSection, config.Key, appConfig);
                    }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal void SaveConfiguration(DataContracts.Configuration configuration, string parentSection, string childSection)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(parentSection) || !string.IsNullOrEmpty(childSection))
                    RemoveCache(parentSection, childSection);
                MasterConfigBO masterConfigBO = GetConfigurationBOs(configuration.ConfigType, configuration.UserName, configuration.Domain, string.IsNullOrEmpty(childSection) ? parentSection : childSection);
                ConfigBO configBO = null;
                DataContracts.Configuration newConfig = null;
                List<DataContracts.Element> elementList = null;

                if (masterConfigBO != null && masterConfigBO.ConfigList.Count > 0)
                {
                    for (int i = masterConfigBO.ConfigList.Count - 1; i >= 0; i--)
                    {
                        ConfigBO configElement = masterConfigBO.ConfigList[i];
                        if (configuration.SectionList.Keys.Contains(configElement.SECTION.Value))
                        {
                            Element el = configuration.SectionList[configElement.SECTION.Value].Where(a => a.Key == configElement.CONFIG_KEY.Value).FirstOrDefault();
                            if (el == null)
                                masterConfigBO.ConfigList.Remove(configElement);
                        }
                        else
                            masterConfigBO.ConfigList.Remove(configElement);
                    }
                }
                else
                    masterConfigBO = (MasterConfigBO)MTServices.Instance[Resources.MT_MasterConfig].CreateNew();

                foreach (KeyValuePair<string, List<DataContracts.Element>> kvp in configuration.SectionList)
                {
                    if (kvp.Value != null)
                        foreach (DataContracts.Element element in kvp.Value)
                        {
                            newConfig = new DataContracts.Configuration();
                            newConfig.ConfigType = configuration.ConfigType;
                            newConfig.Domain = configuration.Domain;
                            newConfig.UserName = string.IsNullOrEmpty(configuration.UserName) ? string.Empty : configuration.UserName;
                            elementList = new List<Element>();
                            elementList.Add(element);
                            newConfig.SectionList = new Dictionary<string, List<Element>>();
                            newConfig.SectionList.Add(kvp.Key, elementList);

                            if (masterConfigBO != null && masterConfigBO.ConfigList.Count > 0)
                            {
                                ConfigBO oldBO = masterConfigBO.ConfigList.FirstOrDefault(a => a.SECTION.Value == kvp.Key && a.CONFIG_KEY.Value == element.Key);
                                if (oldBO != null)
                                {
                                    newConfig.CopyTo(oldBO);
                                }
                                else
                                {
                                    configBO = masterConfigBO.ConfigList.AddNew();
                                    newConfig.CopyTo(configBO);
                                }
                            }
                            else
                            {
                                configBO = masterConfigBO.ConfigList.AddNew();
                                newConfig.CopyTo(configBO);
                            }
                        }
                }
                masterConfigBO.Save();
                if (!string.IsNullOrEmpty(parentSection) || !string.IsNullOrEmpty(childSection))
                    Cache(parentSection, configuration);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal MasterConfigBO GetConfigurationBOs(ConfigType configType, string userName, string domainId, string section)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IEntityManager entMan = MTServices.Instance[Resources.MT_MasterConfig];
                ConfigurationParPO par = (ConfigurationParPO)entMan.Parameters;
                par.USER_NAME.Value = userName;
                par.CONFIG_TYPE.Value = configType.ToString();
                par.DOMAIN_ID.Value = domainId;
                if (!string.IsNullOrEmpty(section))
                    par.SECTION.Value = section;

                MasterConfigBO masterConfigBO = (MasterConfigBO)entMan.GetEntity(par);

                return masterConfigBO;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
