﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Facade.Service;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.BMService;

namespace BMS.Facade.Implementation
{
    public class ReportsOperationsImplementation : IReportsOperations
    {
        //static object lockObject = new Object();  // dead code per fortify
        static readonly string OTHER_REPORTS_PATH = "/Others/";

        #region IReportsQuery Members

        /// <summary>
        /// Fetches the list of available reports from the RS service securely
        /// </summary>
        /// <returns>the list of available reports </returns>
        public IList<IReportInfo> GetReports(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetAllReports(fullUserName).Where(r => r.Path.IndexOf(OTHER_REPORTS_PATH, StringComparison.InvariantCultureIgnoreCase) < 0).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fetches the list of available reports in folder Others from the RS service securely
        /// </summary>
        /// <returns>the list of available reports </returns>
        public IList<IReportInfo> GetOtherReports(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return GetAllReports(fullUserName).Where(r => r.Path.IndexOf(OTHER_REPORTS_PATH, StringComparison.InvariantCultureIgnoreCase) >= 0).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IReportInfo GetPatientInquiryReport(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<IReportInfo> reports = GetAllReports(fullUserName);
                IReportInfo patientInquiryReport = reports.Where(r => r.Name == Constants.PatientInquiry).FirstOrDefault();
                return patientInquiryReport;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        /// <summary>
        /// Retrieves the list of reports from the RS service.
        /// </summary>
        /// <returns>A list of all IReportInfo objects.</returns>
        private List<IReportInfo> FetchReports(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string reportsPath = System.Configuration.ConfigurationManager.AppSettings["ReportsPath"];
                if (string.IsNullOrEmpty(reportsPath))
                    reportsPath = "/";

                var tree = BMSFactory.ReportingServiceClient.GetReports(reportsPath);
                var serverUrl = BMSFactory.ReportingServiceClient.GetReportingServerURL();
                
                List<IReportInfo> reports = new List<IReportInfo>();
                AddChildReports(reports, tree.Root, fullUserName);
                
                ReportUtil.SetDefaultProperties(serverUrl.ToString(), null, null, null);

                return reports;                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the list of reports from cache, or retrieves it and caches it.
        /// </summary>
        /// <returns>A list of all IReportInfo objects.</returns>
        private List<IReportInfo> GetAllReports(string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //lock (lockObject)
                {
                    List<IReportInfo> cachedReports = GetFromCache<List<IReportInfo>>("ReportInfoList");
                    if (cachedReports != null)
                    {
                        //clear unwanted cached parameters
                        foreach (IReportInfo ri in cachedReports)
                            if (ri.Parameters != null) { ri.Parameters.Clear(); ri.Parameters = null; }

                        return cachedReports;
                    }
                    else
                    {
                        var reports = FetchReports(fullUserName);
                        SetToCache<List<IReportInfo>>("ReportInfoList", reports);
                        return reports;
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves a value in the user Session for later use
        /// </summary>
        /// <typeparam name="T1">Type of object to save</typeparam>
        /// <param name="key">the key to find with</param>
        /// <param name="value">the value to save</param>
        private void SetToCache<T1>(string key, T1 value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                System.Web.HttpContext.Current.Session.Add(key, value);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets a value from cache
        /// </summary>
        /// <typeparam name="T1">type of object to get</typeparam>
        /// <param name="key">the key of the object</param>
        /// <returns>the object from session</returns>
        private T1 GetFromCache<T1>(string key)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return (T1)System.Web.HttpContext.Current.Session[key];
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Adds a hierarchical structure of report nodes to a collection
        /// </summary>
        /// <param name="reports">the collection of reports</param>
        /// <param name="node">the tree of nodes</param>
        private void AddChildReports(List<IReportInfo> reports, InfoWorld.RS.DataContracts.ReportsTree.Node node, string fullUserName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (node.Lnk == null)
                    reports.Add(new ReportInfo(fullUserName) { Name = node.Name, Path = node.Path });
                else
                    foreach (var v in node.Lnk)
                        AddChildReports(reports, v, fullUserName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
