using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.Serialization;
using BMS.Utils;
using BMS.Facade;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;

namespace BMS.Facade.Data
{
    /// <summary>
    /// Class used as model for the whiteboard view.
    /// </summary>
    [DataContract]
    public class WhiteboardBedInfo
    {
        #region Data Members

        [DataMember]
        public string BedIdRoot { get; set; }

        [DataMember]
        public string BedIdExtension { get; set; }

        [DataMember]
        public string BedName { get; set; }

        [DataMember]
        public string PatientIdRoot { get; set; }

        [DataMember]
        public string PatientIdExtension { get; set; }

        [DataMember]
        public string PatientLastName { get; set; }

        public string Last4SSN { get; set; }
        public string First4LastName { get; set; }

        [DataMember]
        public string StaffName { get; set; }

        [DataMember]
        public string Attending { get; set; }

        [DataMember]
        public string StayLength { get; set; }

        [DataMember]
        public string ObservationCountdown { get; set; }

        [DataMember]
        public string GenderColor { get; set; }

        [DataMember]
        public string DisplayPatientGender { get; set; }

        [DataMember]
        public string PT { get; set; }

        [DataMember]
        public string WardName { get; set; }

        [DataMember]
        public IconInfo DischargeIcon { get; set; }

        [DataMember]
        public string DischargeIconBytes { get; set; }

        [DataMember]
        public IconInfo BedAvailabilityStatusIcon { get; set; }

        [DataMember]
        public string BedAvailabilityStatusIconBytes { get; set; }

        [DataMember]
        public IconInfo BedCleaningStatusIcon { get; set; }

        [DataMember]
        public string BedCleaningStatusIconBytes { get; set; }

        [DataMember]
        public IList<IconInfo> IconFlagsList { get; set; }

        [DataMember]
        public byte[] CombineIconBinaryList { get; set; }

        [DataMember]
        public bool HasMultipleComments { get; set; }

        [DataMember]
        public bool HasMultipleDivisionsAssigned { get; set; }

        [DataMember]
        public string SimilarNameIconBytes { get; set; }

        [DataMember]
        public bool IsSimilarName { get; set; }

        [DataMember]
        public IconInfo SimilarNameIcon { get; set; }

        [DataMember]
        public bool IsAttendingVisible { get; set; }

        [DataMember]
        public bool IsStaffVisible { get; set; }

        [DataMember]
        public string Reason { get; set; }

        [DataMember]
        public string WaitingListInfo { get; set; }

        [DataMember]
        public string FlowId { get; set; }

        [DataMember]
        public bool HasEvacuationIcon { get; set; }

        [DataMember]
        public IconInfo LevelOfCareIcon { get; set; }

        [DataMember]
        public string LevelOfCareIconBytes { get; set; }

        [DataMember]
        public string LevelOfCareDays { get; set; }

        [DataMember]
        public string LevelOfCareChangeDays { get; set; }

        [DataMember]
        public string EvacuationDispostionStatus { get; set; }

        #endregion

        #region Methods

        public static List<WhiteboardBedInfo> GetWhiteboardData(List<IconInfo> iconFlagsList, string genderColor, string displayPT, string displayStaffAttending, IList<BedInfo> bedList, bool isFromReport, List<Ward> wards, Dictionary<string, SpecialtiesHidden> specialtiesHidden, bool isKiosk, List<EMSNotification> emsList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<WhiteboardBedInfo> bedViewModels = new List<WhiteboardBedInfo>();
                if (bedList != null)
                {
                    List<string> similarities = Similarities(bedList);
                    foreach (BedInfo bed in bedList)
                    {
                        bool isSimilar = IsPatientLastNameOrPatientSsnSimilar(similarities, bed);
                        bedViewModels.Add(GetWhiteboardData(iconFlagsList, genderColor, displayPT, displayStaffAttending, bed, isSimilar, isFromReport, wards, specialtiesHidden, isKiosk, emsList));
                    }
                }
                return bedViewModels;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        private static WhiteboardBedInfo GetWhiteboardData(List<IconInfo> iconFlagsList, string genderColor, string displayPT, string displayStaffAttending, BedInfo model, bool isSimilar, bool isFromReport, List<Ward> wards, Dictionary<string, SpecialtiesHidden> specialtiesHidden, bool isKiosk, List<EMSNotification> emsList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model == null)
                    return null;
                WhiteboardBedInfo res = new WhiteboardBedInfo();

                res.BedIdRoot = model.BedId.root;
                res.BedIdExtension = model.BedId.extension;
                res.PatientIdRoot = model.PatientId.root;
                res.PatientIdExtension = model.PatientId.extension;
                res.BedName = model.BedName;
                res.PatientLastName = model.PatientLastName;
                res.DisplayPatientGender = model.DisplayPatientGender;
                res.StaffName = model.StaffName;
                res.WaitingListInfo = model.WaitingListInfo;
                res.HasMultipleComments = model.HasMultipleComments;
                res.HasMultipleDivisionsAssigned = model.HasMultipleDivisionsAssigned;

                if (!string.IsNullOrEmpty(model.PatientGender))
                {
                    res.FlowId = model.FlowId.Value.ToString();
                }
                else
                    res.FlowId = string.Empty;

                // set similar name icon
                if (isSimilar && iconFlagsList != null)
                    res.SimilarNameIcon = iconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == Constants.SIMILAR_SAME_NAME).First();
                if (res.SimilarNameIcon == null)
                    res.SimilarNameIcon = new IconInfo();
                else
                {
                    res.SimilarNameIconBytes = res.SimilarNameIcon.ImageBytes;
                    res.IsSimilarName = isSimilar == true ? false : true;
                }

                // set bed availability status icon
                if (iconFlagsList != null)
                    res.BedAvailabilityStatusIcon = iconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.BedAvailabilityStatusCode).FirstOrDefault();
                if (res.BedAvailabilityStatusIcon == null)
                    res.BedAvailabilityStatusIcon = new IconInfo();
                else
                    res.BedAvailabilityStatusIconBytes = res.BedAvailabilityStatusIcon.ImageBytes;

                // set discharge status icon - interward transfer icon is a discharge icon
                res.DischargeIcon = iconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.DischargeStatusCode).FirstOrDefault();
                if (res.DischargeIcon == null)
                    res.DischargeIcon = new IconInfo();
                else
                    res.DischargeIconBytes = res.DischargeIcon.ImageBytes;

                // set bed cleaning status icon
                res.BedCleaningStatusIcon = iconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.BedCleaningStatusCode).FirstOrDefault();
                if (res.BedCleaningStatusIcon == null)
                    res.BedCleaningStatusIcon = new IconInfo();
                else
                {
                    // If IsBedCleanManual, then assign the bed clean image and proceed
                    if (model.IsBedCleanManual == true)
                    {
                        res.BedCleaningStatusIconBytes = res.BedCleaningStatusIcon.ImageBytes;
                    }
                    // If IsBedCleanManual is false (i.e, automated bed clean request), then 
                    // verify if there is a EMS Notification Config for the specific ward group.
                    else if (model.IsBedCleanManual == false && emsList != null)
                    {
                        foreach (EMSNotification ems in emsList)
                        {
                            // Get ems notification config location and match with model.DivisionName 
                            // Alternatively :
                            //   Get VistA ward name based on BMS ward group and match with model.WardName
                            if (ems.Location.Equals(model.DivisionName))  
                            {
                                res.BedCleaningStatusIconBytes = res.BedCleaningStatusIcon.ImageBytes;
                                break;
                            }
                        }
                        if (res.BedCleaningStatusIconBytes == null)
                        {
                            res.BedCleaningStatusIcon = new IconInfo();
                        }
                    }
                    else
                    {
                        res.BedCleaningStatusIcon = new IconInfo();
                    }
                }
                    
                // set icons for flags
                res.IconFlagsList = new List<IconInfo>();
                List<int> iconList = new List<int>();
                iconList.AddRange(model.IconFlagIdsOfBedList);
                iconList.AddRange(model.IconFlagIdsOfPatientList);
                iconList.AddRange(model.IconFlagFacilityIdsOfPatientList);
                iconList.AddRange(model.IconFlagFacilityIdsOfBedList);
                res.IconFlagsList = (from icon in iconFlagsList
                                     where (from iconAssociationId in iconList
                                            select iconAssociationId).Contains(icon.IconId)
                                     select icon).OrderBy(o => o.Subgroup == Constants.STANDARD_ICON ? 1 : (o.Subgroup == Constants.EMERGENCY_ICON ? 2 : 3)).ThenBy(o => o.OrderIndex).ToList();
                if (res.IconFlagsList != null && res.IconFlagsList.Where(a => a.IconName.ToLower().Contains(Constants.EVACUATION)).FirstOrDefault() != null)
                    res.HasEvacuationIcon = true;
                else
                    res.HasEvacuationIcon = false;

                //combine all iconFlags for report
                if (isFromReport && res.IconFlagsList != null && res.IconFlagsList.Count > 0)
                {
                    List<string> imagesParam = new List<string>();
                    foreach (IconInfo icon in res.IconFlagsList)
                        imagesParam.Add(icon.ImageBytes);
                    res.CombineIconBinaryList = ImageConverter.MergeMultipleImages(imagesParam);
                }

                string classColor = string.Empty;
                model.Reason = model.Reason ?? string.Empty;
                model.Comment = model.Comment ?? string.Empty;
                if (model.Reason.Contains("TEMPORARILY UNAVAILABLE") || (!string.IsNullOrEmpty(model.BedAvailabilityStatusCode) && (model.BedAvailabilityStatusCode == "OOS" || model.BedAvailabilityStatusCode == "DND" || model.BedAvailabilityStatusCode == "BED_OOS_VISTA" || model.BedAvailabilityStatusCode == "BED_OOS_BEDBOARD")))
                    classColor = "class=\"colorRed SmallText\"";
                else if (model.Reason.Contains("BED ASSIGNED") || model.Comment.Contains("BED ASSIGNED") || !string.IsNullOrEmpty(model.WaitingListInfo))
                    classColor = "class=\"colorOrange SmallText\"";
                else
                    classColor = "class=\"colorYellowOrange SmallText\"";

                //HHG: Fix Whiteboad issue when GenderColor is set to none all rows are set to grey.
                if (genderColor == Constants.GenderColorNone)
                {
                    if ((model.PatientGender == "Male") || (model.PatientGender == "Female"))
                    {
                        classColor = "class=\"colorLightGray SmallText\"";

                    }

                }
                else {
                    if (genderColor == Constants.GenderColorBlueAndPink)
                    {
                        if (!string.IsNullOrEmpty(model.PatientGender))
                        {
                            if (model.PatientGender == "Male")
                                classColor = "class=\"colorBlue SmallText\"";
                            else if (model.PatientGender == "Female")
                                classColor = "class=\"colorPink SmallText\"";
                            else
                                classColor = "class=\"colorLightGray SmallText\"";
                        }
                    }                               
                }

                res.GenderColor = classColor;

                string reasonOfBed = string.Empty;
                //set reason
                if (!string.IsNullOrEmpty(model.Comment) && !string.IsNullOrEmpty(model.Reason))
                    reasonOfBed = model.Reason + " - " + model.Comment;
                else if (string.IsNullOrEmpty(model.Comment) && !string.IsNullOrEmpty(model.Reason))
                    reasonOfBed = model.Reason;
                else if (!string.IsNullOrEmpty(model.Comment) && string.IsNullOrEmpty(model.Reason))
                    reasonOfBed = model.Comment;
                else
                    reasonOfBed = string.Empty;

                if (!string.IsNullOrEmpty(model.WaitingListInfo))
                    res.Reason = model.WaitingListInfo + " " + reasonOfBed;
                else
                    res.Reason = reasonOfBed;

                //Construct the PT based on the Display PT selected value.
                StringBuilder pt = new StringBuilder();

                if (!string.IsNullOrEmpty(model.PatientLastName) && !string.IsNullOrEmpty(model.SSN))
                {
                    if (res.IconFlagsList != null && res.IconFlagsList.Where(a => (a.PatientOrRoomBed == 'P' && (a.IconName.ToLower().Contains("optout") || a.IconName.ToLower().Contains("opt_out") || a.IconName.ToLower().Contains("opt-out") || a.IconName.ToLower().Contains("opt out")))).FirstOrDefault() != null)
                        pt.Append("X X X X");
                    else
                    {
                        if (displayPT == Constants.DISPLAY_PT_FIRST_AND_LAST4_CODE)
                        {
                            pt.Append(model.PatientLastName.Substring(0, 1));
                            pt.Append(model.SSN.Substring(model.SSN.Length - 4));
                        }
                        if (displayPT == Constants.DISPLAY_PT_LAST_NAME_CODE)
                            pt.Append(model.PatientLastName);
                    }
                    res.StayLength = model.LosWard.ElapsedTime();

                    bool shouldHideSpecialties = false;
                    if (specialtiesHidden.ContainsKey(model.WardId.extension))
                    {
                        SpecialtiesHidden hidden = specialtiesHidden[model.WardId.extension];

                        if (hidden == SpecialtiesHidden.Kiosk && isKiosk)
                            shouldHideSpecialties = true;
                        else if (hidden == SpecialtiesHidden.Desktop && !isKiosk)
                            shouldHideSpecialties = true;
                        else if (hidden == SpecialtiesHidden.All)
                            shouldHideSpecialties = true;
                    }

                    if (model.PatientTreatingSpecialty == null || shouldHideSpecialties)
                        res.WardName = model.WardName;
                    else
                        res.WardName = model.WardName + " - " + model.PatientTreatingSpecialty.displayName.ToUpper();
                }
                res.PT = pt.ToString();

                res.Attending = model.Provider;
                switch (displayStaffAttending)
                {
                    case Constants.STAFF:
                        res.IsAttendingVisible = false;
                        res.IsStaffVisible = true;
                        break;
                    case Constants.ATTENDING:
                        res.IsAttendingVisible = true;
                        res.IsStaffVisible = false;
                        break;
                    case Constants.STAFF_ATTENDING:
                        res.IsAttendingVisible = true;
                        res.IsStaffVisible = true;
                        break;
                }
                if (!string.IsNullOrEmpty(model.LevelOfCareCode))
                    res.LevelOfCareIcon = iconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.LevelOfCareCode).FirstOrDefault();
                if (res.LevelOfCareIcon == null)
                    res.LevelOfCareIcon = new IconInfo();
                else
                    res.LevelOfCareIconBytes = res.LevelOfCareIcon.ImageBytes;
                if (model.LevelOfCareDays > 0)
                    res.LevelOfCareDays = model.LevelOfCareDays.ToString();

                if (model.LevelOfCareChangeDays > 0)
                    res.LevelOfCareChangeDays = model.LevelOfCareChangeDays.ToString();

                if (!string.IsNullOrEmpty(model.EvacuationDispositionStatus) && model.IsEvacuationPatient)
                    res.EvacuationDispostionStatus = model.EvacuationDispositionStatus;
                else
                    res.EvacuationDispostionStatus = string.Empty;

                return res;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// List of similar patient last names and similar SSNs.
        /// </summary>
        /// <param name="bedList">The bed info list.</param>
        /// <returns>List of similarities.</returns>
        private static List<string> Similarities(IList<BedInfo> bedList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> similarPatientLastNames = bedList.GroupBy(bed =>
                {
                    if (bed.PatientLastName.Length > 3)
                    {
                        return bed.PatientLastName.ToLower().Substring(0, 4);
                    }
                    return string.Empty;
                }).Where(bed => !string.IsNullOrEmpty(bed.Key) && bed.Count() > 1)
                    .Select(bed => bed.Key)
                    .ToList();

                List<string> similarSSNs = bedList.GroupBy(bed =>
                {
                    if (bed.SSN.Length > 3)
                    {
                        return bed.SSN.ToLower().Substring(bed.SSN.Length - 4, 4);
                    }
                    return string.Empty;
                }).Where(bed => !string.IsNullOrEmpty(bed.Key) && bed.Count() > 1)
                    .Select(bed => bed.Key)
                    .ToList();

                similarPatientLastNames.AddRange(similarSSNs);
                return similarPatientLastNames;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Checks if first four characters of the last name of the patient are the same or the last four characters of patient SSN are the same.
        /// </summary>
        /// <param name="similarities">List of similarities.</param>
        /// <param name="bed">The bed.</param>
        /// <returns>
        /// True - if exists / Otherwise false.
        /// </returns>
        private static bool IsPatientLastNameOrPatientSsnSimilar(List<string> similarities, BedInfo bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool exists = false;
                if (similarities != null && similarities.Count > 0)
                {
                    foreach (string item in similarities)
                    {
                        if ((bed.PatientLastName.Length > 3 && item == bed.PatientLastName.ToLower().Substring(0, 4)) || (bed.SSN.Length > 3 && item == bed.SSN.ToLower().Substring(bed.SSN.Length - 4, 4)))
                        {
                            exists = true;
                            break;
                        }
                    }
                }
                return exists;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        #endregion //Private Methods

        #endregion
    }
}
