﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Mdws;
using BMS.Facade;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Via.Commands.WF;

namespace BMS.VistaIntegration.Via.Tests
{
    [TestClass]
    public class ViaVistASessionFactoryTest
    {
        private bool _compareToMDWS = false;
        private SiteParameters _siteParams = new SiteParameters();
        private VistASite _vistaSite = null;
        private string _site = "CHEYL19";

        [TestInitialize()]
        public void InitTests()
        {
            switch (_site)
            {
                case "HOUTEST":
                    _vistaSite = new VistASite("test", "V16HOU", "580", TimeZoneInfo.Local, "http://DNS.URL/QuerySvc.asmx");
                    BuildParamsForHouston();
                    break;
                case "CHEYL19":
                    _vistaSite = new VistASite("test", "CHEYL19", "516", TimeZoneInfo.Local, "http://DNS.URL/QuerySvc.asmx");
                    BuildParamsForCheyl19();
                    break;
            }
        }

        public void BuildParamsForCheyl19()
        {
            _siteParams.PatientIEN = "3977";
            _siteParams.PatientIENWithAdmission = "7185012"; //7185012^DRALEAU,LORENE^F^2670331^101625668^3041105^73^202^204908~jhkdksdf^0^40
            _siteParams.NewPersonIEN = "202";
        }

        public void BuildParamsForHouston()
        {
            _siteParams = new SiteParameters();
            _siteParams.PatientIEN = "52";
        }

        public IVistAQuery BuildVistAQuery(DataRetrievalMethod retrievalMethod)
        {
            IVistAQuery vistaQuery = null;
            switch (retrievalMethod)
            {
                case DataRetrievalMethod.MDWS:
                    MdwsVistASession session = new MdwsVistASession(_vistaSite);
                    session.Open(new VistAConnectionInfo(_vistaSite.MdwsEndpointUrl, null, null));
                    vistaQuery = new MdwsVistAQuery(session);
                    break;
                case DataRetrievalMethod.VIA:
                    vistaQuery = new ViaVistAQuery(new ViaVistASession(_vistaSite));
                    break;
            }

            return vistaQuery;
        }

        [TestMethod]
        public void TestListBedSwitch()
        {
            try
            {
                List<string> iens = new List<string>() { }; // TODO - add IENS

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<Data.WF.BedSwitch> bedSwitchVIA = viaQuery.GetBedsSwitch(iens);

                Assert.IsNotNull(bedSwitchVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Data.WF.BedSwitch> bedSwitchMDWS = mdwsQuery.GetBedsSwitch(iens);

                    Assert.IsNotNull(bedSwitchMDWS);

                    Assert.AreEqual(bedSwitchVIA.Count, bedSwitchMDWS.Count);
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientByDate()
        {
            try
            {
                DateTime startDate = new DateTime(2003, 1, 1); // TODO - set dates
                DateTime endDate = new DateTime(2003, 2, 1);

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(startDate, endDate);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(startDate, endDate);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientByIEN()
        {
            // spiders out on Attending Physician to "NewPerson" and FacilityTreatingSpecialty - "TreatingSpecialty"
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(null, null, _siteParams.PatientIEN);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(null, null, _siteParams.PatientIEN);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatient()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                Patient patientVIA = viaQuery.GetPatientByIen(_siteParams.PatientIEN);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    Patient patientMDWS = mdwsQuery.GetPatientByIen(_siteParams.PatientIEN);

                    Assert.IsNotNull(patientMDWS);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListAdmittedPatientsForUpdate()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetAdmittedPatientsForUpdate();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetAdmittedPatientsForUpdate();

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientBySSN()
        {
            try
            {
                IList<string> patientSSNs = new List<string>() { }; // TODO - add ssns

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Patient> patientsVIA = viaQuery.GetPatients(patientSSNs);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Patient> patientsMDWS = mdwsQuery.GetPatients(patientSSNs);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListAdmittedPatients()
        {
            try
            {
                string lastPatientMovementIEN = ""; // TODO - add movement IEN

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<AdmittedPatient> patientsVIA = viaQuery.GetAdmittedPatients(lastPatientMovementIEN);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(patientsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<AdmittedPatient> patientsMDWS = mdwsQuery.GetAdmittedPatients(lastPatientMovementIEN);

                    Assert.IsNotNull(patientsMDWS);

                    Assert.AreEqual(patientsVIA.Count, patientsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetWardLocations()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<WardLocation> wardLocationsVIA = viaQuery.GetWardLocations();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(wardLocationsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<WardLocation> wardLocationsMDWS = mdwsQuery.GetWardLocations();

                    Assert.IsNotNull(wardLocationsMDWS);

                    Assert.AreEqual(wardLocationsVIA.Count, wardLocationsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetHospitalLocations()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<HospitalLocation> hospitalLocationsVIA = viaQuery.GetHospitalLocations();

                Assert.IsNotNull(hospitalLocationsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<HospitalLocation> hospitalLocationsMDWS = mdwsQuery.GetHospitalLocations();

                    Assert.IsNotNull(hospitalLocationsMDWS);

                    Assert.AreEqual(hospitalLocationsVIA.Count, hospitalLocationsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListNewPersonByDateRange()
        {
            try
            {
                DateTime startDate = new DateTime(2003, 1, 1); // TODO - set dates
                DateTime endDate = new DateTime(2003, 2, 1);

                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<NewPerson> newPersonsVIA = viaQuery.GetNewPersons(startDate, endDate);
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(newPersonsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<NewPerson> newPersonsMDWS = mdwsQuery.GetNewPersons(startDate, endDate);

                    Assert.IsNotNull(newPersonsMDWS);

                    Assert.AreEqual(newPersonsVIA.Count, newPersonsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListNewPersonByIEN()
        {
            try
            {
                ViaVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA) as ViaVistAQuery;

                // make dynamic call to VIA command for new person
                List<NewPerson> newPersonsVIA = viaQuery.GetResults(new BMS.VistaIntegration.Via.Commands.EntitySetCache.BulkEntitiesListCommand<NewPerson>(viaQuery, BMS.VistaIntegration.Via.Commands.EIS.ListNewPersonCommand.Target, new string[] { _siteParams.NewPersonIEN }, BMS.VistaIntegration.Via.Commands.EIS.ListNewPersonCommand.ArgumentsCount, null));
                Assert.IsNotNull(newPersonsVIA);

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListSpecialty()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);

                DateTime beforeVIA = DateTime.Now;
                IList<Specialty> specialtiesVIA = viaQuery.GetSpecialties();
                DateTime afterVIA = DateTime.Now;
                TimeSpan durationVIA = afterVIA - beforeVIA;

                Assert.IsNotNull(specialtiesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<Specialty> specialtiesMDWS = mdwsQuery.GetSpecialties();

                    Assert.IsNotNull(specialtiesMDWS);

                    Assert.AreEqual(specialtiesVIA.Count, specialtiesMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListWardLocation()
        {
            // can't find MDWS or VIA implementation. There is a GetWardLocations, but that's already tested above.
            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListFacilityMovementTypes()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<FacilityMovementType> facilityMovementTypesVIA = viaQuery.GetFacilityMovementTypes();

                Assert.IsNotNull(facilityMovementTypesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<FacilityMovementType> facilityMovementTypesMDWS = mdwsQuery.GetFacilityMovementTypes();

                    Assert.IsNotNull(facilityMovementTypesMDWS);

                    Assert.AreEqual(facilityMovementTypesVIA.Count, facilityMovementTypesMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListFacilityTreatingSpecialties()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<FacilityTreatingSpecialty> facilityTreatingSpecialtiesVIA = viaQuery.GetFacilityTreatingSpecialties();

                Assert.IsNotNull(facilityTreatingSpecialtiesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<FacilityTreatingSpecialty> facilityTreatingSpecialtiesMDWS = mdwsQuery.GetFacilityTreatingSpecialties();

                    Assert.IsNotNull(facilityTreatingSpecialtiesMDWS);

                    Assert.AreEqual(facilityTreatingSpecialtiesVIA.Count, facilityTreatingSpecialtiesMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListMasMovementTransactionType()
        {
            // can't find MDWS or VIA implementation
            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListMedicalCenterDivision()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<MedicalCenterDivision> medicalCenterDivisionsVIA = viaQuery.GetMedicalCenterDivisions();

                Assert.IsNotNull(medicalCenterDivisionsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<MedicalCenterDivision> medicalCenterDivisionMDWS = mdwsQuery.GetMedicalCenterDivisions();

                    Assert.IsNotNull(medicalCenterDivisionMDWS);

                    Assert.AreEqual(medicalCenterDivisionsVIA.Count, medicalCenterDivisionMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListOrderableItems()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderableItem> orderableItemsVIA = viaQuery.GetOrderableItems();

                Assert.IsNotNull(orderableItemsVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderableItem> orderableItemsMDWS = mdwsQuery.GetOrderableItems();

                    Assert.IsNotNull(orderableItemsMDWS);

                    Assert.AreEqual(orderableItemsVIA.Count, orderableItemsMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListOrderStatuses()
        {
            try
            {
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderStatus> orderStatusesVIA = viaQuery.GetOrderStatuses();

                Assert.IsNotNull(orderStatusesVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderStatus> orderStatusesMDWS = mdwsQuery.GetOrderStatuses();

                    Assert.IsNotNull(orderStatusesMDWS);

                    Assert.AreEqual(orderStatusesVIA.Count, orderStatusesMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        private void TestGetPatientMovement(string patientIEN, DateTime movementDate, MovementTransactionType movementType)
        {
            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            PatientMovement patientMovementVIA = viaQuery.GetPatientMovement(patientIEN, movementDate, movementType);

            Assert.IsNotNull(patientMovementVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                PatientMovement patientMovementMDWS = mdwsQuery.GetPatientMovement(patientIEN, movementDate, movementType);

                Assert.IsNotNull(patientMovementMDWS);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestGetPatientMovementForAdmission()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.ADMISSION);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementForDischarge()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.DISCHARGE);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementForSpecialtyTransfer()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.SPECIALTY_TRANSFER);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementForTransfer()
        {
            try
            {
                DateTime movementDate = new DateTime(2010, 1, 1); // TODO get date

                TestGetPatientMovement(_siteParams.PatientIEN, movementDate, MovementTransactionType.TRANSFER);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestGetPatientMovementByIEN()
        {
            string patientMovementIen = ""; // TODO - get IEN

            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            PatientMovement patientMovementVIA = viaQuery.GetPatientMovementByIen(patientMovementIen);

            Assert.IsNotNull(patientMovementVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                PatientMovement patientMovementMDWS = mdwsQuery.GetPatientMovementByIen(patientMovementIen);

                Assert.IsNotNull(patientMovementMDWS);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestGetPatientMovementByDateRange()
        {
            DateTime startDate = new DateTime(2003, 1, 1); // TODO - set dates
            DateTime endDate = new DateTime(2003, 2, 1);

            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            IList<PatientMovement> patientMovementsVIA = viaQuery.GetPatientMovements(startDate, endDate);

            Assert.IsNotNull(patientMovementsVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                IList<PatientMovement> patientMovementsMDWS = mdwsQuery.GetPatientMovements(startDate, endDate);

                Assert.IsNotNull(patientMovementsMDWS);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestGetPatientMovementIENs()
        {
            DateTime startDate = new DateTime(2003, 1, 1); // TODO - set dates
            DateTime endDate = new DateTime(2003, 2, 1);

            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            IList<PatientMovementIen> patientMovementIensVIA = viaQuery.GetPatientMovementIens(startDate, endDate);

            Assert.IsNotNull(patientMovementIensVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                IList<PatientMovementIen> patientMovementIensMDWS = mdwsQuery.GetPatientMovementIens(startDate, endDate);

                Assert.IsNotNull(patientMovementIensMDWS);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestGetPatientMovementByAdmission()
        {
            string admissionIen = ""; // TODO - get ien

            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            IList<PatientMovement> patientMovementsVIA = viaQuery.GetPatientMovementsForAdmission(admissionIen);

            Assert.IsNotNull(patientMovementsVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                IList<PatientMovement> patientMovementsMDWS = mdwsQuery.GetPatientMovementsForAdmission(admissionIen);

                Assert.IsNotNull(patientMovementsMDWS);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        private void TestListScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIEN = null)
        {
            // "PATIEN","SDATE","EDATE","MAX","FROM"
            // <criteria>7200876^3100510^3130520^^</criteria>
            IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
            IList<ScheduledAdmission> callVIA = viaQuery.GetScheduledAdmissions(startDate, endDate, patientIEN);
            Assert.IsNotNull(callVIA);

            if (_compareToMDWS)
            {
                IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                IList<ScheduledAdmission> callMDWS = mdwsQuery.GetScheduledAdmissions(startDate, endDate, patientIEN);

                Assert.IsNotNull(callMDWS);

                Assert.AreEqual(callVIA.Count, callMDWS.Count);
            }

            // TODO - check fields have expected values and match between VIA/MDWS
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByStartDate()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 5, 10); // TODO - get date

                TestListScheduledAdmissions(startDate);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByDateRange()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 5, 10);
                DateTime endDate = new DateTime(2013, 5, 20);

                TestListScheduledAdmissions(startDate, endDate);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListScheduledAdmissionsByPatientIEN()
        {
            try
            {
                TestListScheduledAdmissions(null, null, _siteParams.PatientIEN);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        private void TestListPatientAppointments(DateTime? startDate, DateTime? endDate, string patientIEN, IEnumerable<string> clinics)
        {
            try
            {
                // "IENSB","MAX","SDATE","EDATE","CLNIEN","FROM"
                // <criteria>7208163^^3101206^3101206^1690^</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<PatientAppointment> callVIA = viaQuery.GetPatientAppointments(startDate, endDate, patientIEN, clinics);

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<PatientAppointment> callMDWS = mdwsQuery.GetPatientAppointments(startDate, endDate, patientIEN, clinics);

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientAppointmentByStartDate()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 12, 6);
                IEnumerable<string> clinics = new[] { "1690" };

                TestListPatientAppointments(startDate, null, null, clinics);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListPatientAppointmentByDateRange()
        {
            try
            {
                DateTime startDate = new DateTime(2010, 12, 6);
                DateTime endDate = new DateTime(2010, 12, 6);
                IEnumerable<string> clinics = new[] { "1690" }; // TODO - get clinics for HOU

                TestListPatientAppointments(startDate, endDate, null, clinics);
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListCancelOrders()
        {
            //  doesn't return all the data
            try
            {
                // "IENSM"
                // <criteria>5587620</criteria>
                IEnumerable<string> iens = new[] { "5587620" }; // TODO - get iens for HOU
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<string> callVIA = viaQuery.GetCanceledOrders(iens);

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<string> ordersMDWS = mdwsQuery.GetCanceledOrders(iens);

                    Assert.IsNotNull(ordersMDWS);

                    Assert.AreEqual(callVIA.Count, ordersMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListRoomBed()
        {
            try
            {
                // "IENSM","FROM","MAX"
                // <criteria>^^1000</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<RoomBed> callVIA = viaQuery.GetRoomBeds();

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<RoomBed> callMDWS = mdwsQuery.GetRoomBeds();

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }

        [TestMethod]
        public void TestListClinicAppointments()
        {
            // NOTE: This is not implemented specifically in any IVistAQuery. This is called by GetPatientAppointments, so this test is somewhat redundant.
            // There is no MDWS command for it.

            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListOrders()
        {
            // No VIA command for ListOrders. The main use case appears to be ListOrderActions.
            throw new NotImplementedException();
        }

        [TestMethod]
        public void TestListOrderActions()
        {
            //  times out
            try
            {
                // main use case has null patient IEN

                DateTime startDate = new DateTime(2002, 2, 20); // TODO - get paramters for HOU
                DateTime endDate = new DateTime(2002, 2, 21);
                IEnumerable<string> orderIENs = new[] { "73", "75", "76", "360", "740" };
                // "ORDIEN","IENSB","SDATE","EDATE","VALUE","MAX","FROM"
                // <criteria>73,75,76,360,740^1336650^3020220^3020220^1^^</criteria>
                IVistAQuery viaQuery = BuildVistAQuery(DataRetrievalMethod.VIA);
                IList<OrderAction> callVIA = viaQuery.GetOrderActions(startDate, endDate, "1336650", orderIENs);

                Assert.IsNotNull(callVIA);

                if (_compareToMDWS)
                {
                    IVistAQuery mdwsQuery = BuildVistAQuery(DataRetrievalMethod.MDWS);
                    IList<OrderAction> callMDWS = mdwsQuery.GetOrderActions(startDate, endDate, "1336650", orderIENs);

                    Assert.IsNotNull(callMDWS);

                    Assert.AreEqual(callVIA.Count, callMDWS.Count);
                }

                // TODO - check fields have expected values and match between VIA/MDWS
            }
            catch (Exception ex)
            {
                Assert.Fail(ex.ToString());
            }
        }
    }
}
