﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using BMS.Web.Models;
using InfoWorld.HL7.ITS;
using BMS.Web.App_GlobalResource;
using BMS.Facade.Data;
using BMS.Facade;
using BMS.Utils;
using Microsoft.Web.Mvc;
using BMS.Web.Models.Shared;
using BMS.Web.Views;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Admin Controller class.
    /// </summary>    
    [ValidateInput(false)]
    public class AdminController : BaseController
    {
        /// <summary>
        ///  Default action for the Admin controller.
        /// </summary>
        /// <returns>The index view.</returns>
        [ReadPermissionAuthFilter]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BaseViewModel model = new BaseViewModel();
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region User

        [ReadPermissionAuthFilter]
        public ActionResult AddEditUser(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdminUserAddEditViewModel model = new AdminUserAddEditViewModel();
                User user = GetUser(string.IsNullOrEmpty(p) ? null : QueryStrings["id"]);
                model.Data = user;                
                if (!model.IsDefaultUser)
                    FacadeManager.UserInterface.FillUserData(model.Data);
                model.RegionId = (user.Region != null) ? user.Region.Id.extension : null;
                model.VisnId = (user.Visn != null) ? user.Visn.Id.extension : null;
                model.FacilityId = (user.DefaultFacility != null) ? user.DefaultFacility.Id.extension : (user.Facility != null) ? user.Facility.Id.extension : null;
                model.DisplayOnlySelectedFacilities = true;
                FillAddEditUserModel(model);
                if (!model.IsDefaultUser)
                {
                    model.PermissionsFacilities = new List<PermissionFacility>();
                    GetUserPermissions(model);
                }
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult AddEditUser(AdminUserAddEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(model.ButtonCancel) || !string.IsNullOrEmpty(model.ButtonReturnToMenu))
                    return this.RedirectToAction<AdminController>(act => act.Index());
                if (!string.IsNullOrEmpty(model.ButtonLogout))
                    return this.RedirectToAction<AccountController>(act => act.LogOff());
                if (!string.IsNullOrEmpty(model.ButtonReturnToRegional))
                    return this.RedirectToAction<NationalAndRegionalController>(act => act.Index((string)null));

                model.LoggedUser = loggedUser;
                if (!string.IsNullOrEmpty(model.FillControls) || !string.IsNullOrEmpty(model.ButtonReadAccess) || !string.IsNullOrEmpty(model.ButtonWriteAccess))
                {
                    ModelState.Clear();
                    FillAddEditUserModel(model);
                    if (!model.IsDefaultUser)
                        ManageUserPermissions(model);
                    else
                        model.PermissionsFacilities = new List<PermissionFacility>();
                    return View(model);
                }

                if (model.PermissionsFacilities != null && model.PermissionsFacilities.Count > 0)
                {
                    int countHasNotReadPermissionButHasWritePermission = model.PermissionsFacilities.Where(a => a.WriteAccess && !a.ReadAccess).Count();
                    if (countHasNotReadPermissionButHasWritePermission > 0)
                    {
                        ModelState.AddModelError(string.Empty, "Read Access is required for facilities with Write Access.");
                    }
                }

                if (!ModelState.IsValid)
                {
                    FillAddEditUserModel(model);
                    return View(model);
                }

                model.Data.Facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, model.FacilityId));
                FacadeManager.ConfigurationInterface.SaveUserProfile(model.Data);
                if (!model.IsDefaultUser)
                {
                    if (model.PermissionsFacilities != null && model.PermissionsFacilities.Count > 0)
                        model.PermissionsFacilities = model.PermissionsFacilities.Where(a => a.ReadAccess || a.WriteAccess).ToList();                    
                    FacadeManager.UserInterface.SaveUserData(model.Data, model.PermissionsFacilities);
                }
                return View("UserEditHasSaved", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void GetUserPermissions(AdminUserAddEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> allfacilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                List<PermissionFacility> permissionFacilities = FacadeManager.UserInterface.GetFacilitiesForReadAndWrite(model.Data.UserName, allfacilities);
                List<VistaSite> vistaSites = FacadeManager.EntityInterface.GetVistaSites().ToList();
                foreach (PermissionFacility pf in permissionFacilities)
                {
                    pf.Facility.VistaSite = vistaSites.Where(a => a.Id.extension.Equals(pf.Facility.VistaSite.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.PermissionsFacilities.Add(pf);
                }
                List<Facility> remainingFacilities = (from item in allfacilities
                                                      where !(from f in model.PermissionsFacilities select f.Facility.Id.extension).Contains(item.Id.extension)
                                                      select item).ToList();
                foreach (Facility f in remainingFacilities)
                {
                    f.VistaSite = vistaSites.Where(a => a.Id.extension.Equals(f.VistaSite.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    model.PermissionsFacilities.Add(new PermissionFacility() { Facility = f, ReadAccess = false, WriteAccess = false });
                }
                model.PermissionsFacilities = model.PermissionsFacilities.OrderBy(a => ((a.Facility.Address1 != null) ? (!string.IsNullOrEmpty(a.Facility.Address1.city) ? a.Facility.Address1.city : string.Empty) : string.Empty)).ToList();

                if (model.Data.ReadAccess == null || model.Data.WriteAccess == null)
                {
                    if (model.Data.ReadAccess == null)
                        model.Data.ReadAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (model.Data.WriteAccess == null)
                        model.Data.WriteAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (model.PermissionsFacilities != null && model.PermissionsFacilities.Count > 0)
                    {
                        if (model.PermissionsFacilities.Where(a => a.ReadAccess == true).ToList().Count > 0)
                            model.Data.ReadAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (model.PermissionsFacilities.Where(a => a.WriteAccess == true).ToList().Count > 0)
                            model.Data.WriteAccess = model.StrictDecisions.Where(a => a.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    }
                }
                if (string.IsNullOrEmpty(model.FillControls) && string.IsNullOrEmpty(model.ButtonReadAccess) && string.IsNullOrEmpty(model.ButtonWriteAccess))
                {
                    if (model.DisplayOnlySelectedFacilities)
                        model.PermissionsFacilities = model.PermissionsFacilities.Where(a => a.ReadAccess || a.WriteAccess).ToList();
                }
                model.PermissionsFacilities = (from item in model.PermissionsFacilities
                                               select new PermissionFacility()
                                               {
                                                   Facility = new Facility() { Id = item.Facility.Id, VistaSite = item.Facility.VistaSite, Name = ((item.Facility.Address1 != null) ? (!string.IsNullOrEmpty(item.Facility.Address1.city) ? item.Facility.Address1.city : string.Empty) : string.Empty) + " (" + item.Facility.SiteNumber + ", " + item.Facility.Code + ")" },
                                                   ReadAccess = item.ReadAccess,
                                                   WriteAccess = item.WriteAccess
                                               }).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void ManageUserPermissions(AdminUserAddEditViewModel model)
        {
            model.PermissionsFacilities = new List<PermissionFacility>();
            GetUserPermissions(model);
            bool readAccess = model.Data.ReadAccess.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? true : false;
            bool writeAccess = model.Data.WriteAccess.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? true : false;
            if (model.Data.SupportUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)
                || model.Data.NationalUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)
                || model.Data.GuestUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase))
            {
                for (int i = 0; i < model.PermissionsFacilities.Count; i++)
                    model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = readAccess, WriteAccess = writeAccess };
                return;
            }
            if (model.Data.RegionalUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase))
            {
                if (model.DisplayOnlySelectedFacilities)
                    model.PermissionsFacilities = model.PermissionsFacilities.Where(a => a.Facility.VistaSite.Visn.Region.Id.extension.Equals(model.RegionId, StringComparison.InvariantCultureIgnoreCase)).ToList();
                for (int i = 0; i < model.PermissionsFacilities.Count; i++)
                {
                    if (model.PermissionsFacilities[i].Facility.VistaSite.Visn.Region.Id.extension.Equals(model.RegionId, StringComparison.InvariantCultureIgnoreCase))
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = readAccess, WriteAccess = writeAccess };
                    else
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = false, WriteAccess = false };
                }
                model.PermissionsFacilities = model.PermissionsFacilities.OrderBy(a => ((a.Facility.Address1 != null) ? (!string.IsNullOrEmpty(a.Facility.Address1.city) ? a.Facility.Address1.city : string.Empty) : string.Empty)).ToList();
                return;
            }
            if (model.Data.VisnUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase))
            {
                if (model.DisplayOnlySelectedFacilities)
                    model.PermissionsFacilities = model.PermissionsFacilities.Where(a => a.Facility.VistaSite.Visn.Id.extension.Equals(model.VisnId, StringComparison.InvariantCultureIgnoreCase)).ToList();
                for (int i = 0; i < model.PermissionsFacilities.Count; i++)
                {
                    if (model.PermissionsFacilities[i].Facility.VistaSite.Visn.Id.extension.Equals(model.VisnId, StringComparison.InvariantCultureIgnoreCase))
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = readAccess, WriteAccess = writeAccess };
                    else
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = false, WriteAccess = false };
                }
                model.PermissionsFacilities = model.PermissionsFacilities.OrderBy(a => ((a.Facility.Address1 != null) ? (!string.IsNullOrEmpty(a.Facility.Address1.city) ? a.Facility.Address1.city : string.Empty) : string.Empty)).ToList();
                return;
            }
            if (model.Data.SiteUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) || model.Data.EMSUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase)
                || model.Data.EMSSupervisorUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) || model.Data.AdminUser.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase))
            {
                if (model.DisplayOnlySelectedFacilities)
                    model.PermissionsFacilities = model.PermissionsFacilities.Where(a => a.Facility.Id.extension.Equals(model.FacilityId, StringComparison.InvariantCultureIgnoreCase)).ToList();
                for (int i = 0; i < model.PermissionsFacilities.Count; i++)
                {
                    if (model.PermissionsFacilities[i].Facility.Id.extension.Equals(model.FacilityId, StringComparison.InvariantCultureIgnoreCase))
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = readAccess, WriteAccess = writeAccess };
                    else
                        model.PermissionsFacilities[i] = new PermissionFacility() { Facility = model.PermissionsFacilities[i].Facility, ReadAccess = false, WriteAccess = false };
                }
                model.PermissionsFacilities = model.PermissionsFacilities.OrderBy(a => ((a.Facility.Address1 != null) ? (!string.IsNullOrEmpty(a.Facility.Address1.city) ? a.Facility.Address1.city : string.Empty) : string.Empty)).ToList();
                return;
            }
            model.PermissionsFacilities = new List<PermissionFacility>();
        }

        private void FillAddEditUserModel(AdminUserAddEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.StrictDecisions = Facade.FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.StrictDecision);
                
                model.Regions = Facade.FacadeManager.EntityInterface.GetRegions();
                if (string.IsNullOrEmpty(model.RegionId) || model.RegionId.Equals(Guid.Empty.ToString(), StringComparison.InvariantCultureIgnoreCase))
                    model.RegionId = model.Regions.FirstOrDefault().Id.extension;
                
                model.Visns = Facade.FacadeManager.EntityInterface.GetVisns(new II(this.loggedUser.Domain, model.RegionId));
                if (string.IsNullOrEmpty(model.VisnId) || model.VisnId.Equals(Guid.Empty.ToString(), StringComparison.InvariantCultureIgnoreCase)
                    || model.Visns.Where(a => a.Id.extension.Equals(model.VisnId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                    model.VisnId = model.Visns.FirstOrDefault().Id.extension;
                
                model.Facilities = FacadeManager.EntityInterface.GetFacilitiesByVisn(new II(this.loggedUser.Domain, model.VisnId)).Select(a => CreateSite(a)).ToList();
                if (string.IsNullOrEmpty(model.FacilityId) || model.FacilityId.Equals(Guid.Empty.ToString(), StringComparison.InvariantCultureIgnoreCase)
                    || model.Facilities.Where(a => a.Id.extension.Equals(model.FacilityId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                    model.FacilityId = model.Facilities.FirstOrDefault().Id.extension;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region SisterSite

        [ReadPermissionAuthFilter]
        public ActionResult SisterSiteAddEdit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdminSisterSiteAddEditViewModel model = new AdminSisterSiteAddEditViewModel();
                FillSisterSiteViewModel(model);
                SetDefaultSisterSitesGroup(model, string.IsNullOrEmpty(p) ? (int?)null : Int32.Parse(QueryStrings["id"]));
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult SisterSiteAddEdit(AdminSisterSiteAddEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(model.BtnSelectSisterSite))
                    return this.RedirectToAction<AdminController>(act => act.SisterSiteAddEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.SisterSiteNumber.ToString() }, loggedUser.Salt)));
                FillSisterSiteViewModel(model);
                SisterSitesGroup outGroup;
                string codesDoNotExist = string.Empty;
                GetSistersIdentifierFromCodes(out outGroup, out codesDoNotExist, new SisterSitesGroup() { RecordNumber = model.RecordNumber.Value, SiteCodeList = model.SisterSiteCodes });

                //set message error
                if (model.SisterSitesGroups.Where(a => a.RecordNumber == model.RecordNumber).FirstOrDefault() == null && string.IsNullOrWhiteSpace(model.SisterSiteCodes))
                    model.MessageForSisterSiteEditSaved = Strings.NothingADDEDUPDATEDMissingOrBlankDataFields;
                else if (!string.IsNullOrWhiteSpace(codesDoNotExist))
                    model.MessageForSisterSiteEditSaved = string.Format(Strings.NothingADDEDUPDATEDTheFollowingCodesDoNotExistInBMSDatabase, codesDoNotExist);
                else
                {
                    FacadeManager.ConfigurationInterface.SaveSisterSites(model.SisterSiteNumber, outGroup);
                    SetDefaultSisterSitesGroup(model, model.RecordNumber);
                    model.MessageForSisterSiteEditSaved = string.Format(Strings.SisterSiteEditSavedInfo, model.RecordNumber, model.SisterSiteCodes.ToUpper());
                }
                return View("SisterSiteEditSaved", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        /// <summary>
        /// Fills the sister site view model.
        /// </summary>
        /// <param name="model">The model.</param>
        private void FillSisterSiteViewModel(AdminSisterSiteAddEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<SisterSitesGroup> sisterSitesGroups = GetSisterSites();
                model.SisterSitesGroups = sisterSitesGroups.Select(x => new SisterSiteGroupViewModel() { Model = x }).ToList();
                model.SisterSitesGroups.Insert(0, new SisterSiteGroupViewModel());
                model.LoggedUser = loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the default sister sites group.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="id">The id.</param>
        private void SetDefaultSisterSitesGroup(AdminSisterSiteAddEditViewModel model, int? id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (id != null)
                {
                    SisterSiteGroupViewModel sisterModel = model.SisterSitesGroups.Where(a => a.RecordNumber == id.Value).FirstOrDefault();
                    if (sisterModel != null)
                    {
                        model.SisterSiteNumber = sisterModel.RecordNumber;
                        if (sisterModel.Model != null)
                        {
                            model.RecordNumber = sisterModel.Model.RecordNumber;
                            model.SisterSiteCodes = sisterModel.Model.SiteCodeList;
                        }
                        else
                        {
                            model.RecordNumber = null;
                            model.SisterSiteCodes = null;
                        }
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the sister sites.
        /// </summary>
        /// <returns></returns>
        private IList<SisterSitesGroup> GetSisterSites()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<SisterSitesGroup> sisterSites = new List<SisterSitesGroup>();

                foreach (SisterSitesGroup sisterSite in FacadeManager.ConfigurationInterface.GetSisterSites())
                    sisterSites.Add(GetSistersCodeFromIdentifier(sisterSite));

                return sisterSites;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #endregion

        #region Facility

        [ReadPermissionAuthFilter]
        public ActionResult FacilityEdit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdminFacilityEditViewModel model = new AdminFacilityEditViewModel();

                model.StrictDecisions = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                model.States = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.State);
                model.TimeZones = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                model.TimeZones.Insert(0, new CD { code = string.Empty, displayName = string.Empty });

                model.Facilities = FacadeManager.EntityInterface.GetFacilities().OrderByDescending(f => f.IsAlive)
                                                                                .ThenBy(f => f.Code)
                                                                                .Select(f => new Facility { Id = f.Id, Name = f.Code + " - " + ((f.Address1 != null) ? (!string.IsNullOrEmpty(f.Address1.city) ? f.Address1.city : string.Empty) : string.Empty) + " (" + f.SiteNumber + " - ACTIVE: " + FormatIsAlive(f.IsAlive) + ")" })
                                                                                .ToList();

                Facility facility = string.IsNullOrEmpty(p) ? null : model.Facilities.FirstOrDefault(f => f.Id.extension.Equals(QueryStrings["id"], StringComparison.InvariantCultureIgnoreCase));

                model.Facilities.Insert(0, new Facility() { Id = new II() { extension = (default(Guid)).ToString() }, Name = Strings.SelectFacility });

                if (model.SisterSitesGroups == null)
                    model.SisterSitesGroups = new List<SisterSiteGroupViewModel>();
                //set default integrated sister site id
                model.SisterSitesGroups.Insert(0, new SisterSiteGroupViewModel());

                if (facility != null)
                {
                    facility = FacadeManager.EntityInterface.GetFacility(facility.Id);

                    model.VistaDivisionList = FacadeManager.VocabularyInterface.GetVocabularyByVistaCode(Util.Vocabulary.VistaMedicalCenterDivision, facility.VistaSite.Code).Select(a => new CD() { code = a.code, codeSystem = a.codeSystem, codeSystemName = a.codeSystemName, displayName = a.displayName.ToUpper() + " {" + Strings.IEN + ":" + a.code.Substring(a.code.IndexOf("_") + 1) + "}" }).ToList();
                    if (model.VistaDivisionList == null)
                        model.VistaDivisionList = new List<CD>();

                    model.FacilityCode = facility.Code;
                    model.FacilityName = facility.Name;
                    model.FacilityPOCContact = (facility.PersonContactName != null) ? facility.PersonContactName.text : string.Empty;
                    model.FacilityPOCEmail = (facility.PersonContactEmail != null) ? facility.PersonContactEmail.value : string.Empty;
                    model.FacilityPOCTelephone = (facility.PersonContactPhone != null) ? facility.PersonContactPhone.value : string.Empty;
                    model.FacilityAddress1 = (facility.Address1 != null) ? facility.Address1.streetName : string.Empty;
                    model.FacilityAddress2 = (facility.Address2 != null) ? facility.Address2.streetName : string.Empty;
                    model.FacilityCity = (facility.Address1 != null) ? facility.Address1.city : string.Empty;
                    model.FacilityState = (facility.Address1 != null) ? facility.Address1.county : string.Empty;
                    model.FacilityZip = (facility.Address1 != null) ? facility.Address1.postalCode : string.Empty;
                    model.FacilityIdentifier = facility.Id.extension;
                    model.FacilityShortName = facility.Code + " (" + facility.SiteNumber + (string.IsNullOrEmpty(model.FacilityCity) ? "" : " - " + model.FacilityCity) + ")";
                    model.VISNCode = facility.VistaSite.Visn.Number;
                    model.RegionCode = facility.VistaSite.Visn.Region.Number.ToString();
                    model.IsAliveSite = CreateStrictDecisionCD(facility.IsAlive);
                    FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(facility.Id);
                    if (settings != null)
                    {
                        model.BMSServerTimeZone = TimeZoneInfo.Local.StandardName;
                        model.LocalTimeAdjust = settings.LocalTimeAdjust;

                        model.WardPrefix = settings.WardPrefix;
                        model.ADTPrefix = settings.ADTPrefix;
                        model.WardSuffix = settings.WardSuffix;
                        model.ADTSuffix = settings.ADTSuffix;
                        model.EMSMailSender = settings.EMSMailSender;
                        model.EventMailSender = settings.EventMailSender;
                        model.SiteAlias1 = settings.SiteAlias1;
                        model.SiteAlias2 = settings.SiteAlias2;
                        model.SiteAlias3 = settings.SiteAlias3;
                        model.FacilitySiteTimeZoneCode = settings.FacilitySiteTimeZone != null ? settings.FacilitySiteTimeZone.code : null;

                        //fill sister sites
                        foreach (SisterSitesGroup group in settings.SisterSitesGroups)
                            model.SisterSitesGroups.Add(new SisterSiteGroupViewModel() { Model = GetSistersCodeFromIdentifier(group) });

                        //set selected integrated site sister id
                        model.IntegratedSisterSiteId = settings.IntegratedSiteSisterGroupId;
                        model.IntegratedFacilityCode = settings.IntegratedFacility != null ? settings.IntegratedFacility.code : Constants.No;
                        model.AutoRemovalWaitingListCode = settings.AutoRemovalWaitingList != null ? settings.AutoRemovalWaitingList.code : Constants.No;
                        //set integrated sites
                        if (settings.IntegratedFacility != null && settings.IntegratedFacility.code == Constants.Yes && settings.IntegratedSiteSisterGroupId != 0)
                        {
                            model.ShowAllowedAccessSites = true;
                            model.IntegratedSiteList = BuildIntegratedSitesList(settings, facility.Code);
                        }
                        else
                        {
                            model.ShowAllowedAccessSites = false;
                        }

                        if (settings.MedicalCenterID != null)
                        {
                            model.VistaDivisionList.Insert(0, new CD() { code = Strings.ClearMedicalCenterId, displayName = Strings.ClearMedicalCenterId });
                            model.MedicalCenterID = settings.MedicalCenterID;
                        }
                        else
                        {
                            model.VistaDivisionList.Insert(0, new CD() { code = Strings.SelectMedicalCenterId, displayName = Strings.SelectMedicalCenterId });
                        }
                        model.EMSUserName = settings.EMSDefaultUserName;
                        model.KioskUserName = settings.KioskDefaultUserName;
                    }

                    II facilityId = new II(this.loggedUser.Domain, model.FacilityIdentifier);
                    model.UsersOperations = FacadeManager.UserInterface.GetUserOperations(facilityId).Select(i => new CheckItem<UserOperations>() { Item = i }).ToList();
                }
                else
                {
                    model.FacilityShortName = Strings.NoFacilitySelected;
                }
                model.LoggedUser = loggedUser;
                if (model.VistaDivisionList == null)
                    model.VistaDivisionList = new List<CD>();
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Builds the integrated sites list.
        /// </summary>
        /// <param name="facilitySettings">The facility settings.</param>
        /// <param name="facilityIdentifier">The facility identifier.</param>
        /// <returns></returns>
        private IList<FacilitySisterSitesGroupViewModel> BuildIntegratedSitesList(FacilitySettings facilitySettings, string facilityCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<FacilitySisterSitesGroupViewModel> integratedSites = new List<FacilitySisterSitesGroupViewModel>();
                string allSisterIdentifiers = facilitySettings.SisterSitesGroups.Where(a => a.RecordNumber == facilitySettings.IntegratedSiteSisterGroupId).Select(a => a.SiteCodeList).FirstOrDefault();
                if (!string.IsNullOrWhiteSpace(allSisterIdentifiers))
                {
                    if (allSisterIdentifiers.Contains(facilityCode))
                    {
                        string[] facilitiesCodes = allSisterIdentifiers.Split(',');
                        foreach (string code in facilitiesCodes)
                        {
                            if (code.Trim() != facilityCode)
                            {
                                Facility facility = FacadeManager.EntityInterface.GetFacilityByCode(code.Trim());
                                if (facility != null)
                                {
                                    if (!string.IsNullOrWhiteSpace(facilitySettings.IntegratedSiteSisterList) && facilitySettings.IntegratedSiteSisterList.Contains(facility.Id.extension))
                                        integratedSites.Add(new FacilitySisterSitesGroupViewModel() { Site = facility, IsIntegratedSite = true });
                                    else
                                        integratedSites.Add(new FacilitySisterSitesGroupViewModel() { Site = facility, IsIntegratedSite = false });
                                }
                            }

                        }
                    }
                }
                return integratedSites;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilter]
        public ActionResult AddUserOperations(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AddUserOperationsModel model = new AddUserOperationsModel();
                model.FacilityId = QueryStrings["id"];
                model.Domains = FacadeManager.UserInterface.GetUserDomains();
                model.SelectedDomain = model.Domains.First();
                II id = new II(this.loggedUser.Domain, model.FacilityId);
                model.FacilityName = FacadeManager.EntityInterface.GetFacility(id).Name;
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilter]
        public ActionResult SearchUsers(AddUserOperationsModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.Domains = FacadeManager.UserInterface.GetUserDomains();
                II id = new II(this.loggedUser.Domain, model.FacilityId);
                var users = FacadeManager.UserInterface.SearchUsers(model.SelectedDomain, !string.IsNullOrEmpty(model.SearchText) ? model.SearchText.Trim() : model.SearchText);
                if (users != null)
                    model.Users = users.Select(u => new CheckItem<User>() { Item = u }).ToList();
                model.LoggedUser = loggedUser;
                return View("AddUserOperations", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static UserOperations ToUserOperations(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new UserOperations()
                {
                    UserId = user.Id.extension,
                    BmsRead = true,
                    BmsWrite = true,
                    UserName = user.UserName
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult AddUserOperations(AddUserOperationsModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II facilityId = new II(this.loggedUser.Domain, model.FacilityId);
                List<PermissionFacility> facilities = new List<PermissionFacility>();
                facilities.Add(new PermissionFacility() { Facility = new Facility() { Id = facilityId }, ReadAccess = true, WriteAccess = true });
                foreach (CheckItem<User> item in model.Users.Where(a => a.IsChecked).ToList())
                    FacadeManager.UserInterface.AddUserOperations(item.Item.UserName, facilities);

                return this.RedirectToAction<AdminController>(act => act.FacilityEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.FacilityId }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private ActionResult RemoveUsers(AdminFacilityEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(model.BtnSelectFacilityEdit))
                    this.RedirectToAction<AdminController>(act => act.FacilityEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.FacilityIdentifier }, loggedUser.Salt)));
                if (model.UsersOperations == null)
                    return this.RedirectToAction<AdminController>(act => act.FacilityEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.FacilityIdentifier }, loggedUser.Salt)));

                II facilityId = new II(this.loggedUser.Domain, model.FacilityIdentifier);
                RemoveUserOperationsModel removeModel = new RemoveUserOperationsModel();
                removeModel.FacilityId = model.FacilityIdentifier;
                removeModel.FacilityName = FacadeManager.EntityInterface.GetFacility(facilityId).Name;
                IList<UserOperations> users = model.UsersOperations.Where(c => c.IsChecked).Select(c => c.Item).ToList();
                removeModel.UserOperations = users;
                removeModel.LoggedUser = loggedUser;
                return View("RemoveUserOperations", removeModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult RemoveUserOperations(RemoveUserOperationsModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II facilityId = new II(this.loggedUser.Domain, model.FacilityId);
                List<PermissionFacility> facilities = new List<PermissionFacility>();
                facilities.Add(new PermissionFacility() { Facility = new Facility() { Id = facilityId }, ReadAccess = false, WriteAccess = false });
                List<UserOperations> operations = FacadeManager.UserInterface.GetUserOperations(facilityId).ToList();
                UserOperations uOp = null;
                foreach (UserOperations op in model.UserOperations)
                {
                    uOp = operations.Where(a => a.UserId == op.UserId).FirstOrDefault();
                    if (uOp != null)
                    {
                        facilities[0].ReadAccess = uOp.BmsRead;
                        facilities[0].WriteAccess = uOp.BmsWrite;
                        FacadeManager.UserInterface.RemoveUserOperations(op.UserName, facilities);
                    }
                    uOp = null;
                }
                return this.RedirectToAction<AdminController>(act => act.FacilityEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.FacilityId }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CD CreateStrictDecisionCD(bool value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return value ? new CD() { code = "Yes", displayName = "Yes" } : new CD() { code = "No", displayName = "No" };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult FacilityEdit(AdminFacilityEditViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(model.BtnSelectFacilityEdit))
                    return this.RedirectToAction<AdminController>(act => act.FacilityEdit(EncryptQueryString(new string[] { "id" }, new string[] { model.FacilityIdentifier }, loggedUser.Salt)));
                if (!string.IsNullOrEmpty(model.BtnRemoveUsers))
                    return RemoveUsers(model);
                model.LoggedUser = loggedUser;
                if (model.FacilityIdentifier == null)
                    return View("FacilityEditSaved", model);

                if (model.UsersOperations != null)
                {
                    foreach (CheckItem<UserOperations> item in model.UsersOperations)
                    {
                        if (!item.Item.BmsRead && item.Item.BmsWrite)
                        {
                            ViewData[InputHelpers.FormNameKey] = "AddEditUserOperations";
                            ModelState.AddModelError(string.Empty, "Read Access is required for facilities with Write Access.");
                            break;
                        }
                    }
                }

                model.StrictDecisions = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                model.States = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.State);
                model.TimeZones = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);

                model.Facilities = FacadeManager.EntityInterface.GetFacilities();
                Facility f = model.Facilities.First(x => x.Id.extension.Equals(model.FacilityIdentifier, StringComparison.InvariantCultureIgnoreCase));
                model.Facilities.Insert(0, new Facility() { Id = new II(), Name = Strings.SelectFacilityName });
                model.VistaDivisionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaMedicalCenterDivision).Select(a => new CD() { code = a.code, codeSystem = a.codeSystem, codeSystemName = a.codeSystemName, displayName = a.displayName.ToUpper() + " " + "IEN: " + a.code.Substring(a.code.IndexOf("_") + 1) }).ToList();
                FacilitySettings facilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(f.Id);
                if (!string.IsNullOrEmpty(model.EMSUserName) && string.IsNullOrEmpty(model.EMSPassword) && string.IsNullOrEmpty(facilitySettings.EMSDefaultPassword))
                {
                    ViewData[InputHelpers.FormNameKey] = Strings.EMS;
                    ModelState.AddModelError("EMSPassword", "EMS Password is required!");
                }
                if (!string.IsNullOrEmpty(model.KioskUserName) && string.IsNullOrEmpty(model.KioskPassword) && string.IsNullOrEmpty(facilitySettings.KioskDefaultPassword))
                {
                    ViewData[InputHelpers.FormNameKey] = Strings.Kiosk;
                    ModelState.AddModelError("KioskPassword", "Whiteboard Kiosk Password is required!");
                }

                if (!ModelState.IsValid)
                {
                    model.BMSServerTimeZone = TimeZoneInfo.Local.StandardName;
                    if (model.SisterSitesGroups == null)
                        model.SisterSitesGroups = new List<SisterSiteGroupViewModel>();
                    model.SisterSitesGroups.Insert(0, new SisterSiteGroupViewModel());
                    if (facilitySettings != null)
                    {
                        //fill sister sites
                        foreach (SisterSitesGroup group in facilitySettings.SisterSitesGroups)
                            model.SisterSitesGroups.Add(new SisterSiteGroupViewModel() { Model = GetSistersCodeFromIdentifier(group) });

                        //set selected integrated site sister id
                        model.IntegratedSisterSiteId = facilitySettings.IntegratedSiteSisterGroupId;

                        //set integrated sites
                        if (facilitySettings.IntegratedFacility != null && facilitySettings.IntegratedFacility.code == Constants.Yes && facilitySettings.IntegratedSiteSisterGroupId != 0)
                        {
                            model.ShowAllowedAccessSites = true;
                            model.IntegratedSiteList = BuildIntegratedSitesList(facilitySettings, f.Code);
                        }
                        else
                        {
                            model.ShowAllowedAccessSites = false;
                        }
                    }
                    return View(model);
                }

                f = FacadeManager.EntityInterface.GetFacility(f.Id);
                if (f.Address1 == null)
                    f.Address1 = new AD();
                f.Address1.streetName = !string.IsNullOrEmpty(model.FacilityAddress1) ? model.FacilityAddress1.ToUpper() : string.Empty;
                f.Address1.city = !string.IsNullOrEmpty(model.FacilityCity) ? model.FacilityCity.ToUpper() : string.Empty;
                f.Address1.county = !string.IsNullOrEmpty(model.FacilityState) ? model.FacilityState.ToUpper() : string.Empty;
                f.Address1.postalCode = !string.IsNullOrEmpty(model.FacilityZip) ? model.FacilityZip.ToUpper() : string.Empty;

                if (f.Address2 == null)
                    f.Address2 = new AD();
                f.Address2.streetName = !string.IsNullOrEmpty(model.FacilityAddress2) ? model.FacilityAddress2.ToUpper() : string.Empty;

                f.Code = model.FacilityCode;
                f.Name = (model.FacilityName != null) ? model.FacilityName.ToUpper() : string.Empty;
                f.IsAlive = (model.IsAliveSite != null) ? ((model.IsAliveSite.code == Constants.Yes) ? true : false) : false;

                if (f.PersonContactName == null)
                    f.PersonContactName = new ST();
                f.PersonContactName.text = !string.IsNullOrWhiteSpace(model.FacilityPOCContact) ? model.FacilityPOCContact.ToUpper() : string.Empty;


                if (f.PersonContactEmail == null)
                    f.PersonContactEmail = new TEL();
                f.PersonContactEmail.value = !string.IsNullOrWhiteSpace(model.FacilityPOCEmail) ? model.FacilityPOCEmail.ToLower() : string.Empty;


                if (f.PersonContactPhone == null)
                    f.PersonContactPhone = new TEL();
                f.PersonContactPhone.value = !string.IsNullOrWhiteSpace(model.FacilityPOCTelephone) ? model.FacilityPOCTelephone.ToLower() : string.Empty;

                Visn visn = FacadeManager.EntityInterface.GetVisns().FirstOrDefault(x => x.Number == model.VISNCode);
                f.VistaSite.Visn = visn;

                FacadeManager.EntityInterface.SaveFacility(f);
                //refresh facility from session
                if (f.Id.extension.Equals(this.loggedUser.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase) && f.Id.root.Equals(this.loggedUser.Facility.Id.root, StringComparison.InvariantCultureIgnoreCase))
                    this.loggedUser.Facility = f;
                FacilitySettings settings = new FacilitySettings();
                settings.LocalTimeAdjust = model.LocalTimeAdjust;
                settings.IntegratedSiteSisterGroupId = model.IntegratedSisterSiteId;
                settings.BMSServerTimeZone = TimeZoneInfo.Local.StandardName;
                settings.WardPrefix = model.WardPrefix;
                settings.ADTPrefix = model.ADTPrefix;
                settings.WardSuffix = model.WardSuffix;
                settings.ADTSuffix = model.ADTSuffix;
                settings.EMSMailSender = model.EMSMailSender;
                settings.EventMailSender = model.EventMailSender;
                settings.SiteAlias1 = model.SiteAlias1;
                settings.SiteAlias2 = model.SiteAlias2;
                settings.SiteAlias3 = model.SiteAlias3;
                settings.FacilitySiteTimeZone = !string.IsNullOrEmpty(model.FacilitySiteTimeZoneCode) ? new CD() { code = model.FacilitySiteTimeZoneCode } : null;
                settings.IntegratedFacility = model.StrictDecisions.Where(a => a.code.Equals(model.IntegratedFacilityCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                settings.AutoRemovalWaitingList = model.StrictDecisions.Where(a => a.code.Equals(model.AutoRemovalWaitingListCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();

                //set integrated sites list
                if (settings.IntegratedFacility == null || settings.IntegratedFacility.code == Constants.No)
                {
                    // delete integrated sites list
                    settings.IntegratedSiteSisterList = string.Empty;
                }
                else
                {
                    //save integrated sites list
                    if (model.IntegratedSiteList != null && model.IntegratedSiteList.Count > 0)
                    {
                        List<string> integratedSites = model.IntegratedSiteList.Where(a => a.IsIntegratedSite == true).Select(a => a.Site.Id.extension).ToList();
                        string integratedSiteList = string.Empty;

                        foreach (string identifier in integratedSites)
                        {
                            if (string.IsNullOrWhiteSpace(integratedSiteList))
                                integratedSiteList = identifier;
                            else
                                integratedSiteList += ", " + identifier;
                        }
                        settings.IntegratedSiteSisterList = integratedSiteList;
                    }
                }

                //delete medical center id or save
                if (model.MedicalCenterID == Strings.ClearMedicalCenterId || model.MedicalCenterID == Strings.SelectMedicalCenterId)
                    settings.MedicalCenterID = null;
                else
                    settings.MedicalCenterID = model.MedicalCenterID;
                settings.EMSDefaultUserName = model.EMSUserName;
                if (!string.IsNullOrEmpty(model.EMSPassword))
                    settings.EMSDefaultPassword = Encryptor.Encrypt(model.EMSPassword);
                else
                    settings.EMSDefaultPassword = facilitySettings.EMSDefaultPassword;
                settings.KioskDefaultUserName = model.KioskUserName;
                if (!string.IsNullOrEmpty(model.KioskPassword))
                    settings.KioskDefaultPassword = Encryptor.Encrypt(model.KioskPassword);
                else
                    settings.KioskDefaultPassword = facilitySettings.KioskDefaultPassword;


                FacadeManager.ConfigurationInterface.SaveFacilitySettings(f.Id, settings);

                if (model.UsersOperations != null)
                {
                    List<PermissionFacility> facilities = new List<PermissionFacility>();
                    facilities.Add(new PermissionFacility() { Facility = new Facility() { Id = f.Id }, ReadAccess = false, WriteAccess = false });
                    List<UserOperations> operations = FacadeManager.UserInterface.GetUserOperations(f.Id).ToList();
                    UserOperations uOp = null;
                    foreach (CheckItem<UserOperations> item in model.UsersOperations)
                    {
                        uOp = operations.Where(a => a.UserId == item.Item.UserId).FirstOrDefault();
                        if (uOp != null)
                        {
                            if (uOp.BmsRead != item.Item.BmsRead || uOp.BmsWrite != item.Item.BmsWrite)
                            {
                                facilities[0].ReadAccess = uOp.BmsRead;
                                facilities[0].WriteAccess = uOp.BmsWrite;
                                FacadeManager.UserInterface.RemoveUserOperations(item.Item.UserName, facilities);

                                facilities[0].ReadAccess = item.Item.BmsRead;
                                facilities[0].WriteAccess = item.Item.BmsWrite;
                                if (facilities[0].ReadAccess || facilities[0].WriteAccess)
                                    FacadeManager.UserInterface.AddUserOperations(item.Item.UserName, facilities);
                            }
                        }
                        else
                        {
                            facilities[0].ReadAccess = item.Item.BmsRead;
                            facilities[0].WriteAccess = item.Item.BmsWrite;
                            if (facilities[0].ReadAccess || facilities[0].WriteAccess)
                                FacadeManager.UserInterface.AddUserOperations(item.Item.UserName, facilities);
                        }
                        uOp = null;
                    }
                }

                VistASettings vistASettings = new VistASettings()
                {
                    LocalTimeAdjust = settings.LocalTimeAdjust,
                    SiteTimeZone = new CD() { code = model.FacilitySiteTimeZoneCode }
                };

                FacadeManager.ConfigurationInterface.SaveVistASettings(vistASettings, f.VistaSite.Id);
                model.LoggedUser = loggedUser;
                return View("FacilityEditSaved", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Cache

        [ReadPermissionAuthFilter]
        public ActionResult ClearCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMS.ServicesWrapper.BMService.BMSFactory.BedManagerCacheClientFromWCF.RefreshCache();
                BMS.Utils.Utilities.ClearEVSImages();
                TimeZoneUtil.ClearCache();
                return View("CacheConfirmation");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        private Site CreateSite(Facility f)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new Site { Id = f.Id, Name = f.Code + " - " + f.Name };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilter]
        public ActionResult SelectUser()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SelectUserModel model = new SelectUserModel();
                model.Domains = FacadeManager.UserInterface.GetUserDomains();
                model.SelectedDomain = model.Domains.First();
                model.Users = FacadeManager.UserInterface.SearchUsers(model.SelectedDomain, model.SearchText);
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilter]
        public ActionResult SearchUser(SelectUserModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.Domains = FacadeManager.UserInterface.GetUserDomains();
                model.Users = FacadeManager.UserInterface.SearchUsers(model.SelectedDomain, !string.IsNullOrEmpty(model.SearchText) ? model.SearchText.Trim() : model.SearchText);
                model.LoggedUser = loggedUser;
                return View("SelectUser", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [ReadPermissionAuthFilter]
        public ActionResult SelectUser(SelectUserModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return this.RedirectToAction<AdminController>(act => act.AddEditUser(EncryptQueryString(new string[] { "id" }, new string[] { model.SelectedUserId }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private string FormatIsAlive(bool isAlive)
        {
            if (isAlive)
                return Constants.Yes.ToUpper();
            else
                return Constants.No.ToUpper();
        }
    }

}
