﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using InfoWorld.EVS.CTSMAPI;
using System.Text;
using System.Web.Mvc;
using BMS.Utils;

namespace BMS.Web.Models
{
    /// <summary>
    /// Ward Occupancy Model
    /// </summary>
    public class WardOccupancyViewModel : BaseViewModel
    {
        #region Public Members
        /// <summary>
        /// Gets or sets the division.
        /// </summary>
        /// <value>
        /// The division.
        /// </value>
        public string Division { get; set; }

        /// <summary>
        /// Gets or sets the last updated date.
        /// </summary>
        /// <value>
        /// The last updated date.
        /// </value>
        public string LastUpdatedDate { get; set; }

        /// <summary>
        /// Gets or sets the facility site time zone.
        /// </summary>
        /// <value>
        /// The facility site time zone.
        /// </value>
        public string FacilitySiteCodeTimeZone { get; set; }

        /// <summary>
        /// Gets or sets the bed info list.
        /// </summary>
        /// <value>
        /// The bed info list.
        /// </value>
        public IList<WardOccupancyBedViewModel> BedInfoList { get; set; }

        /// <summary>
        /// Gets or sets the bed status code.
        /// </summary>
        /// <value>
        /// The bed status code.
        /// </value>
        public string BedStatusCode { get; set; }

        /// <summary>
        /// Gets or sets the root division id.
        /// </summary>
        /// <value>
        /// The root division id.
        /// </value>
        public string RootDivisionId { get; set; }

        /// <summary>
        /// Gets or sets the extension division id.
        /// </summary>
        /// <value>
        /// The extension division id.
        /// </value>
        public string ExtensionDivisionId { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is from summary report.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is from summary report; otherwise, <c>false</c>.
        /// </value>
        public bool IsFromSummaryReport { get; set; }

        /// <summary>
        /// Gets or sets the summary report return path.
        /// </summary>
        /// <value>
        /// The summary report return path.
        /// </value>
        public string SummaryReportReturnPath { get; set; }

        /// <summary>
        /// Gets or sets the summary report return text.
        /// </summary>
        /// <value>
        /// The summary report return text.
        /// </value>
        public string SummaryReportReturnText { get; set; }

        /// <summary>
        /// Gets or sets the visn root used in visn summary report.
        /// </summary>
        /// <value>
        /// The visn root used in visn summary report.
        /// </value>
        public string VisnRoot { get; set; }

        /// <summary>
        /// Gets or sets the visn extension used in visn summary report.
        /// </summary>
        /// <value>
        /// The visn extension used in visn summary report.
        /// </value>
        public string VisnExtension { get; set; }

        /// <summary>
        /// Gets or sets the bed summary report URL.
        /// </summary>
        /// <value>
        /// The bed summary report URL.
        /// </value>
        public string BedSummaryReportUrl { get; set; }

        /// <summary>
        /// Gets or sets the icon information list.
        /// </summary>
        /// <value>
        /// The icon information list.
        /// </value>
        public List<IconInfo> IconFlagsList { get; set; }

        /// <summary>
        /// Gets or sets the total LOS ward.
        /// </summary>
        /// <value>
        /// The total LOS ward.
        /// </value>
        public string TotalLOSWard { get; set; }

        /// <summary>
        /// Gets or sets the total los admin.
        /// </summary>
        /// <value>
        /// The total los admin.
        /// </value>
        public string TotalLosAdmin { get; set; }

        /// <summary>
        /// Gets or sets the ward occupancy report URL.
        /// </summary>
        /// <value>
        /// The ward occupancy report URL.
        /// </value>
        public string WardOccupancyReportUrl { get; set; }

        #endregion
    }

    /// <summary>
    /// Bed view model used in WardOccupancy view
    /// </summary>
    public class WardOccupancyBedViewModel
    {
        public static List<WardOccupancyBedViewModel> GetViewModel(WardOccupancyViewModel parentModel, IList<BedInfo> bedList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var bedViewModels = new List<WardOccupancyBedViewModel>();
                if (bedList != null)
                {
                    List<string> similarities = Similarities(bedList);

                    foreach (var bed in bedList)
                    {
                        bool isSimilar = IsPatientLastNameOrPatientSsnSimilar(similarities, bed);

                        bedViewModels.Add(GetViewModel(parentModel, bed, isSimilar));
                    }
                }
                return bedViewModels;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        static WardOccupancyBedViewModel GetViewModel(WardOccupancyViewModel parentModel, BedInfo model, bool iSsimilar)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model == null)
                    return null;
                WardOccupancyBedViewModel res = new WardOccupancyBedViewModel();

                res.BedIdRoot = model.BedId.root;
                res.BedIdExtension = model.BedId.extension;
                res.PatientIdRoot = model.PatientId.root;
                res.PatientIdExtension = model.PatientId.extension;
                res.BedName = model.BedName;
                res.PatientLastName = model.PatientLastName;
                res.DisplayPatientGender = model.DisplayPatientGender;
                res.HasMultipleComments = model.HasMultipleComments;
                res.HasMultipleDivisionsAssigned = model.HasMultipleDivisionsAssigned;

                //set reason
                if (!string.IsNullOrEmpty(model.Comment) && !string.IsNullOrEmpty(model.Reason))
                    res.Reason = model.Reason + " - " + model.Comment;
                else if (string.IsNullOrEmpty(model.Comment) && !string.IsNullOrEmpty(model.Reason))
                    res.Reason = model.Reason;
                else if (!string.IsNullOrEmpty(model.Comment) && string.IsNullOrEmpty(model.Reason))
                    res.Reason = model.Comment;
                else
                    res.Reason = string.Empty;
                res.IsBedHold = model.IsBedHold;
                if (model.IsBedHold)
                    res.Reason = (string.IsNullOrEmpty(res.Reason)) ? Constants.BED_HOLD_TEXT : res.Reason + " - " + Constants.BED_HOLD_TEXT;
                res.BedHoldIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == Constants.BED_HOLD_CONCEPT_CODE).FirstOrDefault();
                if (res.BedHoldIcon == null)
                    res.BedHoldIcon = new IconInfo();
                else
                    res.BedHoldIconBytes = res.BedHoldIcon.ImageBytes;
                if (iSsimilar && parentModel.IconFlagsList != null)
                    res.SimilarNameIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == Constants.SIMILAR_SAME_NAME).First();
                else
                    res.SimilarNameIcon = new IconInfo();
                res.SimilarNameIconBytes = res.SimilarNameIcon.ImageBytes;

                res.BedAvailabilityStatusIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.BedAvailabilityStatusCode).FirstOrDefault();
                if (res.BedAvailabilityStatusIcon == null)
                    res.BedAvailabilityStatusIcon = new IconInfo();
                else
                    res.BedAvailabilityStatusIconBytes = res.BedAvailabilityStatusIcon.ImageBytes;
                res.DischargeIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.DischargeStatusCode).FirstOrDefault();
                if (res.DischargeIcon == null)
                    res.DischargeIcon = new IconInfo();
                else
                    res.DischargeIconBytes = res.DischargeIcon.ImageBytes;
                res.BedCleaningStatusIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.BedCleaningStatusCode).FirstOrDefault();
                if (res.BedCleaningStatusIcon == null)
                    res.BedCleaningStatusIcon = new IconInfo();
                else
                    res.BedCleaningStatusIconBytes = res.BedCleaningStatusIcon.ImageBytes;

                if (!string.IsNullOrEmpty(model.LevelOfCareCode))
                {
                    res.LevelOfCareIcon = parentModel.IconFlagsList.Where(a => a.Group == Constants.APPLICATION_ICON && a.IconType != null && a.IconType.code == model.LevelOfCareCode).FirstOrDefault();
                    if (res.LevelOfCareIcon == null)
                        res.LevelOfCareIcon = new IconInfo();
                    else
                        res.LevelOfCareIconBytes = res.LevelOfCareIcon.ImageBytes;
                }
                else
                    res.LevelOfCareIcon = new IconInfo();
                if (model.LevelOfCareDays > 0)
                    res.LevelOfCareDays = model.LevelOfCareDays.ToString();

                if (model.LevelOfCareChangeDays > 0)
                    res.LevelOfCareChangeDays = model.LevelOfCareChangeDays.ToString();

                res.IconFlagsList = new List<IconInfo>();
                List<int> iconList = new List<int>();
                iconList.AddRange(model.IconFlagIdsOfBedList);
                iconList.AddRange(model.IconFlagIdsOfPatientList);
                iconList.AddRange(model.IconFlagFacilityIdsOfPatientList);
                iconList.AddRange(model.IconFlagFacilityIdsOfBedList);

                res.IconFlagsList = (from icon in parentModel.IconFlagsList
                                     where (from iconAssociationId in iconList
                                            select iconAssociationId).Contains(icon.IconId)
                                     select icon).OrderBy(o => o.Subgroup == Constants.STANDARD_ICON ? 1 : (o.Subgroup == Constants.EMERGENCY_ICON ? 2 : 3)).ThenBy(o => o.OrderIndex).ToList();

                //combine all iconFlags for report
                if (res.IconFlagsList != null && res.IconFlagsList.Count > 0)
                {
                    List<string> imagesParam = new List<string>();
                    foreach (IconInfo icon in res.IconFlagsList)
                        imagesParam.Add(icon.ImageBytes);
                    res.CombineIconBinaryList = ImageConverter.MergeMultipleImages(imagesParam);
                }

                if (string.IsNullOrEmpty(model.PatientLastName) || string.IsNullOrEmpty(model.SSN))
                {
                    res.EditLinkString = string.Empty;
                    //set the values for properties used in view
                    res.LosAdmin = string.Empty;
                    res.LosWard = string.Empty;
                    res.LosDrg = string.Empty;
                    res.WardName = string.Empty;
                }
                else
                {
                    res.EditLinkString = (model.PatientLastName.Substring(0, 1) + model.SSN.Substring(model.SSN.Length - 4));
                    //set the values for properties used in view
                    res.LosAdmin = model.LosAdmin.ElapsedTime();
                    res.LosWard = model.LosWard.ElapsedTime();
                    res.LosDrg = string.Empty; //model.LosDrg.ElapsedTime(); LosDrg is not calculated in the cold fusion code.
                    if (model.PatientTreatingSpecialty != null)
                        res.WardName = model.WardName + " - " + model.PatientTreatingSpecialty.displayName.ToUpper();
                    else
                        res.WardName = model.WardName;
                }

                string classColor = string.Empty;
                model.Reason = model.Reason ?? string.Empty;
                model.Comment = model.Comment ?? string.Empty;
                if (model.Reason.Contains("TEMPORARILY UNAVAILABLE") || (!string.IsNullOrEmpty(model.BedAvailabilityStatusCode) && (model.BedAvailabilityStatusCode == "OOS" || model.BedAvailabilityStatusCode == "DND" || model.BedAvailabilityStatusCode == "BED_OOS_VISTA" || model.BedAvailabilityStatusCode == "BED_OOS_BEDBOARD")))
                    classColor = "class=\"colorRed SmallText alignCenter\"";
                else if (model.Reason.Contains("BED ASSIGNED") || model.Comment.Contains("BED ASSIGNED") || !string.IsNullOrEmpty(model.WaitingListInfo))
                    classColor = "class=\"colorOrange SmallText alignCenter\"";
                else
                    classColor = "class=\"colorYellowOrange SmallText alignCenter\"";

                if (!string.IsNullOrEmpty(model.WaitingListInfo))
                    res.Reason = model.WaitingListInfo + " " + res.Reason;

                if (!string.IsNullOrEmpty(model.PatientGender))
                {
                    if (model.PatientGender == "Male")
                        classColor = "class=\"colorBlue SmallText alignCenter\"";
                    else if (model.PatientGender == "Female")
                        classColor = "class=\"colorPink SmallText alignCenter\"";
                    else
                        classColor = "class=\"colorLightGray SmallText alignCenter\"";
                }

                res.GenderColor = classColor;

                //Construct the PT based on the Display PT selected value.
                StringBuilder pt = new StringBuilder();

                if (!string.IsNullOrEmpty(model.PatientLastName) && !string.IsNullOrEmpty(model.SSN))
                {
                    if (res.IconFlagsList != null && res.IconFlagsList.Where(a => (a.PatientOrRoomBed == 'P' && (a.IconName.ToLower().Contains("optout") || a.IconName.ToLower().Contains("opt_out") || a.IconName.ToLower().Contains("opt-out") || a.IconName.ToLower().Contains("opt out")))).FirstOrDefault() != null)
                        pt.Append("X X X X");
                    else
                    {
                        pt.Append(model.PatientLastName.Substring(0, 1));
                        pt.Append(model.SSN.Substring(model.SSN.Length - 4));
                    }
                    if (model.PatientTreatingSpecialty == null)
                        res.WardName = model.WardName;
                    else
                        res.WardName = model.WardName + " - " + model.PatientTreatingSpecialty.displayName.ToUpper();
                }
                res.PT = pt.ToString();

                res.Reason = res.Reason;
                return res;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// List of similar patient last names and similar SSNs.
        /// </summary>
        /// <param name="bedList">The bed info list.</param>
        /// <returns>List of similarities.</returns>
        private static List<string> Similarities(IList<BedInfo> bedList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> similarPatientLastNames = bedList.GroupBy(bed =>
                {
                    if (bed.PatientLastName.Length > 3)
                    {
                        return bed.PatientLastName.ToLower().Substring(0, 4);
                    }

                    return string.Empty;
                }
                                                                      )
                                                              .Where(bed => !string.IsNullOrEmpty(bed.Key) && bed.Count() > 1)
                                                              .Select(bed => bed.Key)
                                                              .ToList();

                List<string> similarSSNs = bedList.GroupBy(bed =>
                {
                    if (bed.SSN.Length > 3)
                    {
                        return bed.SSN.ToLower().Substring(bed.SSN.Length - 4, 4);
                    }

                    return string.Empty;
                }
                                                          )
                                                  .Where(bed => !string.IsNullOrEmpty(bed.Key) && bed.Count() > 1)
                                                  .Select(bed => bed.Key)
                                                  .ToList();

                similarPatientLastNames.AddRange(similarSSNs);

                return similarPatientLastNames;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Checks if first four characters of the last name of the patient are the same or the last four characters of patient SSN are the same.
        /// </summary>
        /// <param name="similarities">List of similarities.</param>
        /// <param name="bed">The bed.</param>
        /// <returns>
        /// True - if exists / Otherwise false.
        /// </returns>
        private static bool IsPatientLastNameOrPatientSsnSimilar(List<string> similarities, BedInfo bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool exists = false;

                if (similarities != null && similarities.Count > 0)
                {
                    foreach (string item in similarities)
                    {
                        if ((bed.PatientLastName.Length > 3 && item == bed.PatientLastName.ToLower().Substring(0, 4)) || (bed.SSN.Length > 3 && item == bed.SSN.ToLower().Substring(bed.SSN.Length - 4, 4)))
                        {
                            exists = true;
                            break;
                        }
                    }
                }

                return exists;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IconInfo SimilarNameIcon { get; set; }

        public string SimilarNameIconBytes { get; set; }

        public string BedIdRoot { get; set; }

        public string BedIdExtension { get; set; }

        public string BedName { get; set; }

        public string PatientIdRoot { get; set; }

        public string PatientIdExtension { get; set; }

        public string PatientLastName { get; set; }

        public string GenderColor { get; set; }

        public string DisplayPatientGender { get; set; }

        public string PT { get; set; }

        public string LosAdmin { get; set; }

        public string LosDrg { get; set; }

        public string LosWard { get; set; }

        public string WardName { get; set; }

        public string EditLinkString { get; set; }

        public string Reason { get; set; }

        public IconInfo DischargeIcon { get; set; }

        public string DischargeIconBytes { get; set; }

        public IconInfo BedAvailabilityStatusIcon { get; set; }

        public string BedAvailabilityStatusIconBytes { get; set; }

        public IconInfo BedCleaningStatusIcon { get; set; }

        public string BedCleaningStatusIconBytes { get; set; }

        public IconInfo LevelOfCareIcon { get; set; }

        public string LevelOfCareIconBytes { get; set; }

        public IList<IconInfo> IconFlagsList { get; set; }

        public byte[] CombineIconBinaryList { get; set; }

        public bool HasMultipleComments { get; set; }

        public bool HasMultipleDivisionsAssigned { get; set; }

        public IconInfo BedHoldIcon { get; set; }

        public string BedHoldIconBytes { get; set; }

        public bool IsBedHold { get; set; }

        public string LevelOfCareDays { get; set; }

        public string LevelOfCareChangeDays { get; set; }
    }
}