﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Facade;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using BMS.Facade.Service;
using System.ComponentModel.DataAnnotations;
using BMS.Web.App_GlobalResource;
using BMS.Web.Controllers;
using BMS.Utils;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Models
{
    public class PatientViewModel : BaseViewModel
    {
        /// <summary>
        /// Gets or sets the patient extension.
        /// </summary>
        /// <value>
        /// The patient extension.
        /// </value>
        public string PatientExtension { get; set; }

        /// <summary>
        /// Gets or sets the patient root.
        /// </summary>
        /// <value>
        /// The patient root.
        /// </value>
        public string PatientRoot { get; set; }

        /// <summary>
        /// Gets or sets the patient SSN.
        /// </summary>
        /// <value>
        /// The patient SSN.
        /// </value>
        public string PatientSSN { get; set; }

        /// <summary>
        /// Gets or sets the patient gender.
        /// </summary>
        /// <value>
        /// The patient gender.
        /// </value>
        public string PatientGender { get; set; }

        /// <summary>
        /// Gets or sets the id act.
        /// </summary>
        /// <value>
        /// The id act.
        /// </value>
        public string IdAct { get; set; }

        /// <summary>
        /// Gets or sets the full name of the patient.
        /// </summary>
        /// <value>
        /// The full name of the patient.
        /// </value>
        public string PatientFullName { get; set; }
        /// <summary>
        /// Gets or sets the entry date time.
        /// </summary>
        /// <value>
        /// The entry date time.
        /// </value>
        public DateTime EntryDateTime { get; set; }

        /// <summary>
        /// Gets or sets the date time bed requested.
        /// </summary>
        /// <value>
        /// The date time bed requested.
        /// </value>
        public DateTime DateTimeBedRequested { get; set; }
        public string DateTimeBedRequestedString
        {
            get { return DateTimeBedRequested.ToString(Strings.USDate); }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        DateTimeBedRequested = DateTime.Parse(value);
                    }
                    catch
                    {
                        DateTimeBedRequested = BaseController.DateTimeNowOnCurrentFacility;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the present problem.
        /// </summary>
        /// <value>
        /// The present problem.
        /// </value>
        [ValidateXss]
        public string PresentProblem { get; set; }
        /// <summary>
        /// Gets or sets the type of bed.
        /// </summary>
        /// <value>
        /// The type of bed.
        /// </value>
        [ValidateXss]
        public string TypeOfBed { get; set; }
        /// <summary>
        /// Gets or sets the waiting area.
        /// </summary>
        /// <value>
        /// The waiting area.
        /// </value>
        public string WaitingAreaCode { get; set; }
        
        /// <summary>
        /// Gets or sets the display name of the waiting area.
        /// </summary>
        /// <value>
        /// The display name of the waiting area.
        /// </value>
        public string WaitingAreaDisplayName { get; set; }
        /// <summary>
        /// Gets or sets the disposition.
        /// </summary>
        /// <value>
        /// The disposition.
        /// </value>
        public string DispositionCode { get; set; }
        /// <summary>
        /// Gets or sets the contract.
        /// </summary>
        /// <value>
        /// The contract.
        /// </value>
        public string ContractCode { get; set; }
        /// <summary>
        /// Gets or sets the authorized.
        /// </summary>
        /// <value>
        /// The authorized.
        /// </value>
        public string AuthorizedCode { get; set; }
        /// <summary>
        /// Gets or sets the vista specialty code.
        /// </summary>
        /// <value>
        /// The vista specialty code.
        /// </value>
        public string VistaSpecialtyCode { get; set; }

        public string FeeAccepted { get; set; }
        
        /// <summary>
        /// Gets or sets the reasons using fee.
        /// </summary>
        /// <value>
        /// The reasons using fee.
        /// </value>
        public string ReasonsUsingFeeCode { get; set; }

        public string FeeAcceptingMD { get; set; }

        /// <summary>
        /// Gets or sets the fee comments.
        /// </summary>
        /// <value>
        /// The fee comments.
        /// </value>
        [StringLength(Constants.COMMENTS_MAX_LENGTH, ErrorMessageResourceType = typeof(Strings), ErrorMessageResourceName = "FeeCommentsLengthError")]
        [ValidateXss]
        public string FeeComments { get; set; }
        /// <summary>
        /// Gets or sets the waiting area list.
        /// </summary>
        /// <value>
        /// The waiting area list.
        /// </value>
        public CDList WaitingAreaList { get; set; }
        /// <summary>
        /// Gets or sets the disposition list.
        /// </summary>
        /// <value>
        /// The disposition list.
        /// </value>
        public CDList DispositionList { get; set; }
        /// <summary>
        /// Gets or sets the contract list.
        /// </summary>
        /// <value>
        /// The contract list.
        /// </value>
        public CDList ContractList { get; set; }
        /// <summary>
        /// Gets or sets the authorization list.
        /// </summary>
        /// <value>
        /// The authorization list.
        /// </value>
        public CDList AuthorizationList { get; set; }
        /// <summary>
        /// Gets or sets the vista specialities list.
        /// Property was renamed from ServiceReceivedList to VistaSpecialitiesList.
        /// </summary>
        /// <value>
        /// The vista specialities list.
        /// </value>
        public CDList VistaSpecialitiesList { get; set; }
        /// <summary>
        /// Gets or sets the fee list.
        /// </summary>
        /// <value>
        /// The fee list.
        /// </value>
        public CDList FeeList { get; set; }

        /// <summary>
        /// Gets or sets the hour selected.
        /// </summary>
        /// <value>
        /// The hour selected.
        /// </value>
        public string RequestHourSelected { get; set; }

        /// <summary>
        /// Gets or sets the minute selected.
        /// </summary>
        /// <value>
        /// The minute selected.
        /// </value>
        public string RequestMinuteSelected { get; set; }

        /// <summary>
        /// Gets or sets the room bed assigned.
        /// </summary>
        /// <value>
        /// The room bed assigned.
        /// </value>
        public string RoomBedAssigned { get; set; }

        /// <summary>
        /// Gets or sets the bed assignment date.
        /// </summary>
        /// <value>
        /// The bed assignment date.
        /// </value>
        public DateTime AssignedDate { get; set; }

        /// <summary>
        /// Gets or sets the assignment hour selected.
        /// </summary>
        /// <value>
        /// The hour selected.
        /// </value>
        public string AssignmentHourSelected { get; set; }

        /// <summary>
        /// Gets or sets the assignment minute selected.
        /// </summary>
        /// <value>
        /// The minute selected.
        /// </value>
        public string AssignmentMinuteSelected { get; set; }

        /// <summary>
        /// Gets or sets the date time bed assigned.
        /// </summary>
        /// <value>
        /// The date time bed assigned.
        /// </value>
        public DateTime? DateTimeBedAssigned { get; set; }
        public string DateTimeBedAssignedString
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (DateTimeBedAssigned == null)
                        return string.Empty;
                    else
                        return DateTimeBedAssigned.Value.ToString(Strings.USDate);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        if (string.IsNullOrEmpty(value))
                            DateTimeBedAssigned = null;
                        else
                            DateTimeBedAssigned = DateTime.Parse(value);
                    }
                    catch
                    {
                        DateTimeBedAssigned = BaseController.DateTimeNowOnCurrentFacility;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the selected bed id (room/bed assigned).
        /// </summary>
        /// <value>
        /// The selected bed id.
        /// </value>
        public string SelectedBedId { get; set; }
        /// <summary>
        /// Gets or sets the bed list.
        /// </summary>
        /// <value>
        /// The bed list.
        /// </value>
        public IList<Bed> BedList { get; set; }

        /// <summary>
        /// Gets or sets the bed assignment display.
        /// </summary>
        /// <value>
        /// The bed assignment display.
        /// </value>
        public string BedAssignmentDisplay { get; set; }

        /// <summary>
        /// Gets or sets the removed date.
        /// </summary>
        /// <value>
        /// The removed date.
        /// </value>
        public DateTime? RemovedDate { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is in house.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is in house; otherwise, <c>false</c>.
        /// </value>
        public bool IsInHouse { get; set; }

        /// <summary>
        /// Gets or sets the comments.
        /// </summary>
        /// <value>
        /// The comments.
        /// </value>
        [StringLength(Constants.COMMENTS_MAX_LENGTH, ErrorMessageResourceType = typeof(Strings), ErrorMessageResourceName = "FacilityDiversionErrorCommentsBad")]
        [ValidateXss]
        public string Comments { get; set; }

        /// <summary>
        /// Gets or sets the selected ward group (BMS Ward Assigned drop-down).
        /// </summary>
        /// <value>
        /// The selected ward group.
        /// </value>
        public string SelectedWardGroup { get; set; }

        /// <summary>
        /// Gets or sets the ward group list.
        /// </summary>
        /// <value>
        /// The ward group list.
        /// </value>
        public IList<Division> WardGroupList { get; set; }

        /// <summary>
        /// Gets or sets the selected division.
        /// </summary>
        /// <value>
        /// The selected division.
        /// </value>
        public Division SelectedDivision { get; set; }

        /// <summary>
        /// Gets or sets the bed list display.
        /// </summary>
        /// <value>
        /// The bed list display.
        /// </value>
        public string BedListDisplay { get; set; }

        /// <summary>
        /// Gets or sets the refresh page.
        /// </summary>
        /// <value>
        /// The refresh page.
        /// </value>
        public string RefreshPage { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether bed is in isolation or not.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if bed is in isolation; otherwise, <c>false</c>.
        /// </value>
        public bool IsBedInIsolation { get; set; }

        /// <summary>
        /// Gets or sets the wait time alert date.
        /// </summary>
        /// <value>
        /// The wait time alert date.
        /// </value>
        public DateTime? WaitTimeAlertDate { get; set; }

        /// <summary>
        /// Gets or sets the wait time alert int.
        /// </summary>
        /// <value>
        /// The wait time alert int.
        /// </value>
        [Range(0, 99, ErrorMessage = "Wait Time Alert must be between 0-99")]
        [Display(Name = " ")]
        public int? WaitTimeAlertInt
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    if (WaitTimeAlertDate == null)
                        return null;
                    else
                    {
                        int remainingWaitTimeAlert = RemovedDate == null ? (int)(WaitTimeAlertDate.Value - BaseController.DateTimeNowOnCurrentFacility).TotalHours : (int)(RemovedDate.Value - BaseController.DateTimeNowOnCurrentFacility).TotalHours;
                        if (remainingWaitTimeAlert <= 0)
                            return 0;
                        else
                            return remainingWaitTimeAlert;
                    }

                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        if (value == null)
                            WaitTimeAlertDate = null;
                        else
                        {
                            DateTime dateFacility = BaseController.DateTimeNowOnCurrentFacility.AddHours(1);
                            WaitTimeAlertDate = dateFacility.AddHours(value.Value);
                            int test = (int)(WaitTimeAlertDate.Value - dateFacility).TotalHours;
                        }
                    }
                    catch
                    {
                        WaitTimeAlertDate = null;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is CLC patient.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is CLC patient; otherwise, <c>false</c>.
        /// </value>
        public bool IsCLCPatient { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is evacuation patient.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is evacuation patient; otherwise, <c>false</c>.
        /// </value>
        public bool IsEvacuationPatient { get; set; }

        public bool IsMentalHealthPatient { get; set; }

        /// <summary>
        /// Gets or sets the evac disposition status list.
        /// </summary>
        /// <value>
        /// The evac disposition status list.
        /// </value>
        public CDList EvacDispositionStatusList { get; set; }

        /// <summary>
        /// Gets or sets the selected evac disposition status code.
        /// </summary>
        /// <value>
        /// The selected evac disposition status code.
        /// </value>
        public string SelectedEvacDispositionStatusCode { get; set; }

        /// <summary>
        /// Gets or sets the evac expected date.
        /// </summary>
        /// <value>
        /// The evac expected date.
        /// </value>
        public DateTime? EvacExpectedDate { get; set; }

        /// <summary>
        /// Gets or sets the evac expected date string.
        /// </summary>
        /// <value>
        /// The evac expected date string.
        /// </value>
        public string EvacExpectedDateString
        {
            get
            {
                if (EvacExpectedDate.HasValue)
                    return EvacExpectedDate.Value.ToString(Strings.USDate);
                else
                    return string.Empty;
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        EvacExpectedDate = DateTime.Parse(value);
                    }
                    catch
                    {
                        EvacExpectedDate = null;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the evac expected date hour selected.
        /// </summary>
        /// <value>
        /// The evac expected date hour selected.
        /// </value>
        public string EvacExpectedDateHourSelected { get; set; }

        /// <summary>
        /// Gets or sets the evac expected date minute selected.
        /// </summary>
        /// <value>
        /// The evac expected date minute selected.
        /// </value>
        public string EvacExpectedDateMinuteSelected { get; set; }

        /// <summary>
        /// Gets or sets the evac actual date.
        /// </summary>
        /// <value>
        /// The evac actual date.
        /// </value>
        public DateTime? EvacActualDate { get; set; }

        /// <summary>
        /// Gets or sets the evac actual date string.
        /// </summary>
        /// <value>
        /// The evac actual date string.
        /// </value>
        public string EvacActualDateString
        {
            get
            {
                if (EvacActualDate.HasValue)
                    return EvacActualDate.Value.ToString(Strings.USDate);
                else
                    return string.Empty;
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        EvacActualDate = DateTime.Parse(value);
                    }
                    catch
                    {
                        EvacActualDate = null;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the evac actual date hour selected.
        /// </summary>
        /// <value>
        /// The evac actual date hour selected.
        /// </value>
        public string EvacActualDateHourSelected { get; set; }

        /// <summary>
        /// Gets or sets the evac actual date minute selected.
        /// </summary>
        /// <value>
        /// The evac actual date minute selected.
        /// </value>
        public string EvacActualDateMinuteSelected { get; set; }

        /// <summary>
        /// Gets or sets the evac reason for delay.
        /// </summary>
        /// <value>
        /// The evac reason for delay.
        /// </value>
        [ValidateXss]
        public string EvacReasonForDelay { get; set; }

        /// <summary>
        /// Gets or sets the evac transfer evacuate to.
        /// </summary>
        /// <value>
        /// The evac transfer evacuate to.
        /// </value>
        [ValidateXss]
        public string EvacTransferEvacuateTo { get; set; }

        /// <summary>
        /// Gets or sets the evac equip supply requirements.
        /// </summary>
        /// <value>
        /// The evac equip supply requirements.
        /// </value>
        [ValidateXss]
        public string EvacEquipSupplyRequirements { get; set; }

        /// <summary>
        /// Gets or sets the evac transportation care level.
        /// </summary>
        /// <value>
        /// The evac transportation care level.
        /// </value>
        [ValidateXss]
        public string EvacTransportationCareLevel { get; set; }

        /// <summary>
        /// Gets or sets the evac enroute medical attendant.
        /// </summary>
        /// <value>
        /// The evac enroute medical attendant.
        /// </value>
        [ValidateXss]
        public string EvacEnrouteMedicalAttendant { get; set; }

        /// <summary>
        /// Gets or sets the evac enroute non medical attendant.
        /// </summary>
        /// <value>
        /// The evac enroute non medical attendant.
        /// </value>
        [ValidateXss]
        public string EvacEnrouteNonMedicalAttendant { get; set; }

        /// <summary>
        /// Gets or sets the evac transportation type list.
        /// </summary>
        /// <value>
        /// The evac transportation type list.
        /// </value>
        public CDList EvacTransportationTypeList { get; set; }

        /// <summary>
        /// Gets or sets the selected evac transportation type code.
        /// </summary>
        /// <value>
        /// The selected evac transportation type code.
        /// </value>
        public string SelectedEvacTransportationTypeCode { get; set; }

        /// <summary>
        /// Gets or sets the evac transportation provider list.
        /// </summary>
        /// <value>
        /// The evac transportation provider list.
        /// </value>
        public CDList EvacTransportationProviderList { get; set; }

        /// <summary>
        /// Gets or sets the selected evac transportation provider code.
        /// </summary>
        /// <value>
        /// The selected evac transportation provider code.
        /// </value>
        public string SelectedEvacTransportationProviderCode { get; set; }

        /// <summary>
        /// Gets or sets the evac transportation comments.
        /// </summary>
        /// <value>
        /// The evac transportation comments.
        /// </value>
        [ValidateXss]
        public string EvacTransportationComments { get; set; }

        /// <summary>
        /// Gets or sets the evac destination address.
        /// </summary>
        /// <value>
        /// The evac destination address.
        /// </value>
        [ValidateXss]
        public string EvacDestinationAddress { get; set; }

        /// <summary>
        /// Gets or sets the state of the evac destination city.
        /// </summary>
        /// <value>
        /// The state of the evac destination city.
        /// </value>
        [ValidateXss]
        public string EvacDestinationCityState { get; set; }

        /// <summary>
        /// Gets or sets the evac destination POC telephone.
        /// </summary>
        /// <value>
        /// The evac destination POC telephone.
        /// </value>
        [ValidateXss]
        public string EvacDestinationPOCTelephone { get; set; }

        /// <summary>
        /// Gets or sets the evac originator POC telephone.
        /// </summary>
        /// <value>
        /// The evac originator POC telephone.
        /// </value>
        [ValidateXss]
        public string EvacOriginatorPOCTelephone { get; set; }

        /// <summary>
        /// Gets or sets the evac comments.
        /// </summary>
        /// <value>
        /// The evac comments.
        /// </value>
        [ValidateXss]
        public string EvacComments { get; set; }

        /// <summary>
        /// Gets or sets the evac enroute non medical attendant number.
        /// </summary>
        /// <value>
        /// The evac enroute non medical attendant number.
        /// </value>
        [Range(0, 99, ErrorMessage = "Please enter a number between 0-99")]
        [Display(Name = " ")]
        public int? EvacEnrouteNonMedicalAttendantNumber { get; set; }

        /// <summary>
        /// Gets or sets the evac enroute medical attendant number.
        /// </summary>
        /// <value>
        /// The evac enroute medical attendant number.
        /// </value>
        [Range(0, 99, ErrorMessage = "Please enter a number between 0-99")]
        [Display(Name = " ")]
        public int? EvacEnrouteMedicalAttendantNumber { get; set; }

        /// <summary>
        /// Gets or sets the evacuation arrival time at the destination.
        /// </summary>
        /// <value>
        /// The evacuation arrival time.
        /// </value>
        public DateTime? EvacArrivalTime { get; set; }

        /// <summary>
        /// Gets or sets the evacuation arrival time at the destination as a string.
        /// </summary>
        /// <value>
        /// The evacuation arrival time.
        /// </value>
        public string EvacArrivalTimeString
        {
            get
            {
                if (EvacArrivalTime.HasValue)
                    return EvacArrivalTime.Value.ToString(Strings.USDate);
                else
                    return string.Empty;
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        EvacArrivalTime = DateTime.Parse(value);
                    }
                    catch
                    {
                        EvacArrivalTime = null;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the evac arrival time hour selected.
        /// </summary>
        /// <value>
        /// The evac arrival time hour selected.
        /// </value>
        public string EvacArrivalTimeHourSelected { get; set; }

        /// <summary>
        /// Gets or sets the evac arrival time minute selected.
        /// </summary>
        /// <value>
        /// The evac arrival time minute selected.
        /// </value>
        public string EvacArrivalTimeMinuteSelected { get; set; }

        /// <summary>
        /// Gets or sets the evacuation vehicle's ID.
        /// </summary>
        /// <value>
        /// The evacuation vehicle ID.
        /// </value>
        [ValidateXss]
        public string EvacVehicleID { get; set; }

        /// <summary>
        /// Gets or sets the evacuation vehicle seat position of the evacuee.
        /// </summary>
        /// <value>
        /// The evacuee's seat position.
        /// </value>
        [ValidateXss]
        public string EvacVehicleSeatPosition { get; set; }

        /// <summary>
        /// Gets or sets the evacuation medical attendant's telephone.
        /// </summary>
        /// <value>
        /// The evacuation medical attendant's telephone.
        /// </value>
        [ValidateXss]
        public string EvacMATelephone { get; set; }

        /// <summary>
        /// Gets or sets the evacuation non-medical attendant's telephone.
        /// </summary>
        /// <value>
        /// The evacuation non-medical attendant's telephone.
        /// </value>
        [ValidateXss]
        public string EvacNMATelephone { get; set; }

        /// <summary>
        /// Gets or sets the evacuation originating facility name.
        /// </summary>
        /// <value>
        /// The evacuation originating facility name.
        /// </value>
        [ValidateXss]
        public string EvacOriginatingFacilityName { get; set; }

        /// <summary>
        /// Gets or sets the evacuation originating address.
        /// </summary>
        /// <value>
        /// The evacuation originating address.
        /// </value>
        [ValidateXss]
        public string EvacOriginatingAddress { get; set; }

        /// <summary>
        /// Gets or sets the state of the evacuation originating city.
        /// </summary>
        /// <value>
        /// The state of the evacuation originating city.
        /// </value>
        [ValidateXss]
        public string EvacOriginatingCityState { get; set; }

        /// <summary>
        /// Gets or sets the state of the evacuation originating point of contact.
        /// </summary>
        /// <value>
        /// The state of the evacuation originating point of contact.
        /// </value>
        [ValidateXss]
        public string EvacOriginatingPOC { get; set; }

        /// <summary>
        /// Gets or sets the state of the evacuation vehicle operator name.
        /// </summary>
        /// <value>
        /// The state of the evacuation vehicle operator name.
        /// </value>
        [ValidateXss]
        public string EvacVehicleOperatorName { get; set; }

        /// <summary>
        /// Gets or sets the evacuation vehicle operator's telephone.
        /// </summary>
        /// <value>
        /// The evacuation vehicle operator's telephone.
        /// </value>
        [ValidateXss]
        public string EvacVehicleOperatorTelephone { get; set; }

        /// <summary>
        /// Gets or sets the current ward.
        /// </summary>
        /// <value>
        /// The current ward.
        /// </value>
        public string CurrentWard { get; set; }

        /// <summary>
        /// Gets or sets the current bed.
        /// </summary>
        /// <value>
        /// The current bed.
        /// </value>
        public string CurrentBed { get; set; }

        /// <summary>
        /// Gets or sets the admitting diagnosis.
        /// </summary>
        /// <value>
        /// The admitting diagnosis.
        /// </value>
        [ValidateXss]
        public string AdmittingDiagnosis { get; set; }

        /// <summary>
        /// Gets or sets the facility site code time zone.
        /// </summary>
        /// <value>
        /// The facility site code time zone.
        /// </value>
        public string FacilitySiteCodeTimeZone { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is from evacuation report.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is from evacuation report; otherwise, <c>false</c>.
        /// </value>
        public bool IsFromEvacuationReport { get; set; }

        /// <summary>
        /// Gets or sets the report URL.
        /// </summary>
        /// <value>
        /// The report URL.
        /// </value>
        public string EvacuationReportUrl { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is from home page.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is from home page; otherwise, <c>false</c>.
        /// </value>
        public bool IsFromHomePage { get; set; }

        public bool IsWaitingAreaAvailable { get; set; }

        public DateTime? DateOfScreening { get; set; }

        public string DateOfScreeningString
        {
            get
            {
                if (DateOfScreening.HasValue)
                    return DateOfScreening.Value.ToString(Strings.USDate);
                else
                    return string.Empty;
            }
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    try
                    {
                        DateOfScreening = DateTime.Parse(value);
                    }
                    catch
                    {
                        DateOfScreening = null;
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        public string DateOfScreeningHourSelected { get; set; }
        public string DateOfScreeningMinuteSelected { get; set; }
        public string ContactPhoneNumber { get; set; }
        public string VeteranPreference { get; set; }
        public string MentalHealthComments { get; set; }

        public bool IsFee { get; set; }
        public bool IsCommunityHospital { get; set; }
        public string TransferCoordinator { get; set; }

        [RegularExpression(@"^\(?([0-9]{3})\)?[-. ]?([0-9]{3})[-. ]?([0-9]{4})$", ErrorMessage = "Phone number is not valid.")]
        public string TransferCoordinatorPhone { get; set; }
    }

    public class PatientSelectViewModel : BaseViewModel
    {
        /// <summary>
        /// Gets or sets the patient search text.
        /// </summary>
        /// <value>
        /// The patient search text.
        /// </value>
        [Required(ErrorMessage = "Please enter Last Name, or full SSN, or Last Initial & Last 4 SSN!")]
        [RegularExpression(Constants.PATIENT_SEARCH_REGULAR_EXPRESSION, ErrorMessageResourceType = typeof(Strings), ErrorMessageResourceName = "SearchStringToShort")]
        [ValidateXss]
        public string PatientSearchText { get; set; }

        /// <summary>
        /// Gets or sets the patient list.
        /// </summary>
        /// <value>
        /// The patient list.
        /// </value>
        public List<Patient> PatientList { get; set; }

        /// <summary>
        /// Gets or sets the selected patient extension.
        /// </summary>
        /// <value>
        /// The selected patient extension.
        /// </value>
        public string SelectedPatientExtension { get; set; }

        /// <summary>
        /// Gets or sets the patient not found.
        /// </summary>
        /// <value>
        /// The patient not found.
        /// </value>
        public string PatientNotFound { get; set; }


    }
}
