﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using System.Web.Mvc.Html;
using System.Web.UI;
using System.Web.UI.WebControls;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Models;
using DevExpress.Utils;
using DevExpress.Web;
using DevExpress.Web.Mvc;
using DevExpress.Web.Mvc.UI;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;

namespace BMS.Web.Controllers.Shared
{
    public static class GridHelper
    {
        public static void BuildComboColumn(MVCxGridViewColumn column, string fieldName, string caption, object dataSource, string dataField, int width = 90)
        {
            BuildColumn(column, fieldName, caption, width: 90);

            column.SettingsHeaderFilter.Mode = GridHeaderFilterMode.CheckedList;
            column.Settings.AllowHeaderFilter = DefaultBoolean.True;
            //var comboBoxProperties = column.PropertiesEdit as ComboBoxProperties;
            //comboBoxProperties.DataSource = dataSource;
            //comboBoxProperties.TextField = dataField;
            //comboBoxProperties.ValueField = dataField;
            //comboBoxProperties.ValueType = typeof(string);
        }

        public static void BuildCheckBoxColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)
        {
            BuildColumn(column, fieldName, caption, MVCxGridViewColumnType.CheckBox, width);
        }

        public static void BuildHtmlEncodedColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)  // RRK
        {
            BuildColumn(column, fieldName, caption, width);
            column.PropertiesEdit.EncodeHtml = false;   // Don't re-encode HTML-encoded data...
        }

        public static void BuildHtmlEncodedColumn(MVCxGridViewColumn column, string fieldName, int width = 50) // RRK
        {
            BuildColumn(column, fieldName, width);
            column.PropertiesEdit.EncodeHtml = false;   // Don't re-encode HTML-encoded data...
        }

        public static void BuildColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)
        {
            column.FieldName = fieldName;
            column.Caption = caption;
            column.Width = Unit.Pixel(width);
        }

        public static void BuildColumn(MVCxGridViewColumn column, string fieldName, int width = 50)
        {
            column.FieldName = fieldName;
            column.Width = Unit.Pixel(width);
            column.Settings.AutoFilterCondition = AutoFilterCondition.Contains;
        }

        public static void BuildColumn(MVCxGridViewColumn column, string fieldName, MVCxGridViewColumnType columnType, int width = 50)
        {
            BuildColumn(column, fieldName, width);
            column.ColumnType = columnType;
        }

        public static void BuildColumn(MVCxGridViewColumn column, string fieldName, string caption, MVCxGridViewColumnType columnType, int width = 50)
        {
            BuildColumn(column, fieldName, caption, width);
            column.ColumnType = columnType;
        }

        public static void BuildColumnSimple(MVCxGridViewColumn column, string fieldName, string caption)
        {
            column.Caption = caption;
            BuildColumnSimple(column, fieldName);
        }

        public static void BuildColumnSimple(MVCxGridViewColumn column, string fieldName)
        {
            column.FieldName = fieldName;
            column.Settings.AutoFilterCondition = AutoFilterCondition.Contains;
        }

        public static void BuildWaitTimeColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)
        {
            BuildColumn(column, fieldName, caption, MVCxGridViewColumnType.TimeEdit, width);
            column.Settings.SortMode = DevExpress.XtraGrid.ColumnSortMode.Custom;
        }

        public static void BuildDateTimeColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)
        {
            BuildColumn(column, fieldName, caption, MVCxGridViewColumnType.DateEdit, width);
            column.PropertiesEdit.DisplayFormatString = "MM/dd/yyyy hh:mm tt";
        }

        public static void BuildDateColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 50)
        {
            BuildColumn(column, fieldName, caption, MVCxGridViewColumnType.DateEdit, width);
            column.PropertiesEdit.DisplayFormatString = "MM/dd/yyyy";
        }

        public static void BuildBinaryImageColumn(MVCxGridViewColumn column, string fieldName, string caption, int width = 15, int height = 15)
        {
            BuildColumn(column, fieldName, caption, MVCxGridViewColumnType.BinaryImage, width);

            var properties = column.PropertiesEdit as MVCxBinaryImageEditProperties;
            if (properties != null)
            {
                properties.ImageHeight = height;
                properties.ImageWidth = width;
            }
        }

        public static void BuildBinaryImageSettings(object keyValue, string iconName, string mouseOverText, byte[] imageContent, BinaryImageEditSettings imgSettings, int width = 15, int height = 15)
        {
            if (imageContent != null)
            {
                imgSettings.Name = iconName + keyValue;
                imgSettings.ToolTip = mouseOverText;
                imgSettings.Properties.AlternateText = iconName;
                imgSettings.Width = Unit.Pixel(width);
                imgSettings.Height = Unit.Pixel(height);
                imgSettings.ContentBytes = imageContent;
            }
        }

        public static void BuildTextboxTemplate(MVCxFormLayoutItem itemSettings, string fieldName, string caption = null)
        {
            itemSettings.FieldName = fieldName;
            if (!String.IsNullOrEmpty(caption))
                itemSettings.Caption = caption;
            itemSettings.NestedExtensionType = FormLayoutNestedExtensionItemType.TextBox;
            TextBoxSettings textBoxSettings = (TextBoxSettings)itemSettings.NestedExtensionSettings;
            textBoxSettings.Properties.ValidationSettings.ErrorDisplayMode = ErrorDisplayMode.ImageWithTooltip;
            textBoxSettings.ShowModelErrors = true;
            textBoxSettings.Width = Unit.Percentage(100);
        }

        public static void BuildMemoTemplate(MVCxFormLayoutItem itemSettings, string fieldName, string caption = null, int height = 0)
        {
            itemSettings.FieldName = fieldName;

            if (!String.IsNullOrEmpty(caption))
                itemSettings.Caption = caption;

            itemSettings.NestedExtensionType = FormLayoutNestedExtensionItemType.Memo;
            MemoSettings textBoxSettings = (MemoSettings)itemSettings.NestedExtensionSettings;
            textBoxSettings.Properties.ValidationSettings.ErrorDisplayMode = ErrorDisplayMode.ImageWithTooltip;
            textBoxSettings.ShowModelErrors = true;
            textBoxSettings.Width = Unit.Percentage(100);

            if (height == 0)
                height = 100;

            textBoxSettings.Height = Unit.Pixel(height);
        }

        private static GridViewSettings ConfigurePPBPGrid(HtmlHelper htmlHelper, string controllerName, string actionName, string idField, string salt, string encodedParams = null)
        {
            GridViewSettings settings = new GridViewSettings();
            settings.Name = "grid";

            if (!String.IsNullOrWhiteSpace(encodedParams))
                settings.CallbackRouteValues = new { Controller = controllerName, Action = actionName, p = encodedParams };
            else
                settings.CallbackRouteValues = new { Controller = controllerName, Action = actionName };

            settings.SettingsEditing.Mode = GridViewEditingMode.EditFormAndDisplayRow;
            settings.SettingsBehavior.ConfirmDelete = true;
            settings.SettingsBehavior.AutoExpandAllGroups = true;

            settings.SettingsBehavior.EnableCustomizationWindow = true;
            settings.SettingsText.CustomizationWindowCaption = "Hidden Columns";
            settings.SettingsText.EmptyDataRow = " ";

            settings.CommandColumn.Visible = false;

            settings.SettingsPager.PageSize = 20;
            settings.SettingsPager.PageSizeItemSettings.Visible = true;
            settings.SettingsPager.Visible = true;

            settings.Width = new System.Web.UI.WebControls.Unit("100%");
            settings.Styles.Cell.Font.Size = System.Web.UI.WebControls.FontUnit.Smaller;
            settings.Styles.Cell.VerticalAlign = System.Web.UI.WebControls.VerticalAlign.Middle;
            settings.Styles.Header.Wrap = DefaultBoolean.True;

            settings.Styles.FocusedRow.CssClass = "gridRow-static-height";
            settings.Styles.Row.CssClass = "gridRow-static-height";
            settings.Styles.SelectedRow.CssClass = "gridRow-static-height";
            settings.Styles.GroupPanel.ForeColor = System.Drawing.Color.Black;

            settings.KeyFieldName = idField;

            settings.SettingsPager.Visible = true;
            settings.Settings.ShowGroupPanel = true;
            settings.Settings.ShowFilterRow = true;
            settings.SettingsBehavior.AllowSelectByRowClick = true;

            settings.SettingsAdaptivity.AdaptivityMode = GridViewAdaptivityMode.Off;
            settings.SettingsAdaptivity.AdaptiveColumnPosition = GridViewAdaptiveColumnPosition.Right;
            settings.SettingsAdaptivity.AdaptiveDetailColumnCount = 1;
            settings.SettingsAdaptivity.AllowOnlyOneAdaptiveDetailExpanded = false;
            settings.SettingsAdaptivity.HideDataCellsAtWindowInnerWidth = 0;

            settings.SettingsExport.Landscape = true;
            settings.SettingsExport.LeftMargin = 0;
            settings.SettingsExport.RightMargin = 0;
            settings.SettingsExport.PageHeader.Center = "Export Title";

            if (IconInfo.SameSimilarNameIcon == null)
            {
                IconInfo similarNameIcon = FacadeManager.IconInterface.GetIconByCode(Constants.SIMILAR_SAME_NAME);
                IconInfo.SameSimilarNameIcon = Convert.FromBase64String(similarNameIcon.ImageBytes);
            }

            settings.AccessibilityCompliant = true;

            settings.CustomColumnSort = (sender, e) =>
            {
                if (e.Column.FieldName == "WaitTimeDisplay")
                {
                    string v1 = e.Value1.ToString();
                    string v2 = e.Value2.ToString();
                    
                    // -1 => v1  > v2
                    //  0 => v1 == v2
                    //  1 => v1  < v2

                    string[] v1Vals = v1.Split(':');
                    string[] v2Vals = v2.Split(':');

                    int v1Hours = Convert.ToInt32(v1[0]);
                    int v2Hours = Convert.ToInt32(v2[0]);

                    if (v1Hours > v2Hours)
                        e.Result = -1;
                    else if (v1Hours < v2Hours)
                        e.Result = 1;
                    else // hours are equal
                    {
                        int v1Mins = Convert.ToInt32(v1[1]);
                        int v2Mins = Convert.ToInt32(v2[1]);

                        if (v1Mins > v2Mins)
                            e.Result = -1;
                        else if (v1Mins < v2Mins)
                            e.Result = 1;
                        else // hours and minutes are equal
                            e.Result = 0;
                    }
                }
            };

            return settings;
        }

        private static GridViewSettings ConfigureFacilityPPBPGrid(HtmlHelper htmlHelper, string actionName, HomeViewModel model)
        {
            bool isReset = model.FilterOperation == "Reset";
            string encodedParameters = BaseController.EncryptQueryString(new string[] { "filter", "view", "lastView", "resetGrid" }, new string[] { model.FilterOperation == null || isReset ? model.CurrentFilter : model.FilterOperation, model.View2, model.LastSelectedView, isReset ? "True" : "False" }, model.LoggedUser.Salt);
            GridViewSettings settings = ConfigurePPBPGrid(htmlHelper, "Home", actionName, "IdAct", model.LoggedUser.Salt, encodedParameters);

            if (IconInfo.BedInIsolationIcon == null)
            {
                IconInfo isolationIcon = FacadeManager.IconInterface.GetIconByCode(Constants.BED_ISOLATION);
                IconInfo.BedInIsolationIcon = Convert.FromBase64String(isolationIcon.ImageBytes);
            }

            settings.Columns.Add(column =>
            {
                column.Name = "Actions";
                column.Caption = "Actions";
                column.Width = System.Web.UI.WebControls.Unit.Pixel(60);

                column.SetDataItemTemplateContent(content =>
                {
                    string idValue = (string)content.KeyValue;
                    string editEncodedParams = BaseController.EncryptQueryString(new string[] { "idAdmission", "isFromEvacuationReport", "selectedDivisionName", "selectedDivisionId", "isRosterReport", "isRegulateReport", "isManifestReport" }, new string[] { idValue, "false", string.Empty, "-1", "true", "false", "false" }, model.LoggedUser.Salt);
                    string removeEncodedParams = BaseController.EncryptQueryString(new string[] { "idAdmission" }, new string[] { idValue }, model.LoggedUser.Salt);

                    htmlHelper.ViewContext.Writer.Write(htmlHelper.ActionLink("Edit", "Admission", "Patient", new { p = editEncodedParams }, new { @class = "smlink", style = "padding-right: 5px" }));
                    htmlHelper.ViewContext.Writer.Write(htmlHelper.ActionLink("Remove", "RemoveAdmission", "Home", new { p = removeEncodedParams }, new { @class = "smlink" }));

                    if (!model.CurrentViewSelected)
                    {
                        AdmissionInfo info = ((AdmissionInfo)content.Grid.GetRow(content.VisibleIndex));
                        if (info.Removed.HasValue)
                            htmlHelper.ViewContext.Writer.Write(htmlHelper.ActionLink("Undo", "UndoRemoveAdmission", "Home", new { p = removeEncodedParams }, new { @class = "smlink", style = "padding-left: 5px" }));
                    }
                });
            });

            settings.CustomActionRouteValues = new { Controller = "Home", Action = "CustomAction" };
            settings.CommandColumn.Width = 30;
            settings.CommandColumn.Visible = true;
            settings.CommandColumn.ShowSelectCheckbox = true;
            settings.CommandColumn.SelectAllCheckboxMode = GridViewSelectAllCheckBoxMode.Page;

            settings.ClientSideEvents.SelectionChanged = "OnGridRowSelectionChanged";

            return settings;
        }

        public static GridViewSettings CreateStandardAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "StandardAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered, 110));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Requested", Strings.Requested, 110));

            if (model.CurrentFilter == Constants.YESTERDAY || model.CurrentFilter == Constants.PAST_30_DAYS)
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Removed", Strings.Removed, 50));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, Strings.Comment, 120)); // RRK
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Gender", "SX", 15));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Problem", Strings.PresentingProblem, 180)); // RRK

            BuildTypeOfBedColumn(htmlHelper, settings, isExport);

            settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "WaitingArea", Strings.WaitingArea, model.WaitingAreaList, Strings.CD_DisplayName, 120));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "WaitTimeDisplay", Strings.WaitTime + " " + Strings.hmTimeFormat, 40));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "RemainingWaitTimeAlertHoursDisplay", Strings.WaitTimeAlert + " " + Strings.hTimeFormat, 25));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsCommunityLivingCenter", Strings.CLC, 15));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreatePatientFlowAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "PatientFlowAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered + " " + Strings.OnListDT, 110));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Requested", Strings.Anticipated + " " + Strings.RemovalDT, 110));

            if (model.CurrentFilter == Constants.YESTERDAY || model.CurrentFilter == Constants.PAST_30_DAYS)
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Removed", Strings.Removed + " " + Strings.FromListDT, 50));
            
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "LastEditBy", Strings.LastEdit + " " + Strings.DoneBy, 120));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "BedAssignedDate", Strings.Assigned + " " + Strings.ToRoomDT, 110));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Gender", "SX", 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "RoomBedAssigned", Strings.Room_Bed + " " + Strings.Assigned, 50));

            BuildTypeOfBedColumn(htmlHelper, settings, isExport);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "RemainingWaitTimeAlertHoursDisplay", Strings.WaitTimeAlert + " " + Strings.hTimeFormat, 25));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsCommunityLivingCenter", Strings.CLC, 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "PatientFlowComment", Strings.Comments, 150));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreateFeeUtilizationAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "FeeUtilizationAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered, 110));

            if (model.CurrentFilter == Constants.YESTERDAY || model.CurrentFilter == Constants.PAST_30_DAYS)
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Removed", Strings.Removed, 50));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Gender", "SX", 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "ServiceReceivingFee", Strings.TreatingSpecialty, 110));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Fee", 70));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "ReasonUsingFee", Strings.Reason, 110));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Acute", 110));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Contract", 70));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "FeeAccepted", "Accepted", 70));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "FeeComments", Strings.Comments, 110));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "RemainingWaitTimeAlertHoursDisplay", Strings.WaitTimeAlert + " " + Strings.hTimeFormat, 25));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsCommunityLivingCenter", Strings.CLC, 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsCommunityHospital", "Community Hospital?", 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "TransferCoordinator", 100));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "TransferCoordinatorPhone", 70));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreatePatientInHouseAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "PatientInHouseAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered, 110));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Requested", Strings.Requested, 110));

            if (model.CurrentFilter == Constants.YESTERDAY || model.CurrentFilter == Constants.PAST_30_DAYS)
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Removed", Strings.Removed, 50));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Gender", "SX", 15));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Problem", Strings.PresentingProblem, 180)); // RRK

            BuildTypeOfBedColumn(htmlHelper, settings, isExport);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "WaitingArea", Strings.WaitingArea, 120));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "WaitTimeDisplay", Strings.WaitTime + " " + Strings.hmTimeFormat, 40));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "RemainingWaitTimeAlertHoursDisplay", Strings.WaitTimeAlert + " " + Strings.hTimeFormat, 25));

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsCommunityLivingCenter", Strings.CLC, 15));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreateMentalHealthAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "MentalHealthAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered, 110));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Requested", Strings.Requested, 110));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => { GridHelper.BuildColumn(column, "VeteranPreference", "VP?", 120); column.Settings.AllowHeaderFilter = DefaultBoolean.True; });
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "DateOfScreening", "Date of Screening", 110));
            BuildTypeOfBedColumn(htmlHelper, settings, isExport);            
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "MentalHealthComment", Strings.Comments, 120));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreateEmergencyManagementAdmissionGrid(HtmlHelper htmlHelper, HomeViewModel model, bool isExport)
        {
            GridViewSettings settings = GridHelper.ConfigureFacilityPPBPGrid(htmlHelper, "EmergencyManagementAdmissionGrid", model);

            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "Entered", Strings.Entered, 110));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "EvacExpectedPickUp", "Expected Pick-up", 110));

            BuildPatientColumn(htmlHelper, settings, isExport, true, model.LoggedUser.Salt);

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Gender", "Sex", 120));
            settings.Columns.Add(column => GridHelper.BuildDateTimeColumn(column, "DateOfBirth", "DOB", 110));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Weight", 80));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Problem", "Presenting Problem (Diagnosis)", 120));

            BuildTypeOfBedColumn(htmlHelper, settings, isExport);

            BuildEmergencyIconColumn(htmlHelper, settings, isExport);

            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "TransportationComment", Strings.Comments, 120));

            BuildFacilityGridLayout(settings, model);

            return settings;
        }

        public static GridViewSettings CreateVISNPPBPGrid(HtmlHelper htmlHelper, BedBoardViewModel model, bool isExport)
        {
            GridViewSettings settings = ConfigurePPBPGrid(htmlHelper, "BedBoard", "BedBoardGrid", "IdExtension", model.LoggedUser.Salt);

            settings.Columns.Add(column =>
            {
                column.Name = "Actions";
                column.Caption = "Actions";
                column.Width = System.Web.UI.WebControls.Unit.Pixel(60);

                column.SetDataItemTemplateContent(content =>
                {
                    string idValue = (string)content.KeyValue;
                    string editEncodedParams = BaseController.EncryptQueryString(new string[] { "visn", "facility", "transferId" }, new string[] { model.VISN.Id.extension, model.SelectedFacilityId, idValue }, model.LoggedUser.Salt);
                    string finalizeEncodedParams = BaseController.EncryptQueryString(new string[] { "visn", "facility", "transferId" }, new string[] { model.VISN.Id.extension, model.SelectedFacilityId, idValue }, model.LoggedUser.Salt);

                    htmlHelper.ViewContext.Writer.Write(htmlHelper.ActionLink("Edit", "EditTransfer", "Transfer", new { p = editEncodedParams }, new { @class = "smlink", style = "padding-right: 5px" }));
                    htmlHelper.ViewContext.Writer.Write(htmlHelper.ActionLink("Finalize", "FinalizeTransfer", "Transfer", new { p = finalizeEncodedParams }, new { @class = "smlink" }));
                });
            });

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsNationalWaitList", Strings.NationalWaitList, 15));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "FacilityCodeFrom", Strings.FacShort, 30));
            GridHelper.BuildPatientColumn(htmlHelper, settings, isExport, false, model.LoggedUser.Salt);
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "PatientLastFourDigitsOfSSN", Strings.SSN, 30));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "SCRating", Strings.Service + " " + Strings.Connected, 30));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Era", Strings.OnlyEra, 30));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "Contract", Strings.Contract, 30));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Diagnosis", Strings.Diagnosis, 80)); // RRK
            settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "Specialty", Strings.TreatingSpecialty, model.SpecialtyList, Strings.CD_DisplayName, 180));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "CurrentLocation", Strings.Current + " " + Strings.Location, 100)); // RRK
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "TypeOfBedWard", Strings.TypeOfBed + " / " + Strings.WardRequired, 120));
            settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "AdmissionDate", Strings.ChCl + " " + Strings.Admission + " " + Strings.Date, 60));
            settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Comment", Strings.Comments, 200)); // RRK
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "WaitingTimeDisplay", Strings.WaitTime + " " + Strings.hmTimeFormat, 50));
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "IsFee", "Fee?", 15));

            settings.HtmlDataCellPrepared = (s, e) => { if (e.DataColumn.FieldName == "IsFee") e.Cell.ToolTip = e.GetValue("FeeTooltip").ToString(); };


            settings.CustomColumnDisplayText = (sender, eventArgs) =>
            {
                if (eventArgs.Column.FieldName != "IsNationalWaitList")
                    return;

                if ((bool)eventArgs.Value == true)
                    eventArgs.DisplayText = "X";
                else
                    eventArgs.DisplayText = String.Empty;
            };

            settings.ClientLayout = (@object, eventArgs) =>
            {
                string saveArea = "VISNPPBP" + model.VISN.Number;
                if (eventArgs.LayoutMode == ClientLayoutMode.Saving)
                {
                    SaveUserLayout(model.LoggedUser, saveArea, eventArgs.LayoutData, false);
                }
                else
                {
                    string layoutData = LoadUserLayout(model.LoggedUser, saveArea);

                    if (model.IsFirstLoad)
                    {
                        string[] layoutSplit = layoutData.Split('|');
                        if (layoutSplit.Length != 0 && layoutSplit[0].StartsWith("page"))
                            layoutSplit[0] = "page1";

                        layoutData = String.Join("|", layoutSplit);
                    }

                    eventArgs.LayoutData = layoutData;
                }
            };

            settings.CustomActionRouteValues = new { Controller = "BedBoard", Action = "CustomAction" };
            settings.CommandColumn.Width = 30;
            settings.CommandColumn.Visible = true;
            settings.CommandColumn.ShowSelectCheckbox = true;
            settings.CommandColumn.SelectAllCheckboxMode = GridViewSelectAllCheckBoxMode.Page;

            settings.ClientSideEvents.SelectionChanged = "OnGridRowSelectionChanged";

            return settings;
        }

        public static GridViewSettings CreateNationalPPBPGrid(HtmlHelper htmlHelper, NationalAndRegionalViewModel model, bool isExport)
        {
            //string selectedSpecialty = model.SelectedSpeciality != null ? model.SelectedSpeciality.code : null;
            string selectedWaitingView = model.SelectedWaitingView != null ? model.SelectedWaitingView.code : null;
            string selectedWaitingTime = model.SelectedWaitingTime != null ? model.SelectedWaitingTime.code : null;
            string encodedParams = BaseController.EncryptQueryString(new string[] { "waitingView", "waitingTime", "fillVisn" }, new string[] { selectedWaitingView, selectedWaitingTime, model.FillVisnsByRegion }, model.LoggedUser.Salt);
            GridViewSettings settings = ConfigurePPBPGrid(htmlHelper, "NationalAndRegional", "NationalAndRegionalGrid", "IdExtension", model.LoggedUser.Salt, encodedParams);

            settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "IsNationalWaitListDisplay", Strings.NationalWaitList, new List<string> { "YES", "NO" }, null, 55));
            settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "RegionDisplay", Strings.R, model.Regions, "Name", 55));
            settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "VisnDisplay", Strings.V, model.Visns, "Name", 55));
            BuildPatientColumn(htmlHelper, settings, isExport, false, model.LoggedUser.Salt);
            settings.Columns.Add(column => GridHelper.BuildColumn(column, "PatientLastFourDigitsOfSSN", Strings.SSN, 30));

            if (model.SelectedWaitingView.code.Equals(BMS.Utils.Constants.STANDARD))
            {
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "SCRating", Strings.SVC, 30));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Era", Strings.OnlyEra, 30));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "Contract", Strings.Contract, 30));
                settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Diagnosis", Strings.Diagnosis, 80)); // RRK
                settings.Columns.Add(column => GridHelper.BuildComboColumn(column, "Specialty", Strings.TreatingSpecialty, model.Specialities, Strings.CD_DisplayName, 180));
                settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "CurrentLocation", Strings.Current + " " + Strings.Location, 100)); // RRK
                settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "EventDate", Strings.ChCl + " " + Strings.Admission + " " + Strings.Date, 60));
                settings.Columns.Add(column => GridHelper.BuildHtmlEncodedColumn(column, "Comment", Strings.Comments, 200)); // RRK
            }
            else
            {
                settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "RequestDate", Strings.ReqDate, 60));
                settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "DispositionDate", Strings.DispositionDate, 60));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "EntryFacilityName", Strings.EntryFac, 40));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "AdmittedFacilityName", Strings.AdmitFac, 40));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "EnterBy", Strings.EnterBy, 40));
                settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "EntryDTDisplay", Strings.EntryDT, 60));
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "LastEditBy", Strings.LastEditBy, 40));
                settings.Columns.Add(column => GridHelper.BuildDateColumn(column, "EntryDTDisplay", Strings.LastEditDT, 60));
            }

            settings.Columns.Add(column => GridHelper.BuildColumn(column, "WaitingTimeDisplay", Strings.WaitTime + " " + Strings.hmTimeFormat, 50));

            settings.ClientLayout = (@object, eventArgs) =>
            {
                string saveArea = "NationalPPBP" + model.SelectedWaitingView.code;
                if (eventArgs.LayoutMode == ClientLayoutMode.Saving)
                {
                    SaveUserLayout(model.LoggedUser, saveArea, eventArgs.LayoutData, false);
                }
                else
                {
                    eventArgs.LayoutData = LoadUserLayout(model.LoggedUser, saveArea);
                }
            };

            settings.PreRender = (s, e) =>
            {
                ASPxGridView grid = (ASPxGridView)s;
                grid.FilterExpression = "Contains([IsNationalWaitListDisplay], 'YES')";
            };

            return settings;
        }

        private static void BuildPatientColumn(HtmlHelper htmlHelper, GridViewSettings settings, bool isExport, bool showSSNLink, string salt)
        {
            if (!isExport)
            {
                settings.Columns.Add(column =>
                {
                    GridHelper.BuildColumn(column, "PatientName", Strings.Patient, 120);
                    column.SetDataItemTemplateContent(content =>
                    {
                        string patientId = (string)DataBinder.Eval(content.DataItem, "ExtensionPatientId");
                        string patientName = (string)DataBinder.Eval(content.DataItem, "PatientName");
                        string last4SSN = (string)DataBinder.Eval(content.DataItem, "PatientLastFourDigitsOfSSN");
                        string encodedParams = BaseController.EncryptQueryString(new string[] { "patientId" }, new string[] { patientId }, salt);

                        htmlHelper.ViewContext.Writer.Write("<table><tr height='21px'>");
                        bool needsIcon = (bool)DataBinder.Eval(content.DataItem, "HasSimilarNameOrSSN");
                        if (needsIcon)
                        {
                            BinaryImageEditExtension binaryImage = htmlHelper.DevExpress().BinaryImage(imgSettings => { GridHelper.BuildBinaryImageSettings(content.KeyValue, "Similar Name or SSN", "Similar Name or SSN", BMS.Facade.Data.IconInfo.SameSimilarNameIcon, imgSettings, 20, 20); });
                            htmlHelper.ViewContext.Writer.Write("<td style='padding-right: 5px; position: relative; top: 2px';>");
                            binaryImage.Render();
                            htmlHelper.ViewContext.Writer.Write("</td>");
                        }

                        htmlHelper.ViewContext.Writer.Write("<td>" + patientName);

                        if (showSSNLink && !String.IsNullOrEmpty(last4SSN))
                            htmlHelper.ViewContext.Writer.Write("&nbsp;" + htmlHelper.ActionLink(last4SSN, "PatientInquiry", "Home", new { p = encodedParams }, new { @class = "smlink", @target = "_blank" }));

                        htmlHelper.ViewContext.Writer.Write("</td></tr></table>");
                    });
                });
            }
            else
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "PatientName", "Patient", 120));
        }

        private static void BuildTypeOfBedColumn(HtmlHelper htmlHelper, GridViewSettings settings, bool isExport)
        {
            if (!isExport)
            {
                settings.Columns.Add(column =>
                {
                    GridHelper.BuildColumn(column, "TypeOfBedWard", Strings.TypeOfBed + " / " + Strings.WardRequired, 120);
                    column.CellStyle.VerticalAlign = System.Web.UI.WebControls.VerticalAlign.Middle;
                    column.SetDataItemTemplateContent(c =>
                    {
                        htmlHelper.ViewContext.Writer.Write("<table><tr height='21px'>");
                        bool needsIcon = (bool)DataBinder.Eval(c.DataItem, "IsBedInIsolation");

                        if (needsIcon)
                        {
                            BinaryImageEditExtension binaryImage = htmlHelper.DevExpress().BinaryImage(imgSettings => { GridHelper.BuildBinaryImageSettings(c.KeyValue, Strings.IsolationRequired, Strings.IsolationRequired, BMS.Facade.Data.IconInfo.BedInIsolationIcon, imgSettings, 20, 20); });

                            htmlHelper.ViewContext.Writer.Write("<td style='padding-right: 5px; position: relative; top: 2px';>");
                            binaryImage.Render();
                            htmlHelper.ViewContext.Writer.Write("</td>");
                        }

                        htmlHelper.ViewContext.Writer.Write("<td>" + DataBinder.Eval(c.DataItem, "TypeOfBedWard") + "</td></tr></table>");
                    });
                });
            }
            else
                settings.Columns.Add(column => GridHelper.BuildColumn(column, "TypeOfBedWard", 120));
        }

        private static void BuildEmergencyIconColumn(HtmlHelper htmlHelper, GridViewSettings settings, bool isExport)
        {
            if (!isExport)
            {
                settings.Columns.Add(column =>
                {
                    column.Caption = "Emergency Icons";
                    column.Width = Unit.Pixel(120);
                    column.CellStyle.VerticalAlign = System.Web.UI.WebControls.VerticalAlign.Middle;
                    column.SetDataItemTemplateContent(c =>
                    {
                        htmlHelper.ViewContext.Writer.Write("<table><tr height='21px'>");
                        List<IconInfo> icons = ((AdmissionInfo)c.Grid.GetRow(c.VisibleIndex)).EmergencyIcons;

                        if (icons != null && icons.Count > 0)
                        {
                            foreach (IconInfo icon in icons)
                            {
                                if (!IconInfo.IconBytes.ContainsKey(icon.IconId))
                                    IconInfo.IconBytes.Add(icon.IconId, Convert.FromBase64String(icon.ImageBytes));

                                BinaryImageEditExtension binaryImage = htmlHelper.DevExpress().BinaryImage(imgSettings => { GridHelper.BuildBinaryImageSettings(c.KeyValue, icon.IconName, icon.MouseOverText, IconInfo.IconBytes[icon.IconId], imgSettings, 20, 20); });

                                htmlHelper.ViewContext.Writer.Write("<td style='padding-right: 2px; position: relative; top: 2px';>");
                                binaryImage.Render();
                                htmlHelper.ViewContext.Writer.Write("</td>");
                            }
                        }

                        htmlHelper.ViewContext.Writer.Write("</tr></table>");
                    });
                });
            }
        }

        private static void BuildFacilityGridLayout(GridViewSettings settings, HomeViewModel model)
        {
            settings.HtmlDataCellPrepared = (sender, e) =>
            {
                if (e.DataColumn.FieldName == "RemainingWaitTimeAlertHoursDisplay")
                {
                    int? value = (int?)e.GetValue("RemainingWaitTimeAlertHours");
                    if (value.HasValue && value <= model.Threshold)
                        e.Cell.BackColor = System.Drawing.Color.Red;
                }
            };

            settings.HtmlRowPrepared = (sender, e) =>
            {
                string roomBedAssigned = (string)e.GetValue("RoomBedAssigned");
                if (!String.IsNullOrEmpty(roomBedAssigned))
                    e.Row.BackColor = System.Drawing.Color.FromArgb(255, 165, 0);
            };

            settings.ClientLayout = (@object, eventArgs) =>
            {
                string saveArea = HomeController.GetSaveArea(model);
                if (eventArgs.LayoutMode == ClientLayoutMode.Saving)
                {
                    SaveUserLayout(model.LoggedUser, saveArea, eventArgs.LayoutData, false);
                }
                else
                {
                    string layoutData = LoadUserLayout(model.LoggedUser, saveArea);

                    if (model.IsFirstLoad)
                    {
                        string[] layoutSplit = layoutData.Split('|');
                        if (layoutSplit.Length != 0 && layoutSplit[0].StartsWith("page"))
                            layoutSplit[0] = "page1";

                        layoutData = String.Join("|", layoutSplit);
                    }

                    eventArgs.LayoutData = layoutData;
                }
            };

            settings.CustomColumnDisplayText = (sender, eventArgs) =>
            {
                if (eventArgs.Column.FieldName != "IsCommunityLivingCenter")
                    return;

                if ((bool)eventArgs.Value == true)
                    eventArgs.DisplayText = "Yes";
                else
                    eventArgs.DisplayText = "No";
            };
        }

        public static void SaveUserLayout(User loggedUser, string saveArea, string layoutData, bool toDatabase)
        {
            string key = loggedUser.Id.extension + saveArea;

            HttpContext.Current.Session[key] = layoutData;
            if (toDatabase)
                FacadeManager.ConfigurationInterface.SaveUserPPBPSettings(loggedUser, key, layoutData);
        }

        public static string LoadUserLayout(User loggedUser, string saveArea)
        {
            string key = loggedUser.Id.extension + saveArea;

            string layoutData = String.Empty;
            if (HttpContext.Current.Session[key] == null)
            {
                layoutData = FacadeManager.ConfigurationInterface.GetUserPPBPSettings(loggedUser, key);
                HttpContext.Current.Session[key] = layoutData;
            }
            else
                layoutData = HttpContext.Current.Session[key].ToString();

            return layoutData;
        }
    }
}