﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.Facade.Translators;
using BMS.ServicesWrapper.BMService;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Controllers.Shared;
using BMS.Web.Models;
using DevExpress.Web.Mvc;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;
using VI = BMS.VistaIntegration.FacadeContracts;


namespace BMS.Web.Controllers
{
    /// <summary>
    /// Home Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class HomeController : BaseController
    {
        /// <summary>
        /// Default action for the Home controller.
        /// </summary>
        /// <returns>Returns the Home view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return Index2(null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Default action for the Home controller.
        /// </summary>
        /// <returns>Returns the Home view result.</returns>
        [ReadPermissionAuthFilter]
        public ActionResult Index2(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                HomeViewModel hm = new HomeViewModel();
                hm.FacilityTime = DateTimeNowOnCurrentFacility;
                hm.View2 = string.IsNullOrEmpty(p) ? Strings.View_Standard : string.IsNullOrEmpty(QueryStrings["view"]) ? Strings.View_Standard : QueryStrings["view"];
                hm.LastSelectedView = string.IsNullOrEmpty(p) ? Strings.View_Standard : string.IsNullOrEmpty(QueryStrings["lastView"]) ? Strings.View_Standard : QueryStrings["lastView"];
                hm.CurrentFilter = string.IsNullOrEmpty(p) ? Strings.WaitingList_Current : string.IsNullOrEmpty(QueryStrings["filter"]) ? Strings.WaitingList_Current : QueryStrings["filter"];
                hm.SelectedCensusCategory = string.IsNullOrEmpty(p) ? Strings.Facility : string.IsNullOrEmpty(QueryStrings["censusCategory"]) ? Strings.Facility : QueryStrings["censusCategory"];
                FillViewModel(hm);

                return View("Index", hm);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Home controller  - used for the waiting list buttons and drop downs.
        /// </summary>        
        /// <param name="buttonName">Name of the button.</param>
        /// <param name="buttonHiddenView">The button hidden view.</param>
        /// <param name="buttonHiddenSort">The button hidden sort.</param>
        /// <param name="input">The input (home model).</param>
        /// <returns>Returns the Home view result.</returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index([Bind(Exclude = "RememberMe")] HomeViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrWhiteSpace(input.GoToSelectedFacilitySite))
                {
                    if (input.SelectedFacilitySiteExtension != Strings.PickADifferentSite)
                    {
                        Facility facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, input.SelectedFacilitySiteExtension));
                        if (facility != null)
                        {
                            if (facility.IsAlive == false)
                                input.ChangeIntegratedSiteError = Strings.FacilityIsNotAlive;
                            if (facility.VistaSite.Visn == null)
                                input.ChangeIntegratedSiteError = Strings.TheFacilityDoesntHaveAnAssociatedVISN;
                            if (facility.VistaSite.Visn.Region == null)
                                input.ChangeIntegratedSiteError = Strings.TheVISNOnFacilityDoesntHaveAnAssociatedRegion;
                        }
                        else
                        {
                            input.ChangeIntegratedSiteError = Strings.FacilityNotFound;
                        }

                        if (!string.IsNullOrEmpty(input.ChangeIntegratedSiteError))
                        {
                            return this.RedirectToAction<HomeController>(act => act.ChangeIntegratedSiteError(EncryptQueryString(new string[] { "message", "facilityName", "facilityCode" }, new string[] { input.ChangeIntegratedSiteError, this.loggedUser.Facility.Name, this.loggedUser.Facility.Code }, loggedUser.Salt)));
                        }
                        else
                        {
                            this.loggedUser.Facility = facility;
                            this.loggedUser.Visn = facility.VistaSite.Visn;
                            this.loggedUser.Region = facility.VistaSite.Visn.Region;
                            this.loggedUser.VistaSite = facility.VistaSite;
                            return this.RedirectToAction<HomeController>(act => act.Index());
                        }
                    }
                }

                if (!String.IsNullOrEmpty(input.HeaderCommand))
                {
                    input.LoggedUser = loggedUser;
                    SetView2Selected(input);

                    string saveArea = GetSaveArea(input);
                    string key = loggedUser.Id.extension + saveArea;
                    if (input.HeaderCommand == "Save" && Session[key] != null)
                    {
                        string layoutData = Session[key].ToString();
                        GridHelper.SaveUserLayout(this.loggedUser, saveArea, layoutData, true); // save settings from session to database
                    }
                    else if (input.HeaderCommand == "Reset")
                        GridHelper.SaveUserLayout(this.loggedUser, saveArea, String.Empty, true); // clear saved settings and reset grid
                }

                return this.RedirectToAction<HomeController>(act => act.Index2(EncryptQueryString(new string[] { "filter", "view", "lastView", "censusCategory" }, new string[] { input.FilterOperation == null ? input.CurrentFilter : input.FilterOperation, input.View2, input.LastSelectedView, input.SelectedCensusCategory }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public ActionResult ChangeIntegratedSiteError(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ViewBag.ErrorMessage = QueryStrings["message"];
                ViewBag.FacilityName = QueryStrings["facilityName"];
                ViewBag.FacilityCode = QueryStrings["facilityCode"];
                return View();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for rendering the AdmissionList in Home partial View
        /// </summary>
        /// <param name="input">The input (home model).</param>
        /// <returns>Returns the AdmissionList partial view result.</returns>
        public ActionResult AdmissionList([Bind(Exclude = "RememberMe")] HomeViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //Generate the table columns based on the selected value from the View drop down.
                input.LoggedUser = loggedUser;

                input.CurrentGridView = GetSaveNameFromView(input.View2Selected.displayName);
                return PartialView(input.CurrentGridView, input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public ActionResult ExportTo(string p)
        {
            string currentGrid = QueryStrings["currentGrid"];
            string outputFormat = QueryStrings["outputFormat"];
            HomeViewModel model = BuildModelForGrid(currentGrid);

            GridViewSettings settings = null;
            switch (currentGrid)
            {
                case "StandardAdmissionGrid":
                    settings = GridHelper.CreateStandardAdmissionGrid(null, model, true);
                    break;
                case "PatientFlowAdmissionGrid":
                    settings = GridHelper.CreatePatientFlowAdmissionGrid(null, model, true);
                    break;
                case "FeeUtilizationAdmissionGrid":
                    settings = GridHelper.CreateFeeUtilizationAdmissionGrid(null, model, true);
                    break;
                case "PatientInHouseAdmissionGrid":
                    settings = GridHelper.CreatePatientInHouseAdmissionGrid(null, model, true);
                    break;
                case "MentalHealthAdmissionGrid":
                    settings = GridHelper.CreateMentalHealthAdmissionGrid(null, model, true);
                    break;
                case "EmergencyManagementAdmissionGrid":
                    settings = GridHelper.CreateEmergencyManagementAdmissionGrid(null, model, true);
                    break;
            }

            switch (outputFormat.ToUpper())
            {
                case "CSV":
                    return GridViewExtension.ExportToCsv(settings, model.AdmissionList);
                case "PDF":
                    return GridViewExtension.ExportToPdf(settings, model.AdmissionList);
                case "RTF":
                    return GridViewExtension.ExportToRtf(settings, model.AdmissionList);
                case "XLS":
                    return GridViewExtension.ExportToXls(settings, model.AdmissionList);
                case "XLSX":
                    return GridViewExtension.ExportToXlsx(settings, model.AdmissionList);
                default:
                    return RedirectToAction(currentGrid);
            }
        }

        [ValidateInput(false)]
        public ActionResult StandardAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("StandardAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ValidateInput(false)]
        public ActionResult PatientFlowAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("PatientFlowAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ValidateInput(false)]
        public ActionResult FeeUtilizationAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("FeeUtilizationAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ValidateInput(false)]
        public ActionResult PatientInHouseAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("PatientInHouseAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ValidateInput(false)]
        public ActionResult MentalHealthAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("MentalHealthAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ValidateInput(false)]
        public ActionResult EmergencyManagementAdmissionGrid()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BuildViewForGrid("EmergencyManagementAdmissionGrid");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private HomeViewModel BuildModelForGrid(string gridViewName)
        {
            HomeViewModel model = new HomeViewModel();

            bool onlyCurrentWaitingAdmissions = true;
            bool onlyInHouseWaitingAdmissions = gridViewName == "PatientInHouseAdmissionGrid";
            bool isMentalHealth = gridViewName == "MentalHealthAdmissionGrid";
            bool isEmergencyManagement = gridViewName == "EmergencyManagementAdmissionGrid";
            bool isFee = gridViewName == "FeeUtilizationAdmissionGrid";
            DateTime facilityAdmissionDate = DateTimeNowOnCurrentFacility;

            model.CurrentFilter = QueryStrings == null || QueryStrings.Count == 0 || !QueryStrings.ContainsKey("filter") ? Strings.WaitingList_Current : string.IsNullOrEmpty(QueryStrings["filter"]) ? Strings.WaitingList_Current : QueryStrings["filter"];
            facilityAdmissionDate = GetAdmissionDatePeriod(model, out onlyCurrentWaitingAdmissions);
            model.CurrentViewSelected = onlyCurrentWaitingAdmissions;

            model.AdmissionList = FacadeManager.ADTInterface.GetAdmissionList(this.loggedUser.Facility.Id, facilityAdmissionDate, onlyCurrentWaitingAdmissions, onlyInHouseWaitingAdmissions, isMentalHealth, isEmergencyManagement, isFee);
            model.LoggedUser = this.loggedUser;
            model.Threshold = Convert.ToInt32(System.Configuration.ConfigurationManager.AppSettings["THRESHOLD"]);
            model.CurrentGridView = gridViewName;
            model.WaitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);

            return model;
        }

        private ActionResult BuildViewForGrid(string gridViewName)
        {
            HomeViewModel model = BuildModelForGrid(gridViewName);

            return PartialView(gridViewName, model);
        }

        /// <summary>
        /// Action for redirecting to the edit view.
        /// </summary>
        /// <returns>The Edit View.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Edit()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return View();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the remove view.
        /// </summary>
        /// <returns>The Remove View</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult RemoveAdmission(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                HomeRemoveAdmissionViewModel model = new HomeRemoveAdmissionViewModel();
                model.LoggedUser = loggedUser;
                model.IdAdmission = p;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the undo remove view.
        /// </summary>
        /// <returns>The Undo Remove View</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult UndoRemoveAdmission(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                HomeRemoveAdmissionViewModel model = new HomeRemoveAdmissionViewModel();
                model.LoggedUser = loggedUser;
                model.IdAdmission = p;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the remove view.
        /// </summary>
        /// <returns>The Remove View</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult RemoveAdmissionPost(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingListItem waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, QueryStrings["idAdmission"]), loggedUser);
                waitingListItem.RemovedDate = DateTimeNowOnCurrentFacility;
                waitingListItem.LastEditBy = this.loggedUser.UserName;
                FacadeManager.WorkflowInterface.RemoveFromWaitingList(waitingListItem, loggedUser.Facility);
                return View("AdmissionSuccessRemove", (object)QueryStrings["idAdmission"]);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        public ActionResult CustomAction(List<string> actIds, string action, string gridViewName)
        {
            switch (action)
            {
                case "transferToVisn":
                    {
                        if (!FacadeManager.UserInterface.CheckPermission("Transfer, AddTranfer Update"))
                            return BaseController.UnauthorizedWriteRequestResult(loggedUser.Facility.Name);

                        //CD otherEra = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Era).Where(e => e.code == Constants.OTHER).FirstOrDefault();
                        //CD noContract = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision)[0];
                        //CD defaultSpecialty = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.VistaSpecialty)[0];

                        foreach (string actId in actIds)
                        {
                            try
                            {
                                WaitingListItem waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, actId), loggedUser);
                                if (waitingListItem == null)
                                    continue;

                                Transfer transfer = new Transfer()
                                {
                                    Id = new II(this.loggedUser.Domain, null),
                                    CreationDate = DateTime.UtcNow,
                                    CanceledDate = null,
                                    Patient = waitingListItem.Patient,
                                    FacilityFrom = loggedUser.Facility,
                                    Era = null, // model.EraList.Where(a => a.code == model.SelectedEraCode).FirstOrDefault(),
                                    Contract = waitingListItem.Contract, // model.ContractList.Where(a => a.code == model.SelectedContractCode).FirstOrDefault(),
                                    Diagnosis = waitingListItem.AdmittingDiagnosis, // model.Diagnosis,
                                    CurrentLocation = null,
                                    AdmissionDate = null,
                                    Comment = waitingListItem.Comments,
                                    Speciality = waitingListItem.Patient.TreatingSpecialty,
                                    Edited = DateTime.UtcNow,
                                    EnteredBy = this.loggedUser.UserName,
                                    LastEditBy = this.loggedUser.UserName,
                                    RequestedDate = DateTime.Today,
                                    IsNationalWaitList = false,
                                    TypeOfBedWard = waitingListItem.TypeOfBedWard,
                                    IsFee = waitingListItem.IsFee,
                                    TransferCoordinator = waitingListItem.TransferCoordinator,
                                    TransferCoordinatorPhone = waitingListItem.TransferCoordinatorPhone
                                };

                                FacadeManager.WorkflowInterface.CreateTransferRequest(transfer, transfer.FacilityFrom);

                                waitingListItem.RemovedDate = DateTimeNowOnCurrentFacility;
                                waitingListItem.LastEditBy = this.loggedUser.UserName;
                                FacadeManager.WorkflowInterface.RemoveFromWaitingList(waitingListItem, loggedUser.Facility);
                            }
                            catch 
                            {
                                // errors will be logged at the service layer if GetWaitingListItem, CreateTransferRequest or RemoveFromWaitingList have issues. Meanwhile other transfers in the loop should be processed
                            }
                        }
                        break;
                    }
                case "remove":
                    {
                        if (!FacadeManager.UserInterface.CheckPermission("Home, RemoveAdmission Read"))
                            return BaseController.UnauthorizedWriteRequestResult(loggedUser.Facility.Name);

                        foreach (string actId in actIds)
                        {
                            try
                            {
                                WaitingListItem waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, actId), loggedUser);
                                if (waitingListItem == null)
                                    continue;

                                waitingListItem.RemovedDate = DateTimeNowOnCurrentFacility;
                                waitingListItem.LastEditBy = this.loggedUser.UserName;
                                FacadeManager.WorkflowInterface.RemoveFromWaitingList(waitingListItem, loggedUser.Facility);
                            }
                            catch
                            {
                                // errors will be logged at the service layer if GetWaitingListItem or RemoveFromWaitingList have issues. Meanwhile other transfers in the loop should be processed
                            }
                        }
                        break;
                    }
                default:
                    break;
            }

            return BuildViewForGrid(gridViewName);
        }

        /// <summary>
        /// Action for redirecting to the undo remove view.
        /// </summary>
        /// <returns>The undo Remove View</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult UndoRemoveAdmissionPost(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingListItem waitingListItem = FacadeManager.ADTInterface.GetWaitingListItem(new II(this.loggedUser.Domain, QueryStrings["idAdmission"]), loggedUser);
                waitingListItem.RemovedDate = null;
                waitingListItem.LastEditBy = this.loggedUser.UserName;
                FacadeManager.WorkflowInterface.UndoRemoveFromWaitingList(waitingListItem, loggedUser.Facility);
                return View("AdmissionSuccessUndoRemove", (object)QueryStrings["idAdmission"]);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action to load the time view from the top of the page.
        /// </summary>
        /// <returns>TimeRefresh partial view.</returns>
        public PartialViewResult TimeRefresh()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return PartialView("TimeRefresh");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute]
        public ActionResult PatientInquiry(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                PatientInquiryViewModel model = new PatientInquiryViewModel();
                model.SelectedPatient = FacadeManager.EntityInterface.GetPatientById(new II(loggedUser.Domain, QueryStrings["patientId"]), null);
                model.Header = Strings.ReportDate + ": " + DateTimeNowOnCurrentFacility.ToString();
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [ReadPermissionAuthFilterAttribute(OperationOverride = "Home, Index")]
        [HttpPost]
        public ActionResult SelectCensusCategory(string censusCatgory)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                return Json("Done");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        /// <summary>
        /// Fills the controls with data from Facade.
        /// </summary>
        /// <param name="hm">The home model.</param>
        private void FillViewModel(HomeViewModel hm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SetView2Selected(hm);
                hm.WaitingAreaList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.WaitingArea);
                hm.LoggedUser = this.loggedUser;
                FacilitySettings facilitySettings = null;
                bool isEvacuation = false;
                string marquee = null;
                List<Module> flags = null;
                FacadeManager.ConfigurationInterface.GetConfigurationsForHomePage(this.loggedUser.Facility.Id, out isEvacuation, out marquee, out facilitySettings, out flags);
                hm.MarqueeText = marquee;
                hm.IsEvacuation = isEvacuation;

                if (facilitySettings != null)
                {
                    if (facilitySettings.IntegratedFacility != null && facilitySettings.IntegratedFacility.code == Constants.Yes && facilitySettings.IntegratedSiteSisterGroupId != 0)
                    {
                        SisterSitesGroup group = facilitySettings.SisterSitesGroups.Where(a => a.RecordNumber == facilitySettings.IntegratedSiteSisterGroupId && !string.IsNullOrEmpty(a.SiteCodeList) && a.SiteCodeList.Contains(this.loggedUser.Facility.Id.extension)).FirstOrDefault();
                        if (group != null)
                        {
                            if (!string.IsNullOrEmpty(facilitySettings.IntegratedSiteSisterList))
                                hm.IntegratedSites = BuildIntegratedSites(facilitySettings.IntegratedSiteSisterList, this.loggedUser.Facility);
                            else
                                hm.IntegratedSites = new List<Facility>();
                            hm.IntegratedSites.Insert(0, new Facility() { Id = new II() { root = this.loggedUser.Facility.Id.root, extension = Strings.PickADifferentSite }, Name = Strings.PickADifferentSite });
                        }
                    }
                }

                hm.AdmissionList = new List<AdmissionInfo>();
                IList<Diversion> diversions = FacadeManager.DiversionInterface.GetDiversions(this.loggedUser.Facility.Id).Where(div => div.Status == Constants.Yes && div.StartDate != null).ToList();
                hm.DiversionExists = diversions.Count > 0 ? true : false;

                if (hm.DiversionExists)
                {
                    hm.DiversionNo = Strings.StyleDisplayNone;
                    hm.DiversionYes = string.Empty;
                    foreach (Diversion diversion in diversions)
                    {
                        string diversionInfo = String.Format("- {0}|{1}|{2} Diverted since {3}: {4}\r\n\r\n", diversion.Location.displayName,
                            String.IsNullOrEmpty(diversion.WardGroup) ? "All" : diversion.WardGroup,
                            String.IsNullOrEmpty(diversion.TreatingSpecialty) ? "All" : diversion.TreatingSpecialty,
                            diversion.StartDate.ToShortDateString(), diversion.Comments);

                        hm.DiversionInfo += diversionInfo;
                    }
                }
                else
                {
                    hm.DiversionNo = string.Empty;
                    hm.DiversionYes = Strings.StyleDisplayNone;
                }

                hm.OccupancyPercent = 0;
                List<Ward> wards = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).ToList();

                hm.CensusCategories = new Dictionary<string, CD>();
                hm.CensusCategories.Add(Strings.Facility, new CD { code = string.Empty, displayName = Strings.Facility });

                StringBuilder wardIdList = new StringBuilder();
                IList<II> facilityIdList = new List<II>();
                facilityIdList.Add(this.loggedUser.Facility.Id);
                if (wards != null)
                {
                    foreach (Ward w in wards)
                    {
                        wardIdList.Append(w.Id.extension);
                        wardIdList.Append(",");

                        if (w.CensusCategory != null && !hm.CensusCategories.ContainsKey(w.CensusCategory.displayName))
                            hm.CensusCategories.Add(w.CensusCategory.displayName, w.CensusCategory);
                    }
                }
                int timeZoneMinutesOffset = Convert.ToInt32(RepTimeZoneMinutesOffset);
                hm.Threshold = Convert.ToInt32(System.Configuration.ConfigurationManager.AppSettings["THRESHOLD"]);
                Guid facilityUid = Guid.Parse(this.loggedUser.Facility.Id.extension);
                DateTime? facilityAdmissionDate = DateTimeNowOnCurrentFacility;

                bool onlyCurrentWaitingAdmissions = true;
                bool onlyInHouseWaitingAdmissions = hm.View2Selected != null && hm.View2Selected.displayName.Equals(Strings.View_InHouse);
                bool isMentalHealth = hm.View2Selected != null && hm.View2Selected.displayName.Equals(Strings.View_MentalHealth);
                bool isEmergencyManagement = hm.View2Selected != null && hm.View2Selected.displayName.Equals(Strings.View_EmergencyManagement);
                bool isFee = hm.View2Selected != null && hm.View2Selected.displayName.Equals(Strings.View_FeeUtilization);

                facilityAdmissionDate = GetAdmissionDatePeriod(hm, out onlyCurrentWaitingAdmissions);
                hm.CurrentViewSelected = onlyCurrentWaitingAdmissions;

                List<BedOccupancyCount> bmsBedOccupancylist = new List<BedOccupancyCount>();
                List<NewEventInfo> newEventsInfo;
                IList<AdmissionInfo> admissionInfoList = new List<AdmissionInfo>();

                FacadeManager.ADTInterface.GetHomePageBMSData(this.loggedUser.VistaSite.Id, this.loggedUser.Facility, wardIdList.ToString(), flags, timeZoneMinutesOffset,
                                                                facilityAdmissionDate.Value, onlyCurrentWaitingAdmissions, onlyInHouseWaitingAdmissions, isMentalHealth, isEmergencyManagement, isFee,
                                                                FullUserName, out bmsBedOccupancylist, out newEventsInfo, out admissionInfoList);

                if (bmsBedOccupancylist.Count > 0)
                {
                    if (hm.WardCensusByCategory == null)
                        hm.WardCensusByCategory = new Dictionary<string, int>();

                    Dictionary<string, BedOccupancyCount> wardCensusByCategory = new Dictionary<string, BedOccupancyCount>();

                    // get ward census
                    foreach (BedOccupancyCount boc in bmsBedOccupancylist)
                    {
                        if (boc.DivisionName == "All")
                            hm.OccupancyPercent = boc.OccupancyPercent;
                        else
                        {
                            if (hm.WardCensus == null)
                                hm.WardCensus = new Dictionary<string, int>();
                            hm.WardCensus.Add(boc.DivisionName, boc.OccupancyPercent);
                        }

                        if (!String.IsNullOrEmpty(boc.CensusCategory) && boc.DivisionName != "All")
                        {
                            if (!wardCensusByCategory.ContainsKey(boc.CensusCategory))
                                wardCensusByCategory.Add(boc.CensusCategory, new BedOccupancyCount { CensusCategory = boc.CensusCategory });

                            wardCensusByCategory[boc.CensusCategory].TotalBeds += boc.TotalBeds;
                            wardCensusByCategory[boc.CensusCategory].TotalOccupiedBeds += boc.TotalOccupiedBeds;
                            wardCensusByCategory[boc.CensusCategory].TotalUnavailableBeds += boc.TotalUnavailableBeds;
                        }
                    }

                    hm.WardCensusByCategory.Add(Strings.Facility, hm.OccupancyPercent);
                    foreach (KeyValuePair<string, BedOccupancyCount> kvp in wardCensusByCategory)
                    {
                        hm.WardCensusByCategory.Add(kvp.Key, kvp.Value.OccupancyPercent);
                    }
                }

                // new event configuration
                List<EventInfo> events = new List<EventInfo>();
                foreach (NewEventInfo newEvent in newEventsInfo)
                {
                    NewEventsController.AddDefaultParameters(newEvent.Report, HttpContext.Request.Url.ToString(), Strings.ReturnToHomePage, this.loggedUser);

                    if (newEvent.EventName == Utils.NewEvent.PatientDischarged)
                        newEvent.Report.Parameters.Add(Constants.REP_ADMITTED_PATIENT_TRANS, "Discharges");

                    else if (newEvent.EventName == Utils.NewEvent.AdmittedPatient)
                        newEvent.Report.Parameters.Add(Constants.REP_ADMITTED_PATIENT_TRANS, "Admissions");

                    EventInfo eventInfo = new EventInfo();
                    eventInfo.ReportLink = newEvent.Report.Url;
                    eventInfo.DisplayValue = String.Format("{0} ({1})", EventInfo.GetDisplayNameFor(newEvent.EventName), newEvent.EventCount);
                    events.Add(eventInfo);
                }

                hm.EventsCount = events;
                hm.AdmissionList = admissionInfoList;

                hm.CPU = MvcApplication.CPUStatus.ToString("0.00");

                //Last BMS Update;
                // JMM - This code was moved from HomeController, ADTOperatinsImplementation, and BedManagerQueryCore - It's incredibly inefficient, but will be used as-is until I have a better understanding of the data retrieval involved
                List<SelectableEnum<DataRetrievalMethod>> retrievalMethods = GetRetrievalMethods();
                IList<DataRetrievalMethod> retrievalMethodList = retrievalMethods.Select<SelectableEnum<DataRetrievalMethod>, DataRetrievalMethod>(sel => sel.EnumValue).ToList();

                IList<BMS.Utils.VistaDataType> dataTypes = new List<BMS.Utils.VistaDataType>() { BMS.Utils.VistaDataType.PatientMovement };

                List<SelectableEnum<JobStatus>> jobStatus = GetJobStatus().Where(a => a.EnumValue.ToString() == JobStatus.Succeeded.ToString() || a.EnumValue.ToString() == JobStatus.PartiallySucceeded.ToString()).ToList();
                IList<JobStatus> jobStatusList = jobStatus.Select<SelectableEnum<JobStatus>, JobStatus>(sel => sel.EnumValue).ToList();

                DateTime? bpLastRunDate = BMSFactory.BedManagerQueryClient.FilterBPLastRun(this.loggedUser.VistaSite.Id.extension, retrievalMethodList, dataTypes, jobStatusList);

                string bpLastRunDateText = string.Empty;
                bool isValidDateBPLastRan = false;
                if (bpLastRunDate.HasValue)
                {
                    bpLastRunDateText = string.Format("{0} {1}", bpLastRunDate.Value.ToString(Strings.USDate), " @ ");
                    if ((DateTimeNowOnCurrentFacility - bpLastRunDate.Value).TotalMinutes > 30)
                    {
                        isValidDateBPLastRan = false;
                        bpLastRunDateText += bpLastRunDate.Value.ToString(Strings.HHmmssDateTime);
                    }
                    else
                    {
                        isValidDateBPLastRan = true;
                        bpLastRunDateText += bpLastRunDate.Value.ToString(Strings.HHmmDateTime);
                    }
                }

                hm.BPLastRanDateTime = bpLastRunDateText;
                hm.IsValidDateBPLastRan = isValidDateBPLastRan;

                string firstLoadKey = "IsFirstLoad" + this.loggedUser.Facility.SiteNumber;
                if (!TempData.ContainsKey(firstLoadKey))
                    hm.IsFirstLoad = true;
                else
                    hm.IsFirstLoad = false;

                TempData[firstLoadKey] = false; // Any value will do. If there is any value at this key, we know this site has been loaded during this user session.
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static DateTime GetAdmissionDatePeriod(HomeViewModel model, out bool onlyCurrentWaitingAdmissions)
        {
            onlyCurrentWaitingAdmissions = false;
            DateTime facilityAdmissionDate = DateTimeNowOnCurrentFacility;
            switch (model.CurrentFilter)
            {
                case Constants.CURRENT2:
                    model.WaitingListFilterType = Strings.CurrentlyWaiting;
                    facilityAdmissionDate = DateTimeNowOnCurrentFacility;
                    onlyCurrentWaitingAdmissions = true;
                    break;
                case Constants.TOMORROW:
                    onlyCurrentWaitingAdmissions = false;
                    model.WaitingListFilterType = Strings.WaitingList_Tomorrow;
                    facilityAdmissionDate = DateTimeNowOnCurrentFacility.AddDays(1);
                    break;
                case Constants.YESTERDAY:
                    onlyCurrentWaitingAdmissions = false;
                    model.WaitingListFilterType = Strings.WaitingList_Yesterday;
                    facilityAdmissionDate = DateTimeNowOnCurrentFacility.AddDays(-1);
                    break;
                case Constants.PAST_30_DAYS:
                    onlyCurrentWaitingAdmissions = false;
                    model.WaitingListFilterType = Strings.WaitingList_Past30;
                    facilityAdmissionDate = DateTimeNowOnCurrentFacility.AddDays(-30);
                    break;
            }

            return facilityAdmissionDate;
        }

        /// <summary>
        /// Builds the integrated sites.
        /// </summary>
        /// <param name="integratedSites">The integrated sites.</param>
        /// <param name="currentFacility">The current facility.</param>
        /// <returns></returns>
        private IList<Facility> BuildIntegratedSites(string integratedSites, Facility currentFacility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Facility> list = new List<Facility>();
                string[] identifiers = integratedSites.Split(',');
                Facility facility = null;
                II facilityId = null;
                foreach (string facilityIdentifier in identifiers)
                {
                    if (facilityIdentifier.Trim() != currentFacility.Id.extension)
                    {
                        facilityId = new II() { root = currentFacility.Id.root, extension = facilityIdentifier.Trim() };
                        if (this.loggedUser.IsSuperUser)
                            facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                        else
                            if (FacadeManager.UserInterface.CheckBMSReadPermission(this.loggedUser, facilityId))
                                facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                            else
                                continue;
                        if (facility != null)
                            list.Add(facility);
                    }
                }
                return list;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the view2 selected.
        /// </summary>
        /// <param name="hm">The home model.</param>
        private void SetView2Selected(HomeViewModel hm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (hm.View2List == null)
                    hm.View2List = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.PatientWaitingViews);
                foreach (CD obj in hm.View2List)
                {
                    if (obj.code == hm.View2 || obj.displayName == hm.View2)
                    {
                        if (string.IsNullOrEmpty(hm.LastSelectedView) || hm.LastSelectedView != obj.code)
                        {
                            hm.LastSelectedView = obj.code;
                            hm.IsChangeSelectedView = true;
                        }
                        else
                        {
                            hm.IsChangeSelectedView = false;
                        }
                        hm.View2Selected = obj;
                        hm.View2 = obj.code;
                        break;
                    }
                }

                hm.CurrentGridView = GetSaveNameFromView(hm.View2Selected.displayName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        public static string GetSaveArea([Bind(Exclude = "RememberMe")] HomeViewModel model)
        {
            string saveArea = "FacilityPPBP";
            if (!String.IsNullOrEmpty(model.CurrentGridView))
                saveArea += model.CurrentGridView;
            else
                saveArea += "StandardAdmissionGrid";

            return saveArea + model.LoggedUser.Facility.SiteNumber;
        }

        public static string GetSaveNameFromView(string view)
        {
            if (view.Equals(Strings.View_Standard))
                return "StandardAdmissionGrid";
            else if (view.Equals(Strings.View_FeeUtilization))
                return "FeeUtilizationAdmissionGrid";
            else if (view.Equals(Strings.View_PatientFlow))
                return "PatientFlowAdmissionGrid";
            else if (view.Equals(Strings.View_InHouse))
                return "PatientInHouseAdmissionGrid";
            else if (view.Equals(Strings.View_MentalHealth))
                return "MentalHealthAdmissionGrid";
            else if (view.Equals(Strings.View_EmergencyManagement))
                return "EmergencyManagementAdmissionGrid";
            else
                return "StandardAdmissionGrid";
        }
    }
}
