using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Models.Shared;
using BMS.Web.Views;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class FacilitySettingsController : BaseController
    {
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacilitySettingsViewModel vm = new FacilitySettingsViewModel();
                FillData(vm);
                return View(vm);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(FacilitySettingsViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                input.LoggedUser = this.loggedUser;
                FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(input.LoggedUser.Facility.Id);
                input.FacilitySiteTimeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);                
                input.StrictDecisionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                input.AutoRemovalWaitingList = input.StrictDecisionList.Where(a => a.code == input.AutoRemovalWaitingList.code).FirstOrDefault();
                input.AutoPlacementTransferList = input.StrictDecisionList.Where(a => a.code == input.AutoPlacementTransferList.code).FirstOrDefault();
                settings.AutoPlacementTransferList = input.AutoPlacementTransferList;
                settings.AutoRemovalWaitingList = input.AutoRemovalWaitingList;
                settings.FacilitySiteTimeZone = new CD() { code = input.FacilitySiteTimeZoneCode };                
                settings.LocalTimeAdjust = input.LocalTimeAdjust;
                settings.IntegratedFacility = input.Settings.IntegratedFacility;

                if (!string.IsNullOrEmpty(input.EMSUserName) && string.IsNullOrEmpty(input.EMSPassword) && string.IsNullOrEmpty(settings.EMSDefaultPwd))
                {
                    ViewData[InputHelpers.FormNameKey] = Strings.EMS;
                    ModelState.AddModelError("EMSPassword", "EMS Password is required!");
                    return View(input);
                }

                if (!string.IsNullOrEmpty(input.KioskUserName) && string.IsNullOrEmpty(input.KioskPassword) && string.IsNullOrEmpty(settings.KioskDefaultPwd))
                {
                    ViewData[InputHelpers.FormNameKey] = Strings.Kiosk;
                    ModelState.AddModelError("KioskPassword", "Kiosk Password is required!");
                    return View(input);
                }

                #region EIS Fields

                this.loggedUser.Facility = FacadeManager.EntityInterface.GetFacility(this.loggedUser.Facility.Id);

                this.loggedUser.Facility.Name = (input.FacilityName != null) ? input.FacilityName.ToUpper() : string.Empty;
                if ((this.loggedUser.Facility as Place).Address1 != null)
                    (this.loggedUser.Facility as Place).Address1.streetName = (input.FacilityAddress1 != null) ? input.FacilityAddress1.ToUpper() : string.Empty;
                else
                {
                    (this.loggedUser.Facility as Place).Address1 = new AD();
                    (this.loggedUser.Facility as Place).Address1.streetName = (input.FacilityAddress1 != null) ? input.FacilityAddress1.ToUpper() : string.Empty;
                }
                if ((this.loggedUser.Facility as Place).Address2 != null)
                    (this.loggedUser.Facility as Place).Address2.streetName = (input.FacilityAddress2 != null) ? input.FacilityAddress2.ToUpper() : string.Empty;
                else
                {
                    (this.loggedUser.Facility as Place).Address2 = new AD();
                    (this.loggedUser.Facility as Place).Address2.streetName = (input.FacilityAddress2 != null) ? input.FacilityAddress2.ToUpper() : string.Empty;
                }

                //set facility point of contact
                if ((this.loggedUser.Facility as Place).PersonContactName == null)
                    (this.loggedUser.Facility as Place).PersonContactName = new ST();
                (this.loggedUser.Facility as Place).PersonContactName.text = !string.IsNullOrWhiteSpace(input.FacilityPOCContact) ? input.FacilityPOCContact.ToUpper() : string.Empty;

                //set facility POC email
                if ((this.loggedUser.Facility as Place).PersonContactEmail == null)
                    (this.loggedUser.Facility as Place).PersonContactEmail = new TEL();
                (this.loggedUser.Facility as Place).PersonContactEmail.value = !string.IsNullOrWhiteSpace(input.FacilityPOCEmail) ? input.FacilityPOCEmail.ToLower() : string.Empty;

                //set facility POC telephone
                if ((this.loggedUser.Facility as Place).PersonContactPhone == null)
                    (this.loggedUser.Facility as Place).PersonContactPhone = new TEL();
                (this.loggedUser.Facility as Place).PersonContactPhone.value = !string.IsNullOrWhiteSpace(input.FacilityPOCTelephone) ? input.FacilityPOCTelephone.ToLower() : string.Empty;

                FacadeManager.EntityInterface.SaveFacility(this.loggedUser.Facility);

                #endregion

                //set integrated sites list
                if (input.Settings.IntegratedFacility == null || input.Settings.IntegratedFacility.code == Constants.No)
                {
                    // delete integrated sites list
                    settings.IntegratedSiteSisterList = string.Empty;
                    //delete medical center id
                    settings.MedicalCenterID = null;
                    settings.WardPrefix = null;
                    settings.WardSuffix = null;
                    settings.ADTPrefix = null;
                    settings.ADTSuffix = null;
                }
                else
                {
                    //save integrated sites list
                    if (input.IntegratedSiteList != null && input.IntegratedSiteList.Count > 0)
                    {
                        List<string> integratedSites = input.IntegratedSiteList.Where(a => a.IsIntegratedSite == true).Select(a => a.Site.Id.extension).ToList();
                        string integratedSiteList = string.Empty;

                        foreach (string identifier in integratedSites)
                        {
                            if (string.IsNullOrWhiteSpace(integratedSiteList))
                                integratedSiteList = identifier;
                            else
                                integratedSiteList += ", " + identifier;
                        }
                        settings.IntegratedSiteSisterList = integratedSiteList;
                    }


                    //delete medical center id
                    if (input.MedicalCenterID == Strings.ClearMedicalCenterId || input.MedicalCenterID == Strings.SelectMedicalCenterId)
                        settings.MedicalCenterID = null;
                    else
                        settings.MedicalCenterID = input.MedicalCenterID;
                    settings.WardPrefix = input.WardPrefix;
                    settings.WardSuffix = input.WardSuffix;
                    settings.ADTPrefix = input.ADTPrefix;
                    settings.ADTSuffix = input.ADTSuffix;
                }

                settings.EMSDefaultUserName = input.EMSUserName;
                var encryptionPwd = System.Configuration.ConfigurationManager.AppSettings["EncryptionPassword"];

                if (encryptionPwd != null && !string.IsNullOrEmpty(input.EMSPassword))
                    settings.EMSDefaultPwd = Encryptor.Encrypt(input.EMSPassword, encryptionPwd.ToString());                           

                settings.KioskDefaultUserName = input.KioskUserName;
                if (encryptionPwd != null && !string.IsNullOrEmpty(input.KioskPassword))
                    settings.KioskDefaultPwd = Encryptor.Encrypt(input.KioskPassword, encryptionPwd.ToString());

                FacadeManager.ConfigurationInterface.SaveFacilitySettings(this.loggedUser.Facility.Id, settings);

                VistASettings vistASettings = new VistASettings()
                {
                    LocalTimeAdjust = input.LocalTimeAdjust,
                    SiteTimeZone = new CD() { code = input.FacilitySiteTimeZoneCode }
                };

                FacadeManager.ConfigurationInterface.SaveVistASettings(vistASettings, this.loggedUser.Facility.VistaSite.Id);

                return View("SaveConfirmation", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillData(FacilitySettingsViewModel vm)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                vm.FacilitySiteTimeZoneList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.TimeZone);
                vm.StrictDecisionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                vm.LoggedUser = this.loggedUser;
                vm.Settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(vm.LoggedUser.Facility.Id);
                vm.AutoRemovalWaitingList = vm.Settings.AutoRemovalWaitingList;
                vm.AutoPlacementTransferList = vm.Settings.AutoPlacementTransferList;
                vm.BMSServerTimeZone = TimeZoneInfo.Local.StandardName.ToUpper();

                #region EIS Fields
                Place facility = this.loggedUser.Facility as Place;
                vm.FacilityAddress1 = facility.Address1 != null ? facility.Address1.streetName : string.Empty;
                vm.FacilityAddress2 = facility.Address2 != null ? facility.Address2.streetName : string.Empty;
                vm.FacilityName = facility.Name ?? string.Empty;
                vm.FacilityPOCContact = facility.PersonContactName != null ? facility.PersonContactName.text : string.Empty;
                vm.FacilityPOCEmail = facility.PersonContactEmail != null ? facility.PersonContactEmail.value.Contains(EISConstants.EMAIL_CONTACT_VALUE) == true ? facility.PersonContactEmail.value.Substring(EISConstants.EMAIL_CONTACT_VALUE.Length) : facility.PersonContactEmail.value : string.Empty;
                vm.FacilityPOCTelephone = facility.PersonContactPhone != null ? facility.PersonContactPhone.value.Contains(EISConstants.TELEPHONE_CONTACT_VALUE) == true ? facility.PersonContactPhone.value.Substring(EISConstants.TELEPHONE_CONTACT_VALUE.Length) : facility.PersonContactPhone.value : string.Empty;
                #endregion

                vm.FacilitySiteTimeZoneCode = vm.Settings.FacilitySiteTimeZone != null ? vm.Settings.FacilitySiteTimeZone.code : null;
                vm.IntegratedFacility = vm.Settings.IntegratedFacility;
                vm.LocalTimeAdjust = vm.Settings.LocalTimeAdjust;

                if (vm.Settings.IntegratedFacility != null && vm.Settings.IntegratedFacility.code == Constants.Yes)
                    vm.IsIntegratedFacility = true;
                else
                    vm.IsIntegratedFacility = false;

                vm.EMSUserName = vm.Settings.EMSDefaultUserName;
                vm.KioskUserName = vm.Settings.KioskDefaultUserName;

                //set integrated sites
                if (vm.IsIntegratedFacility && vm.Settings.IntegratedSiteSisterGroupId != 0)
                {
                    vm.DisplayAllowedSites = true;
                    vm.IntegratedSiteList = BuildIntegratedSitesList(vm.Settings.SisterSitesGroups.Where(a => a.RecordNumber == vm.Settings.IntegratedSiteSisterGroupId).FirstOrDefault(), vm.LoggedUser.Facility.Id.extension, vm.Settings.IntegratedSiteSisterList);
                }
                else
                {
                    vm.DisplayAllowedSites = false;
                }

                if (vm.IsIntegratedFacility)
                {
                    vm.ADTPrefix = vm.Settings.ADTPrefix;
                    vm.ADTSuffix = vm.Settings.ADTSuffix;
                    vm.WardPrefix = vm.Settings.WardPrefix;
                    vm.WardSuffix = vm.Settings.WardSuffix;
                    vm.VistaDivisionList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaMedicalCenterDivision).Select(a => new CD() { code = a.code, codeSystem = a.codeSystem, codeSystemName = a.codeSystemName, displayName = a.displayName.ToUpper() + " {" + Strings.IEN + ":" + a.code.Substring(a.code.IndexOf("_") + 1) + "}" }).ToList();
                    if (vm.VistaDivisionList == null)
                        vm.VistaDivisionList = new List<CD>();
                    if (vm.Settings.MedicalCenterID != null)
                    {
                        vm.VistaDivisionList.Insert(0, new CD() { code = Strings.ClearMedicalCenterId, displayName = Strings.ClearMedicalCenterId });
                        vm.MedicalCenterID = vm.Settings.MedicalCenterID;
                    }
                    else
                    {
                        vm.VistaDivisionList.Insert(0, new CD() { code = Strings.SelectMedicalCenterId, displayName = Strings.SelectMedicalCenterId });
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Builds the integrated sites list.
        /// </summary>
        /// <param name="sisterSitesGroup">The sister sites group.</param>
        /// <param name="facilityIdentifier">The facility identifier.</param>
        /// <param name="integratedSiteSisterList">The integrated site sister list.</param>
        /// <returns></returns>
        private IList<FacilitySisterSitesGroupViewModel> BuildIntegratedSitesList(SisterSitesGroup sisterSitesGroup, string facilityIdentifier, string integratedSiteSisterList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<FacilitySisterSitesGroupViewModel> integratedSites = new List<FacilitySisterSitesGroupViewModel>();
                if (sisterSitesGroup != null)
                {
                    if (!string.IsNullOrWhiteSpace(sisterSitesGroup.SiteCodeList))
                    {
                        if (sisterSitesGroup.SiteCodeList.Contains(facilityIdentifier))
                        {
                            string[] facilitiesIdentifiers = sisterSitesGroup.SiteCodeList.Split(',');
                            foreach (string identifier in facilitiesIdentifiers)
                            {
                                if (identifier.Trim() != facilityIdentifier)
                                {
                                    Facility facility = FacadeManager.EntityInterface.GetFacility(new II() { root = this.loggedUser.Facility.Id.root, extension = identifier.Trim() });
                                    if (facility != null)
                                    {
                                        if (!string.IsNullOrWhiteSpace(integratedSiteSisterList) && integratedSiteSisterList.Contains(identifier.Trim()))
                                            integratedSites.Add(new FacilitySisterSitesGroupViewModel() { Site = facility, IsIntegratedSite = true });
                                        else
                                            integratedSites.Add(new FacilitySisterSitesGroupViewModel() { Site = facility, IsIntegratedSite = false });
                                    }
                                }

                            }
                        }
                    }
                }
                return integratedSites;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
