﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using Microsoft.Web.Mvc;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.ServicesWrapper.WF;
using InfoWorld.EVS.CTSMAPI;
using BMS.ServicesWrapper.BMService;
using BMS.Facade.Translators;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Bed Information Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class BedInformationController : BaseController
    {
        /// <summary>
        /// Default action for the BedInformationController.
        /// </summary>
        /// <param name="rootBedId">The root bed id.</param>
        /// <param name="extensionBedId">The extension bed id.</param>
        /// <param name="bedStatusCode">The bed status code.</param>
        /// <param name="extensionDivisionId">The extension division id.</param>
        /// <param name="rootDivisionId">The root division id.</param>
        /// <param name="divisionName">Name of the division.</param>
        /// <param name="isFromSummaryReport">if set to <c>true</c> [is from summary report].</param>
        /// <param name="summaryReportPath">The summary report path.</param>
        /// <param name="summaryReportText">The summary report text.</param>
        /// <param name="visnRoot">The visn root used in visn summary report.</param>
        /// <param name="visnExtension">The visn extension used in visn summary report.</param>
        /// <returns>
        /// Returns the Bed Information view result.
        /// </returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedInfoViewModel bedInfoModel = new BedInfoViewModel();
                bedInfoModel.RootBedId = loggedUser.Domain;
                bedInfoModel.ExtensionBedId = QueryStrings["bedId"];
                bedInfoModel.BedStatusCode = QueryStrings["bedStatus"];
                bedInfoModel.ExtensionWardGroupId = QueryStrings["divisionId"];
                bedInfoModel.RootWardGroupId = loggedUser.Domain;
                bedInfoModel.SelectedWardGroupName = QueryStrings["divisionName"];
                if (!string.IsNullOrEmpty(bedInfoModel.ExtensionWardGroupId) && !bedInfoModel.ExtensionWardGroupId.Equals(Guid.Empty.ToString()))
                {
                    bedInfoModel.WardGroupName = bedInfoModel.SelectedWardGroupName;
                    bedInfoModel.Division = new Division() { Id = new II(bedInfoModel.RootWardGroupId, bedInfoModel.ExtensionWardGroupId), Name = bedInfoModel.SelectedWardGroupName };
                }
                else
                {
                    IList<Division> divisionList = FacadeManager.EntityInterface.GetDivisionsByBedAndFacility(new II(bedInfoModel.RootBedId, bedInfoModel.ExtensionBedId), this.loggedUser.Facility.Id);
                    if (divisionList != null && divisionList.Count > 0)
                    {
                        bedInfoModel.Division = divisionList[0];
                        bedInfoModel.WardGroupName = bedInfoModel.Division.Name;
                    }
                }
                bedInfoModel.RealDivisionSelectedExtension = bedInfoModel.Division.Id.extension;
                bedInfoModel.RealDivisionSelectedRoot = bedInfoModel.Division.Id.root;
                bedInfoModel.IsFromSummaryReport = bool.Parse(QueryStrings["isFromSummaryReport"]);
                bedInfoModel.SummaryReportReturnPath = QueryStrings["summaryReportReturnPath"];
                bedInfoModel.SummaryReportReturnText = QueryStrings["summaryReportReturnText"];
                bedInfoModel.VisnRoot = loggedUser.Domain;
                bedInfoModel.VisnExtension = QueryStrings["visnId"];
                FillViewModel(bedInfoModel);

                return View(bedInfoModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the BedInformationController.
        /// </summary>
        /// <param name="bedInfoModel">The bed information model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilter]
        public ActionResult Index(BedInfoViewModel bedInfoModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (bedInfoModel.SetVisibilityForExpectedOrActualCompletedDate != null)
                {
                    bedInfoModel.Bed = FacadeManager.BedInterface.GetBed(new II() { extension = bedInfoModel.ExtensionBedId, root = bedInfoModel.RootBedId }, loggedUser.VistaSite.Id);
                    bedInfoModel.ReasonList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.UnavailableReason);
                    bedInfoModel.LoggedUser = this.loggedUser;
                    //if bed has unavailable reason add option to delete this reason
                    if (!string.IsNullOrEmpty(bedInfoModel.UnavailableReasonCode))
                        bedInfoModel.ReasonList.Insert(0, new CD() { code = Strings.DeleteThisReason, displayName = Strings.DeleteThisReason });

                    IList<Ward> wardList = FacadeManager.EntityInterface.GetWardsByDivisionId(new II() { extension = bedInfoModel.RealDivisionSelectedExtension, root = bedInfoModel.RealDivisionSelectedRoot });
                    bedInfoModel.WardList = new List<Ward>();
                    Ward ward = null;
                    foreach (Ward w in bedInfoModel.Bed.WardList)
                    {
                        ward = wardList.Where(a => a.Id.extension.Equals(w.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ward != null)
                            bedInfoModel.WardList.Add(ward);
                    }

                    if (GetTypeCode(bedInfoModel.SelectedReason) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                    {
                        if (bedInfoModel.IsActualCompletedDateVisible)
                            bedInfoModel.IsActualCompletedDateVisible = false;

                        bedInfoModel.IsExpectedCompletedDateVisible = true;
                        if (bedInfoModel.ExpectedCompletedDate.HasValue)
                        {
                            bedInfoModel.ExpectedCompletedDateHourSelected = (bedInfoModel.ExpectedCompletedDate.Value.Hour < 10) ? "0" + bedInfoModel.ExpectedCompletedDate.Value.Hour.ToString() : bedInfoModel.ExpectedCompletedDate.Value.Hour.ToString();
                            bedInfoModel.ExpectedCompletedDateMinuteSelected = (bedInfoModel.ExpectedCompletedDate.Value.Minute < 10) ? "0" + bedInfoModel.ExpectedCompletedDate.Value.Minute.ToString() : bedInfoModel.ExpectedCompletedDate.Value.Minute.ToString();
                        }
                        else
                        {
                            bedInfoModel.ExpectedCompletedDateHourSelected = "00";
                            bedInfoModel.ExpectedCompletedDateMinuteSelected = "00";
                        }
                    }
                    else
                    {

                        bedInfoModel.IsExpectedCompletedDateVisible = false;
                        bedInfoModel.ExpectedCompletedDateHourSelected = "00";
                        bedInfoModel.ExpectedCompletedDateMinuteSelected = "00";

                        if (GetTypeCode(bedInfoModel.UnavailableReasonCode) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE && (string.IsNullOrWhiteSpace(bedInfoModel.SelectedReason) || bedInfoModel.SelectedReason == Strings.DeleteThisReason))
                        {
                            bedInfoModel.IsActualCompletedDateVisible = true;
                            bedInfoModel.ActualCompletedDate = DateTimeNowOnCurrentFacility;
                            bedInfoModel.ActualCompletedDateHourSelected = (bedInfoModel.ActualCompletedDate.Hour < 10) ? "0" + bedInfoModel.ActualCompletedDate.Hour.ToString() : bedInfoModel.ActualCompletedDate.Hour.ToString();
                            bedInfoModel.ActualCompletedDateMinuteSelected = (bedInfoModel.ActualCompletedDate.Minute < 10) ? "0" + bedInfoModel.ActualCompletedDate.Minute.ToString() : bedInfoModel.ActualCompletedDate.Minute.ToString();
                        }
                        else
                            bedInfoModel.IsActualCompletedDateVisible = false;
                    }
                    bedInfoModel.BedClean = FacadeManager.BedInterface.GetBedCleanByBedId(new II(bedInfoModel.RootBedId, bedInfoModel.ExtensionBedId), loggedUser.Facility.Id);
                    bedInfoModel.EMSNotifyIcon = FacadeManager.IconInterface.GetIconByCode(Constants.EMS_NOTIFIED);
                    if (bedInfoModel.BedClean != null)
                    {
                        if (bedInfoModel.BedClean.CompletedDate.HasValue)
                            bedInfoModel.IsBedCleanCompleted = true;
                        else
                        {
                            if (!string.IsNullOrEmpty(bedInfoModel.BedClean.AcceptedBy))
                                bedInfoModel.IsBedCleanAccepted = true;
                            else
                                bedInfoModel.IsBedCleanRequested = true;
                        }
                        if (!bedInfoModel.IsBedCleanCompleted)
                        {
                            bedInfoModel.RequestCleaning = bedInfoModel.BedClean.TypeOfClean.code;
                            bedInfoModel.Ward = bedInfoModel.BedClean.Ward.Id.extension;
                            bedInfoModel.CleaningDate = (bedInfoModel.BedClean.RequestedDate.HasValue) ? bedInfoModel.BedClean.RequestedDate.Value : DateTimeNowOnCurrentFacility;
                            bedInfoModel.SpecialInstructions = bedInfoModel.BedClean.SpecialInstructions;
                            if (bedInfoModel.CleaningDate.Hour.ToString().Length == 1)
                                bedInfoModel.HourSelected = bedInfoModel.CleaningDate.Hour.ToString().Replace(bedInfoModel.CleaningDate.Hour.ToString(), "0" + bedInfoModel.CleaningDate.Hour.ToString());
                            else
                                bedInfoModel.HourSelected = bedInfoModel.CleaningDate.Hour.ToString();
                            if (bedInfoModel.CleaningDate.Minute.ToString().Length == 1)
                                bedInfoModel.MinuteSelected = bedInfoModel.CleaningDate.Minute.ToString().Replace(bedInfoModel.CleaningDate.Minute.ToString(), "0" + bedInfoModel.CleaningDate.Minute.ToString());
                            else
                                bedInfoModel.MinuteSelected = bedInfoModel.CleaningDate.Minute.ToString();
                        }
                    }
                    else
                    {
                        bedInfoModel.IsBedCleanRequested = false;
                        bedInfoModel.IsBedCleanAccepted = false;
                    }

                    List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(this.loggedUser.Domain, this.loggedUser.Facility.Id).ToList();
                    if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                    {
                        bedInfoModel.EMSModuleInactive = true;
                        bedInfoModel.DisplayEMSModuleInactive = @"style='display:none;'";
                    }
                    else
                    {
                        bedInfoModel.EMSModuleInactive = false;
                        bedInfoModel.DisplayEMSModuleInactive = string.Empty;
                    }
                    bedInfoModel.LoggedUser = loggedUser;
                    return View(bedInfoModel);
                }
                else
                {
                    if (!string.IsNullOrEmpty(bedInfoModel.CancelButton))
                        return this.RedirectToAction<WardOccupancyController>(act => act.Index(EncryptQueryString(new string[] { "divisionId", "bedStatus", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { bedInfoModel.ExtensionWardGroupId, bedInfoModel.BedStatusCode, bedInfoModel.IsFromSummaryReport.ToString(), bedInfoModel.SummaryReportReturnPath, bedInfoModel.SummaryReportReturnText, bedInfoModel.VisnExtension }, loggedUser.Salt)));

                    CD bedStatus = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.BedStatus).Where(a => a.code == bedInfoModel.BedStatusCode).FirstOrDefault();
                    Bed bed = FacadeManager.BedInterface.GetBed(new II(bedInfoModel.RootBedId, bedInfoModel.ExtensionBedId), loggedUser.VistaSite.Id);
                    bool isFirstComment = true;
                    bool isChangedComment = false;
                    string lastComment = bed.CommentList.Where(a => a.Division.Id.root.Equals(bedInfoModel.RealDivisionSelectedRoot, StringComparison.InvariantCultureIgnoreCase) && a.Division.Id.extension.Equals(bedInfoModel.RealDivisionSelectedExtension)).FirstOrDefault().Comment;
                    if (!string.IsNullOrWhiteSpace(lastComment))
                        isFirstComment = false;

                    //verify if comment changed
                    if (bedInfoModel.Comments != null && lastComment != bedInfoModel.Comments)
                        isChangedComment = true;

                    DateTime dateNow = DateTimeNowOnCurrentFacility;

                    if (!string.IsNullOrEmpty(bedInfoModel.ClearButton))
                    {
                        bedInfoModel.Comments = string.Empty;
                        bedInfoModel.Bed.Reason = null;
                        bedInfoModel.SelectedReason = string.Empty;
                    }

                    //get new reason and type
                    CDWithProperties reason = null;
                    CD type = null;
                    bool isReasonAvailable = true;
                    GetReason(bedInfoModel.SelectedReason, out reason, out type, out isReasonAvailable);
                    bedInfoModel.IsSelectedReasonAvailable = isReasonAvailable;
                    if (bedInfoModel.IsSelectedReasonAvailable)
                    {
                        BedUnavailable bedUnavailable = SaveBedUnavailable(bedInfoModel, bed, dateNow, isChangedComment, reason, type);

                        #region Save EIS Bed Traits
                        if (!string.IsNullOrEmpty(bedInfoModel.ClearButton))
                        {
                            List<Division> divisions = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                            foreach (BedComment bc in bed.CommentList)
                            {
                                if (bc.Division != null && bc.Division.Id != null && divisions.Where(a => a.Id.extension.Equals(bc.Division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                                    bc.Comment = string.Empty;
                            }
                        }
                        else
                        {
                            if (bed.CommentList.Where(a => a.Division.Id.extension.Equals(bedInfoModel.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                                bed.CommentList.Where(a => a.Division.Id.extension.Equals(bedInfoModel.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Comment = bedInfoModel.Comments;
                            else
                                bed.CommentList.Add(new BedComment() { Division = new Division() { Id = new II(bedInfoModel.RealDivisionSelectedRoot, bedInfoModel.RealDivisionSelectedExtension) }, Comment = bedInfoModel.Comments });
                        }
                        FacadeManager.BedInterface.UpdateBed(bed);

                        #endregion

                        if (isChangedComment || !bedInfoModel.BedUnavailableHistory.IsChangedOnlyComment)
                        {
                            //insert into history table
                            if (bedUnavailable != null)
                            {
                                if (!bedInfoModel.BedUnavailableHistory.EditedDate.HasValue && !bedInfoModel.BedUnavailableHistory.CanceledDate.HasValue)
                                {
                                    bedInfoModel.BedUnavailableHistory.EditedDate = dateNow;
                                    bedInfoModel.BedUnavailableHistory.EditedBy = this.loggedUser.UserName;
                                }
                            }
                            else
                            {
                                bedInfoModel.BedUnavailableHistory.Bed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null;
                                bedInfoModel.BedUnavailableHistory.CanceledBy = null;
                                bedInfoModel.BedUnavailableHistory.CanceledDate = null;
                                bedInfoModel.BedUnavailableHistory.CreatedBy = this.loggedUser.UserName;
                                bedInfoModel.BedUnavailableHistory.CreationDate = dateNow;
                                if (isFirstComment)
                                {
                                    bedInfoModel.BedUnavailableHistory.EditedBy = null;
                                    bedInfoModel.BedUnavailableHistory.EditedDate = null;
                                }
                                else
                                {
                                    bedInfoModel.BedUnavailableHistory.EditedBy = this.loggedUser.UserName;
                                    bedInfoModel.BedUnavailableHistory.EditedDate = dateNow;
                                }
                                bedInfoModel.BedUnavailableHistory.Id = new II(this.loggedUser.Domain, null);
                                bedInfoModel.BedUnavailableHistory.Parent = null;
                                bedInfoModel.BedUnavailableHistory.Patient = null;
                                bedInfoModel.BedUnavailableHistory.Reason = null;
                                bedInfoModel.BedUnavailableHistory.Type = null;
                                bedInfoModel.BedUnavailableHistory.VistaSite = this.loggedUser.VistaSite;
                            }

                            bedInfoModel.BedUnavailableHistory.Comment = bedInfoModel.Comments;
                            bedInfoModel.BedUnavailableHistory.Division = new Division() { Id = new II(bedInfoModel.RealDivisionSelectedRoot, bedInfoModel.RealDivisionSelectedExtension) };
                            if (!string.IsNullOrEmpty(bedInfoModel.ClearButton))
                                bedInfoModel.BedUnavailableHistory.IsClearAll = true;
                            foreach (BedComment bc in bed.CommentList)
                            {
                                if (bedInfoModel.BedUnavailableHistory.DivisionList == null)
                                    bedInfoModel.BedUnavailableHistory.DivisionList = new List<Division>();
                                if (bedInfoModel.BedUnavailableHistory.DivisionList.Where(a => a.Id.root.Equals(bc.Division.Id.root, StringComparison.InvariantCultureIgnoreCase) && a.Id.extension.Equals(bc.Division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                                    bedInfoModel.BedUnavailableHistory.DivisionList.Add(bc.Division);
                            }
                            FacadeManager.BedInterface.InsertBedUnavailableHistory(bedInfoModel.BedUnavailableHistory, this.loggedUser.Facility.Id);
                        }
                    }
                    else
                    {
                        //display message
                        return this.RedirectToAction<BedInformationController>(act => act.NotifyChange(EncryptQueryString(new string[] { "bedId", "bedStatus", "divisionId", "divisionName", "bedName", "wardName", "isManualCleaning", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId", "isReasonAvailable" }, new string[] { bedInfoModel.ExtensionBedId, bedInfoModel.BedStatusCode, bedInfoModel.ExtensionWardGroupId, bedInfoModel.WardGroupName, bedInfoModel.Bed.Name, string.Empty, "false", bedInfoModel.IsFromSummaryReport.ToString(), bedInfoModel.SummaryReportReturnPath, bedInfoModel.SummaryReportReturnText, bedInfoModel.VisnExtension, bedInfoModel.IsSelectedReasonAvailable.ToString() }, loggedUser.Salt)));
                    }

                    //display clear all view
                    if (!string.IsNullOrEmpty(bedInfoModel.ClearButton))
                        return this.RedirectToAction<BedInformationController>(act => act.ClearAll(EncryptQueryString(new string[] { "divisionId", "divisionName", "bedName", "bedId", "bedStatus", "wardName", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { bedInfoModel.ExtensionWardGroupId, bedInfoModel.SelectedWardGroupName, bedInfoModel.Bed.Name, bedInfoModel.ExtensionBedId, bedInfoModel.BedStatusCode, string.Empty, bedInfoModel.IsFromSummaryReport.ToString(), bedInfoModel.SummaryReportReturnPath, bedInfoModel.SummaryReportReturnText, bedInfoModel.VisnExtension }, loggedUser.Salt)));

                    bool isBedCleanRequested = false;
                    if (!bedInfoModel.RequestCleaning.Equals(Strings.No))
                    {
                        if (!bedInfoModel.IsBedCleanAccepted)
                        {
                            SaveManualCleaningRequest(bedInfoModel, bed, dateNow);
                            if (!bedInfoModel.IsBedCleanRequested)
                                isBedCleanRequested = true;
                        }
                    }

                    return this.RedirectToAction<BedInformationController>(act => act.NotifyChange(EncryptQueryString(new string[] { "divisionId", "divisionName", "bedStatus", "bedName", "wardName", "isManualCleaning", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId", "isReasonAvailable" }, new string[] { bedInfoModel.ExtensionWardGroupId, bedInfoModel.SelectedWardGroupName, bedInfoModel.BedStatusCode, bedInfoModel.Bed.Name, string.Empty, isBedCleanRequested.ToString(), bedInfoModel.IsFromSummaryReport.ToString(), bedInfoModel.SummaryReportReturnPath, bedInfoModel.SummaryReportReturnText, bedInfoModel.VisnExtension, bedInfoModel.IsSelectedReasonAvailable.ToString() }, loggedUser.Salt)));
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Clears all.
        /// </summary>
        /// <param name="rootWardId">The root ward id.</param>
        /// <param name="extensionWardId">The extension ward id.</param>
        /// <param name="wardGroupName">Name of the ward.</param>
        /// <param name="bedName">Name of the bed.</param>
        /// <param name="rootBedId">The root bed id.</param>
        /// <param name="extensionBedId">The extension bed id.</param>
        /// <param name="bedStatus">The bed status code.</param>
        /// <param name="bedStatus">The bed status name.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ClearAll(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedInfoClearAllCommentsViewModel model = new BedInfoClearAllCommentsViewModel();
                model.RootWardGroupId = loggedUser.Domain;
                model.ExtensionWardGroupId = QueryStrings["divisionId"];
                model.BedStatusCode = QueryStrings["bedStatus"];
                model.WardGroupName = QueryStrings["divisionName"];
                model.BedName = QueryStrings["bedName"];
                model.RootBedId = loggedUser.Domain;
                model.ExtensionBedId = QueryStrings["bedId"];
                model.WardName = QueryStrings["wardName"];
                model.IsFromSummaryReport = bool.Parse(QueryStrings["isFromSummaryReport"]);
                model.SummaryReportReturnPath = QueryStrings["summaryReportReturnPath"];
                model.SummaryReportReturnText = QueryStrings["summaryReportReturnText"];
                model.VisnRoot = loggedUser.Domain;
                model.VisnExtension = QueryStrings["visnId"];
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Notifies the change.
        /// </summary>
        /// <param name="rootWardId">The root ward id.</param>
        /// <param name="extensionWardId">The extension ward id.</param>
        /// <param name="wardGroupName">Name of the ward group.</param>
        /// <param name="bedStatusCode">The bed status code.</param>
        /// <param name="bedName">Name of the bed.</param>
        /// <param name="wardName">Name of the ward.</param>
        /// <param name="isManualCleaning">if set to <c>true</c> [is manual cleaning].</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult NotifyChange(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedInfoNotifyChangeViewModel model = new BedInfoNotifyChangeViewModel();
                model.BedName = QueryStrings["bedName"];
                model.BedStatusCode = QueryStrings["bedStatus"];
                model.ExtensionWardGroupId = QueryStrings["divisionId"];
                model.RootWardGroupId = loggedUser.Domain;
                model.WardGroupName = QueryStrings["divisionName"];
                model.WardName = QueryStrings["wardName"];
                model.IsFromSummaryReport = bool.Parse(QueryStrings["isFromSummaryReport"]);
                model.SummaryReportReturnPath = QueryStrings["summaryReportReturnPath"];
                model.SummaryReportReturnText = QueryStrings["summaryReportReturnText"];
                model.VisnRoot = loggedUser.Domain;
                model.VisnExtension = QueryStrings["visnId"];
                if (QueryStrings.ContainsKey("bedId"))
                    model.BedId = QueryStrings["bedId"];
                model.IsSelectedReasonAvailable = bool.Parse(QueryStrings["isReasonAvailable"]);
                if (bool.Parse(QueryStrings["isManualCleaning"]))
                    model.DisplayManualCleaningMessage = Strings.DisplayActionLinkBlockValue;
                else
                    model.DisplayManualCleaningMessage = Strings.DiplayActionLinkNoneValue;
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for Notify change view.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult NotifyChange(BedInfoNotifyChangeViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model.IsSelectedReasonAvailable)
                    //return to ward whiteboard page
                    return this.RedirectToAction<WardOccupancyController>(act => act.Index(EncryptQueryString(new string[] { "divisionId", "bedStatus", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { model.ExtensionWardGroupId, model.BedStatusCode, model.IsFromSummaryReport.ToString(), model.SummaryReportReturnPath, model.SummaryReportReturnText, model.VisnExtension }, loggedUser.Salt)));
                else
                    return this.RedirectToAction<BedInformationController>(act => act.Index(EncryptQueryString(new string[] { "bedId", "bedStatus", "divisionId", "divisionName", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { model.BedId, model.BedStatusCode, model.ExtensionWardGroupId, model.WardGroupName, model.IsFromSummaryReport.ToString(), model.SummaryReportReturnPath, model.SummaryReportReturnText, model.VisnExtension }, loggedUser.Salt)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for Clear all view.
        /// </summary>
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="btnReturnToWardWhiteboardPage">The button return to ward occupancy page.</param>
        /// <returns></returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ClearAll([Bind(Exclude = "RememberMe")] BedInfoClearAllCommentsViewModel model, string btnReturnToWardOccupancyPage)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(btnReturnToWardOccupancyPage))
                    //return to ward whiteboard page
                    return this.RedirectToAction<WardOccupancyController>(act => act.Index(EncryptQueryString(new string[] { "divisionId", "bedStatus", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { model.ExtensionWardGroupId, model.BedStatusCode, model.IsFromSummaryReport.ToString(), model.SummaryReportReturnPath, model.SummaryReportReturnText, model.VisnExtension }, loggedUser.Salt)));
                else
                    // return to edit bed unavailable page
                    return this.RedirectToAction<BedInformationController>(act => act.Index(EncryptQueryString(new string[] { "bedId", "bedStatus", "divisionId", "divisionName", "isFromSummaryReport", "summaryReportReturnPath", "summaryReportReturnText", "visnId" }, new string[] { model.ExtensionBedId, model.BedStatusCode, model.ExtensionWardGroupId, model.WardGroupName, model.IsFromSummaryReport.ToString(), model.SummaryReportReturnPath, model.SummaryReportReturnText, model.VisnExtension }, loggedUser.Salt)));

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the view model.
        /// </summary>
        /// <param name="bedInfoModel">The bed info model.</param>
        private void FillViewModel(BedInfoViewModel bedInfoModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II bedId = new II(bedInfoModel.RootBedId, bedInfoModel.ExtensionBedId);
                bedInfoModel.Bed = FacadeManager.BedInterface.GetBed(bedId, loggedUser.VistaSite.Id);
                bedInfoModel.LoggedUser = this.loggedUser;
                BedUnavailable bedUnavailable = FacadeManager.BedInterface.GetBedUnavailable(bedId, loggedUser.Facility.Id);
                if (bedUnavailable != null)
                {
                    bedInfoModel.Bed.Reason = bedUnavailable.Reason;
                    bedInfoModel.UnavailableReasonCode = bedUnavailable.Reason.code;
                    bedInfoModel.SelectedReason = bedUnavailable.Reason.code;

                    bedInfoModel.ExpectedCompletedDate = bedUnavailable.ExpectedCompletedDate;
                    if (bedUnavailable.ExpectedCompletedDate.HasValue)
                    {
                        bedInfoModel.ExpectedCompletedDateHourSelected = (bedInfoModel.ExpectedCompletedDate.Value.Hour < 10) ? "0" + bedInfoModel.ExpectedCompletedDate.Value.Hour.ToString() : bedInfoModel.ExpectedCompletedDate.Value.Hour.ToString();
                        bedInfoModel.ExpectedCompletedDateMinuteSelected = (bedInfoModel.ExpectedCompletedDate.Value.Minute < 10) ? "0" + bedInfoModel.ExpectedCompletedDate.Value.Minute.ToString() : bedInfoModel.ExpectedCompletedDate.Value.Minute.ToString();
                    }
                    else
                    {
                        bedInfoModel.ExpectedCompletedDateHourSelected = "00";
                        bedInfoModel.ExpectedCompletedDateMinuteSelected = "00";
                    }
                }
                else
                {
                    bedInfoModel.UnavailableReasonCode = string.Empty;
                    bedInfoModel.SelectedReason = string.Empty;
                }

                if (string.IsNullOrEmpty(bedInfoModel.UnavailableReasonCode))
                {
                    bedInfoModel.IsExpectedCompletedDateVisible = false;
                    bedInfoModel.ExpectedCompletedDateHourSelected = "00";
                    bedInfoModel.ExpectedCompletedDateMinuteSelected = "00";
                    bedInfoModel.IsBedUnavailableReasonAddOperation = true;
                    bedInfoModel.UnavailableReasonTableHeader = Strings.AddAnUnavailableReason;
                }
                else
                {
                    //set visibility for expected completed date
                    if (GetTypeCode(bedInfoModel.UnavailableReasonCode) == Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                        bedInfoModel.IsExpectedCompletedDateVisible = true;

                    bedInfoModel.IsBedUnavailableReasonAddOperation = false;
                    bedInfoModel.UnavailableReasonTableHeader = Strings.EditExistingUnavailableReason;
                }
                BedComment bc = bedInfoModel.Bed.CommentList.Where(a => a.Division.Id.extension.Equals(bedInfoModel.RealDivisionSelectedExtension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                bedInfoModel.Comments = (bc != null) ? bc.Comment : string.Empty;
                bedInfoModel.ReasonList = FacadeManager.VocabularyInterface.GetVocabulary(BMS.Facade.Data.Util.Vocabulary.UnavailableReason);

                //if bed has unavailable reason add option to delete this reason
                if (!string.IsNullOrEmpty(bedInfoModel.UnavailableReasonCode))
                {
                    bedInfoModel.ReasonList.Insert(0, new CD() { code = Strings.DeleteThisReason, displayName = Strings.DeleteThisReason });
                    if (bedInfoModel.ReasonList.Where(a => a.code.Equals(bedInfoModel.UnavailableReasonCode, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                        bedInfoModel.ReasonList.Add(bedInfoModel.Bed.Reason);
                }

                IList<Ward> wardList = FacadeManager.EntityInterface.GetWardsByDivisionId(bedInfoModel.Division.Id);
                bedInfoModel.WardList = new List<Ward>();
                Ward ward = null;
                foreach (Ward w in bedInfoModel.Bed.WardList)
                {
                    ward = wardList.Where(a => a.Id.extension.Equals(w.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (ward != null)
                        bedInfoModel.WardList.Add(ward);
                }
                bedInfoModel.IsActualCompletedDateVisible = false;

                bedInfoModel.BedClean = FacadeManager.BedInterface.GetBedCleanByBedId(bedId, loggedUser.Facility.Id);
                bedInfoModel.EMSNotifyIcon = FacadeManager.IconInterface.GetIconByCode(Constants.EMS_NOTIFIED);
                if (bedInfoModel.BedClean != null)
                {
                    if (bedInfoModel.BedClean.CompletedDate.HasValue)
                        bedInfoModel.IsBedCleanCompleted = true;
                    else
                    {
                        if (!string.IsNullOrEmpty(bedInfoModel.BedClean.AcceptedBy))
                            bedInfoModel.IsBedCleanAccepted = true;
                        else
                            bedInfoModel.IsBedCleanRequested = true;
                    }
                    if (!bedInfoModel.IsBedCleanCompleted)
                    {
                        bedInfoModel.RequestCleaning = bedInfoModel.BedClean.TypeOfClean.code;
                        bedInfoModel.Ward = bedInfoModel.BedClean.Ward.Id.extension;
                        bedInfoModel.CleaningDate = (bedInfoModel.BedClean.RequestedDate.HasValue) ? bedInfoModel.BedClean.RequestedDate.Value : DateTimeNowOnCurrentFacility;
                        bedInfoModel.SpecialInstructions = bedInfoModel.BedClean.SpecialInstructions;
                    }
                    else
                    {
                        bedInfoModel.RequestCleaning = Strings.No;
                        bedInfoModel.CleaningDate = DateTimeNowOnCurrentFacility;
                        bedInfoModel.IsBedCleanRequested = false;
                        bedInfoModel.IsBedCleanAccepted = false;
                    }
                }
                else
                {
                    bedInfoModel.RequestCleaning = Strings.No;
                    bedInfoModel.CleaningDate = DateTimeNowOnCurrentFacility;
                    bedInfoModel.IsBedCleanRequested = false;
                    bedInfoModel.IsBedCleanAccepted = false;
                }
                if (bedInfoModel.CleaningDate.Hour.ToString().Length == 1)
                    bedInfoModel.HourSelected = bedInfoModel.CleaningDate.Hour.ToString().Replace(bedInfoModel.CleaningDate.Hour.ToString(), "0" + bedInfoModel.CleaningDate.Hour.ToString());
                else
                    bedInfoModel.HourSelected = bedInfoModel.CleaningDate.Hour.ToString();
                if (bedInfoModel.CleaningDate.Minute.ToString().Length == 1)
                    bedInfoModel.MinuteSelected = bedInfoModel.CleaningDate.Minute.ToString().Replace(bedInfoModel.CleaningDate.Minute.ToString(), "0" + bedInfoModel.CleaningDate.Minute.ToString());
                else
                    bedInfoModel.MinuteSelected = bedInfoModel.CleaningDate.Minute.ToString();

                List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(this.loggedUser.Domain, this.loggedUser.Facility.Id).ToList();
                if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                {
                    bedInfoModel.EMSModuleInactive = true;
                    bedInfoModel.DisplayEMSModuleInactive = @"style='display:none;'";
                }
                else
                {
                    bedInfoModel.EMSModuleInactive = false;
                    bedInfoModel.DisplayEMSModuleInactive = string.Empty;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the manual cleaning request.
        /// </summary>
        /// <param name="bedInfoModel">The bed info model.</param>
        /// <param name="bed">The bed.</param>
        /// <param name="ward">The ward.</param>
        /// <param name="division">The division.</param>
        /// <param name="facility">The facility.</param>
        /// <param name="dateNow">The date now.</param>
        private void SaveManualCleaningRequest(BedInfoViewModel bedInfoModel, Bed bed, DateTime dateNow)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (bedInfoModel.RequestCleaning.Equals(Strings.Yes) || bedInfoModel.RequestCleaning.Equals(Strings.Stat))
                {
                    if (bedInfoModel.IsBedCleanRequested)
                    {
                        BedClean bc = FacadeManager.BedInterface.GetBedCleanByBedId(bed.Id, this.loggedUser.Facility.Id);
                        if (bc != null)
                        {
                            if (!bc.TypeOfClean.code.Equals(bedInfoModel.RequestCleaning, StringComparison.InvariantCultureIgnoreCase) ||
                                bc.SpecialInstructions != bedInfoModel.SpecialInstructions)
                            {
                                bc.TypeOfClean = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Decision).ToList().Where(a => a.displayName == bedInfoModel.RequestCleaning).FirstOrDefault();
                                bc.SpecialInstructions = bedInfoModel.SpecialInstructions;
                                DateTimeConverter.ConvertDateToUTC(bc, loggedUser.Facility.Id);
                                BMSFactory.BedManagerOperationsClient.UpdateBedCleaningOperation(bc.ToDataContract());
                            }
                        }
                    }
                    else
                    {
                        DateTime? requestDate = bedInfoModel.CleaningDate;
                        double value = 0;
                        if (double.TryParse(bedInfoModel.HourSelected, out value))
                            requestDate = requestDate.Value.AddHours(value);
                        if (double.TryParse(bedInfoModel.MinuteSelected, out value))
                            requestDate = requestDate.Value.AddMinutes(value);

                        BedClean bedClean = new BedClean()
                        {
                            Id = new II(this.loggedUser.Domain, null),
                            Bed = new Place() { Id = bed.Id, Name = bed.Name, Ien = bed.Ien },
                            LastEditDate = dateNow,
                            LastEditedBy = this.loggedUser.UserName,
                            ManualRequest = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision).ToList().Where(a => a.displayName == Strings.Yes).FirstOrDefault(),
                            RequestedDate = requestDate,
                            TypeOfClean = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Decision).ToList().Where(a => a.displayName == bedInfoModel.RequestCleaning).FirstOrDefault(),
                            Ward = FacadeManager.EntityInterface.GetWardById(new II(bed.Id.root, bedInfoModel.Ward)),
                            VistaSite = this.loggedUser.VistaSite,
                            SpecialInstructions = bedInfoModel.SpecialInstructions
                        };

                        DateTimeConverter.ConvertDateToUTC(bedClean, loggedUser.Facility.Id);
                        bedClean.Id = BMSFactory.BedManagerOperationsClient.CreateBedCleaningOperation(bedClean.ToDataContract());
                        FacadeManager.BedInterface.SendCleanDirtyBedNotifications(bedClean);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Saves the bed unavailable.
        /// </summary>
        /// <param name="bedInfoModel">The bed info model.</param>
        /// <param name="bed">The bed.</param>
        /// <param name="ward">The ward.</param>
        /// <param name="division">The division.</param>
        /// <param name="facility">The facility.</param>
        /// <param name="dateNow">The date now.</param>
        /// <param name="reason">The reason.</param>
        /// <param name="type">The type.</param>
        private BedUnavailable SaveBedUnavailable(BedInfoViewModel bedInfoModel, Bed bed, DateTime dateNow, bool isChangedComment, CDWithProperties reason, CD type)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedUnavailable bedUnavailable = FacadeManager.BedInterface.GetBedUnavailable(new II(bedInfoModel.RootBedId, bedInfoModel.ExtensionBedId), loggedUser.Facility.Id);
                bedInfoModel.BedUnavailableHistory = new BedUnavailableHistory();
                bedInfoModel.BedUnavailableHistory.IsChangedOnlyComment = true;
                if (bedUnavailable != null)
                    bedInfoModel.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, bedInfoModel.BedUnavailableHistory);
                if (string.IsNullOrEmpty(bedInfoModel.UnavailableReasonCode))
                {
                    if (!string.IsNullOrEmpty(bedInfoModel.SelectedReason))
                    {
                        //create
                        bedUnavailable = new BedUnavailable()
                        {
                            Bed = (bed != null) ? new Place() { Id = bed.Id, Name = bed.Name } : null,
                            CanceledBy = null,
                            CanceledDate = null,
                            CreatedBy = this.loggedUser.UserName,
                            CreationDate = dateNow,
                            EditedBy = null,
                            EditedDate = null,
                            Id = new II(this.loggedUser.Domain, null),
                            Parent = null,
                            Patient = null,
                            Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null),
                            Type = type,
                            OutOfServiceVistA = false,
                            WardList = bed.WardList,
                            VistaSite = this.loggedUser.VistaSite,
                            ExpectedCompletedDate = CreateExpectedCompletedDate(type, bedInfoModel.ExpectedCompletedDate, bedInfoModel.ExpectedCompletedDateHourSelected, bedInfoModel.ExpectedCompletedDateMinuteSelected)
                        };
                        bedInfoModel.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, bedInfoModel.BedUnavailableHistory);
                        bedInfoModel.BedUnavailableHistory.IsChangedOnlyComment = false;
                        FacadeManager.WorkflowInterface.CreateBedUnavailable(bedUnavailable, loggedUser.Facility);
                    }
                }
                else
                {
                    string newReason = string.Empty;
                    if (!string.IsNullOrEmpty(bedInfoModel.SelectedReason))
                        newReason = bedInfoModel.SelectedReason;
                    
                    DateTime? expectedCompletedDate = CreateExpectedCompletedDate(type, bedInfoModel.ExpectedCompletedDate, bedInfoModel.ExpectedCompletedDateHourSelected, bedInfoModel.ExpectedCompletedDateMinuteSelected);
                    
                    if (bedUnavailable.Reason != null && (newReason != bedUnavailable.Reason.code || expectedCompletedDate != bedUnavailable.ExpectedCompletedDate || isChangedComment))
                    {
                        if (!string.IsNullOrEmpty(newReason) && newReason != Strings.DeleteThisReason)
                        {
                            bedUnavailable.EditedBy = this.loggedUser.UserName;
                            bedUnavailable.EditedDate = dateNow;
                            bedUnavailable.Reason = new CD(reason.code, reason.codeSystem, reason.codeSystemName, null, reason.displayName, null, null, null);
                            bedUnavailable.Type = type;
                            bedUnavailable.OutOfServiceVistA = false;
                            bedUnavailable.ExpectedCompletedDate = expectedCompletedDate;
                            bedInfoModel.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, bedInfoModel.BedUnavailableHistory);
                            FacadeManager.WorkflowInterface.UpdateBedUnavailable(bedUnavailable, loggedUser.Facility);

                        }
                        else
                        {
                            //cancel
                            bedUnavailable.CanceledBy = this.loggedUser.UserName;
                            bedUnavailable.CanceledDate = CreateActualOOSCompletedDate(bedInfoModel.ActualCompletedDate, bedInfoModel.ActualCompletedDateHourSelected, bedInfoModel.ActualCompletedDateMinuteSelected, dateNow);
                            bedUnavailable.OutOfServiceVistA = false;
                            bedInfoModel.BedUnavailableHistory = TranslateBedUnavailableHistory(bedUnavailable, bedInfoModel.BedUnavailableHistory);
                            FacadeManager.WorkflowInterface.CancelBedUnavailable(bedUnavailable, loggedUser.Facility);

                        }
                        bedInfoModel.BedUnavailableHistory.IsChangedOnlyComment = false;
                    }
                }
                return bedUnavailable;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }

        }

        /// <summary>
        /// Gets the reason.
        /// </summary>
        /// <param name="reasonCode">The reason code.</param>
        /// <param name="reason">The reason.</param>
        /// <param name="type">The type.</param>
        private void GetReason(string reasonCode, out CDWithProperties reason, out CD type, out bool isReasonAvailable)
        {
            List<CDWithProperties> reasonList = FacadeManager.VocabularyInterface.GetReasons().ToList();
            List<CD> typeList = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.AdminURType).ToList();
            reason = null;
            type = null;
            isReasonAvailable = true;
            if (!string.IsNullOrWhiteSpace(reasonCode) && reasonCode != Strings.DeleteThisReason)
            {
                reason = reasonList.Where(a => a.code == reasonCode).FirstOrDefault();
                if (reason == null)
                {
                    isReasonAvailable = false;
                }
                else
                {
                    string typeCode = reason.Properties.Where(a => a.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text;
                    type = typeList.Where(a => a.code == typeCode).FirstOrDefault();
                }
            }
        }
        /// <summary>
        /// Translates the bed unavailable history.
        /// </summary>
        /// <param name="bedUnavailable">The bed unavailable.</param>
        /// <param name="bedUnavailableHistory">The bed unavailable history.</param>
        /// <returns></returns>
        private BedUnavailableHistory TranslateBedUnavailableHistory(BedUnavailable bedUnavailable, BedUnavailableHistory bedUnavailableHistory)
        {
            bedUnavailableHistory.Bed = bedUnavailable.Bed;
            bedUnavailableHistory.CanceledBy = bedUnavailable.CanceledBy;
            bedUnavailableHistory.CanceledDate = bedUnavailable.CanceledDate;
            bedUnavailableHistory.CreatedBy = bedUnavailable.CreatedBy;
            bedUnavailableHistory.CreationDate = bedUnavailable.CreationDate;
            bedUnavailableHistory.EditedBy = bedUnavailable.EditedBy;
            bedUnavailableHistory.EditedDate = bedUnavailable.EditedDate;
            bedUnavailableHistory.Id = bedUnavailable.Id;
            bedUnavailableHistory.Parent = bedUnavailable.Parent;
            bedUnavailableHistory.Patient = bedUnavailable.Patient;
            bedUnavailableHistory.Reason = bedUnavailable.Reason;
            bedUnavailableHistory.Type = bedUnavailable.Type;
            bedUnavailableHistory.VistaSite = bedUnavailable.VistaSite;
            bedUnavailableHistory.ExpectedCompletedDate = bedUnavailable.ExpectedCompletedDate;
            return bedUnavailableHistory;
        }

        /// <summary>
        /// Gets the type code.
        /// </summary>
        /// <param name="reasonCode">The reason code.</param>
        /// <returns></returns>
        private string GetTypeCode(string reasonCode)
        {
            List<CDWithProperties> reasonList = FacadeManager.VocabularyInterface.GetReasons().ToList();
            if (!string.IsNullOrWhiteSpace(reasonCode))
            {
                CDWithProperties reason = reasonList.Where(a => a.code == reasonCode).FirstOrDefault();
                if (reason != null)
                    return reason.Properties.Where(a => a.PropertyName.text == "Type").FirstOrDefault().PropertyValue.text;
                else
                    return string.Empty;
            }
            return string.Empty;
        }

        /// <summary>
        /// Creates the expected completed date.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <param name="expectedDate">The expected date.</param>
        /// <param name="expectedHour">The expected hour.</param>
        /// <param name="expectedMinute">The expected minute.</param>
        /// <returns></returns>
        private DateTime? CreateExpectedCompletedDate(CD type, DateTime? expectedDate, string expectedHour, string expectedMinute)
        {
            DateTime? returnExpectedDate = null;
            if (type != null && type.code == Constants.BED_OUT_OF_SERVICE_TYPE_CODE && expectedDate.HasValue)
            {
                returnExpectedDate = expectedDate;
                double value = 0;
                if (double.TryParse(expectedHour, out value))
                    returnExpectedDate = returnExpectedDate.Value.AddHours(value);
                if (double.TryParse(expectedMinute, out value))
                    returnExpectedDate = returnExpectedDate.Value.AddMinutes(value);
            }
            return returnExpectedDate;
        }

        /// <summary>
        /// Creates the actual OOS completed date.
        /// </summary>
        /// <param name="actualDate">The actual date.</param>
        /// <param name="actualHour">The actual hour.</param>
        /// <param name="actualMinute">The actual minute.</param>
        /// <param name="dateNow">The date now.</param>
        /// <returns></returns>
        private DateTime CreateActualOOSCompletedDate(DateTime actualDate, string actualHour, string actualMinute, DateTime dateNow)
        {
            if (actualDate == null || actualDate == DateTime.MinValue)
                return dateNow;
            else
            {
                DateTime returnActualDate = actualDate;
                double value = 0;
                if (double.TryParse(actualHour, out value))
                    returnActualDate = returnActualDate.AddHours(value);
                if (double.TryParse(actualMinute, out value))
                    returnActualDate = returnActualDate.AddMinutes(value);
                return returnActualDate;
            }
        }
    }
}
