﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.Utils;
using BMS.VistaWorker2.Writer;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.VistaIntegration.VistA;
using System.Threading;
using System.Configuration;

namespace BMS.VistaIntegration.Operations
{
    public abstract class BaseOperation
    {
        protected delegate void ProcessTypeCallback(VistaDataType type, Func<Parameter, IEntryProcessor> processorMaker);

        public int MaxCountExceptions { get; set; }
        public readonly VistASite VistASite;
        public readonly IList<Parameter> Parameters;
        public event EventHandler<ParameterEventArgs> TypeUpdated;
        public event EventHandler<ParameterEventArgs> TypeUpdating;
        public event EventHandler Ended;
        private readonly BmsLogger logger;
        private CancellationToken cancellationToken;
        protected IVistAQuery query;
        protected IWriterManager writer = null;
        protected Dictionary<VistaDataType, ParameterEventArgs> eventArgsList = new Dictionary<VistaDataType, ParameterEventArgs>();

        public BaseOperation(VistASite site, IList<Parameter> parameters)
        {
            this.logger = new BmsLogger(string.Format("VistaIntegaration Operation Message (Site {0}): ", site.Name));

            VistASite = site;
            Parameters = parameters;
            MaxCountExceptions = 4;
        }

        protected void OnTypeUpdated(ParameterEventArgs eventArgs)
        {
            if (TypeUpdated != null)
                TypeUpdated(this, eventArgs);
        }

        protected void OnTypeUpdating(ParameterEventArgs eventArgs)
        {
            if (TypeUpdating != null)
                TypeUpdating(this, eventArgs);
        }

        private void OnEnded()
        {
            if (Ended != null)
                Ended(this, EventArgs.Empty);
        }

        private bool CheckIsAborted(ParameterEventArgs eventArgs)
        {
            if (cancellationToken.IsCancellationRequested)
            {
                eventArgs.State = ParameterEventArgs.OperationState.Failed;
                eventArgs.ErrorMessage = "Operation was aborted";
                OnTypeUpdated(eventArgs);
                return true;
            }
            return false;
        }

        protected ParameterEventArgs.Info GetInfo(string type, ParameterEventArgs eventArgs)
        {
            ParameterEventArgs.Info info;
            if (!eventArgs.InfoDictionary.TryGetValue(type, out info))
            {
                info = new ParameterEventArgs.Info();
                eventArgs.InfoDictionary.Add(type, info);
            }
            return info;
        }

        protected void FinaliseAuditLog(ParameterEventArgs eventArgs)
        {
            if (eventArgs.State == ParameterEventArgs.OperationState.Failed)
                OnTypeUpdated(eventArgs);
            else if (eventArgs.WriterExceptions == null || eventArgs.WriterExceptions.Count == 0)
            {
                if (eventArgs.State == ParameterEventArgs.OperationState.Running)
                    eventArgs.State = ParameterEventArgs.OperationState.Succeeded;
                OnTypeUpdated(eventArgs);
            }
            else
            {
                eventArgs.State = ParameterEventArgs.OperationState.PartiallySucceeded;
                eventArgs.ErrorMessage = string.Format("{0} writer errors occurred", eventArgs.WriterExceptions.Count);
                OnTypeUpdated(eventArgs);
            }
        }

        private void ProcessType(Parameter param, IEntryProcessor processor)
        {
            VistaDataType originalType = param.Type;
            try
            {
                int writerExceptionsCount = 0;
                logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - BaseOperation - Start processing entities", VistASite.Name, param.Type.ToString());
                if (originalType == VistaDataType.Order || originalType == VistaDataType.PatientMovement || originalType == VistaDataType.PatientAppointment || originalType == VistaDataType.ScheduledAdmission || originalType == VistaDataType.PatientMovementIen || originalType == VistaDataType.EDISPatientAdmission)
                    originalType = VistaDataType.ADT;
                while (processor.HasMoreEntries)
                {
                    if (CheckIsAborted(eventArgsList[originalType]))
                        return;
                    try
                    {
                        string type = processor.ProcessNextEntry();
                        if (type != null)
                        {
                            lock (eventArgsList)
                            {
                                eventArgsList[originalType].RetrievedEntries++;

                                //logger.LogFormat(BmsLogger.Level.Warning, "Incremented 'RetrievedEntries' for {0} ({1},{2}) Current count: {3}", originalType, param.Type, type, eventArgsList[originalType].RetrievedEntries);

                                ParameterEventArgs.Info info = GetInfo(type, eventArgsList[originalType]);
                                info.ProcessedWithSuccesEntries++;

                                //logger.LogFormat(BmsLogger.Level.Warning, "Incremented 'ProcessedWithSuccesEntries' for {0} ({1},{2}) Current count: {3}", originalType, param.Type, type, info.ProcessedWithSuccesEntries);
                            }
                        }
                    }
                    catch (WriterException e)
                    {
                        lock (eventArgsList)
                        {
                            logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - BaseOperation - Catch WriterException= {2}", VistASite.Name, param.Type.ToString(), e.ToString());
                            eventArgsList[originalType].RetrievedEntries++;

                            //logger.LogFormat(BmsLogger.Level.Warning, "Incremented 'RetrievedEntries' for {0} ({1}) Current count: {2}", originalType, param.Type, eventArgsList[originalType].RetrievedEntries);

                            ParameterEventArgs.Info info = GetInfo(e.Entity.GetType().Name, eventArgsList[originalType]);
                            info.ProcessedWithFaildEntries++;

                            //logger.LogFormat(BmsLogger.Level.Warning, "Incremented 'ProcessedWithSuccesEntries' for {0} ({1}) Current count: {2}", originalType, param.Type, info.ProcessedWithFaildEntries);

                            writerExceptionsCount++;
                            if (eventArgsList[originalType].WriterExceptions.Count < MaxCountExceptions)
                            {
                                eventArgsList[originalType].WriterExceptions.Add(e);
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        lock (eventArgsList)
                        {
                            logger.LogFormat(BmsLogger.Level.Error, "{0} - {1} - BaseOperation - Catch General Exception= {2}", VistASite.Name, param.Type.ToString(), e.ToString());
                            eventArgsList[originalType].State = ParameterEventArgs.OperationState.Failed;
                            eventArgsList[originalType].ErrorMessage = e.ToString();
                        }
                        return;
                    }
                }

                logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - BaseOperation - End processing entities", VistASite.Name, param.Type.ToString());
            }
            finally
            {
                lock (eventArgsList)
                {
                    eventArgsList[originalType].CompletedOperations += 1;
                    if ((originalType != VistaDataType.ADT && originalType != VistaDataType.Patient)
                            || (originalType == VistaDataType.ADT && eventArgsList[originalType].CompletedOperations == 6)
                            || (originalType == VistaDataType.Patient && eventArgsList[originalType].CompletedOperations == 2))
                    {
                        FinaliseAuditLog(eventArgsList[originalType]);
                        eventArgsList.Remove(originalType);
                        logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - BaseOperation - Updated audit log", VistASite.Name, param.Type.ToString());
                    }
                }

                processor.Dispose();
                processor = null;
                logger.LogFormat(BmsLogger.Level.Info, "{0} - {1} - BaseOperation - Exit processing entities", VistASite.Name, param.Type.ToString());
            }
        }

        private void ProcessType(VistaDataType type, Func<Parameter, IEntryProcessor> processorMaker)
        {
            Parameter param = Parameters.FirstOrDefault(s => s.Type == type);
            VistaDataType originalType = VistaDataType.None;
            if (param == null)
            {
                if (type == VistaDataType.Order || type == VistaDataType.PatientMovement || type == VistaDataType.PatientAppointment || type == VistaDataType.ScheduledAdmission || type == VistaDataType.PatientMovementIen || type == VistaDataType.EDISPatientAdmission)
                {
                    if (Parameters.FirstOrDefault(s => s.Type == VistaDataType.ADT) == null)
                        return;
                    else
                    {
                        param = Parameters.FirstOrDefault(s => s.Type == VistaDataType.ADT);
                        originalType = VistaDataType.ADT;
                    }
                }
                else
                    return;
            }
            else
                originalType = param.Type;

            DateTime? now = DateTime.UtcNow;
            JobLaunchType launchType = param.JobLaunchType;
            if (param.JobLaunchType == JobLaunchType.Automatic)
            {
                if (param.EndDateParam.HasValue)
                    now = VistASite.ConvertToSiteDateTime(new DateTime(now.Value.Year, now.Value.Month, now.Value.Day, now.Value.Hour, now.Value.Minute, 0));
                else
                    now = param.EndDateParam;
                if (type == VistaDataType.ScheduledAdmission || type == VistaDataType.PatientAppointment || type == VistaDataType.WaitingList)
                    now = param.EndDateParam;
                param = new Parameter(param.Type, param.StartDateParam, now);
            }

            lock (eventArgsList)
            {
                if (!eventArgsList.ContainsKey(originalType))
                {
                    eventArgsList.Add(originalType, new ParameterEventArgs(param));
                    eventArgsList[originalType].State = ParameterEventArgs.OperationState.Running;
                    eventArgsList[originalType].CompletedOperations = 0;
                    OnTypeUpdating(eventArgsList[originalType]);
                }
            }

            if (launchType == JobLaunchType.Automatic)
            {
                DateTime? start = param.StartDateParam;
                if (type == VistaDataType.PatientMovement || type == VistaDataType.PatientMovementIen)
                    now = now.Value.AddMonths(1);
                if (type == VistaDataType.PatientMovementIen)
                {
                    DateTime dt = DateTime.UtcNow;
                    if (start.HasValue)
                    {
                        int PatientMovementIenDays;
                        try
                        {
                            PatientMovementIenDays = int.Parse(ConfigurationManager.AppSettings["VistaIntegration.PatientMovementIenDays"]);
                        }
                        catch (Exception)
                        {
                            PatientMovementIenDays = 30;
                        }
                        dt = dt.AddDays(-PatientMovementIenDays);
                    }

                    else
                        dt = DateTime.UtcNow.AddYears(-100);
                    start = VistASite.ConvertToSiteDateTime(new DateTime(dt.Year, dt.Month, dt.Day, dt.Hour, dt.Minute, 0));
                }
                if (type == VistaDataType.Order)
                {
                    if (start.HasValue)
                        start = start.Value.AddHours(-3);
                }
                param = new Parameter(type, start, now);
            }
            else
                param = new Parameter(type, param.StartDateParam, param.EndDateParam);
            param.JobLaunchType = launchType;

            if (CheckIsAborted(eventArgsList[originalType]))
                return;
            using (IEntryProcessor processor = processorMaker(param))
            {
                ProcessType(param, processor);
            }
        }

        protected abstract void ProcessTypes(ProcessTypeCallback callback);

        public void Run(IVistASession session, IWriterManagerFactory writerManagerFactory, CancellationToken cancellationToken)
        {
            this.cancellationToken = cancellationToken;
            try
            {
                if (cancellationToken.IsCancellationRequested)
                    return;
                writer = writerManagerFactory.MakeWriter();
                writer.Open(VistASite);
                query = session.MakeQuery();
                ProcessTypes(new ProcessTypeCallback(ProcessType));
            }
            catch (Exception ex)
            {
                logger.LogError(VistASite.Name + " - Exception BaseOperation.Run - " + ex.ToString());
                throw;
            }
            finally
            {
                query = null;
                try
                {
                    if (writer != null)
                        writer.Close();
                }
                catch (Exception e)
                {
                    string msj = string.Format("Exception on {0} when closing the writer:\n{1}", this, e);
                    logger.LogError(msj);
                }
                try
                {
                    OnEnded();
                }
                catch (Exception e)
                {
                    string msj = string.Format("Exception on {0} when OnEnded event is called:\n{1}", this, e);
                    logger.LogError(msj);
                }
            }
        }

        public override string ToString()
        {
            return string.Format("Operation on site {0} for types {1}", VistASite.Name, Utilities.ToString(Parameters.Select(s => s.Type)));
        }
    }
}
