﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using Mdws2ORM;
using Mdws2ORM.Core;
using Mdws2ORM.Maps;

namespace BMS.VistaIntegration.Via.Commands
{
    public sealed class EntitySetCache
    {
        private static readonly object siteCacheLock = new object();
        private static readonly Dictionary<string, EntitySetCache> siteCacheDictionary = new Dictionary<string, EntitySetCache>();
        private static readonly BmsLogger Logger = new BmsLogger("EntitySetCache: ");

        private readonly object cacheRegistryLock = new object();
        private readonly Dictionary<Type, IEntityCache> entityCacheRegistry = new Dictionary<Type, IEntityCache>();

        private EntitySetCache()
        {
        }

        public IEntityQuery CreateEntityQuery(ViaVistAQuery query)
        {
            if (query == null)
            {
                throw new ArgumentNullException("query");
            }

            return new CachedEntityQuery(query);
        }

        public void RegisterBulkType<T>(IDependencySource dependencySource = null)
            where T : class, IEntity, new()
        {
            lock (this.cacheRegistryLock)
            {
                if (!this.entityCacheRegistry.ContainsKey(typeof(T)))
                {
                    this.entityCacheRegistry[typeof(T)] = new EntityCache<T, BulkEntitiesListCommand<T>>(dependencySource);
                }
            }
        }

        public T GetEntity<T>(ViaVistAQuery query, string ien) where T : class
        {
            if (query == null)
            {
                throw new ArgumentNullException("query");
            }

            IEntityCache cache;
            if (this.entityCacheRegistry.TryGetValue(typeof(T), out cache))
            {
                return ((IEntityCache<T>)cache).GetItem(query, ien);
            }

            return default(T);
        }

        public IEnumerable<T> GetEntities<T>(ViaVistAQuery query, IEnumerable<string> iens) where T : class
        {
            if (query == null)
            {
                throw new ArgumentNullException("query");
            }

            IEntityCache cache;
            if (this.entityCacheRegistry.TryGetValue(typeof(T), out cache))
            {
                return from ien in iens
                       let entity = ((IEntityCache<T>)cache).GetItem(query, ien)
                       where entity != null
                       select entity;
            }

            return Enumerable.Empty<T>();
        }

        public List<T> RunThroghCache<T>(Func<List<T>> func)
        {
            IEntityCache cache;
            if (this.entityCacheRegistry.TryGetValue(typeof(T), out cache))
            {
                return ((IEntityCache<T>)cache).RunThroughCache(func);
            }

            return func();
        }

        public static EntitySetCache Create(VistASite site)
        {
            if (site == null)
            {
                throw new ArgumentNullException("site");
            }

            lock (siteCacheLock)
            {
                if (!siteCacheDictionary.ContainsKey(site.Number))
                {
                    siteCacheDictionary[site.Number] = new EntitySetCache();
                }

                return siteCacheDictionary[site.Number];
            }
        }

        public static void DownloadEntities(Type type, ViaVistAQuery query, VistASite site, string[] iens, char separator, string target, int argsCount)
        {
            var genericMethod = typeof(EntitySetCache).GetMethod("DownloadEntitiesCore", BindingFlags.NonPublic | BindingFlags.Static);
            var specificMethod = genericMethod.MakeGenericMethod(type);
            specificMethod.Invoke(null, new object[] { query, site, iens, separator, target, argsCount });
        }

        private static void DownloadEntitiesCore<T>(ViaVistAQuery query, VistASite site, string[] iens, char separator, string target, int argumentsCount) where T : class, IEntity, new()
        {
            // This method is invoked using reflection in the method right above
            EntitySetCache entitySetCache;
            if (siteCacheDictionary.TryGetValue(site.Number, out entitySetCache))
            {
                IEntityCache cache;
                if (entitySetCache.entityCacheRegistry.TryGetValue(typeof(T), out cache))
                {
                    if (separator != '\0')
                    {
                        iens = (from entry in iens
                                from ien in entry.Split(separator)
                                select ien).Distinct().ToArray();
                    }

                    ((IEntityCache<T>)cache).RunThroughCache(() =>
                    {
                        Logger.LogFormat(BmsLogger.Level.Verbose, "{1} - EntitySetCache.DownloadEntities<{0}> - Downloading {2} entities ('{4}') from VIA using target '{3}'", typeof(T).Name, query.Site.Name, iens.Length, target, string.Join("','", iens));

                        var cachedKeys = cache.CleanupCacheAndGetValidKeys();
                        iens = iens.Where(x => !cachedKeys.Contains(x)).ToArray();

                        Logger.LogFormat(BmsLogger.Level.Verbose, "{1} - EntitySetCache.DownloadEntities<{0}> - Downloading {2} entities ('{4}') not in cache already from VIA using target '{3}'", typeof(T).Name, query.Site.Name, iens.Length, target, string.Join("','", iens));
                        return query.GetResults(new BulkEntitiesListCommand<T>(query, target, iens, argumentsCount, cache.DependencySource)).ToList();
                    });
                }
            }
        }

        public class BulkEntitiesListCommand<TEntity> : BaseListCommand<TEntity> where TEntity : class, IEntity, new()
        {
            private readonly string target;
            private readonly string[] iens;
            private readonly int argumentsCount;
            private readonly IDependencySource dependencySource;
            private string partialIens;

            public BulkEntitiesListCommand(ViaVistAQuery query, string target, string[] iens, int argumentsCount, IDependencySource dependencySource)
                : base(query)
            {
                this.target = target;
                this.iens = iens;
                this.argumentsCount = argumentsCount;

                this.dependencySource = dependencySource;
            }

            public override List<TEntity> Execute(ViaVistASession session)
            {
                if (this.iens == null || this.iens.Length < 1)
                {
                    return new List<TEntity>(0);
                }

                var fullList = new List<TEntity>();
                foreach (var part in this.iens.JoinAndSplitByLength(',', BaseCommand<TEntity>.MaxCriteriaLength))
                {
                    this.partialIens = part;

                    try
                    {
                        var partialList = base.Execute(session);
                        fullList.AddRange(partialList);
                    }
                    catch (ViaException ex)
                    {
                        Logger.LogWarning(string.Format("Partial retrieval of entities of type {0} for iens '{1}' failed with exception {2}", typeof(TEntity).Name, this.partialIens, ex));
                    }
                }

                return fullList;
            }

            protected override string GetTarget()
            {
                return this.target;
            }

            protected override IEnumerable<object> GetCriteria()
            {
                // Argument 0
                yield return this.partialIens;

                // Remaining arguments
                for (int i = 1; i < this.argumentsCount; i++)
                {
                    yield return string.Empty;
                }
            }

            protected override IDependencySource GetDependencySource()
            {
                return this.dependencySource;
            }
        }

        private interface IEntityCache
        {
            IDependencySource DependencySource
            {
                get;
            }

            bool HasEntry(string x);

            IEnumerable<string> CleanupCacheAndGetValidKeys();
        }

        private interface IEntityCache<T> : IEntityCache
        {
            T GetItem(ViaVistAQuery query, string ien);

            T GetItemFromCache(string ien, bool storeNull);

            List<T> RunThroughCache(Func<List<T>> entitiesFunc);
        }

        private class EntityCache<TEntity, TCommand> : IEntityCache<TEntity>
            where TEntity : class, IEntity, new()
            where TCommand : BaseListCommand<TEntity>
        {
            private object locker = new object();
            private Dictionary<string, CacheEntry<TEntity>> entitiesCache = new Dictionary<string, CacheEntry<TEntity>>();
            private static readonly TimeSpan CacheDuration = TimeSpan.FromDays(1);

            public EntityCache(IDependencySource dependencySource)
            {
                Logger.LogFormat(BmsLogger.Level.Info, "EntityCache<{0}>.ctor - Created cache store with duration '{1}'", typeof(TEntity).Name, CacheDuration);
                this.DependencySource = dependencySource;
            }

            public IDependencySource DependencySource
            {
                get;
                private set;
            }

            public TEntity GetItem(ViaVistAQuery query, string ien)
            {
                return this.GetItemFromCache(ien, true);
            }

            public TEntity GetItemFromCache(string ien, bool storeNull)
            {
                CacheEntry<TEntity> item = null;

                if (string.IsNullOrEmpty(ien))
                {
                    return null;
                }

                lock (this.locker)
                {
                    if (!this.entitiesCache.TryGetValue(ien, out item) && storeNull)
                    {
                        Logger.LogFormat(BmsLogger.Level.Warning, "EntityCache<{0}>.GetItemFromCache - Entity with IEN '{1}' could not be retrieved", typeof(TEntity).Name, ien);
                    }
                    else
                    {
                        Logger.LogFormat(BmsLogger.Level.Verbose, "EntityCache<{0}>.GetItemFromCache - Retrieving entity with IEN '{1}' from cache", typeof(TEntity).Name, ien);
                    }

                    return item == null ? null : item.Entity;
                }
            }

            public List<TEntity> RunThroughCache(Func<List<TEntity>> entitiesFunc)
            {
                lock (this.locker)
                {
                    Logger.LogFormat(BmsLogger.Level.Verbose, "EntityCache<{0}>.RunThroughCache - Retrieving multiple entities", typeof(TEntity).Name);
                    var entities = entitiesFunc();
                    if (entities != null)
                    {
                        var now = DateTime.UtcNow;

                        entities.ForEach(entity =>
                        {
                            if (!string.IsNullOrEmpty(entity.IEN))
                            {
                                this.entitiesCache[entity.IEN] = new CacheEntry<TEntity>(now, (TEntity)entity);
                            }
                            else
                            {
                                Logger.LogFormat(BmsLogger.Level.Verbose, "EntityCache<{0}>.RunThroughCache - Not caching entity with empty IEN", typeof(TEntity).Name);
                            }
                        });
                        Logger.LogFormat(BmsLogger.Level.Verbose, "EntityCache<{0}>.RunThroughCache - Downloaded {1} entities ('{2}') from VIA", typeof(TEntity).Name, entities.Count, string.Join("','", entities.Select(x => x.IEN)));
                    }

                    return entities;
                }
            }

            public bool HasEntry(string ien)
            {
                return this.entitiesCache.ContainsKey(ien);
            }

            public IEnumerable<string> CleanupCacheAndGetValidKeys()
            {
                this.Cleanup();
                return (from entity in this.entitiesCache
                        select entity.Key).ToArray();
            }

            private void Cleanup()
            {
                var now = DateTime.UtcNow;
                var entitiesToDelete = (from entityPair in this.entitiesCache
                                        where (now - entityPair.Value.RetrievalDate) > CacheDuration
                                        select entityPair).ToArray();

                if (entitiesToDelete.Length > 0)
                {
                    Logger.LogFormat(BmsLogger.Level.Verbose, "EntityCache<{0}>.Cleanup - Dropping {1} entities from cache", typeof(TEntity).Name, entitiesToDelete.Length);

                    this.entitiesCache.RemoveRange(entitiesToDelete);
                }
            }
        }

        private class CacheEntry<TEntity>
        {
            public CacheEntry(DateTime dateTime, TEntity entity)
            {
                this.RetrievalDate = dateTime;
                this.Entity = entity;
            }

            public DateTime RetrievalDate
            {
                get;
                private set;
            }

            public TEntity Entity
            {
                get;
                private set;
            }

            public bool Equals(CacheEntry<TEntity> obj)
            {
                return object.Equals(this.RetrievalDate, obj.RetrievalDate) && object.Equals(this.Entity, obj.Entity);
            }

            public override bool Equals(object obj)
            {
                return this.Equals(obj as CacheEntry<TEntity>);
            }

            public override int GetHashCode()
            {
                return this.RetrievalDate.GetHashCode() ^ (this.Entity != null ? this.Entity.GetHashCode() : 0);
            }
        }

        private class CachedEntityQuery : IEntityQuery
        {
            private readonly ViaVistAQuery query;

            public CachedEntityQuery(ViaVistAQuery query)
            {
                this.query = query;
            }

            public IEnumerable<ListItem<T1>> List<T1>(BaseEntityMap<T1> entityMap, ListParam listParam, string iens = "") where T1 : class
            {
                throw new NotImplementedException();
            }

            public T1 Get<T1>(BaseEntityMap<T1> entityMap, GetParam getParam, string ien) where T1 : class
            {
                return this.GetEntity<T1>(ien);
            }

            public T1 GetNotNull<T1>(BaseEntityMap<T1> entityMap, GetParam getParam, string ien = "") where T1 : class
            {
                return this.GetEntity<T1>(ien);
            }

            private T1 GetEntity<T1>(string ien) where T1 : class
            {
                return query.EntitySetCache.GetEntity<T1>(this.query, ien);
            }
        }
    }
}
