﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Via.Commands.EIS;
using BMS.VistaIntegration.Via.Commands.EntityCommands;
using BMS.VistaIntegration.Via.Commands.EVS;

namespace BMS.VistaIntegration.Via.Commands.WF
{
    public class ListPatientMovementCommand : BaseListPeriodCommand<PatientMovement>
    {
        public ListPatientMovementCommand(ViaVistAQuery query)
            : base(query)
        {
        }

        public DateTime? MovementDate
        {
            get;
            set;
        }

        public DateTime? LastEditedDate
        {
            get;
            set;
        }

        protected override string GetTarget()
        {
            return "ListPatientMovement";
        }

        protected override IEnumerable<object> GetCriteria()
        {
            yield return string.Empty;
            yield return this.StartDate.GetValueOrDefault(MinDate);
            yield return this.EndDate.GetValueOrDefault(this.MaxDate);
            yield return this.MovementDate;
            yield return this.LastEditedDate;
            yield return this.MaxCount;
            yield return string.IsNullOrEmpty(this.From) ? (this.StartDate.HasValue ? (object)this.StartDate.Value.AddDays(-1) : null) : (object)this.From;
        }

        protected override IDependencySource GetDependencySource()
        {
            return PatientMovementDependencySource.Instance;
        }
    }

    internal sealed class PatientMovementDependencySource : IDependencySource<PatientMovement>
    {
        private static IDependencySource instance;

        private PatientMovementDependencySource()
        {
        }

        public static IDependencySource Instance
        {
            get
            {
                if (instance == null)
                {
                    instance = new PatientMovementDependencySource();
                }

                return instance;
            }
        }

        public IEnumerable<DependentEntityInfo> GetDependentTypes()
        {
            yield return new DependentEntityInfo<NewPerson>(ListNewPersonCommand.Target, ListNewPersonCommand.ArgumentsCount, "100");
            yield return new DependentEntityInfo<FacilityMovementType>(ListFacilityMovementTypeCommand.Target, ListFacilityMovementTypeCommand.ArgumentsCount, ".04");
            yield return new DependentEntityInfo<Patient>(ListPatientCommand.Target, ListPatientCommand.ArgumentsCount, ListPatientCommand.CheckIen, ".03");
            yield return new DependentEntityInfo<RoomBed>(ListRoomBedCommand.Target, ListRoomBedCommand.ArgumentsCount, ".07");
            yield return new DependentEntityInfo<WardLocation>(ListWardLocationCommand.Target, ListWardLocationCommand.ArgumentsCount, ".06");
        }

        public List<PatientMovement> PostProcessEntities(ViaVistAQuery query, List<PatientMovement> result)
        {
            result = (from movement in result
                      let transaction = movement.TransactionTypeId.ParseDecimal()
                      where (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_ADMISSION_MAX) ||
                            (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_TRANSFER_MAX) ||
                            (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_DISCHARGE_MAX) ||
                            (transaction >= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MIN && transaction <= Constants.PATIENT_MOVEMENT_TRANSACTION_SPECIALTY_TRANSFER_MAX)
                      select movement).ToList();

            result.ForEach(r =>
            {
                if (r.Bed == null && !string.IsNullOrEmpty(r.RoomBedId))
                {
                    r.Bed = query.EntitySetCache.GetEntity<RoomBed>(query, r.RoomBedId);
                }

                if (r.Ward == null && !string.IsNullOrEmpty(r.WardLocationId))
                {
                    r.Ward = query.EntitySetCache.GetEntity<WardLocation>(query, r.WardLocationId);
                }
            });

            return result;
        }
    }
}
