﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Via.Commands.EntityCommands;
using BMS.Utils;
using BMS.VistaIntegration.Via.Commands.EIS;

namespace BMS.VistaIntegration.Via.Commands.WF
{
    public class ListOrderActionsCommand : BaseListPeriodCommand<OrderAction>
    {
        public static TimeSpan SplitDuration = TimeSpan.FromDays(1);

        private bool isAnticipated;

        private IEnumerable<Order> orders;

        private string currentOrdersSet;

        public ListOrderActionsCommand(ViaVistAQuery query)
            : base(query)
        {
        }

        public string PatientIen
        {
            get;
            set;
        }

        public IEnumerable<string> OrderableItemIens
        {
            get;
            set;
        }

        public override List<OrderAction> Execute(ViaVistASession session)
        {
            return Enumerable.Union(
                this.GetOrderActions(session),
                this.GetAnticipatedOrderActions(session),
                EqualityComparer).ToList();
        }

        protected override string GetTarget()
        {
            return "ListOrderActions";
        }

        protected override IEnumerable<object> GetCriteria()
        {
            yield return string.Join(",", this.OrderableItemIens);
            yield return this.currentOrdersSet;
            yield return this.StartDate.GetValueOrDefault(MinDate);
            yield return this.EndDate.GetValueOrDefault(this.MaxDate);
            yield return this.isAnticipated ? 2 : 1;
            yield return this.MaxCount;
            yield return this.From;
        }

        protected override IDependencySource GetDependencySource()
        {
            return OrderActionDependencySource.Instance;
        }

        protected override IEnumerable<OrderAction> ProcessSingleDictionary(ViaVistASession session, string key, Dictionary<string, List<string>> dictionary)
        {
            return base.ProcessSingleDictionary(session, key, dictionary).DoAction(x =>
            {
                x.OrderId = key;
                x.Order = this.orders.Single(y => y.IEN == key);
            });
        }

        private IEnumerable<OrderAction> GetAnticipatedOrderActions(ViaVistASession session)
        {
            var result = Enumerable.Empty<OrderAction>();
            if (!this.OrderableItemIens.IsNullOrEmpty())
            {
                this.isAnticipated = true;
                var startDate = this.StartDate.GetValueOrDefault(MinDate);
                var endDate = this.EndDate.GetValueOrDefault(this.MaxDate);
                try
                {
                    for (var date = startDate; date <= endDate; date += SplitDuration)
                    {
                        this.StartDate = date;
                        this.EndDate = date + SplitDuration;
                        this.orders = this.GetOrders();

                        result = this.MergeResults(session, result);
                    }
                }
                finally
                {
                    this.StartDate = startDate;
                    this.EndDate = endDate;
                }
            }

            return result;
        }

        private IEnumerable<OrderAction> GetOrderActions(ViaVistASession session)
        {
            var result = Enumerable.Empty<OrderAction>();
            if (!this.OrderableItemIens.IsNullOrEmpty())
            {
                var startDate = this.StartDate.GetValueOrDefault(MinDate);
                var endDate = this.EndDate.GetValueOrDefault(this.MaxDate);
                try
                {
                    this.isAnticipated = false;
                    for (var date = startDate; date <= endDate; date += SplitDuration)
                    {
                        this.StartDate = date;
                        this.EndDate = date + SplitDuration;
                        this.orders = this.GetOrders();

                        result = this.MergeResults(session, result);
                    }
                }
                finally
                {
                    this.StartDate = startDate;
                    this.EndDate = endDate;
                }
            }

            return result;
        }

        private IEnumerable<OrderAction> MergeResults(ViaVistASession session, IEnumerable<OrderAction> result)
        {
            foreach (var set in this.orders.Select(x => x.IEN).Distinct().JoinAndSplitByLength(',', MaxCriteriaLength))
            {
                this.currentOrdersSet = set;
                result = result.Union(base.Execute(session), EqualityComparer);
            }

            return result;
        }

        private IEnumerable<Order> GetOrders()
        {
            return this.VistAQuery.GetResults(new ListOrdersCommand(this.VistAQuery)
            {
                StartDate = this.StartDate,
                EndDate = this.EndDate,
                OrderableItemIens = this.OrderableItemIens,
                PatientIen = this.PatientIen,
                IsAnticipated = this.isAnticipated,
            }).Distinct(ListOrdersCommand.EqualityComparer);
        }
    }

    internal sealed class OrderActionDependencySource : IDependencySource
    {
        private static IDependencySource instance;

        private OrderActionDependencySource()
        {
        }

        public static IDependencySource Instance
        {
            get
            {
                if (instance == null)
                {
                    instance = new OrderActionDependencySource();
                }

                return instance;
            }
        }

        public IEnumerable<DependentEntityInfo> GetDependentTypes()
        {
            yield return new DependentEntityInfo<NewPerson>(ListNewPersonCommand.Target, ListNewPersonCommand.ArgumentsCount, "5", "3");
        }
    }
}
