﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.Text.RegularExpressions;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using Mdws2ORM;
using Mdws2ORM.Core;
using Mdws2ORM.Impl;
using Mdws2ORM.Maps;

namespace BMS.VistaIntegration.Via.Commands
{
    public abstract class BaseCommand<T> where T : class, IEntity
    {
        public const int MaxCriteriaLength = 255;

        private readonly IEntityQuery entityQuery;

        protected BaseCommand(ViaVistAQuery query)
        {
            if (query == null)
            {
                throw new ArgumentNullException("query");
            }

            this.Logger = new BmsLogger(string.Format("{0}:{1} - ", this.GetType().GetFriendlyName(), query.Site.Name));

            this.VistAQuery = query;
            this.EntityMap = EntityMapRegistration.InitMediator.GetBaseEntityMap<T>();
            this.entityQuery = query.EntitySetCache.CreateEntityQuery(query);
        }

        public BaseEntityMap<T> EntityMap
        {
            get;
            protected set;
        }

        protected BmsLogger Logger
        {
            get;
            private set;
        }

        protected ViaVistAQuery VistAQuery
        {
            get;
            private set;
        }

        protected static IComparer<T> Comparer
        {
            get
            {
                return new EntityComparer();
            }
        }

        protected static IEqualityComparer<T> EqualityComparer
        {
            get
            {
                return new EntityComparer();
            }
        }

        public virtual List<T> Execute(ViaVistASession session)
        {
            var result = this.GetEntitiesFromSection(session);
            if (result != null)
            {
                var dependencySource = this.GetDependencySource() as IDependencySource<T>;
                if (dependencySource != null)
                {
                    result = dependencySource.PostProcessEntities(this.VistAQuery, result);
                }
            }
            return result;
        }

        protected virtual List<T> GetEntitiesFromSection(ViaVistASession session)
        {
            return this.GetEntitiesFromData(session);
        }

        protected abstract List<T> GetEntitiesFromData(ViaVistASession session);

        protected abstract string GetTarget();

        protected virtual IEnumerable<object> GetCriteria()
        {
            yield return string.Empty;
            yield return string.Empty;
        }

        protected T GetEntity(Entry entry, BaseEntityMap<T> entityMap, string iens)
        {
            var entity = entityMap.NewEntity(entry.Ien);

            Translator.PopulateFields(this.entityQuery, entity, entry, entityMap.AllFields.ToArray());

            /**
            if (entityMap.MultipleFieldsMap.Length > 0)
                AddMultipleFields(entityMap, entity, Utilities.MakeListIens(entry, iens));
            if (entityMap.GetForListFieldsMap.Length > 0)
                AddGetFetchOnlyFields(entityMap, entity, Utilities.MakeGetIens(entry, iens));
             **/

            return entity;
        }

        protected virtual IDependencySource GetDependencySource()
        {
            return null;
        }

        protected void PrepareCache(VistASite site, IList<Entry> entries)
        {
            if (entries.Count < 1)
            {
                // There is no need for downloading dependent entities when there are no entities depending on them.
                return;
            }

            var dependencySource = this.GetDependencySource();

            if (dependencySource != null)
            {
                var ienEntries = (from type in dependencySource.GetDependentTypes()
                                  select new
                                  {
                                      Type = type.EntityType,
                                      IENS = (from item in entries
                                              from field in item.Fields
                                              where type.FieldIens.Contains(field.FieldIen) && !string.IsNullOrWhiteSpace(field.Value) && type.IsValidIen(field.Value)
                                              select field.Value).Distinct().ToArray(),
                                      Target = type.Target,
                                      ArgumentsCount = type.ArgumentsCount,
                                      Separator = type.Separator,
                                  }).ToArray();

                foreach (var ienEntry in ienEntries)
                {
                    this.Logger.LogFormat(BmsLogger.Level.Verbose, string.Format("Preparing cache for type {0} with IENs '{1}'", ienEntry.Type.Name, string.Join(",", ienEntry.IENS)));

                    EntitySetCache.DownloadEntities(ienEntry.Type, this.VistAQuery, site, ienEntry.IENS, ienEntry.Separator, ienEntry.Target, ienEntry.ArgumentsCount);
                }
            }
        }

        protected Dictionary<string, Dictionary<string, List<string>>> GetSectionsFromVia(ViaVistASession session)
        {
            var target = this.GetTarget();
            var criteria = string.Join("^", from criterion in this.GetCriteria() select ConvertToString(criterion));

            var reply = session.GetBMSDataFromVia(target, criteria);

            if (reply.fault != null)
            {
                this.Logger.LogFormat(BmsLogger.Level.Error, "VIA call failed for target '{0}' and criteria '{1}'\r\nRoot error message: {2}", target, criteria, reply.fault.message);
                throw new ViaException(string.Format("Response from VIA for target '{0}' and criteria '{1}': {2}", target, criteria, reply.fault.message));
            }

            var bulkText = string.Empty;
            if (reply.text != null && reply.text.Length > 0)
            {
                var text = reply.text[0];
                bulkText = text.Trim();
            }

            var sections = ParseSections(bulkText);

            var errorKey = sections.Keys.FirstOrDefault(x => x.StartsWith("Error", StringComparison.OrdinalIgnoreCase));
            if (!string.IsNullOrEmpty(errorKey))
            {
                var message = string.Join("\r\n", sections[errorKey]);
                this.Logger.LogFormat(BmsLogger.Level.Error, "VIA call returned error for target '{0}' and criteria '{1}'\r\nError message: {2}", target, criteria, message);
                throw new ViaException(string.Format("Response from VIA for target '{0}' and criteria '{1}': {2}", target, criteria, message));
            }

            return sections;
        }

        private string ConvertToString(object criteria)
        {
            if (criteria == null)
            {
                return string.Empty;
            }

            if (criteria is DateTime)
            {
                return ((DateTime)criteria).ToInternalFormat();
            }

            return criteria.ToString();
        }

        private static Dictionary<string, Dictionary<string, List<string>>> ParseSections(string bulkText)
        {
            var sections = new Dictionary<string, Dictionary<string, List<string>>>();

            var lines = bulkText.Split('\r', '\n').Where(x => !string.IsNullOrWhiteSpace(x));
            var currentSection = string.Empty;
            Dictionary<string, List<string>> currentDictionary = null;
            foreach (var line in lines)
            {
                var sectionHeadMatch = Regex.Match(line, @"^\[(?<head>.*)\](?<rest>.*)$");
                if (sectionHeadMatch != null && sectionHeadMatch.Success)
                {
                    currentSection = sectionHeadMatch.Groups["head"].Value;
                    var tag = string.Empty;
                    var tagMatch = Regex.Match(sectionHeadMatch.Groups["rest"].Value, "^ - (?<tag>.*)$");
                    if (tagMatch != null && tagMatch.Success)
                    {
                        tag = tagMatch.Groups["tag"].Value;
                    }

                    if (!sections.TryGetValue(tag, out currentDictionary))
                    {
                        currentDictionary = new Dictionary<string, List<string>>();
                        sections.Add(tag, currentDictionary);
                    }

                    if (!currentDictionary.ContainsKey(currentSection))
                    {
                        currentDictionary.Add(currentSection, new List<string>());
                    }
                }
                else if (currentDictionary != null)
                {
                    currentDictionary[currentSection].Add(line);
                }
            }

            return sections;
        }

        private sealed class EntityComparer : IComparer<T>, IEqualityComparer<T>
        {
            public int Compare(T x, T y)
            {
                string xIen = null;
                if (x != null)
                {
                    xIen = x.IEN;
                }

                string yIen = null;
                if (y != null)
                {
                    yIen = y.IEN;
                }

                if (string.IsNullOrEmpty(xIen) && string.IsNullOrEmpty(yIen))
                {
                    return 0;
                }

                if (string.IsNullOrEmpty(xIen))
                {
                    return -1;
                }

                if (string.IsNullOrEmpty(yIen))
                {
                    return 1;
                }

                double xIenValue, yIenValue;

                if (!double.TryParse(xIen, out xIenValue) || !double.TryParse(yIen, out yIenValue))
                {
                    return xIen.CompareTo(yIen);
                }

                return xIenValue.CompareTo(yIenValue);
            }

            public bool Equals(T x, T y)
            {
                return this.Compare(x, y) == 0;
            }

            public int GetHashCode(T obj)
            {
                return ~(obj.GetHashCode() ^ (obj.IEN ?? string.Empty).GetHashCode());
            }
        }
    }

    public interface IDependencySource
    {
        IEnumerable<DependentEntityInfo> GetDependentTypes();
    }

    public interface IDependencySource<T> : IDependencySource
    {
        List<T> PostProcessEntities(ViaVistAQuery query, List<T> result);
    }

    public abstract class DependentEntityInfo
    {
        private Func<string, bool> iensFilter;

        protected DependentEntityInfo(Type type, string target, int argumentsCount, string[] fieldIens, char separator = '\0', Func<string, bool> iensFilter = null)
        {
            this.EntityType = type;
            this.Target = target;
            this.ArgumentsCount = argumentsCount;
            this.FieldIens = fieldIens;
            this.Separator = separator;
            this.iensFilter = iensFilter;
        }

        public Type EntityType
        {
            get;
            private set;
        }

        public string Target
        {
            get;
            private set;
        }

        public int ArgumentsCount
        {
            get;
            private set;
        }

        public string[] FieldIens
        {
            get;
            private set;
        }

        public char Separator
        {
            get;
            private set;
        }

        public bool IsValidIen(string ien)
        {
            return this.iensFilter == null || this.iensFilter(ien);
        }
    }

    public sealed class DependentEntityInfo<T> : DependentEntityInfo where T : class, IEntity
    {
        public DependentEntityInfo(string target, int argumentsCount, params string[] fieldIens)
            : base(typeof(T), target, argumentsCount, fieldIens)
        {
        }

        public DependentEntityInfo(string target, int argumentsCount, char separator, params string[] fieldIens)
            : base(typeof(T), target, argumentsCount, fieldIens, separator: separator)
        {
        }

        public DependentEntityInfo(string target, int argumentsCount, Func<string, bool> iensFilter, params string[] fieldIens)
            : base(typeof(T), target, argumentsCount, fieldIens, iensFilter: iensFilter)
        {
        }
    }

    [Serializable]
    internal class ViaException : Exception
    {
        public ViaException()
        {
        }

        public ViaException(string message)
            : base(message)
        {
        }

        public ViaException(string message, Exception innerException)
            : base(message, innerException)
        {
        }

        protected ViaException(SerializationInfo info, StreamingContext context)
            : base(info, context)
        {
        }
    }
}
