﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using System.ServiceModel;
using BMS.VistaIntegration.Exceptions;
using BMS.VistaIntegration.VistA;
using Mdws2ORM.QuerySvcService;
using System.Configuration;


namespace BMS.VistaIntegration.Mdws
{
    public class MdwsVistASession : IVistASession
    {
        private static string userPwd = null, userSiteCode = null, userName = null, userDuz = null, userSsn = null, userContext = null;

        public MdwsVistASession(VistASite site)
        {
            VistASite = site;
        }

        static MdwsVistASession()
        {
            InitUserSettings();
        }

        private QuerySvcSoapClient _client;

        public QuerySvcSoap Client
        {
            get { return _client; }
        }

        public VistASite VistASite { get; private set; }
        private bool _isOpen = false;
        private bool _isDisposable = false;

        private void CheckIsOpen()
        {
            if (_isDisposable)
                throw new ObjectDisposedException("MdwsVistASession");
            if (!_isOpen)
                throw new InvalidOperationException("VistA session is not open");
        }

        private void OnFault(FaultTO fault)
        {

            if (fault.message == "That site id is in use")
                return;
            if (TryAgain(fault))
                throw new VistAConnectionException("Try again later.");
            throw new VistAConnectionException(fault.message);

        }

        private bool TryAgain(FaultTO fault)
        {
            if (fault == null)
                return false;
            else if (fault.message.Equals("The remote procedure XUS INTRO MSG is not registered to the option XUS SIGNON."))
                return true;
            else if (fault.message.Equals("Object synchronization method was called from an unsynchronized block of code."))
                return true;
            return false;
        }

        private UserTO TryConnect(string userPwd, string number, string userSiteCode, string userName, string userDuz, string userSsn, string userContext, int seconds)
        {
            System.Threading.Thread.Sleep(TimeSpan.FromSeconds(seconds));
            return _client.visitSite(userPwd, number, userSiteCode, userName, userDuz, userSsn, userContext);
        }

        private static void InitUserSettings()
        {
            try
            {
                userPwd = ConfigurationManager.AppSettings["visitorAppPwd"];
                userSiteCode = ConfigurationManager.AppSettings["visitorUserSiteCode"];
                userName = ConfigurationManager.AppSettings["visitorUserName"];
                userDuz = ConfigurationManager.AppSettings["visitorUserDuz"];
                userSsn = ConfigurationManager.AppSettings["visitorUserSsn"];
                userContext = ConfigurationManager.AppSettings["visitorContext"];
            }
            catch { throw new VistAConnectionException("Default config parameters are incorrect!"); } 
        }

        private UserTO VisitSite()
        {
            var visit = _client.visitSite(userPwd, VistASite.Number, userSiteCode, userName, userDuz, userSsn, userContext);
            if (TryAgain(visit.fault)) visit = TryConnect(userPwd, VistASite.Number, userSiteCode, userName, userDuz, userSsn, userContext, 1);
            if (TryAgain(visit.fault)) visit = TryConnect(userPwd, VistASite.Number, userSiteCode, userName, userDuz, userSsn, userContext, 5);
            if (TryAgain(visit.fault)) visit = TryConnect(userPwd, VistASite.Number, userSiteCode, userName, userDuz, userSsn, userContext, 10);
            return visit;
        }

        public void Open(VistAConnectionInfo vistAConnection)
        {
            if (_isOpen)
                throw new InvalidOperationException("VistA session is already open");
            if (_isDisposable)
                throw new ObjectDisposedException("MdwsVistASession");
            try
            {
                InternalOpen(vistAConnection);
            }
            catch (Exception ex)
            {
                if ((ex as EndpointNotFoundException) != null && ex.Message.StartsWith("There was no endpoint listening"))
                {
                    try
                    {
                        InternalOpen(vistAConnection);
                    }
                    catch (Exception e) { throw new VistAConnectionException(e); };
                }
                else
                    throw new VistAConnectionException(ex);
            }
            _isOpen = true;
        }

        private void InternalOpen(VistAConnectionInfo vistAConnection)
        {
            _client = new QuerySvcSoapClient();
            if (!string.IsNullOrEmpty(VistASite.MdwsEndpointUrl))
            {
                var serverUri = new Uri(VistASite.MdwsEndpointUrl);
                _client.Endpoint.Address = new EndpointAddress(serverUri, _client.Endpoint.Address.Identity, _client.Endpoint.Address.Headers);
                _client.ChannelFactory.Endpoint.Address = new EndpointAddress(serverUri, _client.Endpoint.Address.Identity, _client.Endpoint.Address.Headers);
            }
            _client.Endpoint.Behaviors.Add(new MdwsManagerEndpointBehavior());

            var visit = VisitSite();
            if (visit.fault != null)
            {
                _client.disconnect();
                OnFault(visit.fault);
            }
        }

        public void RunClientAction(Action action)
        {
            try
            {
                CheckIsOpen();
                action();
            }
            catch (CommunicationException e)
            {
                throw new VistAConnectionException(e);
            }
            catch (TimeoutException e)
            {
                throw new VistAConnectionException(e);
            }
        }

        public void Close()
        {
            if (_isDisposable)
                return;
            _isDisposable = true;
            _isOpen = false;
            if (_client == null)
                return;

            if (_client.State == CommunicationState.Faulted)
            {
                _client.Abort();
            }
            else
            {
                try
                {
                    _client.disconnect();
                    _client.Close();
                }
                catch (Exception)
                {
                    _client.Abort();
                }
            }
        }

        public IVistAQuery MakeQuery()
        {
            return new MdwsVistAQuery(this);
        }

        public void Dispose()
        {
            Close();
        }

        public bool IsAlive()
        {
            try
            {
                if (!_isOpen || _isDisposable) 
                    return false;

                _client.disconnect();
                var connectDto = VisitSite();
                if (connectDto.fault == null) return true;
                return connectDto.fault.message.Contains("already connected");
            }
            catch
            {
                return false;
            }

        }

        public void Test(VistAConnectionInfo connection)
        {
            this.Open(connection);
        }
    }
}
