﻿using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using Mdws2ORM.Maps;
using Mdws2ORM.QuerySvcService;
using BMS.VistaIntegration.Mdws.Commands.EntityCommands;
using BMS.VistaIntegration.Mdws.Commands.SubEntityCommands;
using System;
using Mdws2ORM.Maps.SingleMaps;
using Mdws2ORM;
using BMS.VistaIntegration.Mdws.Commands.Concrete.WF;
using BMS.Utils;
using BMS.VistaIntegration.Mdws.Commands.Concrete;

namespace BMS.VistaIntegration.Mdws.Commands.WF
{
    public class ListPatientAppointmentFromClinicsCommand
    {
        public IEnumerable<string> ClinicIens { get; set; }
        public DateTime? StartDate { get; set; }
        public DateTime? EndDate { get; set; }
        public int? MaxCount { get; set; }
        private static ClinicMap AppClinicMap;
        private static BmsLogger logger = new BmsLogger("Vista Integration: ListPatientAppointmentFromClinicsCommand:");

        static ListPatientAppointmentFromClinicsCommand()
        {
            AppClinicMap = new ClinicMap();
        }

        class ClinicApp
        {
            public DateTime AppDateTime { get; set; }
            private string _patients;
            public string Patients
            {
                get { return _patients; }
                set
                {
                    _patients = value;
                    PatientList = new List<string>();
                    if (!string.IsNullOrEmpty(_patients))
                        PatientList = _patients.Split(',').ToList();
                }
            }
            public List<string> PatientList { get; set; }
        }

        class ClinicMap : SubEntityMap<ClinicApp>
        {

            public override ClinicApp NewEntity(string ien)
            {
                return new ClinicApp();
            }

            public override string File
            {
                get { return "44.001"; }
            }

            protected override void MapFields()
            {
                this.MapDateTime(".01", (m, s) => m.AppDateTime = s);
                this.MapFreeText("2", (m, s) => m.Patients = s);
            }
        }

        class ListClinicsCommand
        {
            public DateTime? StartTime { get; set; }
            public DateTime? EndTime { get; set; }
            public int? MaxCount { get; set; }
            public string ClinicIen { get; set; }
            private string clinicIenFormat;

            private void InitStartTime(ISession session)
            {
                ListParamBuilder builder = new ListParamBuilder();
                builder.Number("1");
                builder.Index("#");
                IList<ClinicApp> start = session.List<ClinicApp>(builder.Build(), ListPatientAppointmentFromClinicsCommand.AppClinicMap, clinicIenFormat).Select(s => s.Entity).ToList();
                if (start.Any())
                    StartTime = start[0].AppDateTime;
            }

            private string GetScreen()
            {
                return EndTime.HasValue ? new ScreenPeriodBuilder().Format("I {0}").Variable("Y").Build(StartTime, EndTime) : string.Empty;
            }

            public IList<ClinicApp> Execute(ISession session)
            {
                clinicIenFormat = string.Format(",{0},", ClinicIen);
                if (!StartTime.HasValue)
                    InitStartTime(session);
                if (!StartTime.HasValue)
                    return new ClinicApp[0];

                ListParamBuilder builder = new ListParamBuilder();
                builder.Index("#");
                builder.Screen(GetScreen());
                if (MaxCount.HasValue)
                    builder.Number(MaxCount.Value.ToString());
                builder.Identity("I $D(^SC([CLINIC_IEN],\"S\",Y,1)) S I=0 F  S I=$O(^SC([CLINIC_IEN],\"S\",Y,1,I)) Q:+I=0  D EN^DDIOL($P(^(I,0),U))".Replace("[CLINIC_IEN]", ClinicIen));

                string from = Converters.ToInternalFormat(StartTime.Value.AddSeconds(-1));

                List<ClinicApp> result = new List<ClinicApp>();
                bool hasMore = true;

                while (hasMore)
                {
                    builder.From(from);
                    int size = result.Count;
                    result.AddRange(session.List<ClinicApp>(builder.Build(), ListPatientAppointmentFromClinicsCommand.AppClinicMap, clinicIenFormat).Select(s => s.Entity));
                    int partialSize = result.Count - size;
                    hasMore = MaxCount.HasValue ? MaxCount.Value == partialSize : false;
                    if (hasMore)
                    {
                        DateTime lastDateTime = result.Last().AppDateTime;
                        if (EndTime.HasValue)
                        {
                            if (lastDateTime <= EndTime)
                                from = Converters.ToInternalFormat(lastDateTime);
                            else
                            {
                                result.RemoveAll(a => a.AppDateTime > EndTime);
                                hasMore = false;
                            }
                        }
                        else
                            from = Converters.ToInternalFormat(lastDateTime);
                    }
                }
                return result;
            }
        }

        private IEnumerable<string> GetPatients(string ien, ISession session)
        {
            List<string> patients = new List<string>();
            ListClinicsCommand command = new ListClinicsCommand();
            command.ClinicIen = ien;
            command.StartTime = StartDate;
            command.EndTime = EndDate;
            command.MaxCount = MaxCount;
            IList<ClinicApp> clinics = command.Execute(session);
            foreach (ClinicApp ca in clinics)
            {
                if (ca.PatientList != null && ca.PatientList.Count > 0)
                    patients.AddRange(ca.PatientList);
            }
            logger.LogFormat(BmsLogger.Level.Info, "ListPatientAppointmentFromClinicsCommand - GetPatients: count - {0} for clinic {1}", patients.Count, ien);
            return patients;
        }

        public List<PatientAppointment> Execute(ISession session)
        {
            if (ClinicIens == null)
                throw new InvalidOperationException();

            List<PatientAppointment> result = new List<PatientAppointment>();
            ListPatientAppointmentFromPatientCommand command = new ListPatientAppointmentFromPatientCommand();            

            foreach (string clinic in ClinicIens)
            {
                command.ClinicIen = clinic;
                foreach (string patient in GetPatients(clinic, session))
                {
                    command.PatientIen = patient;
                    command.StartTime = StartDate;
                    command.EndTime = EndDate;
                    result.AddRange(command.Execute(session));
                }
            }
            return result;
        }
    }
}
