﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Data;
using BMS.ServicesWrapper.EVS;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using FC = BMS.Facade.Data;
using InfoWorld.EVS.CTSMAPI;

using SC = BMS.DataContracts;
using BMS.ServicesWrapper.WF;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.Facade;
using BMS.ServicesWrapper.BMService;

namespace BMS.VistaWorker2.Writer.Implementation.Concrete.WF
{
    /// <summary>
    /// The order writer.
    /// </summary>
    class OrderWriter : BaseWorkFlowWriter<OrderAction>
    {
        List<CD> orderableItemTypeList = new List<CD>();
        List<CDWithProperties> ADTOrderableItemList = new List<CDWithProperties>();        
        List<Facility> facilitiesByVista = new List<Facility>();        

        private string GetOrderType(Order entity, out IList<CDWithProperties> cdpList, out string orderableItem)
        {
            orderableItem = null;
            cdpList = null;

            if (entity.OrderOrderableItemIds == null)
                return null;
            string orderType = null;
            foreach (string ordItem in entity.OrderOrderableItemIds)
            {
                cdpList = ADTOrderableItemList.Where(a => a.code.StartsWith(VistaSite.Code + "_", StringComparison.InvariantCultureIgnoreCase) && a.code.EndsWith("_" + ordItem, StringComparison.InvariantCultureIgnoreCase)).ToList();
                orderType = null;
                if (cdpList != null && cdpList.Count > 0 && cdpList[0] != null)
                {
                    ConceptProperty cp = cdpList[0].Properties.Where(a => a.PropertyName.text == "OrderableItemType").FirstOrDefault();
                    if (cp != null)
                    {
                        CD cd = orderableItemTypeList.Where(a => a.code.Equals(cp.PropertyValue.text, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();                        
                        orderType = cd.displayName;
                    }                    
                    if (!string.IsNullOrEmpty(orderType))
                    {
                        orderableItem = ordItem;
                        return orderType;
                    }
                }
            }
            return null;
        }

        private static bool ContainsString(string text, string value)
        {
            return text.IndexOf(value, StringComparison.OrdinalIgnoreCase) >= 0;
        }

        private bool AnticipatedDischarge(OrderAction orderAction)
        {
            bool isAnticipatedDischarge = (ContainsString(orderAction.OrderText, "ANTICIPATE") || ContainsString(orderAction.OrderText, "PLANNED")) && ContainsString(orderAction.OrderText, "DISCHARGE");

            if (!isAnticipatedDischarge)
                return false;

            Order entity = orderAction.Order;
            FC.HospitalLocation hospitallocation = InsertIfNullOrUpdateIfDirty<BMS.VistaIntegration.Data.HospitalLocation, FC.HospitalLocation>(entity.HospitalLocation);            
            string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();

            SC.DischargeOrder wfDischarge = new SC.DischargeOrder()
            {
                Id = new II(domainId, null),
                PatientId = MakePatient(entity.Patient),                
                OrderedDate = orderAction.DateTimeOrdered,
                SignedDate = orderAction.DateTimeSigned,
                Ien = entity.IEN,
                OrderText = orderAction.OrderText,
                ReleasedDate = orderAction.ReleaseDateTime,
                SignedById = MakePerson(orderAction.SignedBy),
                ProviderId = MakePerson(orderAction.Provider),
                HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                FacilityList = facilitiesByVista.Select(facility => facility.Id).ToList(),
                IsActive = (entity.StatusId != null && entity.StatusId == "6") ? true : false,
                WardId = (hospitallocation != null && !string.IsNullOrEmpty(hospitallocation.WardLocationId)) ? new II(domainId, hospitallocation.WardLocationId) : null
            };

            ConvertToUtc(wfDischarge);
            BMSFactory.BedManagerOperationsClientWindows.CreateDischargeOrder(wfDischarge);
            UtilsInstance.SendMailNotification(orderAction, wfDischarge.VistaSiteId, VistATimeZone, Constants.ANTICIPATED_DISCHARGE_ORDER, null, wfDischarge.PatientId);
            Logger.LogFormat(BmsLogger.Level.Info, "Create anticipated discharge for patient {0} with succes", entity.Patient.Name);
            hospitallocation = null;            
            return true;
        }

        /// <summary>
        /// Sends the event.
        /// </summary>
        /// <param name="entity">The entity.</param>
        protected override bool SendEvent(OrderAction orderAction)
        {
            FillOrdersData();
            IList<CDWithProperties> cdpList = null;
            string orderableItem;
            Order entity = orderAction.Order;
            if (entity == null)
            {
                Tracer.TraceMessage(string.Format("Order with id={0} has no order.", orderAction.OrderId));
                return false;
            }
            string orderType = GetOrderType(entity, out cdpList, out orderableItem);

            if (string.IsNullOrEmpty(orderType))
            {
                bool anticipatedDischarge = AnticipatedDischarge(orderAction);
                if (anticipatedDischarge) return true;

                if (entity.Patient != null)
                    Logger.LogFormat(BmsLogger.Level.Info, "Order with id={0} for patient {1} has no type.", entity.IEN, entity.Patient.Name);
                else
                    Logger.LogFormat(BmsLogger.Level.Info, "Order with id={0} has no type and no patient.", entity.IEN);
                return false;
            }

            if (entity.Patient == null)
                throw new WriterException(entity, string.Format("Order with id={0} has no patient.", entity.IEN));

            FC.HospitalLocation hospitallocation = InsertIfNullOrUpdateIfDirty<BMS.VistaIntegration.Data.HospitalLocation, FC.HospitalLocation>(entity.HospitalLocation);            
            string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();            

            if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_ADMISSION))
            {
                SC.AdmissionOrder wfAdmission = new SC.AdmissionOrder()
                {
                    Id = new II(domainId, null),                    
                    PatientId = MakePatient(entity.Patient),
                    OrderedDate = orderAction.DateTimeOrdered,
                    SignedDate = orderAction.DateTimeSigned,
                    Ien = entity.IEN,
                    OrderText = orderAction.OrderText,
                    ReleasedDate = orderAction.ReleaseDateTime,
                    SignedById = MakePerson(orderAction.SignedBy),
                    ProviderId = MakePerson(orderAction.Provider),
                    HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                    VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                    FacilityList = CreateFacilitiesList(cdpList),
                    IsActive = (entity.StatusId != null && entity.StatusId == "6") ? true : false,
                    WardId = (hospitallocation != null && !string.IsNullOrEmpty(hospitallocation.WardLocationId)) ? new II(domainId, hospitallocation.WardLocationId) : null
                };

                ConvertToUtc(wfAdmission);                
                BMSFactory.BedManagerOperationsClientWindows.CreateAdmissionOrder(wfAdmission);
                UtilsInstance.SendMailNotification(orderAction, wfAdmission.VistaSiteId, VistATimeZone, Constants.ADMISSION_ORDER, null, wfAdmission.PatientId);
                Logger.LogFormat(BmsLogger.Level.Info, "Create admission order for patient {0} with succes", entity.Patient.Name);

                if (entity.Patient.IsOptOut)
                {
                    // associate icon with this patient
                    SC.IconAssociation iconAssociation = new SC.IconAssociation();
                    iconAssociation.PatientId = wfAdmission.PatientId;
                    iconAssociation.ListIconFlagIdsOfPatient = new List<int>() { 18 };
                    BMSFactory.BedManagerOperationsClientWindows.InsertIconAssociation(iconAssociation);
                }

                return true;                
            }

            if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_DISCHARGE))
            {
                SC.DischargeOrder wfDischarge = new SC.DischargeOrder()
                {
                    Id = new II(domainId, null),
                    PatientId = MakePatient(entity.Patient),                    
                    OrderedDate = orderAction.DateTimeOrdered,
                    SignedDate = orderAction.DateTimeSigned,
                    Ien = entity.IEN,
                    OrderText = orderAction.OrderText,
                    ReleasedDate = orderAction.ReleaseDateTime,
                    SignedById = MakePerson(orderAction.SignedBy),
                    ProviderId = MakePerson(orderAction.Provider),
                    OrderableItem = GetVistaOrderableItem(orderableItem),
                    HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                    VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                    FacilityList = CreateFacilitiesList(cdpList),
                    IsActive = (entity.StatusId != null && entity.StatusId == "6") ? true : false,
                    WardId = (hospitallocation != null && !string.IsNullOrEmpty(hospitallocation.WardLocationId)) ? new II(domainId, hospitallocation.WardLocationId) : null
                };

                ConvertToUtc(wfDischarge);
                BMSFactory.BedManagerOperationsClientWindows.CreateDischargeOrder(wfDischarge);
                UtilsInstance.SendMailNotification(orderAction, wfDischarge.VistaSiteId, VistATimeZone, Constants.DISCH_ORDER, null, wfDischarge.PatientId);
                Logger.LogFormat(BmsLogger.Level.Info, "Create discharge request for patient {0} with succes", entity.Patient.Name);

                return true;
            }

            if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_TRANSFER))
            {
                SC.MovementOrder wfMovement = new SC.MovementOrder()
                {
                    Id = new II(domainId, null),                    
                    PatientId = MakePatient(entity.Patient),
                    OrderedDate = orderAction.DateTimeOrdered,
                    SignedDate = orderAction.DateTimeSigned,
                    Ien = entity.IEN,
                    OrderText = orderAction.OrderText,
                    ReleasedDate = orderAction.ReleaseDateTime,
                    SignedById = MakePerson(orderAction.SignedBy),
                    ProviderId = MakePerson(orderAction.Provider),
                    HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                    VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                    FacilityList = CreateFacilitiesList(cdpList),
                    IsActive = (entity.StatusId != null && entity.StatusId == "6") ? true : false,
                    WardId = (hospitallocation != null && !string.IsNullOrEmpty(hospitallocation.WardLocationId)) ? new II(domainId, hospitallocation.WardLocationId) : null
                };
                ConvertToUtc(wfMovement);
                BMSFactory.BedManagerOperationsClientWindows.CreateMovementOrder(wfMovement);
                
                UtilsInstance.SendMailNotification(orderAction, wfMovement.VistaSiteId, VistATimeZone, Constants.TRANSFER_ORDER, null, wfMovement.PatientId);
                Logger.LogFormat(BmsLogger.Level.Info, "Create movement request for patient {0} with succes", entity.Patient.Name);
                hospitallocation = null;

                string orderStatusName = entity.OrderStatus.Name;
                if (orderStatusName == "ACTIVE" || orderStatusName == "RENEWED")
                {
                    FC.WaitingListItem item = new FC.WaitingListItem();
                    IList<CD> cds = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = BMS.Facade.Data.Util.Vocabulary.WaitingArea.ToString(), MaxSelectedCodes = int.MaxValue });
                    item.WaitingArea = cds.FirstOrDefault(cd => cd.code == Constants.WAITING_AREA_SCHEDULED_ADMISSIONS);
                    string domain = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                    item.Id = new II(domain, null);
                    item.Patient = MakeWfPatient(entity.Patient);

                    item.RequestedBedDate = null;

                    Ward ward = FacadeManager.EntityInterface.GetWardById(wfMovement.WardId);
                    if (ward != null)
                    {
                        if (ward.Division != null)
                            item.Facility = ward.Division.Facility;

                        item.TypeOfBedWard = ward.Specialty.displayName;
                    }

                    item.VistaSite = VistaSite;
                    ConvertToUtc(item);
                    item.CreationDate = DateTime.UtcNow;
                    item.RequestedDate = DateTime.UtcNow;
                    WFFactory.WaitingListFlowClientWindows.CreateWaitingListItem(item);
                    Logger.LogFormat(BmsLogger.Level.Info, "Create Waiting List Item for Transfer patient {0}, {1} {2} with success",
                                   item.Patient.LastName, item.Patient.FirstName, item.Patient.MiddleName);
                }

                return true;
            }

            if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_AUTOICON))
            {
                List<IconInfo> allIcons = FacadeManager.IconInterface.GetAllIconsByFacility(Guid.Parse(VistaSite.Id.extension));
                List<IconInfo> autoIcons = new List<IconInfo>();
                if (allIcons != null && allIcons.Count > 0)
                {
                    foreach (IconInfo icon in allIcons)
                    {
                        if (icon.Active && icon.AutoIconActive && !String.IsNullOrWhiteSpace(icon.AutoIconPattern))
                        {
                            autoIcons.Add(icon);
                        }
                    }
                }

                II patientId = MakePatient(entity.Patient);
                int? iconFlagId = null;
                bool isIsolation = false;

                // find auto-icon pattern for icon name
                foreach (IconInfo autoIcon in autoIcons)
                {
                    if (orderAction.OrderText.Contains(autoIcon.AutoIconPattern))
                    {
                        iconFlagId = autoIcon.IconId;

                        if (autoIcon.IsIsolation)
                            isIsolation = true;
                    }
                }

                if (iconFlagId != null)
                {
                    string orderStatusName = entity.OrderStatus.Name;
                    if (orderStatusName == "ACTIVE" || orderStatusName == "RENEWED")
                    {
                        // associate icon with this patient
                        SC.IconAssociation iconAssociation = new SC.IconAssociation();
                        iconAssociation.PatientId = patientId;
                        iconAssociation.ListIconFlagIdsOfPatient = new List<int>() { iconFlagId.Value };

                        if (isIsolation)
                        {
                            SC.PatientLocation patientLocation = BMSFactory.BedManagerOperationsClientWindows.GetPatientLocation(patientId);
                            iconAssociation.ListIconFlagIdsOfBed = new List<int> { 7 };
                            iconAssociation.BedId = patientLocation.CurrentAdmissionBedId;
                        }

                        BMSFactory.BedManagerOperationsClientWindows.InsertIconAssociation(iconAssociation);
                    }
                    else if (orderStatusName == "DISCONTINUED" || orderStatusName == "COMPLETE" || orderStatusName == "EXPIRED" || orderStatusName == "DISCONTINUED/EDIT" || orderStatusName == "CANCELLED" || orderStatusName == "LAPSED")
                    {
                        // remove icon association for this patient
                        BMSFactory.BedManagerOperationsClientWindows.DeleteIconAssociation(Guid.Parse(patientId.extension), iconFlagId);
                    }
                }

                return true;
            }
            

            return false;
        }

        private CD GetVistaOrderableItem(string ordItem)
        {
            if (string.IsNullOrEmpty(ordItem)) 
                return null;
            CodeFilterParameters codeFilterParam = new CodeFilterParameters();
            codeFilterParam.MaxSelectedCodes = int.MaxValue;
            codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItem.ToString();
            codeFilterParam.MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase;
            codeFilterParam.MatchProperty = MatchProperty.Code;
            codeFilterParam.MatchText = VistaSite.Code + "_" + ordItem;
            List<CD> list = EVS.SearchConceptsWithPaging(codeFilterParam);
            if (list != null && list.Count > 0)
                return list[0];
            return null;
        }

        private List<II> CreateFacilitiesList(IList<CDWithProperties> cdpList)
        {
            if (cdpList == null)
                return null;
            List<II> facilitiesList = new List<II>();
            Facility facility = null;
            foreach (CDWithProperties cdp in cdpList)
            {
                string[] cdpCodeArray = cdp.code.Split('_');
                facility = facilitiesByVista.Where(a => a.Code.Equals(cdpCodeArray[1], StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (facility != null)
                    facilitiesList.Add(facility.Id);                
            }
            return facilitiesList;
        }

        protected override SC.VistaIntegrationLog MakeLog(OrderAction entity)
        {
            return new SC.VistaIntegrationLog()
            {
                File = SC.VistaFiles.Order,
                Ien = string.Format("{0}_{1}", entity.OrderId, entity.DateTimeOrdered.ToString("MM/dd/yyyy HH:mm:ss"))
            };
        }

        private void FillOrdersData()
        {
            if (orderableItemTypeList == null || orderableItemTypeList.Count == 0)
            {
                CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                codeFilterParam.MaxSelectedCodes = int.MaxValue;
                codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItemType.ToString();
                orderableItemTypeList = EVS.GetCodes(codeFilterParam).ToList();
            }
            if (ADTOrderableItemList == null || ADTOrderableItemList.Count == 0)
            {
                string vistaCode = VistaSite.Code + "_";
                CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                codeFilterParamWithProperties.MaxSelectedCodes = int.MaxValue;
                codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
                codeFilterParamWithProperties.Properties = new List<string>();
                codeFilterParamWithProperties.Properties.Add(Constants.CONCEPT_PROPERTY_IEN);
                codeFilterParamWithProperties.Properties.Add(Constants.CONCEPT_PROPERTY_ORDERABLE_ITEM_TYPE);
                ADTOrderableItemList = EVS.GetCodesWithProperties(codeFilterParamWithProperties).Where(a => a.code.StartsWith(vistaCode)).ToList();
            }
            if (facilitiesByVista == null || facilitiesByVista.Count == 0)            
                facilitiesByVista = EIS.GetFacilities(VistaSite.Id).ToList();            
        }
    }
}
