﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.EVS;
using InfoWorld.HL7.ITS;
using BMS.Facade;
using System.ServiceModel;
using System.Reflection;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using FC = BMS.Facade.Data;
using BMS.VistaWorker2.Writer.Exceptions;
using BMS.VistaWorker2.Writer.Implementation.Concrete.Dal;

namespace BMS.VistaWorker2.Writer.Implementation.Concrete
{
    /// <summary>
    /// Abstract base writer.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public abstract class BaseEntityWriter<T> : IEntityWriter<T>
    {
        public FC.VistaSite VistaSite { get { return Data.Site; } }
        public BmsLogger Logger { get { return Data.Logger; } }
        public TimeZoneInfo VistATimeZone { get { return Data.TimeZone; } }
        protected CommonData Data { get; private set; }
        protected WF.Utils UtilsInstance { get; set; }

        public virtual void InitData(CommonData data, WF.Utils utilsInstance)
        {
            this.Data = data;
            this.UtilsInstance = utilsInstance;
        }

        public abstract void InitCache();

        public B InsertIfNullOrUpdateIfDirty<E, B>(E entity)
            where E : class,IEntity
            where B : class, new()
        {
            if (entity == null)
                return null;

            IEntityWriter<E> entityWriter = Data.WriterManager.MakeEntityWriter<E>(false);
            BaseEisWriter<E, B> eisEntityWriter = entityWriter as BaseEisWriter<E, B>;
            if (eisEntityWriter == null)
                throw new InvalidOperationException();

            using (eisEntityWriter)
            {
                B bmsEntity = eisEntityWriter.InsertOrUpdate(entity);
                return bmsEntity;
            }
        }

        protected static IEISWrapper EIS { get { return EISFactory.InstanceWindows; } }

        protected static IEVSWrapper EVS { get { return EVSFactory.InstanceWindows; } }

        protected abstract void OnInsert(T entity);

        protected abstract bool OnUpdate(T entity);

        protected abstract void OnDelete(T entity);

        private string GetOtherDetails(Exception e)
        {
            FaultException fault = e as FaultException;
            if (fault == null)
                return null;
            Type type = e.GetType();
            if (!type.IsGenericType)
                return null;
            if (type.GetGenericTypeDefinition() != typeof(FaultException<>))
                return null;
            object detail = WriterUtilities.GetDetail(fault);
            Type detailType = detail.GetType();

            if (detailType.Namespace.StartsWith("InfoWorld.EIS") ||
                detailType.Namespace.StartsWith("InfoWorld.EVS"))
            {
                return Utils.Utilities.ObjectToStringUsingPropertyReflection(detail);
            }
            return null;
        }

        private void OnException(T entity, Exception e)
        {            
            WriterException writerException = e as WriterException;
            if (writerException != null)
            {
                if (writerException.Message.ToLower().StartsWith("bed") && writerException.Message.ToLower().EndsWith("has no wards"))
                {
                    Logger.LogInformation(writerException.Message);
                    return;
                }
                LogException(writerException);
                return;
            }
            Tracer.TraceMessage(VistaSite.Name + " BaseEntityWriter OnException");
            Tracer.TraceException(e);
            string details = GetOtherDetails(e);
            string msj = string.Format("Writer has occurred an exception on entity {0}.", entity);
            if (!string.IsNullOrEmpty(details))
                msj += "\nDetails:" + details;
            writerException = new WriterException(entity, msj, e);
            LogException(writerException);
            throw writerException;
        }

        private void LogException(WriterException e)
        {
            if (e.InnerException == null)
                Logger.LogWarning(e.Message);
            else
                Logger.LogFormat(BmsLogger.Level.Warning, "{0}\n{1}", e.Message, e.InnerException);
        }

        public void Insert(T entity)
        {
            try
            {
                OnInsert(entity);
            }
            catch (Exception e)
            {
                OnException(entity, e);
                if (e.Message.ToLower().StartsWith("bed") && e.Message.ToLower().EndsWith("has no wards"))
                    return;
                else
                    throw;
            }
        }

        public bool Update(T entity)
        {
            try
            {
                return OnUpdate(entity);
            }
            catch (Exception e)
            {
                OnException(entity, e);
                if (e.Message.ToLower().StartsWith("bed") && e.Message.ToLower().EndsWith("has no wards"))
                    return false;
                else
                    throw;
            }
        }

        public void Delete(T entity)
        {
            try
            {
                OnDelete(entity);
            }
            catch (Exception e)
            {
                OnException(entity, e);
                throw;
            }
        }

        /// <summary>
        /// Converts datetimes properties on entity to UTC.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="entity">The entity.</param>
        protected void ConvertToUtc<E>(E entity) where E : class
        {
            TimeZoneUtil.ConvertObjectDates(entity, false, VistATimeZone);
        }

        public virtual void Dispose()
        {
        }

        public virtual IVistaIntegrationLogChecker LogChecker
        {
            get;
            set;
        }
    }
}
