﻿using System;
using System.Configuration;
using System.ServiceModel;
using BMS.FaultContracts;
using BMS.Numi;
using BMS.Schedulers.Managers;
using BMS.ServiceContracts;
using BMS.Utils;
using BMS.VistaIntegration;
using BMS.VistaIntegration.Cache;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Mdws;
using BMS.VistaIntegration.Via;
using BMS.VistaIntegration.VistA;

namespace BMS.ServiceImplementation
{
    public class VistaQueryCore : IVistaWorkerQuery
    {        
        public static Crawler Crawler
        {
            get;
            set;
        }                

        private VistAConnectionInfo ToVistAConnection(DataContracts.VistASite site)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new VistAConnectionInfo(
                    site.ConnectionString,
                    site.AccessCode,
                    site.VerifyCode);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private VistASite TranslateSite(DataContracts.VistASite site)
        {
            return new VistASite(site.Id, site.VistaName, site.Number, TimeZoneInfo.FindSystemTimeZoneById(site.TimeZoneId), ConfigurationManager.AppSettings[site.MdwsEndpointConfigKey]);
        }

        public void RunOnDemand(DataContracts.VistASite site, VistaDataType types, DateTime? startDate, DateTime? endDate, DataRetrievalMethod retrievalMethod)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CheckCrowler();
                VistAConnectionInfo connection = ToVistAConnection(site);
                var command = Crawler.MakeRunOnDemandCommand(retrievalMethod);
                var vistaSite = TranslateSite(site);
                command.Run(vistaSite, connection, types, startDate, endDate);
                if (!command.Succes)
                    throw new FaultException<VistAException>(
                        new VistAException()
                        {
                            ErrorMessage = command.ErrorMessage
                        });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void TestConnection(DataContracts.VistASite site, DataRetrievalMethod retrievalMethod)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CheckCrowler();

                var factory = GetSessionFactory(retrievalMethod);
                if (factory == null)
                {
                    throw new Exception("The given retrieval method not supported");
                }

                VistAConnectionInfo connection = ToVistAConnection(site);
                var command = Crawler.MakeTestConnectionCommand(factory);
                var vistaSite = TranslateSite(site);
                command.Run(vistaSite,connection);
                if (!command.Succes)
                    throw new FaultException<VistAException>(
                        new VistAException()
                        {
                            ErrorMessage = command.ErrorMessage
                        });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CheckCrowler()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (Crawler == null)
                    throw new InvalidOperationException("VistaIntegration not started");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SetRefreshSchedulersFlag()
        {
            SchedulerManager.Instance.SetRefreshSchedulersFlag();
        }

        public void SetRefreshVistACommandsFlag()
        {
            SchedulerManager.Instance.SetRefreshVistACommandsFlag();
        }

        public void SetRefreshNumiCommandsFlag()
        {
            SchedulerManager.Instance.SetRefreshNumiCommandsFlag();
        }

        public void SetRefreshWhiteboardReportCommandsFlag()
        {
            SchedulerManager.Instance.SetRefreshWhiteboardReportCommandsFlag();
        }

        public void RefreshSchedulers()
        {
            SchedulerManager.Instance.RefreshSchedulers();
            RefreshVistACommands();
            RefreshNumiCommands();
            RefreshWhiteboardReportCommands();
        }

        public void RefreshVistACommands()
        {
            if (Crawler == null) return;
            Crawler.RefreshCommands();
        }

        public void RefreshNumiCommands()
        {
            NumiManager.Instance.RefreshCommands();
        }

        public void RefreshWhiteboardReportCommands()
        {
            WhiteboardReport.WhiteboardReportManager.Instance.RefreshCommands();
        }

        private static IVistASessionFactory GetSessionFactory(DataRetrievalMethod retrievalMethod)
        {
            switch (retrievalMethod)
            {
                case DataRetrievalMethod.ODBC:
                    return new CacheVistASessionFactory();

                case DataRetrievalMethod.MDWS:
                    return new MdwsVistASessionFactory();

                case DataRetrievalMethod.VIA:
                    return new ViaVistASessionFactory();

                default:
                    return null;
            }
        }
    }
}
