﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using BMS.Utils;
using System.Configuration;
using BMS.ServiceImplementation;
using BMS.Workflows.WF;
using InfoWorld.Security.Authorization.AuthorizationSubscriber;
using System.ServiceModel.Activities;
using BMS.VistaIntegration.HL7.ServiceImplementation;
using BMS.VistaIntegration;
using System.Xml.Linq;
using BMS.VistaIntegration.Dal;
using BMS.VistaIntegration.Audit;
using BMS.VistaIntegration.Managers;
using BMS.VistaIntegration.Managers.Implementation;
using BMS.Schedulers;
using BMS.Schedulers.Managers;
using BMS.Numi;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.EVS;
using BMS.ServicesWrapper.BMService;

namespace BMS.Host
{
    public class Host
    {
        public delegate void StartExceptionHandler(Exception ex);
        Dictionary<Type, System.ServiceModel.ServiceHostBase> _bmsServices = new Dictionary<Type, System.ServiceModel.ServiceHostBase>();
        IList<ManualResetEvent> _stopHandles;
        StartExceptionHandler _handler;
        Crawler _crawler;
        const string All = "ALL";
        const string Bms = "BMS";
        const string Bms_Vi = "BMS.VI";

        public Host(StartExceptionHandler handler)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            /*Test*/
            try
            {
                if (handler == null)
                    throw new ArgumentNullException("handler", "Host requires a StartExceptionHandler to send starting exceptions to.");
                _handler = handler;
                try
                {
                    Uri path = new Uri(System.Reflection.Assembly.GetExecutingAssembly().CodeBase);
                    Environment.CurrentDirectory = System.IO.Path.GetDirectoryName(path.LocalPath);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception Host(): Exception trying to set uri or CurrentDirectory.");
                    Tracer.TraceException(ex);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        System.ServiceModel.ServiceHostBase CreateHost(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSThreadPoolState bmsState = (BMSThreadPoolState)state;
                System.ServiceModel.ServiceHostBase host;
                if (!bmsState.IsWFService)
                    host = new System.ServiceModel.ServiceHost(bmsState.ImplementedType);
                else
                    host = new WorkflowServiceHost(
                        Activator.CreateInstance(bmsState.ImplementedType));

                lock (_bmsServices)
                {
                    _bmsServices.Add(bmsState.ImplementedType, host);
                }
                return host;
                
            }
            catch (Exception ex)
            {
                Tracer.TraceMessage("Exception CreateHost(): Exception trying to create host.");
                Tracer.TraceException(ex);
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        System.ServiceModel.ServiceHostBase DisposeHost(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSThreadPoolState bmsState = (BMSThreadPoolState)state;
                System.ServiceModel.ServiceHostBase host = _bmsServices[bmsState.ImplementedType];
                if (host != null)
                {
                    host.Faulted -= new EventHandler(host_Faulted);
                    try { host.Abort(); }
                    finally
                    {
                        lock (_bmsServices)
                        {
                            _bmsServices.Remove(bmsState.ImplementedType);
                        }
                    }
                }
                return host;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        void host_Faulted(object sender, EventArgs e)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                System.ServiceModel.ServiceHostBase host = (System.ServiceModel.ServiceHostBase)sender;
                Type hostType = null;
                BMSThreadPoolState state = null;
                if (sender != null)
                {
                    Dictionary<Type, System.ServiceModel.ServiceHostBase>.Enumerator en = _bmsServices.GetEnumerator();
                    while (en.MoveNext())
                    {
                        if (en.Current.Value == host)
                        {
                            hostType = en.Current.Key;
                            state = new BMSThreadPoolState(hostType, host is WorkflowServiceHost);
                            break;
                        }
                    }
                    if (state != null)
                    {

                        Tracer.TraceMessage("ServiceHost [" + state.ImplementedType.Name + "] faulted. Restarting ...");
                        StopService(state);
                        StartService(state);
                        System.GC.Collect(GC.MaxGeneration, GCCollectionMode.Forced);
                    }
                }
            }
            catch (Exception ex) 
            {
                Tracer.TraceMessage("Exception host_Faulted(): Exception host faulted");
                Tracer.TraceException(ex);
            
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        void StartService(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSThreadPoolState bmsState = (BMSThreadPoolState)state;
                try
                {
                    System.ServiceModel.ServiceHostBase host = CreateHost(state);

                    host.Open();
                    host.Faulted += new EventHandler(host_Faulted);

                    Tracer.TraceServiceStart(host);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception StartService(): Exception try to start service.");
                    Tracer.TraceException(ex);
                    if (_handler != null)
                        _handler(ex);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        void StopService(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSThreadPoolState bmsState = (BMSThreadPoolState)state;
                try
                {
                    System.ServiceModel.ServiceHostBase disposedHost = DisposeHost(state);
                    if (disposedHost != null)
                        Tracer.TraceServiceStop(disposedHost);
                }
                catch (Exception ex)
                {

                    Tracer.TraceMessage("Exception StopService(): Exception try to stop service.");
                    Tracer.TraceException(ex);
                }
                finally
                {
                    if (bmsState.StopHandle != null)
                        bmsState.StopHandle.Set();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private void ThrowInvalidAppConfigException(string key)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new InvalidProgramException("Invalid app config for " + key);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        private bool IsKeyEnabled(string key)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string value = ConfigurationManager.AppSettings[key];
                bool isEnabled = true;
                if (bool.TryParse(value, out isEnabled) == false)
                {
                    ThrowInvalidAppConfigException(key);
                }
                return isEnabled;
            }
            catch (Exception ex) 
            {


                Tracer.TraceMessage("Exception IsKeyEnabled(): Exception try to set isEnabled boolean.");
                Tracer.TraceException(ex);
                return true;
            
            }

            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        private void StartVistaIntegration(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    if (!IsKeyEnabled(AppSettingsKeys.VistaIntegrationEnabled))
                        return;
                    BMS.VistaWorker2.Writer.IWriterManagerFactory writerManagerFactory = new VistaWorker2.Writer.Implementation.WriterManagerFactory();
                    BMS.VistaIntegration.Mdws.MdwsVistASessionFactory vistaSessionFactory = new VistaIntegration.Mdws.MdwsVistASessionFactory();
                    BMS.VistaIntegration.ISessionFactory sessionFactory = new SessionFactory();
                    IOperationManager operationManager = new OperationManager();
                    _crawler = new Crawler(
                        writerManagerFactory,
                        sessionFactory,
                        operationManager
                        );
                    Tracer.TraceMessage("Service created: VistA Integration");
                    _crawler.StartVistASyncronization();
                    Tracer.TraceMessage("Service started: VistA Integration");
                    VistaQueryCore.Crawler = _crawler;
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception StartVistaIntegration(): Exception try to start VistaIntegration.");
                    Tracer.TraceException(ex);
                    if (_crawler != null)
                        _crawler.EndVistASyncronization();
                    if (_handler != null)
                        _handler(ex);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void StopVistaIntegration(object state)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSThreadPoolState bmsState = (BMSThreadPoolState)state;
                try
                {
                    if (_crawler != null)
                    {
                        _crawler.EndVistASyncronization();
                        Tracer.TraceMessage("Service stopped: Vista integration");
                    }
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception StopVistaIntegration(): Exception try to stop VistaIntegration.");
                    Tracer.TraceException(ex);
                }
                finally
                {
                    bmsState.StopHandle.Set();
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #region Stop & Start

        public void OnStop()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string BMSServiceHostStartType = ConfigurationManager.AppSettings["BMSServiceHostStartType"];

                ManualResetEvent stopHandle = new ManualResetEvent(false);
                if (BMSServiceHostStartType.Equals(All, StringComparison.InvariantCultureIgnoreCase) || BMSServiceHostStartType.Equals(Bms_Vi, StringComparison.InvariantCultureIgnoreCase))
                {
                    SchedulerManager.Instance.Close();
                    NumiManager.Instance.Stop();
                    WhiteboardReport.WhiteboardReportManager.Instance.Stop();

                    //Vista Integration                    
                    ThreadPool.QueueUserWorkItem(StopVistaIntegration, new BMSThreadPoolState(stopHandle));
                    stopHandle.WaitOne();
                }

                _stopHandles = new List<ManualResetEvent>();

                //BMS Persistence
                 stopHandle = new ManualResetEvent(false);
                _stopHandles.Add(stopHandle);
                ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(BedManagerOperationsCore), stopHandle));

                stopHandle = new ManualResetEvent(false);
                _stopHandles.Add(stopHandle);
                ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(BedManagerQueryCore), stopHandle));

                if (BMSServiceHostStartType.Equals(All, StringComparison.InvariantCultureIgnoreCase) || BMSServiceHostStartType.Equals(Bms, StringComparison.InvariantCultureIgnoreCase))
                {
                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(ConfigurationOperationsCore), stopHandle));

                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(AuthorizationSubscriber), stopHandle));
                }

                if (BMSServiceHostStartType.Equals(All, StringComparison.InvariantCultureIgnoreCase) || BMSServiceHostStartType.Equals(Bms_Vi, StringComparison.InvariantCultureIgnoreCase))
                {
                    //Vista Integration - HL7Messaging
                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(HL7OperationsCore), stopHandle));

                    //BMS Workflows
                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(TransferFlow), stopHandle));

                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(WaitingListFlow), stopHandle));

                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(BedUnavailableFlow), stopHandle));

                    //BMS Cache
                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(BedManagerCacheCore), stopHandle));

                    //Vista Query - Last Updated Date
                    stopHandle = new ManualResetEvent(false);
                    _stopHandles.Add(stopHandle);
                    ThreadPool.QueueUserWorkItem(StopService, new BMSThreadPoolState(typeof(VistaQueryCore), stopHandle));
                }

                WaitHandle.WaitAll(_stopHandles.ToArray());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        private void StartNumi(object state)
        {
            try
            {
                Tracer.TraceMessage("Starting Numi Service...");
                NumiManager.Instance.Start();
            }
            catch (Exception ex)
            {
                Tracer.TraceMessage("Exception StartNumi(): Exception try to start Numi service.");
                Tracer.TraceException(ex);

                if (_handler != null)
                    _handler(ex);
            }
        }

        private void StartWhiteboardReport(object state)
        {
            try
            {
                Tracer.TraceMessage("Starting WhiteboardReport...");
                WhiteboardReport.WhiteboardReportManager.Instance.Start();
            }
            catch (Exception ex)
            {
                Tracer.TraceMessage("Exception StartWhiteboardReport(): Exception try to start StartWhiteboardReport.");
                Tracer.TraceException(ex);
                if (_handler != null)
                    _handler(ex);
            }
        }

        public void OnStart()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string BMSServiceHostStartType = ConfigurationManager.AppSettings["BMSServiceHostStartType"];

                //BMS Persistence

                ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(BedManagerOperationsCore)));                
                ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(BedManagerQueryCore)));

                if (BMSServiceHostStartType.Equals(All, StringComparison.InvariantCultureIgnoreCase) || BMSServiceHostStartType.Equals(Bms, StringComparison.InvariantCultureIgnoreCase))
                {
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(ConfigurationOperationsCore)));
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(AuthorizationSubscriber)));
                }

                if (BMSServiceHostStartType.Equals(All, StringComparison.InvariantCultureIgnoreCase) || BMSServiceHostStartType.Equals(Bms_Vi, StringComparison.InvariantCultureIgnoreCase))
                {
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(BedManagerCacheCore)));

                    //Vista Integration - HL7Messaging
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(HL7OperationsCore)));

                    //BMS Workflows
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(TransferFlow), true));
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(WaitingListFlow), true));
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(BedUnavailableFlow), true));

                    //Vista worker
                    ThreadPool.QueueUserWorkItem(StartService, new BMSThreadPoolState(typeof(VistaQueryCore)));

                    EVSFactory.InstanceWindows.FillCache();
                    EISFactory.InstanceWindows.FillCache();

                    //Updates audit jobs failed with status running to failed.
                    BMSFactory.BedManagerOperationsClientWindows.UpdateAuditJobsFailedWithStatusRunningToFailed();

                    SchedulerManager.Instance.Start();

                    ThreadPool.QueueUserWorkItem(StartNumi, null);
                    ThreadPool.QueueUserWorkItem(StartWhiteboardReport, null);
                    //Vista integration
                    ThreadPool.QueueUserWorkItem(StartVistaIntegration, null);
                    
                    try
                    {
                        int workerThreads, completionThreads, a;
                        if (int.TryParse(ConfigurationManager.AppSettings["ThreadPoolMaxWorkerThreads"], out workerThreads) && workerThreads > 0)
                        {
                            ThreadPool.GetMaxThreads(out a, out completionThreads);
                            ThreadPool.SetMaxThreads(workerThreads, completionThreads);
                            ThreadPool.GetMaxThreads(out a, out completionThreads);
                            Tracer.TraceMessage("ThreadPool max worker threads set to " + a.ToString());
                        }
                    }
                    catch (Exception ex) 
                    {
                        Tracer.TraceMessage("Exception OnStart(): Trying to set threadpool.");
                        Tracer.TraceException(ex);
                       
                    
                    }
                }
            }
            catch (Exception ex)
            {
                
                Tracer.TraceMessage("Exception OnStart(): Exception occurred while trying to start services."); 
                Tracer.TraceException(ex);
                throw;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}

