﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using System.Web.SessionState;
using InfoWorld.HL7.ITS;

namespace BMS.Facade.Service
{
    /// <summary>
    /// Interface used to obtain information about users.
    /// </summary>
    public interface IUserOperations
    {
        /// <summary>
        /// Gets the profile for the current logged on user.
        /// </summary>
        /// <returns>An user.</returns>
        User GetProfile();

        /// <summary>
        /// Gets the default user profile.
        /// </summary>
        /// <returns></returns>
        User GetDefaultUserProfile();

        /// <summary>
        /// Gets the user.
        /// </summary>
        /// <param name="userId">The user id.</param>
        /// <returns></returns>
        User GetUser(string userId);

        /// <summary>
        /// Check permission for the current logged user.
        /// </summary>
        /// <param name="permissionName"></param>
        /// <returns></returns>
        bool CheckPermission(string permissionName);

        /// <summary>
        /// Gets the user operations for facility.
        /// </summary>
        /// <param name="facilityId"></param>
        /// <returns></returns>
        IList<UserOperations> GetUserOperations(II facilityId);

        /// <summary>
        /// Adds the user operations.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        void AddUserOperations(string userName, List<PermissionFacility> facilities);

        /// <summary>
        /// Removes the user operations.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        void RemoveUserOperations(string userName, List<PermissionFacility> facilities);

        /// <summary>
        /// Gets the user domains.
        /// </summary>
        /// <returns></returns>
        IList<string> GetUserDomains();

        /// <summary>
        /// Fills the user data.
        /// </summary>
        /// <param name="user">The user.</param>
        void FillUserData(User user);

        void RevokeUserPermissions(User user);

        /// <summary>
        /// Saves the user data.
        /// </summary>
        /// <param name="user">The user.</param>
        void SaveUserData(User user, List<PermissionFacility> facilities);

        /// <summary>
        /// Saves the user data.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <param name="facility">The facility.</param>
        void SaveUserData(User user, PermissionFacility facility);

        /// <summary>
        /// Searches the users.
        /// </summary>
        /// <param name="domain">The domain.</param>
        /// <param name="searchUsername">The search username.</param>
        /// <returns></returns>
        IList<User> SearchUsers(string domain, string searchUsername);

        /// <summary>
        /// Checks the BMS read permission.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        bool CheckBMSReadPermission(User user, II facilityId);

        /// <summary>
        /// Checks the BMS write permission.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        bool CheckBMSWritePermission(User user, II facilityId);

        /// <summary>
        /// Checks the BMS write permission for whiteboards.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        bool CheckBMSWritePermissionWhiteboard(User user, II facilityId);

        /// <summary>
        /// Gets the facilities for which the user has read and write permissions.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        /// <returns></returns>
        List<PermissionFacility> GetFacilitiesForReadAndWrite(string userName, List<Facility> facilities);

        /// <summary>
        /// Gets the facilities for which the user has read permission, including the user groups.
        /// </summary>
        /// <param name="user">The User.</param>
        /// <param name="facilities">The facilities.</param>
        /// <returns></returns>
        List<Facility> GetFacilitiesWithReadAccess(User user, List<Facility> facilities);

        /// <summary>
        /// Gets the users by role and facility.
        /// </summary>
        /// <param name="roleName">Name of the role.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        List<User> GetUsersByRoleAndFacility(string roleName, II facilityId);

        /// <summary>
        /// Inserts the logged user.
        /// </summary>
        /// <param name="sessionId">The session id.</param>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facility">The facility.</param>
        /// <param name="visn">The visn.</param>
        void InsertLoggedUser(string sessionId, string userName, Facility facility, Visn visn);

        /// <summary>
        /// Gets the logged users count.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityIdList">The facility id list.</param>
        /// <returns></returns>
        Dictionary<string, int> GetLoggedUsersCount(string domainId, string facilityIdList);

        /// <summary>
        /// Deletes the ems staff.
        /// </summary>
        /// <param name="emsStaffId">The ems staff id.</param>
        void DeleteEmsStaff(II emsStaffId);

        /// <summary>
        /// Creates the ems staff.
        /// </summary>
        /// <param name="emsStaff">The ems staff.</param>
        void CreateEmsStaff(EmsStaff emsStaff);

        /// <summary>
        /// Updates the ems staff.
        /// </summary>
        /// <param name="emsStaff">The ems staff.</param>
        void UpdateEmsStaff(EmsStaff emsStaff);

        /// <summary>
        /// Gets the ems staff.
        /// </summary>
        /// <param name="emsStaffId">The ems staff id.</param>
        /// <returns></returns>
        EmsStaff GetEmsStaff(II emsStaffId);

        /// <summary>
        /// Filters the ems staff.
        /// </summary>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>
        List<EmsStaff> FilterEmsStaff(Facility facility);

        /// <summary>
        /// Checks the user has role.
        /// </summary>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        bool CheckUserHasRole(User user);
    }
}
