﻿using System;
using System.Collections.Generic;
using System.Data;
using BMS.Facade.Data;
using BMS.ServicesWrapper.WF;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.BMService;
using System.ServiceModel;
using BMS.Facade.Fault;
using System.Configuration;
using BMS.Utils;
using System.Linq;
using BMS.Facade;
using BMS.ServicesWrapper.EVS;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;
using System.Text;

namespace Import_Flows
{
    class Program
    {
        static string facilityDB = null;

        public static void Main(string[] args)
        {
            string facilityCode = null;
            string timezone = null;
            TimeZoneInfo facilityTimeZone = null;
            string dataType = null;
            DateTime? startDate = null;
            if (!(args.Length == 5 || args.Length == 4))
            {
                Console.WriteLine("Import Flow cannot run. Please verify the parameters!");
                System.Environment.Exit(-1);
            }

            if (args[0].StartsWith("/dataType="))
            {
                dataType = args[0].Split('=')[1];
                if (string.IsNullOrEmpty(dataType) || (dataType != "facility_waiting_list" && dataType != "visn_waiting_list" && dataType != "bed_unavailable" && dataType != "bed_clean"))
                {
                    Console.WriteLine("Incorrect Data Type!");
                    System.Environment.Exit(-1);
                }
            }
            else
            {
                Console.WriteLine("Incorrect Data Type!");
                System.Environment.Exit(-1);
            }

            if (args[1].StartsWith("/facilityCode="))
            {
                facilityCode = args[1].Split('=')[1];
                if (string.IsNullOrEmpty(facilityCode) || facilityCode.Length != 3)
                {
                    Console.WriteLine("Incorrect Facility Code!");
                    System.Environment.Exit(-1);
                }
            }
            else
            {
                Console.WriteLine("Incorrect Facility Code!");
                System.Environment.Exit(-1);
            }

            if (args[2].StartsWith("/facilityTimeZone="))
            {
                if (string.IsNullOrEmpty(args[2].Split('=')[1]))
                {
                    Console.WriteLine("Incorrect FacilityTimeZone!");
                    System.Environment.Exit(-1);
                }
                else
                {
                    switch (args[2].Split('=')[1])
                    {
                        case "EST":
                            timezone = "Eastern Standard Time";
                            break;
                        case "CST":
                            timezone = "Central Standard Time";
                            break;
                        case "MST":
                            timezone = "Mountain Standard Time";
                            break;
                        case "PST":
                            timezone = "Pacific Standard Time";
                            break;
                        case "HST":
                            timezone = "Hawaiian Standard Time";
                            break;
                        case "AST":
                            timezone = "Atlantic Standard Time";
                            break;
                        case "AKST":
                            timezone = "Alaskan Standard Time";
                            break;
                        default:
                            timezone = string.Empty;
                            break;
                    }
                    if (string.IsNullOrEmpty(timezone) || timezone.Length < 3)
                    {
                        Console.WriteLine("The parameter value provided for FacilityTimeZone is incorrect!");
                        System.Environment.Exit(-1);
                    }
                    else
                    {
                        facilityTimeZone = TimeZoneInfo.FindSystemTimeZoneById(timezone);
                    }
                }
            }
            else
            {
                Console.WriteLine("Incorrect FacilityTimeZone!");
                System.Environment.Exit(-1);
            }

            if (!string.IsNullOrEmpty(dataType) && (dataType == "facility_waiting_list" || dataType == "bed_unavailable" || dataType == "bed_clean"))
            {
                if (args[3].StartsWith("/facilityDB="))
                {
                    facilityDB = args[3].Split('=')[1];
                    if (string.IsNullOrEmpty(facilityDB))
                    {
                        Console.WriteLine("Incorrect facility database!");
                        System.Environment.Exit(-1);
                    }
                }
                else
                {
                    Console.WriteLine("Incorrect facility database!");
                    System.Environment.Exit(-1);
                }
            }

            if (!string.IsNullOrEmpty(dataType))
            {
                string date = string.Empty;
                if (dataType == "bed_unavailable")
                {
                    if (args.Length > 4 && args[4].StartsWith("/startDate="))
                        date = args[4].Split('=')[1];
                }
                else
                {
                    if (dataType == "visn_waiting_list" && args[3].StartsWith("/startDate="))
                        date = args[3].Split('=')[1];

                    else if ((dataType == "facility_waiting_list" || dataType == "bed_clean") && args[4].StartsWith("/startDate="))
                        date = args[4].Split('=')[1];
                    else
                    {
                        Console.WriteLine("Incorrect start date!");
                        System.Environment.Exit(-1);
                    }
                }
                try
                {
                    if (dataType == "bed_unavailable" && string.IsNullOrEmpty(date))
                        startDate = null;
                    else
                        startDate = Convert.ToDateTime(date);
                }
                catch
                {
                    Console.WriteLine("Incorrect start date!");
                    System.Environment.Exit(-1);
                }

            }

            switch (dataType)
            {
                case "facility_waiting_list":
                    ImportFacilityWaitingList(facilityCode, facilityTimeZone, startDate);
                    break;
                case "bed_unavailable":
                    ImportBedUnavailable(facilityCode, facilityTimeZone, startDate);
                    break;
                case "bed_clean":
                    ImportBedClean(facilityCode, facilityTimeZone, startDate);
                    break;
                case "visn_waiting_list":
                    ImportVisnWaitingList(facilityCode, facilityTimeZone, startDate);
                    break;
            }
        }

        #region Private Facility Waiting List Methods

        private static void ImportFacilityWaitingList(string facilityCode, TimeZoneInfo facilityTimeZone, DateTime? startDate)
        {
            bool _isSuccess = true;
            Facility facility = EISFactory.InstanceWindows.GetFacilityByCode(facilityCode);
            if (facility == null || facility.Id == null || string.IsNullOrEmpty(facility.Id.extension) || facility.Id.extension == Guid.Empty.ToString())
            {
                Console.WriteLine("ERROR: Facility not found!");
                return;
            }

            //get existing waiting list from BMS
            List<WaitingListItem> existingWaitingList = new List<WaitingListItem>();
            DataSet dsExistingWaitingList = DataQuery.GetImportData(@"/SqlScripts/GetWaitingListItemsFromBMS.sql", ConfigurationManager.ConnectionStrings["BMS_DB_ConnString"].ConnectionString, null, null, facility.Id.extension, null);

            //get existing waiting list from 3 BMS database.
            string connString = ConfigurationManager.ConnectionStrings["BMSFacilityConnString"].ConnectionString;
            connString = connString.Replace(connString.Split(';')[1], "Initial Catalog=" + facilityDB);
            DataSet ds = DataQuery.GetImportData(@"/SqlScripts/FacilityWaitingList.sql", connString, null, startDate, null, null);
            Dictionary<int, List<WaitingListItem>> waitingListItems = Translators.TranslateToWaitingList(ds, facility, facilityTimeZone);
            if (dsExistingWaitingList != null && dsExistingWaitingList.Tables != null && dsExistingWaitingList.Tables.Count > 0 && dsExistingWaitingList.Tables[0] != null && dsExistingWaitingList.Tables[0].Rows.Count > 0)
            {
                existingWaitingList = Translators.TranslateWaitingListFromBMS(dsExistingWaitingList, facility, facilityTimeZone);
                if (existingWaitingList != null && existingWaitingList.Count > 0)
                {        
                    //remove item from waiting list
                    List<WaitingListItem> removeWaitingListItems = (from itemFromBMS in existingWaitingList
                                                                    join wlRemove in waitingListItems[3]
                                                                    on itemFromBMS.PatientId.extension.ToUpper() equals wlRemove.Patient.Id.extension.ToUpper()
                                                                    select new WaitingListItem
                                                                        {
                                                                            Id = itemFromBMS.Id,
                                                                            VistaSite = facility.VistaSite,
                                                                            Facility = facility,
                                                                            Patient = wlRemove.Patient,
                                                                            CreationDate = itemFromBMS.CreationDate,
                                                                            RequestedDate = itemFromBMS.RequestedDate,
                                                                            Problem = wlRemove.Problem,
                                                                            TypeOfBedWard = wlRemove.TypeOfBedWard,
                                                                            LastEditBy = wlRemove.LastEditBy,
                                                                            WaitingArea = wlRemove.WaitingArea,
                                                                            FeeDisposition = wlRemove.FeeDisposition,
                                                                            Authorization = wlRemove.Authorization,
                                                                            Contract = wlRemove.Contract,
                                                                            ReasonFee = wlRemove.ReasonFee,
                                                                            CommentFee = wlRemove.CommentFee,
                                                                            RequestedBed = wlRemove.RequestedBed,
                                                                            RequestedBedDate = wlRemove.RequestedBedDate,
                                                                            Parent = itemFromBMS.Parent,
                                                                            FlowId = itemFromBMS.FlowId,
                                                                            IEN = itemFromBMS.IEN,
                                                                            RemovedDate = wlRemove.RemovedDate
                                                                        }).ToList();
                    if (removeWaitingListItems != null && removeWaitingListItems.Count > 0)
                    {
                        _isSuccess = RemoveWaitingList(removeWaitingListItems);

                        existingWaitingList = (from existingWL in existingWaitingList
                                               where !(from removeWL in removeWaitingListItems select removeWL.Patient.Id.extension.ToUpper()).Contains(existingWL.PatientId.extension.ToUpper())
                                               select existingWL).ToList();
                    }


                    if (existingWaitingList != null && existingWaitingList.Count > 0)
                    {
                        //update waiting list item
                        List<WaitingListItem> updateWaitingListItems = (from itemFromBMS in existingWaitingList
                                                                        join wlUpdate in waitingListItems[2]
                                                                        on itemFromBMS.PatientId.extension.ToUpper() equals wlUpdate.Patient.Id.extension.ToUpper()
                                                                        where (itemFromBMS.Problem != wlUpdate.Problem || itemFromBMS.TypeOfBedWard != wlUpdate.TypeOfBedWard || itemFromBMS.RequestedBedDate != wlUpdate.RequestedBedDate
                                                                             || (itemFromBMS.WaitingArea == null && wlUpdate.WaitingArea != null) || (itemFromBMS.WaitingArea != null && wlUpdate.WaitingArea == null)
                                                                             || (itemFromBMS.WaitingArea != null && wlUpdate.WaitingArea != null && itemFromBMS.WaitingArea.code != wlUpdate.WaitingArea.code)
                                                                             || (itemFromBMS.FeeDisposition == null && wlUpdate.FeeDisposition != null) || (itemFromBMS.FeeDisposition != null && wlUpdate.FeeDisposition == null)
                                                                             || (itemFromBMS.FeeDisposition != null && wlUpdate.FeeDisposition != null && itemFromBMS.FeeDisposition.code != wlUpdate.FeeDisposition.code)
                                                                             || (itemFromBMS.Authorization == null && wlUpdate.Authorization != null) || (itemFromBMS.Authorization != null && wlUpdate.Authorization == null)
                                                                             || (itemFromBMS.Authorization != null && wlUpdate.Authorization != null && itemFromBMS.Authorization.code != wlUpdate.Authorization.code)
                                                                             || (itemFromBMS.Contract == null && wlUpdate.Contract != null) || (itemFromBMS.Contract != null && wlUpdate.Contract == null)
                                                                             || (itemFromBMS.Contract != null && wlUpdate.Contract != null && itemFromBMS.Contract.code != wlUpdate.Contract.code)
                                                                             || (itemFromBMS.ReasonFee == null && wlUpdate.ReasonFee != null) || (itemFromBMS.ReasonFee != null && wlUpdate.ReasonFee == null)
                                                                             || (itemFromBMS.ReasonFee != null && wlUpdate.ReasonFee != null && itemFromBMS.ReasonFee.code != wlUpdate.ReasonFee.code)
                                                                             || itemFromBMS.CommentFee != wlUpdate.CommentFee
                                                                             || (itemFromBMS.RequestedBed == null && wlUpdate.RequestedBed != null) || (itemFromBMS.RequestedBed != null && wlUpdate.RequestedBed == null)
                                                                             || (itemFromBMS.RequestedBed != null && wlUpdate.RequestedBed != null && itemFromBMS.RequestedBed.Id.extension != wlUpdate.RequestedBed.Id.extension))
                                                                        select new WaitingListItem
                                                                        {
                                                                            Id = itemFromBMS.Id,
                                                                            VistaSite = facility.VistaSite,
                                                                            Facility = facility,
                                                                            Patient = wlUpdate.Patient,
                                                                            CreationDate = itemFromBMS.CreationDate,
                                                                            RequestedDate = itemFromBMS.RequestedDate,
                                                                            Problem = wlUpdate.Problem,
                                                                            TypeOfBedWard = wlUpdate.TypeOfBedWard,
                                                                            LastEditBy = wlUpdate.LastEditBy,
                                                                            WaitingArea = wlUpdate.WaitingArea,
                                                                            FeeDisposition = wlUpdate.FeeDisposition,
                                                                            Authorization = wlUpdate.Authorization,
                                                                            Contract = wlUpdate.Contract,
                                                                            ReasonFee = wlUpdate.ReasonFee,
                                                                            CommentFee = wlUpdate.CommentFee,
                                                                            RequestedBed = wlUpdate.RequestedBed,
                                                                            RequestedBedDate = wlUpdate.RequestedBedDate,
                                                                            Parent = itemFromBMS.Parent,
                                                                            FlowId = itemFromBMS.FlowId,
                                                                            IEN = itemFromBMS.IEN
                                                                        }).ToList();

                        if (updateWaitingListItems != null && updateWaitingListItems.Count > 0)
                        {
                            bool isSuccessfullyUpdated = UpdateWaitingList(updateWaitingListItems);
                            if (_isSuccess)
                                _isSuccess = isSuccessfullyUpdated;
                        }
                    }
                }
            }

            if (waitingListItems[1] != null && waitingListItems[1].Count > 0)
            {
                if (existingWaitingList != null && existingWaitingList.Count > 0)
                {
                    waitingListItems[1] = (from newWL in waitingListItems[1]
                                           where !(from existingWL in existingWaitingList select existingWL.PatientId.extension.ToUpper()).Contains(newWL.Patient.Id.extension.ToUpper())
                                           select newWL).ToList();
                }

                bool isSuccessfullyInserted = InsertWaitingList(waitingListItems[1]);
                if (_isSuccess)
                    _isSuccess = isSuccessfullyInserted;
            }

            if (_isSuccess)
                Console.WriteLine("SUCCESS: Facility Waiting List - Data imported!");
            else
                Console.WriteLine("PARTIAL SUCCESS: Facility Waiting List - Not all data imported!");
            //DataQuery.InsertVistaOperation(facility.VistaSite.Id.extension, "1024", DateTime.UtcNow);
        }

        private static bool InsertWaitingList(List<WaitingListItem> list)
        {
            bool isSuccess = true;
            foreach (WaitingListItem item in list)
            {
                try
                {
                    WFFactory.WaitingListFlowClientWindows.CreateWaitingListItem(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on insert entry in WL for: IEN: " + item.IEN + " SSN: " + item.Patient.SSN.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool UpdateWaitingList(List<WaitingListItem> list)
        {
            bool isSuccess = true;
            foreach (WaitingListItem item in list)
            {
                try
                {
                    WFFactory.WaitingListFlowClientWindows.ModifyWaitingListItem(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on update entry in WL for: IEN: " + item.IEN + " SSN: " + item.Patient.SSN.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool RemoveWaitingList(List<WaitingListItem> list)
        {
            bool isSuccess = true;
            foreach (WaitingListItem item in list)
            {
                try
                {
                    WFFactory.WaitingListFlowClientWindows.RemoveFromWaitingList(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on remove entry in WL for: IEN: " + item.IEN + " SSN: " + item.Patient.SSN.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        #endregion

        #region Private VISN Waiting List Methods

        private static void ImportVisnWaitingList(string facilityCode, TimeZoneInfo facilityTimeZone, DateTime? startDate)
        {
            bool _isSuccess = true;
            Facility facility = EISFactory.InstanceWindows.GetFacilityByCode(facilityCode);
            if (facility == null || facility.Id == null || string.IsNullOrEmpty(facility.Id.extension) || facility.Id.extension == Guid.Empty.ToString())
            {
                Console.WriteLine("ERROR: Facility not found!");
                return;
            }
            facility.VistaSite = EISFactory.InstanceWindows.GetVistaSite(facility.VistaSite.Id);
            facility.VistaSite.Visn = EISFactory.InstanceWindows.GetVisn(facility.VistaSite.Visn.Id);
            facility.VistaSite.Visn.Region = EISFactory.InstanceWindows.GetRegion(facility.VistaSite.Visn.Region.Id);
            if (facility.VistaSite.Visn == null || facility.VistaSite.Visn.Id == null || string.IsNullOrEmpty(facility.VistaSite.Visn.Id.extension) || facility.VistaSite.Visn.Id.extension == Guid.Empty.ToString())
            {
                Console.WriteLine("ERROR: Visn not found!");
                return;
            }

            //get existing transfers from BMS
            List<Transfer> existingTransfers = new List<Transfer>();
            DataSet dsExistingTransfers = DataQuery.GetImportData(@"/SqlScripts/GetVISNWaitingListItemsFromBMS.sql", ConfigurationManager.ConnectionStrings["BMS_DB_ConnString"].ConnectionString, null, null, facility.Id.extension, null);
            //get existing transfers from 3 BMS database.
            DataSet ds = DataQuery.GetImportData(@"/SqlScripts/VisnWaitingList.sql", ConfigurationManager.ConnectionStrings["BMSVisnConnString"].ConnectionString, facilityCode, startDate, null, null);
            Dictionary<int, List<Transfer>> transfersList = Translators.TranslateToTransfer(ds, facility, facilityTimeZone);

            if (dsExistingTransfers != null && dsExistingTransfers.Tables != null && dsExistingTransfers.Tables.Count > 0 && dsExistingTransfers.Tables[0] != null && dsExistingTransfers.Tables[0].Rows.Count > 0)
            {
                existingTransfers = Translators.TranslateTransferFromBMS(dsExistingTransfers, facility, facilityTimeZone);
                if (existingTransfers != null && existingTransfers.Count > 0)
                {
                    List<Transfer> finalizeTransfers = (from item in transfersList[3]
                                                        join transfer in existingTransfers
                                                         on item.Patient.Id.extension.ToUpper() equals transfer.PatientId.extension.ToUpper()
                                                        select new Transfer
                                                        {
                                                            Id = transfer.Id,
                                                            CanceledDate = transfer.CanceledDate,
                                                            CreationDate = transfer.CreationDate,
                                                            FlowId = transfer.FlowId,
                                                            Parent = new Act() { Id = transfer.Id },
                                                            IEN = transfer.IEN,
                                                            Patient = item.Patient,
                                                            Comment = item.Comment,
                                                            Era = item.Era,
                                                            FacilityFrom = item.FacilityFrom,
                                                            Contract = item.Contract,
                                                            Diagnosis = item.Diagnosis,
                                                            CurrentLocation = item.CurrentLocation,
                                                            Speciality = item.Speciality,
                                                            AdmissionDate = item.AdmissionDate,
                                                            RequestedDate = item.RequestedDate,
                                                            IsNationalWaitList = false,
                                                            DischargeComment = item.DischargeComment,
                                                            Disposition = item.Disposition,
                                                            DispositionDate = item.DispositionDate,
                                                            FacilityTo = item.FacilityTo
                                                        }).ToList();
                    if (finalizeTransfers != null && finalizeTransfers.Count > 0)
                    {
                        _isSuccess = FinalizeTransfers(finalizeTransfers);
                         existingTransfers = (from transfer in existingTransfers
                                              where !(from fTransfer in finalizeTransfers select fTransfer.Patient.Id.extension.ToUpper()).Contains(transfer.PatientId.extension.ToUpper())
                                              select transfer).ToList();
                    }

                    if (existingTransfers != null && existingTransfers.Count > 0)
                    {

                        List<Transfer> updateTransfers = (from item in transfersList[2]
                                                          join transfer in existingTransfers
                                                             on item.Patient.Id.extension.ToUpper() equals transfer.PatientId.extension.ToUpper()
                                                          where (item.Diagnosis != transfer.Diagnosis || item.CurrentLocation != transfer.CurrentLocation || item.AdmissionDate != transfer.AdmissionDate
                                                                          || item.Comment != transfer.Comment || item.Era.code != transfer.Era.code
                                                                          || (item.Speciality == null && transfer.Speciality != null) || (item.Speciality != null && transfer.Speciality == null)
                                                                          || (item.Speciality != null && transfer.Speciality != null && item.Speciality.code != transfer.Speciality.code)
                                                                          || (item.Contract == null && transfer.Contract != null) || (item.Contract != null && transfer.Contract == null)
                                                                          || (item.Contract != null && transfer.Contract != null && item.Contract.code != transfer.Contract.code))
                                                          select new Transfer
                                                          {
                                                              Id = transfer.Id,
                                                              CanceledDate = transfer.CanceledDate,
                                                              CreationDate = transfer.CreationDate,
                                                              FlowId = transfer.FlowId,
                                                              Parent = transfer.Parent,
                                                              IEN = transfer.IEN,
                                                              Patient = item.Patient,
                                                              Comment = item.Comment,
                                                              Era = item.Era,
                                                              FacilityFrom = item.FacilityFrom,
                                                              Contract = item.Contract,
                                                              Diagnosis = item.Diagnosis,
                                                              CurrentLocation = item.CurrentLocation,
                                                              Speciality = item.Speciality,
                                                              AdmissionDate = item.AdmissionDate,
                                                              RequestedDate = item.RequestedDate,
                                                              IsNationalWaitList = false
                                                          }).ToList();

                        if (updateTransfers != null && updateTransfers.Count > 0)
                        {
                            bool isSuccessfullyUpdated = UpdateTransfers(updateTransfers);
                            if (_isSuccess)
                                _isSuccess = isSuccessfullyUpdated;
                        }
                    }
                }
            }

            if (transfersList[1] != null && transfersList[1].Count > 0)
            {
                if (existingTransfers != null && existingTransfers.Count > 0)
                {
                    transfersList[1] = (from newTra in transfersList[1]
                                        where !(from existingTra in existingTransfers select existingTra.PatientId.extension.ToUpper()).Contains(newTra.Patient.Id.extension.ToUpper())
                                        select newTra).ToList();
                }

                bool isSuccessfullyInserted = InsertTransfers(transfersList[1]);
                if (_isSuccess)
                    _isSuccess = isSuccessfullyInserted;
            }

            if (_isSuccess)
                Console.WriteLine("SUCCESS: Visn Waiting List - Data imported!");
            else
                Console.WriteLine("PARTIAL SUCCESS: Visn Waiting List - Not all data imported!");
        }

        private static bool InsertTransfers(List<Transfer> list)
        {
            bool isSuccess = true;
            foreach (Transfer item in list)
            {
                try
                {
                    WFFactory.TransferWorkflowClientWindows.CreateTransferRequest(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on insert entry in VISN Waiting List: " + item.Patient.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool UpdateTransfers(List<Transfer> list)
        {
            bool isSuccess = true;
            foreach (Transfer item in list)
            {
                try
                {
                    WFFactory.TransferWorkflowClientWindows.UpdateTransferRequest(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on update entry in VISN Waiting List: " + item.Patient.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool FinalizeTransfers(List<Transfer> list)
        {
            bool isSuccess = true;
            List<II> updateIds = new List<II>();
            II eventId = null;
            CD disposition = null;
            foreach (Transfer item in list)
            {
                try
                {
                    if (item.Disposition != null && item.Disposition.code != null && item.Disposition.code == BMS.Utils.Constants.VA_ADMISSION)
                    {
                        if (disposition == null)
                            disposition = item.Disposition;
                        item.Disposition.code = "Expired";
                        item.Disposition.displayName = "EXPIRED";
                        eventId = WFFactory.TransferWorkflowClientWindows.CreateTransferEvent(item);
                        updateIds.Add(eventId);
                    }
                    else
                        WFFactory.TransferWorkflowClientWindows.CreateTransferEvent(item);
                    
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on finalized entry in VISN Waiting List: " + item.Patient.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            if (updateIds != null && updateIds.Count > 0 && disposition != null)
            {
                try
                {
                    StringBuilder sb = new StringBuilder();
                    foreach (II id in updateIds)
                    {
                        sb.Append(id.extension);
                        sb.Append(",");
                    }
                    sb = sb.Remove(sb.Length - 1, 1);
                    DataQuery.UpdateDispositionForTransferEvents(sb.ToString(), disposition);
                }
                catch (Exception ex)
                {
                    isSuccess = false;
                    Tracer.TraceMessage("Exception on update disposition for finalized visn waiting list items: ");
                    Tracer.TraceException(ex);
                }
            }
            return isSuccess;
        }

        #endregion

        #region Private Bed Clean Methods

        private static void ImportBedClean(string facilityCode, TimeZoneInfo facilityTimeZone, DateTime? startDate)
        {
            bool _isSuccess = true;
            Facility facility = EISFactory.InstanceWindows.GetFacilityByCode(facilityCode);
            if (facility == null || facility.Id == null || string.IsNullOrEmpty(facility.Id.extension) || facility.Id.extension == Guid.Empty.ToString())
            {
                Console.WriteLine("ERROR: Facility not found!");
                return;
            }
            facility.VistaSite = EISFactory.InstanceWindows.GetVistaSite(facility.VistaSite.Id);
            List<Ward> wards = EISFactory.InstanceWindows.GetWardsByVistaSiteId(facility.VistaSite.Id).ToList();
            string wardUIDList = string.Empty;
            foreach (Ward ward in wards)
            {
                if (string.IsNullOrEmpty(wardUIDList))
                {
                    wardUIDList += ward.Id.extension;
                }
                else
                {
                    wardUIDList += "'";
                    wardUIDList += "," + "'" + ward.Id.extension;
                }
            }

            //get existing bed clean from 3 BMS database.
            string connString = ConfigurationManager.ConnectionStrings["BMSFacilityConnString"].ConnectionString;
            connString = connString.Replace(connString.Split(';')[1], "Initial Catalog=" + facilityDB);
            DataSet ds = DataQuery.GetImportData(@"/SqlScripts/BedClean.sql", connString, null, startDate, null, null);
            Dictionary<int, List<BedClean>> bcList = Translators.TranslateToBedClean(ds, facility, wards, facilityTimeZone);

            if(bcList[1] != null)
                Tracer.TraceMessage("Insert bed clean list from Local BMS DB: " + bcList[1].Count);

            if (bcList[2] != null)
                Tracer.TraceMessage("Update bed clean list from Local BMS DB: " + bcList[2].Count);
            
            List<BedClean> bedCleanList = new List<BedClean>();
            //get existing bed clean from BMS database.
            DataSet existingBedCleanRequest = DataQuery.GetImportData(@"/SqlScripts/GetBedCleanFromBMS.sql", ConfigurationManager.ConnectionStrings["BMS_DB_ConnString"].ConnectionString, null, null, null, wardUIDList);
            if (existingBedCleanRequest != null && existingBedCleanRequest.Tables != null && existingBedCleanRequest.Tables.Count > 0 && existingBedCleanRequest.Tables[0] != null && existingBedCleanRequest.Tables[0].Rows.Count > 0)
            {
                bedCleanList = Translators.TranslateBedClean(existingBedCleanRequest, facilityTimeZone);
                if (bedCleanList != null && bedCleanList.Count > 0)
                {
                    Tracer.TraceMessage("Update bed clean list from BMS DB: " + bedCleanList.Count);

                    List<BedClean> updateBedCleanList = new List<BedClean>();
                    var updateBedCleanManualRequestNoList = (from bedClean in bcList[2].Where(a => a.EventIen != null)
                                                             join bmsBedClean in bedCleanList.Where(a => a.EventIen != null)
                                                                   on bedClean.EventIen  equals bmsBedClean.EventIen 
                                                             where (bedClean.Ward.Id.extension.ToUpper() == bmsBedClean.Ward.Id.extension.ToUpper() && bedClean.Bed.Id.extension.ToUpper() == bmsBedClean.Bed.Id.extension.ToUpper())
                                                             select new BedClean
                                                             {
                                                                 Id = bmsBedClean.Id,
                                                                 RequestedDate = bmsBedClean.RequestedDate,
                                                                 VacatedDate = bedClean.VacatedDate,
                                                                 AcceptedBy = bedClean.AcceptedBy,
                                                                 AcceptedDate = bedClean.AcceptedDate,
                                                                 EMSNotify = bedClean.EMSNotify,
                                                                 LastEditDate = bedClean.LastEditDate,
                                                                 LastEditedBy = bedClean.LastEditedBy,
                                                                 ManualRequest = bedClean.ManualRequest,
                                                                 TypeOfClean = bedClean.TypeOfClean,
                                                                 VistaSite = facility.VistaSite,
                                                                 EventIen = bmsBedClean.EventIen,
                                                                 CompletedBy = bedClean.CompletedBy,
                                                                 CompletedDate = bedClean.CompletedDate,
                                                                 Bed = bedClean.Bed,
                                                                 EventType = bedClean.EventType,
                                                                 Ward = bedClean.Ward
                                                             }).ToList();

                    var updateBedCleanManualRequestYesList = (from bedClean in bcList[2].Where(a => a.EventIen == null)
                                                              join bmsBedClean in bedCleanList.Where(a => a.EventIen == null)
                                                                    on bedClean.Bed.Id.extension.ToUpper() equals bmsBedClean.Bed.Id.extension.ToUpper()
                                                              where bedClean.Ward.Id.extension.ToUpper() == bmsBedClean.Ward.Id.extension.ToUpper() && bedClean.RequestedDate == bmsBedClean.RequestedDate
                                                              select new BedClean
                                                              {
                                                                  Id = bmsBedClean.Id,
                                                                  RequestedDate = bmsBedClean.RequestedDate,
                                                                  VacatedDate = bedClean.VacatedDate,
                                                                  AcceptedBy = bedClean.AcceptedBy,
                                                                  AcceptedDate = bedClean.AcceptedDate,
                                                                  EMSNotify = bedClean.EMSNotify,
                                                                  LastEditDate = bedClean.LastEditDate,
                                                                  LastEditedBy = bedClean.LastEditedBy,
                                                                  ManualRequest = bedClean.ManualRequest,
                                                                  TypeOfClean = bedClean.TypeOfClean,
                                                                  VistaSite = facility.VistaSite,
                                                                  EventIen = bmsBedClean.EventIen,
                                                                  CompletedBy = bedClean.CompletedBy,
                                                                  CompletedDate = bedClean.CompletedDate,
                                                                  Bed = bedClean.Bed,
                                                                  EventType = bedClean.EventType,
                                                                  Ward = bedClean.Ward
                                                              }).ToList();

                    updateBedCleanList.AddRange(updateBedCleanManualRequestNoList);
                    updateBedCleanList.AddRange(updateBedCleanManualRequestYesList);

                    if (updateBedCleanList != null && updateBedCleanList.Count > 0)
                        _isSuccess = UpdateBedClean(updateBedCleanList);
                }
            }


            //insert bed clean
            if (bcList[1] != null && bcList[1].Count > 0)
            {
                if (bcList[1] != null && bcList[1].Count > 0)
                {
                    bool isSuccessfullyInserted = InsertBedClean(bcList[1].OrderBy(a => a.RequestedDate).ToList());
                    if (_isSuccess)
                        _isSuccess = isSuccessfullyInserted;
                }
            }

            DataQuery.UpdateBedCleanByBMSSystem(wardUIDList);

            if (_isSuccess)
                Console.WriteLine("SUCCESS: Bed Clean - Data imported!");
            else
                Console.WriteLine("PARTIAL SUCCESS: Bed Clean- Not all data imported!");
        }

        private static bool InsertBedClean(List<BedClean> bcList)
        {
            bool isSuccess = true;
            foreach (BedClean item in bcList)
            {
                try
                {
                    DataQuery.InsertBedClean(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on insert bed clean: " + item.Bed.Id.extension + " ward: " + item.Ward.Id.extension);
                    isSuccess = false;
                    Console.WriteLine(ex.Message);
                }
            }
            return isSuccess;
        }

        private static bool UpdateBedClean(List<BedClean> bcList)
        {
            bool isSuccess = true;
            foreach (BedClean item in bcList)
            {
                try
                {
                    DataQuery.UpdateBedClean(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on updated bed clean: " + item.Bed.Id.extension + " ward: " + item.Ward.Id.extension);
                    isSuccess = false;
                    Console.WriteLine(ex.Message);
                }
            }
            return isSuccess;
        }

        #endregion

        #region Private Bed UnavailableMethods

        private static void ImportBedUnavailable(string facilityCode, TimeZoneInfo facilityTimeZone, DateTime? startDate)
        {
            bool _isSuccess = true;
            Facility facility = EISFactory.InstanceWindows.GetFacilityByCode(facilityCode);
            if (facility == null || facility.Id == null || string.IsNullOrEmpty(facility.Id.extension) || facility.Id.extension == Guid.Empty.ToString())
            {
                Console.WriteLine("ERROR: Facility not found!");
                return;
            }

            facility.VistaSite = EISFactory.InstanceWindows.GetVistaSite(facility.VistaSite.Id);
            CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
            codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.UnavailableReason.ToString();
            codeFilterParamWithProperties.Properties = new List<string>();
            codeFilterParamWithProperties.Properties.Add("Type");
            List<CDWithProperties> reasonList = EVSFactory.InstanceWindows.GetCodesWithProperties(codeFilterParamWithProperties).Where(a => a.code.StartsWith(facility.VistaSite.Code)).ToList();
            List<CD> typeList = EVSFactory.InstanceWindows.GetCodes(new CodeFilterParameters() { VocabularyDomain = Util.Vocabulary.AdminURType.ToString() }).ToList();
            List<Division> divisions = EISFactory.InstanceWindows.GetDivisions(facility.Id).ToList();
            List<Bed> beds = new List<Bed>();
            foreach (Division div in divisions)
                beds.AddRange(EISFactory.InstanceWindows.GetBedsInDivision(div.Id, facility.VistaSite.Id));
            List<Ward> wards = EISFactory.InstanceWindows.GetWardsByVistaSiteId(facility.VistaSite.Id).ToList();
            Tracer.TraceMessage("loaded all EIS and EVS collections");
            string wardUIDList = string.Empty;
            foreach (Ward ward in wards)
            {
                if (string.IsNullOrEmpty(wardUIDList))
                {
                    wardUIDList += ward.Id.extension;
                }
                else
                {
                    wardUIDList += "'";
                    wardUIDList += "," + "'" + ward.Id.extension;
                }
            }

            List<BedUnavailable> bedUnavailableListFromBMS = new List<BedUnavailable>();
            //get all bed unavailable that already exists in bms database
            DataSet bedUnavailableFromBMSDataSet = DataQuery.GetImportData(@"/SqlScripts/GetBedUnavailableFromBMS.sql", ConfigurationManager.ConnectionStrings["BMS_DB_ConnString"].ConnectionString, null, null, null, wardUIDList);
            if (bedUnavailableFromBMSDataSet != null && bedUnavailableFromBMSDataSet.Tables != null && bedUnavailableFromBMSDataSet.Tables.Count > 0 && bedUnavailableFromBMSDataSet.Tables[0] != null && bedUnavailableFromBMSDataSet.Tables[0].Rows.Count > 0)
            {
                if (!startDate.HasValue)
                {
                    Console.WriteLine("Insert start date!");
                    System.Environment.Exit(-1);
                }
                else
                {
                    //translate bed unavailable
                    bedUnavailableListFromBMS = Translators.TranslateBedUnavailable(bedUnavailableFromBMSDataSet, beds, facilityTimeZone);                
                }
            }

            string connString = ConfigurationManager.ConnectionStrings["BMSFacilityConnString"].ConnectionString;
            connString = connString.Replace(connString.Split(';')[1], "Initial Catalog=" + facilityDB);
            DataSet ds = DataQuery.GetImportData(@"/SqlScripts/BedUnavailable.sql", connString, null, null, null, null);
            Tracer.TraceMessage("begin import bed unavailable");
            List<BedUnavailable> buList = Translators.TranslateToBedUnavailable(ds, facility.VistaSite, beds, wards, facilityTimeZone, reasonList, typeList);
            if (bedUnavailableListFromBMS != null && bedUnavailableListFromBMS.Count > 0)
            {
                //if buLastInformationList contains items, then OutOfServiceVistA = 1 means that the flow must close.
                List<BedUnavailable> buLastInformationList = new List<BedUnavailable>();
                DataSet bedUnavailableLastInformationDataSet = DataQuery.GetImportData(@"/SqlScripts/GetBedUnavailableLastInformation.sql", connString, null, startDate, null, null);
                if (bedUnavailableLastInformationDataSet != null && bedUnavailableLastInformationDataSet.Tables != null && bedUnavailableLastInformationDataSet.Tables.Count > 0)
                {
                    buLastInformationList = Translators.TranslateBedUnavailableLastInformation(bedUnavailableLastInformationDataSet, beds, reasonList, typeList, facilityTimeZone);
                    if (buLastInformationList != null && buLastInformationList.Count > 0)
                    {
                        //close bed unavailable
                        List<BedUnavailable> closeBedUnavailableList = (from buBms in bedUnavailableListFromBMS
                                                                        join b in buLastInformationList.Where(a => a.OutOfServiceVistA == true)
                                                                                 on buBms.Bed.Id.extension.ToUpper() equals b.Bed.Id.extension.ToUpper()
                                                                        where b.CanceledBy != null && b.CanceledDate != null
                                                                        select new BedUnavailable
                                                                        {
                                                                            CanceledDate = b.CanceledDate,
                                                                            CreationDate = buBms.CreationDate,
                                                                            Id = buBms.Id,
                                                                            Parent = buBms.Parent,
                                                                            FlowId = buBms.FlowId,
                                                                            IEN = buBms.IEN,
                                                                            Type = b.Type,
                                                                            Reason = b.Reason,
                                                                            Division = null,
                                                                            Bed = buBms.Bed,
                                                                            CreatedBy = buBms.CreatedBy,
                                                                            EditedBy = b.EditedBy,
                                                                            EditedDate = b.EditedDate,
                                                                            CanceledBy = b.CanceledBy,
                                                                            VistaSite = facility.VistaSite,
                                                                            OutOfServiceVistA = buBms.OutOfServiceVistA,
                                                                            WardList = buBms.WardList,
                                                                            ExpectedCompletedDate = buBms.ExpectedCompletedDate
                                                                        }).ToList();
                        if (closeBedUnavailableList != null && closeBedUnavailableList.Count > 0)
                        {
                            _isSuccess = CloseBedUnavailable(closeBedUnavailableList);
                            bedUnavailableListFromBMS = (from bed in bedUnavailableListFromBMS
                                                         where !(from b in closeBedUnavailableList select b.Bed.Id.extension.ToUpper()).Contains(bed.Bed.Id.extension.ToUpper())
                                                         select bed).ToList();
                        }



                        if (bedUnavailableListFromBMS != null && bedUnavailableListFromBMS.Count > 0)
                        {

                            //update bed unavailable
                            List<BedUnavailable> updateBedUnavailable = (from buBms in bedUnavailableListFromBMS
                                                                         join b in buLastInformationList.Where(a => a.OutOfServiceVistA == false)
                                                                                  on buBms.Bed.Id.extension.ToUpper() equals b.Bed.Id.extension.ToUpper()
                                                                         where ((buBms.Reason == null && b.Reason != null) || (buBms.Reason != null && b.Reason == null) || (buBms.Reason != null && b.Reason != null && buBms.Reason.code != b.Reason.code)
                                                                                || (buBms.Type == null && b.Type != null) || (buBms.Type != null && b.Type == null) || (buBms.Type != null && b.Type != null && buBms.Type.code != b.Type.code))
                                                                                && b.CanceledBy == null && b.CanceledDate == null
                                                                         select new BedUnavailable
                                                                         {
                                                                             CanceledDate = null,
                                                                             CreationDate = buBms.CreationDate,
                                                                             Id = buBms.Id,
                                                                             Parent = buBms.Parent,
                                                                             FlowId = buBms.FlowId,
                                                                             IEN = buBms.IEN,
                                                                             Type = b.Type,
                                                                             Reason = b.Reason,
                                                                             Division = null,
                                                                             Bed = buBms.Bed,
                                                                             CreatedBy = buBms.CreatedBy,
                                                                             EditedBy = b.EditedBy,
                                                                             EditedDate = b.EditedDate,
                                                                             CanceledBy = null,
                                                                             VistaSite = facility.VistaSite,
                                                                             OutOfServiceVistA = buBms.OutOfServiceVistA,
                                                                             WardList = buBms.WardList,
                                                                             ExpectedCompletedDate = buBms.ExpectedCompletedDate
                                                                         }).ToList();

                            if (updateBedUnavailable != null && updateBedUnavailable.Count > 0)
                            {
                                bool isSuccessfullyUpdated = UpdateBedUnavailable(updateBedUnavailable);
                                if (_isSuccess)
                                    _isSuccess = isSuccessfullyUpdated;
                            }

                            buList = (from bed in buList
                                      where !(from b in bedUnavailableListFromBMS select b.Bed.Id.extension.ToUpper()).Contains(bed.Bed.Id.extension.ToUpper())
                                      select bed).ToList();
                        }
                    }
                }
            }

            //insert bed unavailable
            if (buList != null && buList.Count > 0)
            {
                bool isSuccessfullyInserted = InsertBedUnavailable(buList);
                if (_isSuccess)
                    _isSuccess = isSuccessfullyInserted;
            }

            if (_isSuccess)
            {
                Tracer.TraceMessage("SUCCESS: Bed Unavailable - Data imported!");
                Console.WriteLine("SUCCESS: Bed Unavailable - Data imported!");
            }
            else
            {
                Tracer.TraceMessage("PARTIAL SUCCESS: Bed Unavailable - Not all data imported!");
                Console.WriteLine("PARTIAL SUCCESS: Bed Unavailable - Not all data imported!");
            }
        }

        private static bool InsertBedUnavailable(List<BedUnavailable> buList)
        {
            bool isSuccess = true;
            foreach (BedUnavailable item in buList)
            {
                try
                {
                    WFFactory.BedUnavailableWorkflowClientWindows.CreateBedUnavailable(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on insert entry in Bed Unavailbale: " + item.Bed.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool UpdateBedUnavailable(List<BedUnavailable> buList)
        {
            bool isSuccess = true;
            foreach (BedUnavailable item in buList)
            {
                try
                {
                    WFFactory.BedUnavailableWorkflowClientWindows.UpdateBedUnavailable(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on update entry in Bed Unavailbale: " + item.Bed.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        private static bool CloseBedUnavailable(List<BedUnavailable> buList)
        {
            bool isSuccess = true;
            foreach (BedUnavailable item in buList)
            {
                try
                {
                    WFFactory.BedUnavailableWorkflowClientWindows.CancelBedUnavailable(item);
                }
                catch (Exception ex)
                {
                    Tracer.TraceMessage("Exception on cancel entry in Bed Unavailbale: " + item.Bed.Id.extension);
                    isSuccess = false;
                    OnWorkflowException(ex);
                }
            }
            return isSuccess;
        }

        #endregion

        #region Private Write Exception Methods

        private static void OnWorkflowException(Exception ex)
        {
            FaultException<GenericWFServiceFault> fault = ex as FaultException<GenericWFServiceFault>;
            if (fault == null || fault.Detail == null)
            {
                Console.WriteLine(ex.Message);
                Tracer.TraceMessage(ex.Message);
                Console.WriteLine(ex.StackTrace);
                Tracer.TraceMessage(ex.StackTrace);
            }
            else
            {
                WFInvalidOperation wfEx = fault.Detail.ServiceFault as WFInvalidOperation;
                if (wfEx != null)
                {
                    Console.WriteLine(wfEx.FriendlyMessage);
                    Tracer.TraceMessage(wfEx.FriendlyMessage);
                    Console.WriteLine(wfEx.OperationName);
                    Tracer.TraceMessage(wfEx.OperationName);
                    Console.WriteLine(wfEx.ErrorMessage);
                    Tracer.TraceMessage(wfEx.ErrorMessage);
                    Console.WriteLine(wfEx.StackTrace);
                    Tracer.TraceMessage(wfEx.StackTrace);
                }
                else
                {
                    WfOtherException wfEx2 = fault.Detail.ServiceFault as WfOtherException;
                    if (wfEx2 != null)
                    {
                        Console.WriteLine(wfEx2.ErrorMessage);
                        Tracer.TraceMessage(wfEx2.ErrorMessage);
                        Console.WriteLine(wfEx2.StackTrace);
                        Tracer.TraceMessage(wfEx2.StackTrace);
                    }
                }
            }
        }

        #endregion
    }
}
