﻿using System;
using System.Net;
using System.Windows;
using System.Configuration;
using WinForms = System.Windows.Forms;
using System.IO;
using System.Reflection;
using System.Collections.Generic;
using System.Windows.Input;
using System.Collections.ObjectModel;
using System.Data.SqlClient;
using System.Linq;
using BMSConfigurationWarlock.Base;
using System.Xml.Serialization;
using System.Xml.Linq;
using System.Xml.XPath;
using System.Text.RegularExpressions;
using System.Xml;
using System.ServiceProcess;
using BMSConfigurationWarlock.ServiceLogic;
using System.Threading;
using System.Security.Principal;
using Microsoft.Web.Administration;
using System.ComponentModel;
using System.Diagnostics;
using BMSConfigurationWarlock.ReportingService;

namespace BMSConfigurationWarlock
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window, IModelWindow
    {
        internal MainWindowModel model;

        public IMessageModel MessageModel { get { return model; } }

        public MainWindow(GeneralConfigurationModel generalConfigurationModel)
        {
            InitializeComponent();
            model = new MainWindowModel() { GeneralConfiguration = generalConfigurationModel };
            model.GeneralConfiguration.SaveButtonVisible = false;
            Populate(model);
            this.GeneralConfigurationModule.model = generalConfigurationModel;
            this.DataContext = model;

            Loaded += new RoutedEventHandler(MainWindowLoaded);
            Closing += new System.ComponentModel.CancelEventHandler(MainWindowClosing);
           
            model.GeneralConfiguration.PropertyChanged += new System.ComponentModel.PropertyChangedEventHandler(modelPropertyChanged);
            model.PropertyChanged += new System.ComponentModel.PropertyChangedEventHandler(modelPropertyChanged);
            model.DatabaseConfigurationCollection.ElementPropertyChanged += new PropertyChangedEventHandler(modelPropertyChanged);
        }

        void modelPropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            RefreshVariables();
        }
        
        #region UI Events

        void MainWindowLoaded(object sender, RoutedEventArgs e)
        {
            PasswordBox.Password = model.ReportsUserPassword;
        }

        private void AddServicesLocationButtonClick(object sender, RoutedEventArgs e)
        {
            WinForms.FolderBrowserDialog dialog = new WinForms.FolderBrowserDialog();
            if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                model.SearchServicesLocation = dialog.SelectedPath + ";" + model.SearchServicesLocation;
            }
        }

        private void AddSiteLocationButtonClick(object sender, RoutedEventArgs e)
        {
            WinForms.FolderBrowserDialog dialog = new WinForms.FolderBrowserDialog();
            if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                model.SearchSiteLocation = dialog.SelectedPath + ";" + model.SearchSiteLocation;
            }
        }

        private void LoadSiteConfigButtonClick(object sender, RoutedEventArgs e)
        {
            string validLocations = Utils.GetValidLocations(model.SearchSiteLocation);
            if (string.IsNullOrWhiteSpace(validLocations))
                this.ShowErrorMessage((string)FindResource(Constants.ConfigsNotFoundMessageContent));
            string[] locations = validLocations.Split(new char[] { ';' });
            string assemblyPath = String.Empty;
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                foreach (string location in locations)
                {
                    if(string.IsNullOrWhiteSpace(location))
                        continue;
                    
                    //search for assemblies with correct version and name
                    string[] matchingFiles = Directory.GetFiles(location, Constants.BMSiteAssemblyName, SearchOption.AllDirectories);
                    foreach (string file in matchingFiles)
                    {
                        AssemblyName assemblyName = AssemblyName.GetAssemblyName(file);
                        string version = String.Format("{0}.{1}", assemblyName.Version.Major, assemblyName.Version.Minor);
                        if (version.Equals(model.GeneralConfiguration.SelectedVersion))
                        {
                            assemblyPath = file;
                            break;
                        }
                    }
                    if(!string.IsNullOrEmpty(assemblyPath))
                        break;
                }
            }
            model.SiteAssemblyPath = assemblyPath;
            model.SiteConfigFileLoaded = true;
        }

        private void LoadConfigsButtonClick(object sender, RoutedEventArgs e)
        {
            List<ServiceConfiguration> configFiles = new List<ServiceConfiguration>();
            string validLocations = Utils.GetValidLocations(model.SearchServicesLocation);
            if(string.IsNullOrWhiteSpace(validLocations))
                this.ShowErrorMessage((string)FindResource(Constants.ConfigsNotFoundMessageContent));
            string[] locations = validLocations.Split(new char[] { ';' });
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                foreach (string location in locations)
                {
                    if(string.IsNullOrWhiteSpace(location))
                        continue;
                    foreach (ServiceConfiguration serviceConfiguration in model.ServiceConfigurationCollection)
                    {
                        string serviceAssembly = serviceConfiguration.AssemblyName;
                        //search for assemblies with correct version and name
                        string[] matchingFiles = Directory.GetFiles(location, serviceAssembly, SearchOption.AllDirectories);
                        foreach (string file in matchingFiles)
                        {
                            AssemblyName assemblyName = AssemblyName.GetAssemblyName(file);
                            string version = String.Format("{0}.{1}", assemblyName.Version.Major, assemblyName.Version.Minor);
                            if (version.Equals(model.GeneralConfiguration.SelectedVersion))
                                serviceConfiguration.ExecutablePath = file;
                        }
                    }
                }
            }

            model.ServicesConfigFilesLoaded = true;
        }

        void MainWindowClosing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            MessageBoxResult result = System.Windows.MessageBox.Show(this, (string)FindResource(Constants.MainWindowCloseMessageContent), (string)FindResource(Constants.ConfigurationWarlockMainWindow), MessageBoxButton.YesNoCancel, MessageBoxImage.Information);
            if (result == MessageBoxResult.Cancel)
                e.Cancel = true;
            else 
            if (result == MessageBoxResult.Yes)
            {
                //save configurations.
                SaveConfigurations();
            }
        }

        private void ApplyPatternButtonClick(object sender, RoutedEventArgs e)
        {
            foreach (DatabaseConfiguration dbConfig in model.DatabaseConfigurationCollection)
                dbConfig.DatabaseName = String.Format(model.DatabaseNamePattern, dbConfig.DatabaseName);
        }

        private void TestConnectionsButtonClick(object sender, RoutedEventArgs e)
        {
            string result = string.Empty;
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                foreach (DatabaseConfiguration dbConfig in model.DatabaseConfigurationCollection)
                    try
                    {
                        using (SqlConnection sql = new SqlConnection())
                        {
                            sql.ConnectionString = Utils.GetConnectionString(dbConfig.DatabaseName, model.GeneralConfiguration.DatabaseServer);
                            sql.Open();
                            //verify database version from the appropriate extended attribute
                            string script = File.ReadAllText(Constants.GetDatabaseVersionSql);
                            SqlCommand command = new SqlCommand(script, sql);
                            var queryResult = command.ExecuteScalar();
                            if(queryResult == null)
                                throw new Exception(String.Format((string)FindResource(Constants.DatabaseVersionIncorrect), dbConfig.DatabaseName, (string)FindResource(Constants.DatabaseVersionNoVersion), model.GeneralConfiguration.SelectedVersion));
                            string dbVersion = queryResult as string;
                            if (!dbVersion.Equals(model.GeneralConfiguration.SelectedVersion, StringComparison.InvariantCultureIgnoreCase))
                                throw new Exception(String.Format((string)FindResource(Constants.DatabaseVersionIncorrect), dbConfig.DatabaseName, dbVersion, model.GeneralConfiguration.SelectedVersion));
                        }
                    }
                    catch (Exception exception)
                    {
                        result += dbConfig.DatabaseName + " (" + exception.Message + "); ";
                    }
            }
            if (!string.IsNullOrWhiteSpace(result))
                this.ShowErrorMessage((string)FindResource(Constants.DatabaseConnectionsFailedMessageContent) + ": " + result);
            else
            {
                this.ShowSuccessMessage((string)FindResource(Constants.DatabaseConnectionsSucceededMessageContent));
                model.DatabaseConnectionsChecked = true;
            }
        }

        private void RunSqlScriptsButtonClick(object sender, RoutedEventArgs e)
        {
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                foreach(SqlScript sqlScript in model.SqlScriptCollection)
                    if(sqlScript.Run)
                        try
                        {
                            ProcessStartInfo info = new ProcessStartInfo(Constants.sqlCmd, String.Format("-S {0} -E -i {1} -v {2}", model.GeneralConfiguration.DatabaseServer, sqlScript.Name, Utils.GetVariables(sqlScript, model)));
                            info.CreateNoWindow = true;
                            info.UseShellExecute = false;
                            info.RedirectStandardOutput = true;
                            Process process = new Process();
                            process.StartInfo = info;
                            process.Start();
                            string res = process.StandardOutput.ReadToEnd();
                            this.ShowSuccessMessage(res);
                        }
                        catch (Exception sqlException)
                        {
                            this.ShowErrorMessage(sqlException.Message);
                        }
            }
        }

        private void ModifyConfigsButtonClick(object sender, RoutedEventArgs e)
        {
            if (!model.DatabaseConnectionsChecked)
                this.ShowErrorMessage((string)FindResource(Constants.DatabaseConnectionsNotTestedContent));
            else
            if (model.ServiceConfigurationCollection == null || model.ServiceConfigurationCollection.Count == 0)
                this.ShowErrorMessage((string)FindResource(Constants.ConfigurationFilesNotFound));
            else
            {
                using(OverrideCursor cursor = new OverrideCursor(Cursors.Wait)){
                //modify the identified configuration files
                    foreach (ServiceConfiguration config in model.ServiceConfigurationCollection)
                    {
                        if (string.IsNullOrEmpty(config.ConfigurationPath))
                        {
                            this.ShowErrorMessage(string.Format((string)FindResource(Constants.ConfigurationFileNotFound), config.AssemblyName));
                            continue;
                        }
                        //make sure the file is writeable
                        File.SetAttributes(config.ConfigurationPath, FileAttributes.Normal);
                        //load config as xml
                        XDocument configFile = XDocument.Load(config.ConfigurationPath);
                        MyConnections myConnections = new MyConnections();
                        foreach (DatabaseConnection connection in config.DatabaseConnections)
                        {
                            string databaseName = model.DatabaseConfigurationCollection.First(db => db.DatabaseKey.Equals(connection.DatabaseKey, StringComparison.InvariantCultureIgnoreCase)).DatabaseName;
                            string connectionString = Utils.GetConnectionString(databaseName, model.GeneralConfiguration.DatabaseServer);
                            //connections.xml will be built from the myconnections object
                            foreach (string id in connection.ConnectionIds)
                                myConnections.Add(new MyConnection() { ConnectionId = id, ConnectionString = connectionString });

                            //connection strings directly in the config file
                            foreach (string key in connection.ConnectionKeys)
                            {
                                var query = from c in configFile.XPathSelectElements(Constants.ConnectionStringAddPathInConfig)
                                            where c.Attribute(Constants.Name).Value.Equals(key, StringComparison.InvariantCultureIgnoreCase)
                                            select c;

                                foreach (XElement add in query)
                                    add.Attribute(Constants.ConnectionString).Value = connectionString;
                            }
                        }
                        //save configuration file
                        configFile.Save(config.ConfigurationPath);
                        //update services' addresses
                        Utils.ModifyConfigFile(config.ConfigurationPath, model);
                        //save connections.xml file
                        if (!string.IsNullOrEmpty(config.ConnectionsXmlPath))
                        {
                            //make sure the file is writeable
                            File.SetAttributes(config.ConnectionsXmlPath, FileAttributes.Normal);
                            //write connections to file
                            XmlSerializer xmlSerializer = new XmlSerializer(typeof(MyConnections));
                            using (StreamWriter sw = new StreamWriter(config.ConnectionsXmlPath))
                            {
                                xmlSerializer.Serialize(sw, myConnections);
                                sw.Flush();
                            }
                        }
                    }
                }
                model.ServicesConfigFilesModified = true;
            }
        }

        private void ModifySiteWebConfigButtonClick(object sender, RoutedEventArgs e)
        {
            //make sure the file is writeable
            File.SetAttributes(model.SiteConfigPath, FileAttributes.Normal);
            Utils.ModifyConfigFile(model.SiteConfigPath, model);
            model.SiteConfigFileModified = true;
        }

        private void PasswordBoxPasswordChanged(object sender, RoutedEventArgs e)
        {
            model.ReportsUserPassword = (sender as System.Windows.Controls.PasswordBox).Password;
        }

        private void RestartServicesButtonClick(object sender, RoutedEventArgs e)
        {
            int servicesFound = 0;
            //get only the executable files
            List<ServiceConfiguration> services = model.ServiceConfigurationCollection.Where(serviceConfig => serviceConfig.AssemblyName.Contains(Constants.ExecutableExtension)).ToList();

            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                ServiceController[] serviceList = ServiceController.GetServices(model.GeneralConfiguration.ServicesServer);
                foreach (ServiceController service in serviceList)
                {
                    //check whether all the services have already been found
                    if (servicesFound == services.Count)
                        break;
                    NTServiceInfo serviceInfo = new NTServiceInfo(service.ServiceName, model.GeneralConfiguration.ServicesServer);
                    string servicePath = serviceInfo.PathToExecutable();
                    foreach (ServiceConfiguration serviceConfiguration in services)
                    {
                        string localPath = Path.GetDirectoryName(serviceConfiguration.ExecutablePath);
                        Uri path = new Uri(localPath);
                        if (path.IsUnc)
                        {
                            string localFromUnc;
                            bool success = PathManager.GetPath(localPath, out localFromUnc);
                            if (success)
                                localPath = localFromUnc;
                        }
                        localPath = Path.Combine(localPath, serviceConfiguration.AssemblyName);
                        if (servicePath.Equals(localPath, StringComparison.InvariantCultureIgnoreCase))
                        //found a service, restart it.
                        {
                            servicesFound++;
                            try
                            {
                                //if the service is stopped, only try to start it
                                if (service.Status != ServiceControllerStatus.Stopped)
                                {
                                    //send the stop command
                                    service.Stop();
                                    service.Refresh();
                                    DateTime stopTime = DateTime.Now;
                                    //wait for the service to stop
                                    while (service.Status != ServiceControllerStatus.Stopped && (DateTime.Now - stopTime).Seconds < model.ServiceOperationTimeOutSeconds)
                                    {
                                        Thread.Sleep(Constants.SleepTimeMilliseconds);
                                        service.Refresh();
                                    }
                                    //if timeout occured before the service has stopped, exception
                                    if (service.Status != ServiceControllerStatus.Stopped)
                                        throw new Exception(String.Format((string)FindResource(Constants.ServiceDidNotRespondInATimelyFashion), service.ServiceName));
                                }
                                //start the service
                                service.Start();
                            }
                            catch (Exception exception)
                            {
                                this.ShowErrorMessage(exception.Message);
                            }
                        }
                    }
                }
            }
        }

        private void RestartSiteApplicationButtonClick(object sender, RoutedEventArgs e)
        {
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            using (ServerManager manager = ServerManager.OpenRemote(model.GeneralConfiguration.WebServer))
            {
                foreach (Site site in manager.Sites)
                    if (site.Name.Equals(model.GeneralConfiguration.ApplicationName, StringComparison.InvariantCultureIgnoreCase))
                    {
                        var applicationRoot = site.Applications.Where(a => a.Path == "/").Single();
                        var virtualRoot = applicationRoot.VirtualDirectories.Where(v => v.Path == "/").Single();
                        string physicalPath = virtualRoot.PhysicalPath;
                        
                        //recycle application
                        var applicationPool = manager.ApplicationPools.Where(ap => ap.Name == applicationRoot.ApplicationPoolName).Single();
                        while (applicationPool.State != ObjectState.Stopped)
                            try
                            {
                                applicationPool.Stop();
                            }
                            catch { Thread.Sleep(Constants.SleepTimeMilliseconds); }

                        applicationPool.Start();
                    }
            }
        }

        #endregion

        #region app logic
        
        private void Populate(MainWindowModel model)
        {
            try
            {
                model.SearchServicesLocation = Utils.GetValidLocations(ConfigurationManager.AppSettings[Constants.SuggestedServicesLocation]);
            }
            catch { model.SearchServicesLocation = String.Empty; }
            try
            {
                model.SearchSiteLocation = Utils.GetValidLocations(ConfigurationManager.AppSettings[Constants.SuggestedSiteLocation]);
            }
            catch { model.SearchSiteLocation = String.Empty; }
            try
            {
                model.DatabaseNamePattern = ConfigurationManager.AppSettings[Constants.SuggestedDatabaseNamePattern];
            }
            catch { model.DatabaseNamePattern = "{0}"; }
            try
            {
                model.ReportingServicesUrl = ConfigurationManager.AppSettings[Constants.ReportingServicesUrl];
            }
            catch { model.ReportingServicesUrl = String.Empty; }
            try
            {
                model.ReportsUserDomain = ConfigurationManager.AppSettings[Constants.ReportsUserDomain];
            }
            catch { model.ReportsUserDomain = String.Empty; }
            try
            {
                model.ReportsUserName = ConfigurationManager.AppSettings[Constants.ReportsUserName];
            }
            catch { model.ReportsUserName = String.Empty; }
            try
            {
                model.ReportsUserPassword = ConfigurationManager.AppSettings[Constants.ReportsUserPassword];
            }
            catch { model.ReportsUserPassword = String.Empty; }
            try 
            {
                model.ReportsRootFolder = ConfigurationManager.AppSettings[Constants.ReportsRootFolder];
            }
            catch { model.ReportsRootFolder = String.Empty; }
            try
            {
                model.ServiceOperationTimeOutSeconds = int.Parse(ConfigurationManager.AppSettings[Constants.ServiceOperationTimeoutSeconds]);
            }
            catch { model.ServiceOperationTimeOutSeconds = 10; }
            try
            {
                model.HttpsPortOffset = int.Parse(ConfigurationManager.AppSettings[Constants.HttpsPortOffset]);
            }
            catch { model.HttpsPortOffset = 1; }
            try
            {
                model.ApplicationDomain = ConfigurationManager.AppSettings[Constants.AppDomain];
            }
            catch { model.ApplicationDomain = String.Empty; }
            try
            {
                model.SuperUsers = ConfigurationManager.AppSettings[Constants.SuperUsersList];
            }
            catch { model.SuperUsers = String.Empty; }


            //get services configuration from xml file
            XmlSerializer xmlSerializer = new XmlSerializer(typeof(ServiceConfigurationList));
            model.ServiceConfigurationCollection = new ObservableCollection<ServiceConfiguration>(((ServiceConfigurationList)xmlSerializer.Deserialize(new StringReader(File.ReadAllText(Constants.ServicesConfigurationXml)))).Services);
            
            //get databases configuration from xml file
            xmlSerializer = new XmlSerializer(typeof(DatabaseConfigurationList));
            model.DatabaseConfigurationCollection = new NotifyOnElementChangedObservableCollection<DatabaseConfiguration>(((DatabaseConfigurationList)xmlSerializer.Deserialize(new StringReader(File.ReadAllText(Constants.DatabasesConfigurationXml)))).Databases);

            //get sql scripts
            model.SqlScriptCollection = Utils.GetSqlScriptsCollection();

            model.Variables = new ObservableCollection<DatabaseConfiguration>();
            List<string> allVariables = new List<string>();
            foreach (SqlScript sqlScript in model.SqlScriptCollection)
                if(sqlScript.Variables != null)
                    allVariables.AddRange(sqlScript.Variables);
            allVariables = new List<string>(allVariables.Distinct(StringComparer.InvariantCultureIgnoreCase));
            for (int i = allVariables.Count - 1; i >= 0; i--)
            {
                string variable = allVariables[i];
                //remove variables that represent known databases, they should be in the database collection
                if (model.DatabaseConfigurationCollection.Any(databaseConfig => databaseConfig.DatabaseScriptVariable.Equals(variable, StringComparison.InvariantCultureIgnoreCase) &&
                                                                                !string.IsNullOrEmpty(databaseConfig.DatabaseKey)))
                    allVariables.Remove(variable);
            }

            foreach (string variable in allVariables)
            {
                DatabaseConfiguration configuration =  model.DatabaseConfigurationCollection.FirstOrDefault(dbConfig => dbConfig.DatabaseScriptVariable.Equals(variable, StringComparison.InvariantCultureIgnoreCase));
                if (configuration != null)
                    model.Variables.Add(configuration);
                else
                    model.Variables.Add(new DatabaseConfiguration() { DatabaseScriptVariable = variable });
            }

            for (int i = model.DatabaseConfigurationCollection.Count - 1; i >= 0; i--) 
            {
                if (string.IsNullOrEmpty(model.DatabaseConfigurationCollection[i].DatabaseKey))
                    model.DatabaseConfigurationCollection.RemoveAt(i);
            }
            RefreshVariables();
        }

        private void RefreshVariables() 
        {
            DatabaseConfiguration varNewServicesEndpoint = model.Variables.FirstOrDefault(item => item.DatabaseScriptVariable.Equals(Constants.varNewServicesEndpoint, StringComparison.InvariantCultureIgnoreCase));
            if (varNewServicesEndpoint != null)
                varNewServicesEndpoint.ScriptVariableValue = model.GeneralConfiguration.ServicesServer + ":" + model.GeneralConfiguration.ServicesPort;
            DatabaseConfiguration varNewServer = model.Variables.FirstOrDefault(item => item.DatabaseScriptVariable.Equals(Constants.varNewServer, StringComparison.InvariantCultureIgnoreCase));
            if (varNewServer != null)
                varNewServer.ScriptVariableValue = model.GeneralConfiguration.ServicesServer;
            DatabaseConfiguration varNewDBs = model.Variables.FirstOrDefault(item => item.DatabaseScriptVariable.Equals(Constants.varNewDBs, StringComparison.InvariantCultureIgnoreCase));
            if (varNewDBs != null)
            {
                varNewDBs.ScriptVariableValue = string.Empty;
                foreach (DatabaseConfiguration item in model.DatabaseConfigurationCollection)
                    varNewDBs.ScriptVariableValue += item.DatabaseName + ",";
                if (model.DatabaseConfigurationCollection.Count > 0)
                    varNewDBs.ScriptVariableValue = varNewDBs.ScriptVariableValue.Substring(0, varNewDBs.ScriptVariableValue.Length - 1);
            }
        }

        private void SaveConfigurations()
        {
            System.Configuration.Configuration config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            
            //general configurations window settings
            config.AppSettings.Settings[Constants.DefaultVersion].Value = model.GeneralConfiguration.SelectedVersion;
            config.AppSettings.Settings[Constants.PreferredApplicationName].Value = model.GeneralConfiguration.ApplicationName;
            config.AppSettings.Settings[Constants.PreferredDatabaseServer].Value = model.GeneralConfiguration.DatabaseServer;
            config.AppSettings.Settings[Constants.PreferredServicesPort].Value = model.GeneralConfiguration.ServicesPort;
            config.AppSettings.Settings[Constants.PreferredServicesPrefix].Value = model.GeneralConfiguration.ServicesPrefix;
            config.AppSettings.Settings[Constants.PreferredServicesServer].Value = model.GeneralConfiguration.ServicesServer;
            config.AppSettings.Settings[Constants.PreferredWebServer].Value = model.GeneralConfiguration.WebServer;
            //services tab settings
            config.AppSettings.Settings[Constants.SuggestedServicesLocation].Value = model.SearchServicesLocation;
            //site tab settings
            config.AppSettings.Settings[Constants.SuggestedSiteLocation].Value = Path.GetDirectoryName(model.SiteAssemblyPath);
            //reports tab settings
            config.AppSettings.Settings[Constants.ReportsUserDomain].Value = model.ReportsUserDomain;
            config.AppSettings.Settings[Constants.ReportsUserName].Value = model.ReportsUserName;
            config.AppSettings.Settings[Constants.ReportsUserPassword].Value = model.ReportsUserPassword;
            config.AppSettings.Settings[Constants.ReportingServicesUrl].Value = model.ReportingServicesUrl;
            config.AppSettings.Settings[Constants.ReportsRootFolder].Value = model.ReportsRootFolder;
            //database tab settings
            config.AppSettings.Settings[Constants.SuggestedDatabaseNamePattern].Value = model.DatabaseNamePattern;

            config.Save(ConfigurationSaveMode.Modified);
            ConfigurationManager.RefreshSection("appSettings");

            XmlSerializer xmlSerializer = new XmlSerializer(typeof(DatabaseConfigurationList));
            using (StringWriter sw = new StringWriter())
            {
                DatabaseConfigurationList databaseList = new DatabaseConfigurationList();
                databaseList.Databases = model.DatabaseConfigurationCollection.ToList();
                databaseList.Databases.AddRange(model.Variables);
                xmlSerializer.Serialize(sw, databaseList);
                File.WriteAllText(Constants.DatabasesConfigurationXml, sw.GetStringBuilder().ToString());
            }
        }

        #endregion

        private void UpdateReportsButtonClick(object sender, RoutedEventArgs e)
        {
            if (!model.DatabaseConnectionsChecked)
            {
                this.ShowErrorMessage((string)FindResource(Constants.DatabaseConnectionsNotTestedContent));
                return;
            }
            using (OverrideCursor cursor = new OverrideCursor(Cursors.Wait))
            {
                try
                {
                    

                    ReportingService.ReportingService2005 rs = new ReportingService.ReportingService2005();
                    rs.Url = model.ReportingServicesUrl;
                    //rs.Credentials = System.Net.CredentialCache.DefaultCredentials;
                    rs.Credentials = new NetworkCredential(model.ReportsUserName, model.ReportsUserPassword, model.ReportsUserDomain);

                    DataSourceDefinition definition = new DataSourceDefinition();
                    definition.CredentialRetrieval = CredentialRetrievalEnum.Integrated;

                    definition.ConnectString = "data source=" + model.GeneralConfiguration.DatabaseServer + ";initial catalog=" + model.DatabaseConfigurationCollection.First(db => db.DatabaseKey.Equals(Constants.BMSDWDatabaseKey, StringComparison.InvariantCultureIgnoreCase)).DatabaseName;
                    definition.Enabled = true;
                    definition.EnabledSpecified = true;
                    definition.Extension = "SQL";
                    definition.ImpersonateUser = false;
                    definition.ImpersonateUserSpecified = true;
                    definition.Prompt = null;
                    definition.WindowsCredentials = false;

                    rs.CreateDataSource(Constants.BMSReports, "/" + model.ReportsRootFolder + "/" + Constants.DataSource, true, definition, null);

                    CatalogItem[] items = rs.ListChildren("/" + model.ReportsRootFolder, true);
                    foreach (CatalogItem item in items)
                    {
                        if (item.Type != ItemTypeEnum.Report)
                            continue;
                        DataSource[] dataSources = rs.GetItemDataSources(item.Path);
                        DataSource dataSource = dataSources[0];
                        DataSourceReference dataSourceReference = new DataSourceReference();
                        dataSourceReference.Reference = "/" + model.ReportsRootFolder + "/" + Constants.DataSource + "/" + Constants.BMSReports;
                        dataSource.Item = dataSourceReference;
                        rs.SetItemDataSources(item.Path, dataSources);
                    }
                    this.ShowSuccessMessage((string)FindResource(Constants.UpdateReportsSucceededMessageContent));
                }
                catch (Exception exception)
                {
                    this.ShowErrorMessage(exception.Message);
                }
            }

        }
    }
}
